// DMAPAUSE.C
// This program reads 8 bit PCM sound data from a file.
// The 8 bit sound data is then played back using DMA.
// If a key is struck on the keyboard during the playback,
// then DMA is paused. Another keystoke will restart DMA.
//
// The buffers that are used in this program are allocated 
// using cvxBufferAlloc. Each buffer is set to 16K. 

#include <stdio.h>
#include <conio.h>
#include "cvxdigi.h"
#include "cvxutil.h"

#define _BUFFER_SIZE  0x4000     // Size of each DMA buffer.
#define _BUFFER_COUNT 4          // Number of DMA buffers.
#define _PCM_FILE     "TEST1.V8"  // File containing sound data.

extern _dmaInProgressFlag;          
extern _dmaDevice;

VOID main( VOID )
{
   HANDLE  pcmFileHandle;
   LPSTR   pcmBuffer[ _BUFFER_COUNT ];
   LONG    bytesAvailable;
   LONG    bytesRead;
   WORD    portAddress;
   BYTE    dmaRate;
   WORD    repeatCount = 1;
   WORD    initError;
   WORD    i;

   // Allocate memory for buffers used to read 8 bit PCM
   // data from file.
   for( i = 0; i < _BUFFER_COUNT; i++ )
   {
      pcmBuffer[ i ] =  cvxBufferAlloc( ( LONG )_BUFFER_SIZE,  &bytesAvailable );

      if( pcmBuffer == _NULL )
      {
         printf( "ERROR : Cannot allocate memory!\n" ); 
         exit( 0 );
      }
   }

   // Initialize DMA. Setting each parameter to _AUTODETECT 
   // causes dmaInit to perform a search for the Port,
   // DMA channel, and IRQ setting respectively.
   initError = dmaInit( _AUTODETECT, _AUTODETECT, _AUTODETECT, 
                       &portAddress );

   // If the variable _dmaDevice equals 0 then the DMA
   // sub-system was not initialized correctly.
   if( _dmaDevice == 0 )
      printf( "ERROR = %d : dmaInit failed\n", initError );
   else
   {
      // Open a file containing 8 bit PCM data.
      if( cvxFileOpen( _PCM_FILE, _OPEN_R_ONLY, &pcmFileHandle ) )
         printf( "ERROR : %s not found.\n", _PCM_FILE );
      else
      {
         // Get header information, if it exists, from file.
         cvxFileRead( pcmFileHandle, pcmBuffer[ 0 ],
                      ( LONG )_HEADER_LENGTH, &bytesRead );

         // Get rate from header.
         dmaRate = pcmBuffer[ 0 ][ _HEADER_RATE_OFFSET ];

         //  Loop here until we have queued all 4 buffers.
         for( i = 0; i < _BUFFER_COUNT; i++ )
         {
            // Fill buffer with sound data.
            cvxFileRead( pcmFileHandle, pcmBuffer[ i ],
                          ( LONG )_BUFFER_SIZE, &bytesRead );

            // Insert buffer into DMA queue.
            dmaPlay( ( LPSTR )pcmBuffer[ i ], ( LONG )bytesRead,
                     dmaRate, repeatCount );
         }

         printf( "Press any key to pause dma.\n" );

         // Loop until DMA has completed.
         while( _dmaInProgressFlag )
         {
            // If a key is struck, pause DMA playback.
            if( kbhit() )
            {
               dmaPause();

               // Clear all keystrokes.
               while( kbhit() )
                  getch();

               printf( "\nDMA has been paused.\n" );
               printf( "Press any key to restart.\n" );

               // Loop until a key is struck.
               while( !kbhit() )
               
               // Restart dma playback.
               dmaUnpause();

               // Clear all key stokes.
               while( kbhit() )
                  getch();

               printf( "\nPress any key to pause DMA.\n" );
            }
         }
      }
   }

   // Clear all requests from the Play and Record Queue.
   if( dmaFlushQueue() )
      printf( "DMA uninit failure : could not flush queue.\n" );

   // Uninitialize the DMA system.   
   if( dmaUninit() )
      printf( "DMA uninit failure.\n" );
   
   // Close the sound file.
   cvxFileClose( pcmFileHandle );

   // Free memory used by 8 bit PCM buffer.
   for( i = ( _BUFFER_COUNT - 1 ); i != -1 ; i-- )
   {
      if( cvxBufferFree( pcmBuffer[ i ] ) )
         printf( "ERROR : Cannot free memory!\n" );
   }
}



