// DMAREC.C
// This program records using DMA. Four buffers are queued for
// DMA recording. After each buffer has completed being recorded
// it is written to a file.
//
// The buffers that are used in this program are allocated 
// using cvxBufferAlloc. Each buffer is set to 32K. 

#include <stdio.h>
#include <bios.h>
#include <dos.h>
#include <errno.h>
#include <fcntl.h>
#include "cvxdigi.h"
#include "cvxutil.h"

#define _BUFFER_SIZE  0x8000     // Size of each DMA buffer.
#define _BUFFER_COUNT 4          // Number of DMA buffers.
#define _PCM_FILE     "TEST1.V8"  // File for recorded data.

BYTE cvxHeader[] = { 0xFF,0x55,0xFF,0xAA,0xFF,0x55,0xFF,0xAA,
                     0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 };


extern _dmaInProgressFlag;
extern _dmaDevice;

VOID main( VOID )
{
   HANDLE  pcmFileHandle;
   LPSTR   pcmBuffer[ _BUFFER_COUNT ];
   LONG    bytesAvailable;
   WORD    portAddress;
   LONG    bytesWritten;
   WORD    phase;
   WORD    initError;
   WORD    i;

   // Allocate memory for buffers used to record 8 bit PCM.
   for( i = 0; i < _BUFFER_COUNT; i++ )
   {
      pcmBuffer[ i ] = cvxBufferAlloc( ( LONG )_BUFFER_SIZE, &bytesAvailable );
      if( pcmBuffer == _NULL )
      {
         printf( "ERROR : Cannot allocate memory!\n" );
         exit( 0 );
      }
   }

   // Initialize DMA. Setting each parameter to _AUTODETECT 
   // causes dmaInit to perform a search for the Port,
   // DMA channel, and IRQ setting respectively.
   initError = dmaInit( _AUTODETECT, _AUTODETECT, _AUTODETECT, 
                        &portAddress );


   // If the variable _dmaDevice equals 0 then the DMA
   // sub-system was not initialized correctly.
   if( _dmaDevice == 0 )
      printf( "ERROR = %d : dmaInit failed\n", initError );
   else
   {
      // Prompt user to hit a key to begin recording.
      printf( "Hit any key to begin recording\n" );
      
      // Wait for keystroke.
      while( !kbhit() );

      // Clear keystroke(s) from keyboard buffer.
      while( kbhit() )
         getch();
      
      // Notify user that recording has begun.
      printf( "Recording ..." );

      // Create file for recorded 8 bit PCM data.
      if( cvxFileCreate( _PCM_FILE, _CREATE_NORMAL, &pcmFileHandle ) )
         printf( "ERROR : Cannot create %s.\n", _PCM_FILE );
      else
      {
         // Put rate and format into header.
         cvxHeader[ _HEADER_RATE_OFFSET ]   = _CVX_RATE_DEFAULT;
         cvxHeader[ _HEADER_FORMAT_OFFSET ] = _FORMAT_V8;

         // Write header information to file.
         cvxFileWrite( pcmFileHandle, cvxHeader,
                       ( LONG )_HEADER_LENGTH, &bytesWritten );

         // Queue all buffers for recording.
         for( i = ( _BUFFER_COUNT - 1 ); i != -1; i-- )
            dmaRecord( pcmBuffer[i], ( LONG )_BUFFER_SIZE,
                       ( WORD )_CVX_RATE_DEFAULT );
         
         // The variable 'phase' will indicate when a buffer has 
         // been filled with recorded sound data. One is subtracted
         // from phase so that when a buffer is finished, phase
         // will equal the return value of dmaNumberInQueue.
         phase = _BUFFER_COUNT - 1;

         // Loop here until there are no more buffers left 
         // in queue.
         do
         {
            // Loop until a spot in the queue opens up.
            while( dmaNumberInQueue() > phase );
            
            // Write recorded buffer to file.
            cvxFileWrite( pcmFileHandle, pcmBuffer[ phase ],
                          ( LONG )_BUFFER_SIZE, &bytesWritten );

            // Decrement phase so that we can tell when the
            // next buffer has been recorded.
            phase--;

         } while( phase > 0 );

         // At this point the last buffer is being recorded.  
         while( _dmaInProgressFlag );
         
         // Write last recorded buffer to file.
         cvxFileWrite( pcmFileHandle, pcmBuffer[ phase ],
                       ( LONG )_BUFFER_SIZE, &bytesWritten );

         // Notify user that recording has completed.
         printf( " complete.\n" );

      }
   }

   // Clear all requests from the Play and Record Queue.
   if( dmaFlushQueue() )
      printf( "DMA uninit failure : could not flush queue.\n" );

   // Uninitialize the DMA system.   
   if( dmaUninit() )
      printf( "DMA uninit failure.\n" );
   
   // Close the sound file.
   cvxFileClose( pcmFileHandle );

   // Free memory used by 8 bit PCM buffer.
   for( i = ( _BUFFER_COUNT - 1 ); i != -1; i-- )
   {
      if( cvxBufferFree( pcmBuffer[ i ] ) )
         printf( "ERROR : Cannot free memory!\n" );
   }
}


