//
//    SAY.EXE - (c) 1992 Covox Inc., Eugene, Oregon USA
//
//    File to playback Covox ADPCM and PCM recordings on any Covox hardware
//    from the DOS command line.
//
//      This program or parts of it may be used in other programs written
//      to work with the Covox external digitizer freely and with out prior
//      consent of Covox Inc. The speech output through the parallel printer
//      is a Covox US Patent Number 4812847. Other US and foreign patents
//      are pending. Specifications subject to change without notice.
//
//      SAY.EXE can be compiled using either MicroSoft C 6.0 or Borland
//      C 2.0. See SAY.MAK for more information.
//
//----------------------------------------------------------------------------
//             Copyright (c) 1992, Covox, Inc. All Rights Reserved                   
//----------------------------------------------------------------------------

#include <conio.h>
#include <string.h>
#include <stdio.h>
#include <io.h>
#include <stdlib.h>
#include <dos.h>
#include "parse.h"
#include "cvxdigi.h"
#include "cvxutil.h"


//  Version string for SAY.C
//
PSTR    sayVersionString  = "4.01"; 

// Error codes for SAY.C
//
enum
{
   _ERROR_RATE_EXCEEDED = 1,       // Rate specified on command line exceeded 229
   _ERROR_INVALID_FILE_NAME,       // File name did not begin with a character.
   _ERROR_NO_COMMAND_LINE_OPTIONS, // No options were found on command line.
   _ERROR_INVALID_JUMPER,          // Invalid dma jumper specified on command line.
   _ERROR_INVALID_IRQ_NUMBER,      // Invalid irq specified on command line.
   _ERROR_INVALID_DMA_CHANNEL,     // Invalid dma channel specified on command line.
   _ERROR_INVALID_PORT             // Invalid port specified on command line.
};


// Covox DAC hardware types.
#define   _HARDWARE_INTERNAL_NONDMA   0  // Sound Master II, Voice Master Key System
#define   _HARDWARE_INTERNAL_DMA      1  // Sound Master II, Voice Master Key System
#define   _HARDWARE_EXTERNAL_LPT      4  // Voice Master Key System II
#define   _HARDWARE_ISP               5  // Internal speaker.


// If EXTERNAL is defined at compile time then the default
// output device will be LPT1 otherwise _CVX_VM0
// is the default.
//
#ifdef EXTERNAL
#define   _HARDWARE_DEFAULT      _HARDWARE_EXTERNAL_LPT
#else
#define   _HARDWARE_DEFAULT      _HARDWARE_INTERNAL_NONDMA
#endif

// Global variables.
//
WORD hardwareType = _HARDWARE_DEFAULT;  // Covox playback device.

WORD portOut      = _HARDWARE_DEFAULT;  // Port for non-DMA playback.

WORD dmaJumper    = _AUTODETECT;        // Base port for dma playback.
WORD channel      = _AUTODETECT;        // DMA channel of Sound Master II or Voice Master.
WORD irqNumber    = _AUTODETECT;        // IRQ of Sound Master II or Voice Master.

BOOL trebleFlag   = _FALSE;             // If _TRUE, playback with data differentiation.
BOOL noiseFlag    = _FALSE;             // If _TRUE, generate computer noise during silence periods.

LONG bufferSize   = 0L;                 // Size of playback buffer(s).
WORD bufferCount  = 5;                  // Number of buffers to use for dmaPlay().

BYTE playbackRate = 0;                  // Covox playback rate (0 to _CVX_RATE_OUT_MAXIMUM)

BYTE fileName[80] = "";                 // File path and name.
BYTE fileFormat;                        // Sound data format of playback file.

BOOL quietFlag    = _FALSE;             // If _TRUE, suppress all screen output.

BOOL hookKBFlag   = _TRUE;              // Used by playFile() and dmaPlayFile()
                                        //   to determine use of HOOK_KB.C functions.

//------------------------------------------------------------------------
// SAY.C function prototypes

// Function to display valid command line options to the screen.
//
VOID  displayCommandLineSyntax( VOID );

// This function parses the header and displays the appropriate information
// to the screen.
//
WORD  displayPlayFileInfo( HANDLE );

// Command line parse function prototypes.
//
WORD  parseCommandLine ( WORD argc, PSTR  argv[] );
WORD  parseFilename    ( PSTR );
WORD  parseRate        ( PSTR );
WORD  parseTreble      ( PSTR );
WORD  parseNoise       ( PSTR );
WORD  parsePort        ( PSTR );
WORD  parseJumper      ( PSTR );
WORD  parseChannel     ( PSTR );
WORD  parseQuiet       ( PSTR );
WORD  parseIrqNumber   ( PSTR );

// Used to take over <CTRL-C> and <CTRL-BREAK>
//
VOID ( interrupt far * oldCtrlBreak ) ();  
VOID interrupt far controlBreakHandler( VOID );  

extern _ioStopFlag;

// ------------------------------------------------------------------------
// --------------------------   BEGIN MAIN   ------------------------------
VOID main( WORD argc, PSTR argv[] )
{
   PSTR   string;
   WORD   fileHandle;
   WORD   returnValue = _ERROR_NONE;
   BYTE   filePath[80];
   WORD   i, j;

   struct find_t findTable;  // Stucture for Microsoft _dos_find functions

   // Get all options from the command line that have been entered
   // by user. If illegal options have been used then display  
   // the legal command line syntax and exit.
   //
   if( parseCommandLine(argc, argv) )
   {
      if( !quietFlag )
         displayCommandLineSyntax();

      exit(0);
   }

   // If no file name was given on the command line then display
   // the command line options and exit.
   //
   if( !strlen( fileName ) )
   {
      if( !quietFlag )
         displayCommandLineSyntax();

      exit(0);
   }

   // Trap <CTRL-C> and <CTRL-BREAK>
   // Save off orignal ISR vector address and set temporary
   // handler address.
   //
   oldCtrlBreak = _dos_getvect( 0x1B );
   _dos_setvect( 0x1B, controlBreakHandler );

   if( !quietFlag )
   {
      // Print out SAY.EXE version.
      printf( "\nCovox Inc (c) - Sound file playback utility - Version %s\n", sayVersionString );
   }

   // Find first instance of file specified by user of SAY.EXE
   //
   if( _dos_findfirst( fileName,_A_NORMAL,&findTable ) )
      returnValue = _ERROR_FILE_NOT_FOUND;

   // Get path from file name 
   //
   for(i = 0, j = 0 ; ; i++)    // j is latest back slash 
   {
      if( fileName[i] == '\\' || fileName[i] == ':')
         j = i+1;

      if( fileName[i] == '\0' )
      {
         filePath[j] = '\0';
         break;
      }

      filePath[i] = fileName[i];
   }

   // Do until all files specifeied have been played.
   //
   do
   {
      // Append file path onto file name.
      //
      strcpy(fileName,filePath);
      strcat(fileName,findTable.name);

      // Open sound file.
      //
      returnValue =  cvxFileOpen( fileName, _OPEN_R_ONLY, &fileHandle );

      if( !quietFlag ) 
      {
         // Display file format, rate , etc to screen
         displayPlayFileInfo( fileHandle );
      }

      if( !returnValue )
      {
         // Start playback.
         //
         if( hardwareType == _HARDWARE_INTERNAL_DMA ) 
         {
            // Call DMA playback function.
            //
            returnValue = dmaPlayFile( fileHandle, fileFormat, playbackRate, 
                                       dmaJumper,  channel,    irqNumber,
                                       noiseFlag,  trebleFlag, hookKBFlag,
                                       bufferSize, bufferCount );
         }
         else
         {
            // Call non-DMA playback function.
            //
            returnValue = playFile( fileHandle, fileFormat, playbackRate,
                                    portOut,    noiseFlag,  trebleFlag, 
                                    hookKBFlag, bufferSize );
         }

      }

      // Close file.
      //
      cvxFileClose( fileHandle );

   }while( _dos_findnext(&findTable) == 0  && !_ioStopFlag );

   // If an error occured print out error message.
   //
   if( returnValue )
   {         
      string = cvxGetErrorString( returnValue );
      printf("\nERROR = %d \n", returnValue );
      printf("%s\n", string);
   }

   // Restore orignal <CTRL-C> and <CTRL-BREAK> vector.
   //
   _dos_setvect( 0x1B, oldCtrlBreak );


   //  Clear all keystrokes from the keyboard buffer.
   //
   while( kbhit() )
   {
     getch();
   }
}
// ---------------------------    END MAIN    --------------------------------
//----------------------------------------------------------------------------

// This function parses the header and displays the appropriate information
// to the screen.
//
WORD displayPlayFileInfo( HANDLE fileHandle )
{
   static BYTE tempRate;
   static BYTE tempFormat;

   BYTE   line[10];
   BYTE   parseBuffer[ _HEADER_LENGTH_VMF ];
   LONG   bytesRead;
   WORD   returnValue = _ERROR_NONE;

   // Read data from sound file for cvxHeaderParse.
   //
   returnValue = cvxFileRead( fileHandle, parseBuffer, ( LONG )_HEADER_LENGTH_VMF, &bytesRead );

   if( !returnValue )
   {
      // Reset file back to the beginning.
      //
      lseek( fileHandle, 0L, SEEK_SET );
   
      // Get file format and playback rate.
      //
      cvxHeaderParse( ( LPSTR )parseBuffer, &tempFormat, &tempRate );

      // Assign file format found in header.
      //
      fileFormat = tempFormat;

      // If playbackRate was specified then setup temp variable.
      //
      if( playbackRate )
         tempRate = playbackRate;

      // Display file name to screen.
      //
      printf("\nPlaying File:  %s\n", fileName);
      printf("Output device: ");
      if( hardwareType == _HARDWARE_INTERNAL_NONDMA )
      {
         printf("Logical port %s", itoa(portOut,line,10) );
      }
      else if( hardwareType == _HARDWARE_EXTERNAL_LPT )
      {
         printf("LPT%s", itoa(portOut-3,line,10) );
      }
      else if( hardwareType == _HARDWARE_ISP )
      {
         printf( "Internal Speaker" );
      }
      else if( hardwareType == _HARDWARE_INTERNAL_DMA )
      {
         printf( "DMA" );
      }
   
      printf("\n");
      printf("Format:        ");
      switch( tempFormat )
      {
         case _FORMAT_V2S:
            printf("Two bit ADPCM w/ silence encoding\n");
            break;
   
         case _FORMAT_V3S:
            printf("Three bit ADPCM w/ silence encoding\n");
            break;
   
         case _FORMAT_V4S:
            printf("Four bit ADPCM w/ silence encoding\n");
            break;
   
         case _FORMAT_V8S:
            printf("Eight bit PCM w/ silence encoding\n");
            break;
   
         case _FORMAT_VMF:
         case _FORMAT_V8:
            printf("Eight bit PCM\n");
            break;
   
         case _FORMAT_VOC:
            printf("VOC\n");
            break;
   
      }
   
      // Print out Covox and Hertz rate to screen.
      //
      printf("Rate:          %u  (", tempRate );
      printf("%uHz)\n", cvxRateToHz( tempRate) );
   
      printf("\nPress any key to stop playback\n");
   }

   return( returnValue );
}


// Parse the options entered by user of program.
//
WORD parseCommandLine(WORD argc, PSTR argv[])
{
   BYTE  commandLineToken;
   unsigned short commandLineCount;
   WORD  returnValue;

   static PARSE_INFO sayParse[] =
   {
      { 1,  0 , parseFilename},
      { 1, 'R', parseRate},
      { 0, 'Q', parseQuiet},
      { 0, 'T', parseTreble},
      { 0, 'N', parseNoise},
      { 1, 'P', parsePort},
      { 1, 'J', parseJumper},
      { 1, 'C', parseChannel},
      { 1, 'I', parseIrqNumber},
   };

   commandLineToken = '/';

   commandLineCount = sizeof( sayParse ) / sizeof(PARSE_INFO);

   SetParse(commandLineToken, commandLineCount, sayParse );

   if( argc == 1 )
   {
      returnValue = _ERROR_NO_COMMAND_LINE_OPTIONS;
   }
   else
   {
      returnValue = Parse(argc, argv);
   }

   return( returnValue );

}


WORD parseFilename( PSTR x )
{
   strcpy(fileName,x);

   strlwr( fileName );

   // Check to see if fileName begins with an alpha-numeric character.
   //
   if( strcspn(fileName, ".") == 0 )
      return( _ERROR_INVALID_FILE_NAME );

   // If the last character in fileName is a period then
   // remove it.
   //
   if( fileName[ strlen(fileName) - 1 ] == '.' )
      fileName[ strlen(fileName) - 1 ] == _NULL;

   return( _ERROR_NONE );
}

WORD parseRate( PSTR x )
{
   playbackRate = (BYTE)atoi(x);

   if( playbackRate > ( BYTE )_CVX_RATE_OUT_MAXIMUM )
   {
      return( _ERROR_RATE_EXCEEDED );
   }
   return( _ERROR_NONE );
}

WORD parseTreble( PSTR x )
{
   trebleFlag = _TRUE;
   return( _ERROR_NONE );
}

WORD parseQuiet( PSTR x )
{
   quietFlag = 1;
   return( _ERROR_NONE );
}
WORD parseNoise( PSTR x )
{
   noiseFlag = 1;
   return( _ERROR_NONE );
}

WORD parsePort( PSTR x )
{
   PSTR string1;

   strlwr(x);
   if( strstr(x,"vm") )
   {
      hardwareType = _HARDWARE_INTERNAL_NONDMA;
      portOut = atoi(x+2);
   }
   else if( strstr(x,"lpt") )
   {
      hardwareType = _HARDWARE_EXTERNAL_LPT;

      // Add three to lpt port since routines expect 4 - 6 as
      // lpt1 - lpt3.
      //
      portOut = atoi(x+3) + 3;
   }
   else if( strstr(x,"isp") )
   {
      hardwareType = _HARDWARE_ISP;
      portOut = _CVX_ISP;
   }
   else if( strstr(x,"dma") )
   {
      hardwareType = _HARDWARE_INTERNAL_DMA;
   }
   else
      return( _ERROR_INVALID_PORT );

   return( _ERROR_NONE );
}

WORD parseJumper( PSTR x )
{
   dmaJumper = atoi(x);

   if( (dmaJumper == _AUTODETECT ) ||
       (dmaJumper == _CVX_VM0    ) ||
       (dmaJumper == _CVX_VM1    ) ||
       (dmaJumper == _CVX_VM2    ) ||
       (dmaJumper == _CVX_VM3    ) )
   {
      return( _ERROR_NONE );
   }
   else
   {
      return( _ERROR_INVALID_JUMPER );
   }
}

WORD parseChannel( PSTR x )
{
   channel = atoi(x);

   if( ( channel == _AUTODETECT    ) ||
       ( channel == _DMA_CHANNEL_1 ) ||
       ( channel == _DMA_CHANNEL_3 ) )     
   {                         
      return( _ERROR_NONE );
   }
   else
   {
      return( _ERROR_INVALID_DMA_CHANNEL );
   }
}


WORD parseIrqNumber( PSTR x )
{
   irqNumber = atoi(x);

   if( (irqNumber == _AUTODETECT) ||
       (irqNumber == _IRQ_2)  || 
       (irqNumber == _IRQ_3)  || 
       (irqNumber == _IRQ_4)  || 
       (irqNumber == _IRQ_5)  || 
       (irqNumber == _IRQ_7) )   
   {
      return( _ERROR_NONE );
   }
   else
   {
      return( _ERROR_INVALID_IRQ_NUMBER );
   }
}

// ISR for control break.
//
VOID interrupt far controlBreakHandler( VOID )
{
   _asm nop
}


// Display to the screen the legal command line options avaiable.
//
VOID displayCommandLineSyntax( VOID )
{
   WORD  i, j;
   WORD  messageLength;

   static PSTR commandLineSyntaxMessage[] =
   {
      "        SAY Version ",
      "",
// insert an x in the external play
#ifdef EXTERNAL
      "x"
#endif
      " - Covox Inc (c) 1992\n",
      "Playback a sound file to any Covox speech hardware.\n",
      "SAY <filename> [<switches>]\n",
      "/Rxxx - Where xxx is a playback rate from 0-209.  0 = default rate of 132.\n",
      "/T    - Treble output.\n",
      "/Q    - Quiet mode.\n",
      "/N    - Computer generated noise rather than dead silence.\n",
      "/Pxxx - The port to use for sound output, xxx can be one of the following:\n",
      "          VM0 "
#ifndef EXTERNAL
      "(default)"
#endif
      ", VM1, VM2, VM3 for Voice Master or Sound Master II, or\n",
      "          LPT1 "
#ifdef EXTERNAL
      "(default)"
#endif
      ", LPT2, LPT3 for SpeechThing or VM System II\n",
      "          ISP for internal speaker\n",
      "          DMA for Voice Master or Sound Master II DMA output.\n",
       "The following apply only to Voice Master and Sound Master II with DMA.\n",
      "/Jx   - Select DMA port setting (0-3).  -1 (default) for sequential search.\n",
      "/Cx   - Select DMA channel (1 or 3).  -1 (default) checks 1 then 3.\n",
      "/Ix   - Select IRQ (2,3,4,5 or 7).  -1 (default) for sequential search.\n\n",
      "SAY birds.v4s /Plpt1\n",
      "        Play file BIRDS.V4S to SpeechThing attached to LPT1"
   };

   // Put version into array.
   commandLineSyntaxMessage[ 1 ] = sayVersionString;

   messageLength = sizeof(commandLineSyntaxMessage)/sizeof(PSTR);

   for(i = 0 ; ( (i < messageLength) && (i != 22) ) ; i++)
   {
      printf(commandLineSyntaxMessage[i]);

   }
}


