/* UNTAR.C - Read Un*x format TAR files from disk or DOS device
   $Author:   Phlash  $
   $Date:   04 Nov 1994 21:50:06  $
   $Revision:   1.0  $
*/

#include <stdio.h>
#include <io.h>
#include <fcntl.h>
#include <string.h>
#include <direct.h>
#include <time.h>
#include <dos.h>
#include "tar.h"
#include "ioctl.h"
#include "qic_drv.h"

#define MAXPATH  128
#define WILDCHR '*'

/* check for memory model */
#ifndef M_I86LM
#error Must use large model!
#endif

int verbose=0;
int mode=0;
int device=0;
int qic_tape=0;

static qicDrvIoctl_t ioc;
static unsigned int tsize=0, tpos=0;
static int tnum=0;
static char *tb[2];

int devicecheck(int tf)
{
int i;

   if(ioctl(tf, FIOGETINFO, &i))
   {
      fprintf(stderr, "cannot determine type of input file\n");
      return -1;
   }
   if(i & 0x80)
   {
      i = (i & 0xFF) | 0x20;
      ioctl(tf, CIOSETINFO, &i);
      device=1;
      if(ioctl(tf, CIORDCNTRL(sizeof(ioc)), &ioc))
         return 0;
      if(strncmp(ioc.chk, QIC_CHK_VALUE, 4))
         return 0;
      ioc.async = 1;
      if(ioctl(tf, CIOWRCNTRL(sizeof(ioc)), &ioc))
         return 0;
      qic_tape = 1;
      puts("(AshbySoft *) QIC_TAPE detected, using asyncronous mode");
   }
   return 0;
}

char *getbuffer(void)
{
char *buffer;
unsigned int seg;

   _dos_allocmem(0xFFFF, &seg);
   if(_dos_allocmem(seg, &seg))
   {
      fprintf(stderr, "cannot allocate transfer buffer\n");
      return NULL;
   }
   FP_SEG(buffer) = (seg + 0x0FFF) & ~0x0FFF;
   FP_OFF(buffer) = 0;
   FP_SEG(tb[0]) = FP_SEG(buffer) + 0x1000;
   FP_SEG(tb[1]) = FP_SEG(buffer) + 0x2000;
   FP_OFF(tb[0]) = FP_OFF(tb[1]) = 0;
   return buffer;
}

int openbuffer(char *name, int mode)
{
int tf;

   if((tf=open(name, mode)) < 0)
      return -1;
   if(devicecheck(tf))
      return -1;
   if(qic_tape)
   {
      read(tf, tb[tnum], 0xF000);
      tnum = 1-tnum;
   }
   return tf;
}

int readbuffer(int tf, char *buf, unsigned int len)
{
unsigned int tleft=tsize-tpos;

   if(len < tleft)
   {
      memcpy(buf, &tb[tnum][tpos], len);
      tpos += len;
   }
   else
   {
      memcpy(buf, &tb[tnum][tpos], tleft);
      if(qic_tape)
      {
         ioc.command = QIC_COMPLETE;
         if(ioctl(tf, CIOWRCNTRL(sizeof(ioc)), &ioc))
            return -1;
         if(ioc.command == -1)
            return -1;
         tsize = (unsigned int)ioc.command;
         read(tf, tb[tnum], 0xF000);
         tnum=1-tnum;
      }
      else
      {
         tsize=read(tf, tb[tnum], 0xF000);
         if(tsize == 0xFFFF)
            return -1;
      }
      if(tsize < len-tleft)
         return 0;
      memcpy(&buf[tleft], tb[tnum], len-tleft);
      tpos = len-tleft;
   }
   return len;
}

int wildcmp(char *wilds, char *str, char wc)
{
char *l, *s;
int flag;
char wcrd;

   if(strlen(str) == 0)
      return -1;

   wcrd = wc & 127;
   flag = 0;

   for(l=wilds, s=str; *l != 0; )
   {
      if(*l == wcrd)
      {
         if(flag)
            return 0;
         flag = 1;
         l++;
         s = str + strlen(str) - (strlen(wilds) - (l - wilds));
      }
      else
      {
         if(*l != *s)
            return -1;
         l++;
         s++;
      }
   }
   if(*s != 0)
      return -1;
   return 0;
}

int unix_to_dos(char *unixname, char *dosname, int len)
{
char *u, *d;
int cnt, tot, dir, ext;

/* scan unix name, convert '/' to '\' and truncate to 8.3 format */
   cnt = 0;
   tot = 0;
   dir = 0;
   ext = 0;
   for( u=unixname, d=dosname; (*u!=0) && (tot<len-1); )
   {
      switch( *u )
      {
      case '/':
         *d++ = '\\';
         u++;
         cnt = 0;
         dir = 1;
         ext = 0;
         tot++;
         break;

      case '.':
         if(ext)
         {
            if(cnt < 3)
            {
               *d++ = 'x';
               u++;
               tot++;
               cnt++;
            }
            else
               u++;
            dir = 0;
         }
         else
         {
            *d++ = *u++;
            cnt = 0;
            dir = 0;
            ext = 1;
            tot++;
         }
         break;

      default:
         dir = 0;
         if(ext)
         {
            if(cnt < 3)
            {
               *d++ = *u++;
               cnt++;
               tot++;
            }
            else
               u++;
         }
         else
         {
            if(cnt < 8)
            {
               *d++ = *u++;
               cnt++;
               tot++;
            }
            else
               u++;
         }
         break;
      }
   }
   *d = 0;
   tot++;
   if(tot == len)
      return -1;
   if(dir)
      return 1;
   return 0;
}

long calclong(char *f_size)
{
long size;

   sscanf(f_size, " %lo", &size);
   return size;
}

int calcint(char *pint)
{
int val;

    sscanf(pint, " %o", &val);
    return val;
}

int getdir(int tf, union TAR_BLOCK *ptb)
{
int n,i;

/* read directory block */
   if((n = readbuffer(tf, ptb->raw, BLOCKSIZE)) != BLOCKSIZE)
   {
      fprintf(stderr, "cannot read directory block\n");
      return 1;
   }
   for(i=0; i<n; i++)
      if(ptb->raw[i])
         return 0;
   fprintf(stderr, "End of file\n");
   return 1;
}

int skipfile(int tf, union TAR_BLOCK *ptb)
{
long skip;
int n;

/* work out skip length required from file size */
   skip = calclong(ptb->direct.f_size);
   skip = ((skip + BLOCKSIZE-1)/BLOCKSIZE)*BLOCKSIZE;

/* skip number of blocks required */
   if(0)     /* disabled 'cause it breaks the caching system! */
   {
      if(lseek(tf, skip, SEEK_CUR) < 0)
      {
         fprintf(stderr, "cannot lseek past file\n");
         return 1;
      }
   }
   else
   {
      while(skip)
      {
         n=readbuffer(tf, ptb->raw, (skip < 0xF000) ? (unsigned int)skip : 0xF000);
         if(n == -1)
         {
            fprintf(stderr, "cannot read past file\n");
            return 1;
         }
         if(!n)
         {
            fprintf(stderr, "premature end of file\n");
            return 1;
         }
         skip -= (unsigned int)n;
      }
   }
   return 0;
}

void displayfile(union TAR_BLOCK *ptb)
{
int i, j, perms, gid, uid;
long size, tim;
struct tm *ptm;
char permstr[12], *pchrs="xwr";

/* extract data as binary */
   perms = calcint(ptb->direct.f_perm);
   uid =   calcint(ptb->direct.f_uid);
   gid =   calcint(ptb->direct.f_gid);
   size =  calclong(ptb->direct.f_size);
   tim =   calclong(ptb->direct.f_date);

/* convert permissions to string representation (eg: '-rw-r--r--') */
   if(perms & 0xFE00)
      permstr[0] = 'd';
   else
      permstr[0] = '-';
   for(i=0; i<3; i++)
   {
      for(j=0; j<3; j++)
      {
         if(perms & (1 << (3*i+j)))
            permstr[9-(3*i+j)] = pchrs[j];
         else
            permstr[9-(3*i+j)] = '-';
      }
   }
   permstr[10] = '\0';

/* convert date & time */
   ptm = gmtime(&tim);

/* display file info */
   printf("%s %5d %5d %02d/%02d/%02d %8ld bytes %.100s\n",
      permstr, uid, gid,
      ptm->tm_mday, ptm->tm_mon, ptm->tm_year,
      size, ptb->direct.f_name);
}

int writefile(int tf, union TAR_BLOCK *ptb)
{
char dosname[MAXPATH];
long size, tsize, rsize;
int wf, n;

/* display info if required */
   if(verbose)
      displayfile(ptb);

/* give up if NOT mode 0 (extract) */
   if(mode)
      return skipfile(tf, ptb);

/* map UNIX path/file name to DOS */
   switch(unix_to_dos(ptb->direct.f_name, dosname, MAXPATH))
   {
   case 0:
      break;
   case -1:
      fprintf(stderr, "error mapping to DOS name space\n");
      return 1;
   case 1:
      dosname[strlen(dosname)-1] = 0;
      if(verbose)
         printf("creating %s\n", dosname);
      if(mkdir(dosname))
      {
         fprintf(stderr, "cannot create directory\n");
         return 1;
      }
      return 0;
   }

/* display name if req */
   if(verbose)
      printf("extracting %s\n", dosname);

/* check for name clash */
   while(access(dosname, 0) == 0)
   {
      printf("File %s clashes with existing file, please enter new name: ", dosname);
      gets(dosname);
   }

/* create new file */
   wf = open(dosname, O_RDWR | O_BINARY | O_CREAT, 0666);
   if(wf < 0)
   {
      fprintf(stderr, "cannot create new file\n");
      return 1;
   }

/* work out file size, and tar file size */
   size = calclong(ptb->direct.f_size);

/* copy number of blocks required (truncate last block written) */
   tsize = 0L;
   while(tsize < size)
   {
      rsize = (size-tsize+BLOCKSIZE-1)/BLOCKSIZE*BLOCKSIZE;
      n=readbuffer(tf, ptb->raw, (rsize < 0xF000) ? (unsigned int)rsize : 0xF000);
      if(n == -1)
      {
         fprintf(stderr, "cannot read file data\n");
         close(wf);
         return 1;
      }
      if(!n)
      {
         fprintf(stderr, "premature end of file\n");
         close(wf);
         return 1;
      }
      if(size-tsize < (unsigned int)n)
         write(wf, ptb->raw, (unsigned int)(size-tsize));
      else
         write(wf, ptb->raw, (unsigned int)n);
      tsize += (unsigned int)n;   
   }
   close(wf);
   return 0;
}

int main(int argc, char *argv[])
{
int tf, i, all=0;
union TAR_BLOCK *ptb;

/* check args */
   if(argc < 2)
   {
      puts("usage: untar [-v] [-t] <tar file> [file(s)]");
      puts("where -v displays file names as they are processed");
      puts("      -t displays a table of contents ONLY (implies -v)");
      return 0;
   }

/* process arguments */
   i=0;
   while(!i)
   {
      if(strcmp(argv[1], "-v") == 0)
      {
         verbose++;
         argc--;
         argv++;
      }
      else if(strcmp(argv[1], "-t") == 0)
      {
         verbose++;
         mode++;
         argc--;
         argv++;
      }
      else
         i++;
   }

/* allocate transfer buffer */
   ptb=(union TAR_BLOCK *)getbuffer();
   if(!ptb)
      return 1;

/* open input file / device */
   tf = openbuffer(argv[1], O_RDONLY | O_BINARY);
   if(tf < 0)
   {
      fprintf(stderr, "cannot open tar file %s\n", argv[1]);
      return 1;
   }
   if(argc < 3)
      all++;

/* now read tar file */
   for(;;)
   {
      if(getdir(tf, ptb))
         return 1;
      if(all)
         writefile(tf, ptb);
      else
      {
         for(i=2; i<argc; i++)
         {
            if(wildcmp(argv[i], ptb->direct.f_name, WILDCHR) == 0)
            {
               writefile(tf, ptb);
               break;
            }
         }
         if(i == argc)
            skipfile(tf, ptb);
      }
   }
}

