/* [DOC] File ****************************************************
** MODULE INFORMATION*
**********************
**      FILE    NAME:       PWGPUT.C
**      SYSTEM  NAME:       POWER
**      VERSION NUMBER      3.00
**
** Descr:       Character and string put functions for
**              Programmer Oriented Window Environment Routines.
**              (NoC) July 1989 - Ling Thio Software Productions.
******************************************************************
** ADMINISTRATIVE INFORMATION*
******************************
**      ORIGINAL AUTHOR:    Ling Thio
**      CREATION DATE:      89/07/10
**      CHANGES:            none
*****************************************************************/

#include <stdarg.h>                         /* for (,...) */
#include "pwinc.h"                          /* Main include file */

static void pw_inc_row(PWWIN *win);
static void pw_pc(PWWIN *win, int c);

/* [DOC] Function *********************************************
** Name:        pw_cursor                                 [API]
** SYNOPSIS:    void pw_cursor(win, row, col);
**              PWWIN *win          Pointer to window handle
**              int   row           Row position
**              int   col           Column position
** Descr:       Positions the cursor in window.
**              The 'pw_cursor' function moves the cursor
**                position to 'col','row' of window 'win'.
**              It will only effect successive calls to
**                pw_printf(), pw_putc() and pw_puts().
** RETURNS:     void
** SEE ALSO:    pw_printfat, pw_putcat, pw_putsat
**************************************************************/
void pw_cursor(PWWIN *win, int row, int col)
{
    win->ccol = col;
    win->crow = row;
    pwv_cursor(win);
}

/* [DOC] Function *********************************************
** Name:        pw_putc                                   [API]
** SYNOPSIS:    int pw_putc(win, c);
**              PWWIN *win          Pointer to window handle
**              int   c             Character to print
** Descr:       Prints a character in window.
**              The 'pw_putc' function displays the character
**                'c' at the current cursor position, with the
**                current text attribute of window 'win'.
**              The cursor position is updated afterwards.
** RETURNS:     The displayed character 'c'.
** SEE ALSO:    pw_putcat, pw_puts, pw_putsat
**************************************************************/
int pw_putc(PWWIN *win, int c)
{
    pwv_cursor(win);
    pw_pc(win, c);
    return c;
}

/* [DOC] Function *********************************************
** Name:        pw_putcell                                [API]
** SYNOPSIS:    int pw_putc(win, c, ttype);
**              PWWIN *win          Pointer to window handle
**              int   c             Character to print
**              int   ttype         Text type
** Descr:       Prints a cell (attr+char) in window.
**              The 'pw_putc' function displays the character
**                'c' at the current cursor position, with the
**                attribute 'ttype' of window 'win'.
**              The cursor position is updated afterwards.
** RETURNS:     The displayed character 'c'.
** SEE ALSO:    pw_putcat, pw_puts, pw_putsat
**************************************************************/
int pw_putcell(PWWIN *win, int c, int ttype)
{
    PWCELL cell = (pw_attrib(win->wtype, ttype) << 8) + (c & 0xff);

    pwv_cursor(win);
    if (c == '\n')
        pw_inc_row(win);
    else
    {
        if (win->ccol >= win->ncol)
            pw_inc_row(win);
        pwv_putcell(win, cell);
        (win->ccol)++;
    }
    return c;
}

/* [DOC] Function *********************************************
** Name:        pw_putcat                                 [API]
** SYNOPSIS:    int pw_putcat(win, row, col, c);
**              PWWIN *win          Pointer to window handle
**              int   row           Row position
**              int   col           Column position
**              int   c             Character to print
** Descr:       Prints a character in window at specified position.
**              The 'pw_putcat' function displays the character
**                'c' at the cursor position 'col','row' of
**                window 'win'.
**              The cursor position is updated afterwards.
**              This function does NOT take care of wrap-around
**                and special control characters.  Use
**                pw_cursor() and pw_putc() for that purpose.
**              This function can be used to display characters
**                in the borders.
** RETURNS:     The displayed character 'c'.
** SEE ALSO:    pw_putcat, pw_puts, pw_putsat
**************************************************************/
int pw_putcat(PWWIN *win, int row, int col, int c)
{
    win->ccol = col;
    win->crow = row;
    pwv_cursor(win);
    pwv_putc(win, c);
    (win->ccol)++;
    return c;
}

unsigned char pw_putattrat(PWWIN *win, int row, int col, unsigned char attr)
{
    win->ccol = col;
    win->crow = row;
    pwv_cursor(win);
    pwv_putattr(win, (PWCELL)attr);
    return attr;
}

PWCELL pw_putcellat(PWWIN *win, int row, int col, PWCELL cell)
{
    win->ccol = col;
    win->crow = row;
    pwv_cursor(win);
    pwv_putcell(win, cell);
    return cell;
}

PWCELL pw_getcellat(PWWIN *win, int row, int col)
{
    win->ccol = col;
    win->crow = row;
    pwv_cursor(win);
    return pwv_getcell(win);
}

/* [DOC] Function *********************************************
** Name:        pw_puts                                   [API]
** SYNOPSIS:    char *pw_puts(win, s);
**              PWWIN *win          Pointer to window handle
**              char  *s            Pointer to string
** Descr:       Prints a string in window.
**              The 'pw_puts' function displays the character 
**                string 's' at the cursor position 'row','col'
**                of the window 'win'.
**              The cursor position is updated afterwards.
** RETURNS:     Pointer to character string 's'.
** SEE ALSO:    pw_putsat, pw_printf, pw_printfat, printf
**************************************************************/
char *pw_puts(PWWIN *win, char *s)
{
    unsigned char *ptr = (unsigned char *) s;

    pwv_cursor(win);
    while (*ptr)
        pw_pc(win, *(ptr++));
    return s;
}

/* [DOC] Function *********************************************
** Name:        pw_putsat                                 [API]
** SYNOPSIS:    char *pw_putsat(win, row, col, s);
**              PWWIN *win          Pointer to window handle
**              int   row           Row position
**              int   col           Column position
**              char  *s            Pointer to string
** Descr:       Prints a string in window at specified position.
**              The 'pw_puts' function displays the character 
**                string 's' at the cursor position 'col','row'
**                of the window 'win'.
**              The cursor position is updated afterwards.
**              This function does NOT take care of wrap-around
**                and special control characters.  Use
**                pw_cursor() and pw_puts() for that purpose.
**              This function can be used to display strings
**                in the borders.
** RETURNS:     Pointer to character string 's'.
** SEE ALSO:    pw_puts, pw_printf, pw_printfat, printf
**************************************************************/
char *pw_putsat(PWWIN *win, int row, int col, char *s)
{
    char *ptr = s;

    win->ccol = col;
    win->crow = row;
    pwv_cursor(win);
    while (*ptr)
    {
        pwv_putc(win, *(ptr++));
        (win->ccol)++;
    }
    return s;
}

/* [DOC] Function *********************************************
** Name:        pw_printf                                 [API]
** SYNOPSIS:    void pw_printf(win, s);
**              PWWIN *win          Pointer to window handle
**              char *format        Format of string
** Descr:       Prints a formatted string in window.
**              The 'pw_printf' function displays a character 
**                string  at the current cursor position of
**                the window 'win'.
**              It uses the format string 'format' and its
**                parameters, just like the printf() function.
**              The cursor position is updated afterwards.
** RETURNS:     void
** SEE ALSO:    pw_puts, pw_putsat, pw_printfat, printf
**************************************************************/
void pw_printf(PWWIN *win, char *format, ...)
{
    va_list arg_ptr;
    char _pwv_temp[SCRATCH_LEN];
    unsigned char *s = (unsigned char *) _pwv_temp;

    va_start(arg_ptr, format);
    /* arg_ptr now points to argument after format string */
    vsprintf(_pwv_temp, format, arg_ptr);
    va_end(arg_ptr);

    pwv_cursor(win);
    while (*s)
        pw_pc(win, *(s++));
}

/* [DOC] Function *********************************************
** Name:        pw_printfat                               [API]
** SYNOPSIS:    void pw_printfat(win, row, col, s);
**              PWWIN *win          Pointer to window handle
**              int   row           Row position
**              int   col           Column position
**              char *format        Format of string
** Descr:       Prints a formatted string in window
**              at specified position.
**              The 'pw_printf' function displays a character 
**                string at the cursor position 'col','row'
**                of the window 'win'.
**              It uses the format string 'format' and its
**                parameters, just like the printf() function.
**              The cursor position is updated afterwards.
** RETURNS:     void
** SEE ALSO:    pw_puts, pw_putsat, pw_printf, printf
**************************************************************/
void pw_printfat(PWWIN *win, int row, int col, char *format, ...)
{
    va_list arg_ptr;
    char _pwv_temp[SCRATCH_LEN];
    unsigned char *s = (unsigned char *) _pwv_temp;

    va_start(arg_ptr, format);
    /* arg_ptr now points to argument after format string */
    vsprintf(_pwv_temp, format, arg_ptr);
    va_end(arg_ptr);

    win->ccol = col;
    win->crow = row;
    pwv_cursor(win);
    while (*s)
        pw_pc(win, *(s++));
}

static void pw_inc_row(PWWIN *win)
{
    win->ccol = 0;
    if (win->crow >= win->nrow-1)
        pwb_scroll_up(win, 0, 0, win->nrow, win->ncol, 1); 
    else
        (win->crow)++;
    pwv_cursor(win);
}

static void pw_pc(PWWIN *win, int c)
{
    switch(c)
    {
    case '\a':
        putchar('\a');
        break;
    case '\b':
        if (win->ccol)
        {
            win->ccol--;
        }
        else
        {
            if (win->crow) 
            {
                win->crow--;
                win->ccol=win->ncol;
            }
        }
        pwv_cursor(win);
        break;
    case '\f':
        pw_cls(win);
        break;
    case '\n':
        pw_inc_row(win);
        break;
    case '\t':
        while ((++(win->ccol)) % 8);
        if (win->ccol >= win->ncol)
            pw_inc_row(win);
        break;
    case '\r':
        win->ccol = 0;
        pwv_cursor(win);
        break;
    default:
        if (c == PW_TYPE_CHAR && !(win->status & STAT_TYPECHAR)) {
            win->status |= STAT_TYPECHAR;
            break;
        }
        if (win->status & STAT_TYPECHAR) {
            win->status &= (~STAT_TYPECHAR);
            switch (c) {
            case 'p': win->ttype = PWT_PLAIN;     return;
            case 'n': win->ttype = PWT_NORMAL;    return;
            case 'h': win->ttype = PWT_HIGHLIGHT; return;
            case 'r': win->ttype = PWT_REVERSE;   return;
            case 's': win->ttype = PWT_SPECIAL;   return;
            }
        }
        if (win->ccol >= win->ncol)
            pw_inc_row(win);
        if (win->ttype != PWT_PLAIN) {
            pwv_putattr(win, pwv_attrib(win->wtype, win->ttype));
            pwv_cursor(win);
        }
        pwv_putc(win, c);
        (win->ccol)++;
    }
}

