/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       mib.c
**     SYSTEM   NAME:       Management Information Base
**     ORIGINAL AUTHOR(S):  Dirk Wisse
**     VERSION  NUMBER:     1
**     CREATION DATE:       1990/11/22
**
** DESCRIPTION: Management Information Base
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision:   1.1  $
** WORKFILE:    $Workfile:   MIB.C  $
** LOGINFO:     $Log:   I:/ETSTJAN/CPROG/BEHOLDER/SNMP/MIB2/VCS/MIB.C_V  $
**              
**                 Rev 1.1   01 Feb 1991 13:48:52   etstjan
**              No explicit note
**              
**                 Rev 1.0   26 Nov 1990 16:43:24   etstjan
**              No explicit note
*************************************************************************/
#if ! defined(PRD)
static char _pvcs_hdr[] =
"$Header:   I:/ETSTJAN/CPROG/BEHOLDER/SNMP/MIB2/VCS/MIB.C_V   1.1   01 Feb 1991 13:48:52   etstjan  $";
#endif
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <futil.h>
#include "error.h"
#include "snmp.h"
#include "mib2.h"

mib_sck
    MibSck =
{
    NULL,
    "Description",
    "Contact",
    "Name",
    "Location",
    127,
    0,
    0
};

int
    MibOjiCmp
    (
        unsigned    *Oji1,
        unsigned    OjiLen1,
        unsigned    *Oji2,
        unsigned    OjiLen2
    )
{
    while (OjiLen1 > 0 && OjiLen2 > 0 && *Oji1 == *Oji2)
    {
        OjiLen1--;
        OjiLen2--;
        Oji1++;
        Oji2++;
    }
    if (!OjiLen1 && !OjiLen2)
        return (0);
    if (!OjiLen1)
        return (-1);
    if (!OjiLen2)
        return (1);
    if (*Oji1 < *Oji2)
        return (-2);
    else
        return (2);
}

mib_prf
    *MibGetPrf
    (
        char        *Com,
        unsigned    ComLen
    )
{
    mib_prf
        *Prf;

	Prf = MibSck.SckPrf;
	while (Prf != NULL)
    {
        if (Prf->PrfComLen == ComLen && !memcmp (Prf->PrfCom, Com, ComLen))
            return (Prf);
        Prf = Prf->PrfNxt;
    }
    return (NULL);
}

/**************************************************************
** NAME:        MibAddPrf
** SYNOPSIS:    mib_prf
**                  *MibAddPrf
**                  (
**                      char        *Com,
**                      unsigned    ComLen
**                  )
** DESCRIPTION: Gets or Adds the MIB profile for a community.
**              Parameters:
**              Com:    Community.
**              ComLen: Length of community in characters.
** RETURNS:     NULL --> error
**              else     pointer to desired profile
**************************************************************/
mib_prf
    *MibAddPrf
    (
        char        *Com,
        unsigned    ComLen
    )
{
    mib_prf
        *Prf;

    if ((Prf = MibGetPrf (Com, ComLen)) != NULL)
        return (Prf);
    if (ComLen > MIB_SZECOM)
        return (NULL);
    if ((Prf = (mib_prf *) IPBufGet (sizeof (mib_prf))) == NULL)
        return (NULL);
    memcpy (Prf->PrfCom, Com, ComLen);
    Prf->PrfComLen = ComLen;
    Prf->PrfObj = NULL;
	Prf->PrfNxt = MibSck.SckPrf;
	MibSck.SckPrf = Prf;
    return (Prf);
}

/**************************************************************
** NAME:        MibAddObj   
** SYNOPSIS:    int
**                  MibAddObj
**                  (
**                      mib_prf     *Prf,
**                      unsigned    *Oji,
**                      unsigned    OjiLen,
**                      unsigned    (*RqsGet) (pkt_obj *Obj,
**                                              void *Arg),
**                      unsigned    (*RqsNxt) (pkt_obj *Obj,
**                                              void *Arg),
**                      unsigned    (*RqsSet) (pkt_obj *Obj,
**                                              void *Arg),
**                      void        *RqsArg
**                  )
** DESCRIPTION: Adds a MIB object to a profile.
**              Parameters:
**              Prf:    Profile to add object to.
**              Oji:    Identifier of object.
**              OjiLen: Length of Indentifier.
**              RqsGet: Pointer to Get function or NULL.
**              RqsNxt: Pointer to Get Next function or NULL.
**              RqsSet: Pointer to Set function or NULL.
**              RqsArg: Argument to Get, Next or Set function.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    MibAddObj
    (
        mib_prf     *Prf,
        unsigned    *Oji,
        unsigned    OjiLen,
        unsigned    (*RqsGet) (pkt_obj *Obj, void *Arg),
        unsigned    (*RqsNxt) (pkt_obj *Obj, void *Arg),
        unsigned    (*RqsSet) (pkt_obj *Obj, void *Arg),
        void        *RqsArg
    )
{
    mib_obj
        *Obj,
        **Ins;

    if (OjiLen > MIB_SZEOJI)
        return (-1);
    if ((Obj = (mib_obj *) IPBufGet (sizeof (mib_obj))) == NULL)
        return (-1);
    memcpy (Obj->ObjOji, Oji, OjiLen * sizeof (int));
    Obj->ObjOjiLen = OjiLen;
    Obj->ObjRqsGet = RqsGet;
    Obj->ObjRqsNxt = RqsNxt;
    Obj->ObjRqsSet = RqsSet;
    Obj->ObjRqsArg = RqsArg;
    Ins = &Prf->PrfObj;
    while ((*Ins) != NULL && MibOjiCmp ((*Ins)->ObjOji, (*Ins)->ObjOjiLen,
                       Obj->ObjOji  , Obj->ObjOjiLen) < 0)
    {
        Ins = &(*Ins)->ObjNxt;
    }
    Obj->ObjNxt = *Ins;
    *Ins = Obj;
    return (0);
}


unsigned
    MibGetObj
    (
        mib_prf     *Prf,
        pkt_obj     *Obj
    )
{
    unsigned
        Err;
    mib_obj
        *MibObj;

    Err = PKT_NOSUCHNAME;
    MibObj = Prf->PrfObj;
    while (MibObj != NULL && Err == PKT_NOSUCHNAME)
    {
        switch (MibOjiCmp (MibObj->ObjOji, MibObj->ObjOjiLen,
                           Obj->ObjOji, Obj->ObjOjiLen))
        {
            case -1:
                if (MibObj->ObjRqsGet != NULL && MibObj->ObjRqsNxt != NULL)
                    Err = MibObj->ObjRqsGet (Obj, MibObj->ObjRqsArg);
                break;
            case  0:
                if (MibObj->ObjRqsGet != NULL && MibObj->ObjRqsNxt == NULL)
                    Err = MibObj->ObjRqsGet (Obj, MibObj->ObjRqsArg);
                break;
        }
        MibObj = MibObj->ObjNxt;
    }
    return (Err);
}

unsigned
    MibNxtObj
    (
        mib_prf     *Prf,
        pkt_obj     *Obj
    )
{
    unsigned
        Err;
    mib_obj
        *MibObj;

    Err = PKT_NOSUCHNAME;
    MibObj = Prf->PrfObj;
    while (MibObj != NULL && Err == PKT_NOSUCHNAME)
    {
        switch (MibOjiCmp (MibObj->ObjOji, MibObj->ObjOjiLen,
                           Obj->ObjOji, Obj->ObjOjiLen))
        {
            case -1:
            case  0:
                if (MibObj->ObjRqsNxt != NULL)
                    Err = MibObj->ObjRqsNxt (Obj, MibObj->ObjRqsArg);
                break;
            case  1:
            case  2:
                memcpy (Obj->ObjOji, MibObj->ObjOji,
                        MibObj->ObjOjiLen * sizeof (int));
                Obj->ObjOjiLen = MibObj->ObjOjiLen;
                if (MibObj->ObjRqsNxt != NULL)
                    Err = MibObj->ObjRqsNxt (Obj, MibObj->ObjRqsArg);
                else if (MibObj->ObjRqsGet != NULL)
                    Err = MibObj->ObjRqsGet (Obj, MibObj->ObjRqsArg);
                break;
        }
        MibObj = MibObj->ObjNxt;
    }
    return (Err);
}

unsigned
    MibSetObj
    (
        mib_prf     *Prf,
        pkt_obj     *Obj
    )
{
    unsigned
        Err;
    mib_obj
        *MibObj;

    Err = PKT_NOSUCHNAME;
    MibObj = Prf->PrfObj;
    while (MibObj != NULL && Err == PKT_NOSUCHNAME)
    {
        switch (MibOjiCmp (MibObj->ObjOji, MibObj->ObjOjiLen,
                           Obj->ObjOji, Obj->ObjOjiLen))
        {
            case -1:
                if (MibObj->ObjRqsSet != NULL && MibObj->ObjRqsNxt != NULL)
                    Err = MibObj->ObjRqsSet (Obj, MibObj->ObjRqsArg);
                break;
            case  0:
                if (MibObj->ObjRqsSet != NULL && MibObj->ObjRqsNxt == NULL)
                    Err = MibObj->ObjRqsSet (Obj, MibObj->ObjRqsArg);
                break;
        }
        MibObj = MibObj->ObjNxt;
    }
    return (Err);
}

void
    MibGetLst
    (
        mib_prf     *Prf,    
        unsigned    *ErrSts,
        unsigned    *ErrInd,
        pkt_obj     *Lst,
        unsigned    LstLen
    )
{
    unsigned
        Sts,
        Ind;

    Sts = PKT_NOERROR;
    for (Ind = 0; Ind < LstLen && Sts == PKT_NOERROR; Ind++)
        Sts = MibGetObj (Prf, &Lst [Ind]);
    *ErrSts = Sts;
    if (Sts == PKT_NOERROR)
    {
        *ErrInd = 0;
		MibSck.SckTotGet += Ind;
    }
    else
        *ErrInd = Ind;
    return;
}

void
    MibNxtLst
    (
        mib_prf     *Prf,    
        unsigned    *ErrSts,
        unsigned    *ErrInd,
        pkt_obj     *Lst,
        unsigned    LstLen
    )
{
    unsigned
        Sts,
        Ind;

    Sts = PKT_NOERROR;
    for (Ind = 0; Ind < LstLen && Sts == PKT_NOERROR; Ind++)
        Sts = MibNxtObj (Prf, &Lst [Ind]);
    *ErrSts = Sts;
    if (Sts == PKT_NOERROR)
    {
        *ErrInd = 0;
		MibSck.SckTotGet += Ind;
    }
    else
        *ErrInd = Ind;
    return;
}

void
    MibSetLst
    (
        mib_prf     *Prf,    
        unsigned    *ErrSts,
        unsigned    *ErrInd,
        pkt_obj     *Lst,
        unsigned    LstLen
    )
{
    unsigned
        Sts,
        Ind;

    Sts = PKT_NOERROR;
    for (Ind = 0; Ind < LstLen && Sts == PKT_NOERROR; Ind++)
        Sts = MibSetObj (Prf, &Lst [Ind]);
    *ErrSts = Sts;
    if (Sts == PKT_NOERROR)
    {
        *ErrInd = 0;
		MibSck.SckTotSet += Ind;
    }
    else
        *ErrInd = Ind;
    return;
}

/**************************************************************
** NAME:        MibGetRsp
** SYNOPSIS:    void
**                  MibGetRsp
**                  (
**                      char        *Com,
**                      unsigned    ComLen,
**                      pkt_ror     *Rsp
**                  )
** DESCRIPTION: Gets objects form the MIB.
** RETURNS:     0 -->   no error
**              else    error code
** RETURNS:     void
**************************************************************/
void
    MibGetRsp
    (
        char        *Com,
        unsigned    ComLen,
        pkt_ror     *Rsp
    )
{
    mib_prf
        *Prf;

    ERR_DEB(ERR_MIB, 2, "MibGetRsp");
    if ((Prf = MibGetPrf (Com, ComLen)) == NULL)
    {
		Rsp->RorErrSts = PKT_NOSUCHNAME;
		Rsp->RorErrInd = 1;
        return;
    }
    switch (Rsp->RorTyp)
    {
        case PKT_GETRQS:
			MibGetLst (Prf,
			&Rsp->RorErrSts, &Rsp->RorErrInd, Rsp->RorLst, Rsp->RorLstLen);
            break;
        case PKT_NXTRQS:
			MibNxtLst (Prf,
			&Rsp->RorErrSts, &Rsp->RorErrInd, Rsp->RorLst, Rsp->RorLstLen);
            break;
        case PKT_SETRQS:
			MibSetLst (Prf,
			&Rsp->RorErrSts, &Rsp->RorErrInd, Rsp->RorLst, Rsp->RorLstLen);
            break;
    }
}

/**************************************************************
** NAME:        MibPol       
** SYNOPSIS:    void
**                  MibPol
**                  (
**                      void
**                  )
** DESCRIPTION: MIB main function.
**              Checks if request pending. If so it gets the
**              requested objects from the MIB, it constructs
**              a respones and sends this to the management
**              station. 
** RETURNS:     void
**************************************************************/
void
    MibPol
    (
        void
    )
{
    pkt_ror
        Rqs,
        Rsp;
    char
        Com [AGT_SZECOM];
    unsigned
        ComLen;

    if (AgtRcvRqs (Com, &ComLen, &Rqs) < 0)
        return;
    ERR_DEB (ERR_MIB, 2, "Packet received");
    memcpy (&Rsp, &Rqs, sizeof (pkt_ror));
    MibGetRsp (Com, ComLen, &Rsp);
    if (AgtSndRsp (Com, ComLen, &Rsp) < 0)
    {
        Rqs.RorErrSts = PKT_TOOBIG;
        Rqs.RorErrInd = 0;
        AgtSndRsp (Com, ComLen, &Rqs);
    }
}

unsigned
    MibGetCnt
    (
        pkt_obj *Obj,
        void    *Arg
    )
{
    Obj->ObjTag = PKT_COUNTER;
	Obj->ObjSyn.SynLngInt = *(long unsigned *)Arg;
    return (PKT_NOERROR);
}

unsigned
    MibGetStr
    (
        pkt_obj *Obj,
        void    *Arg
    )
{
    Obj->ObjTag = PKT_OCTETSTRING;
	strcpy (Obj->ObjSyn.SynBufChr, (char *)Arg);
    Obj->ObjSynLen = strlen ((char *)Arg);
    return (PKT_NOERROR);
}

unsigned
    MibSetStr
    (
        pkt_obj *Obj,
        void    *Arg
    )
{
    if (Obj->ObjTag != PKT_OCTETSTRING)
        return (PKT_BADVALUE);
    Obj->ObjSyn.SynBufChr [Obj->ObjSynLen] = '\0';
    strcpy ((char *)Arg, Obj->ObjSyn.SynBufChr);
    return (PKT_NOERROR);
}

unsigned
    MibGetTit
    (
        pkt_obj *Obj,
        void    *Arg
    )
{
    Obj->ObjTag = PKT_TIMETICKS;
	Obj->ObjSyn.SynLngInt = TitGet ();
    return (PKT_NOERROR);
}

unsigned
    MibGetOji
    (
        pkt_obj *Obj,
        void    *Arg
    )
{
    Obj->ObjTag = PKT_OBJECTIDENTIFIER;
    memcpy (Obj->ObjSyn.SynBufInt, AgtSck.SckOji, AgtSck.SckOjiLen * sizeof (int));
	Obj->ObjSynLen = AgtSck.SckOjiLen;
    return (PKT_NOERROR);
}

unsigned
    MibGetSrv
    (
        pkt_obj *Obj,
        void    *Arg
    )
{
    Obj->ObjTag = PKT_INTEGER;
	Obj->ObjSyn.SynLngInt = MibSck.SckSrv;
    return (PKT_NOERROR);
}

unsigned
    MibGetAut
    (
        pkt_obj *Obj,
        void    *Arg
    )
{
    Obj->ObjTag = PKT_INTEGER;
	Obj->ObjSyn.SynLngInt = AgtSck.SckEnaAut ? 1 : 2;
    return (PKT_NOERROR);
}

unsigned
    MibSetAut
    (
        pkt_obj *Obj,
        void    *Arg
    )
{
    if (Obj->ObjTag != PKT_INTEGER)
        return (PKT_BADVALUE);
    if (Obj->ObjSyn.SynLngInt != 1 && Obj->ObjSyn.SynLngInt != 2)
        return (PKT_BADVALUE);
	AgtSck.SckEnaAut = (Obj->ObjSyn.SynLngInt == 1);
    return (PKT_NOERROR);
}

int
    MibIniMib
    (
        void
    )
{                                                 
    unsigned
        Sys [MIB_SZEOJI] = {1, 3, 6, 1, 2, 1, 1,  1, 0},
        Nmp [MIB_SZEOJI] = {1, 3, 6, 1, 2, 1, 11, 1, 0};
    mib_prf
        *Prf;

    if ((Prf = MibAddPrf ("public", 6)) == NULL)
        return (-1);
    if (MibAddObj (Prf, Sys, 9, MibGetStr, NULL, MibSetStr, &MibSck.SckDsc))
        return (-1);
    Sys [7] ++;
    if (MibAddObj (Prf, Sys, 9, MibGetOji, NULL, NULL, NULL))
        return (-1);
    Sys [7] ++;
    if (MibAddObj (Prf, Sys, 9, MibGetTit, NULL, NULL, NULL))
        return (-1);
    Sys [7] ++;
    if (MibAddObj (Prf, Sys, 9, MibGetStr, NULL, MibSetStr, &MibSck.SckCtt))
        return (-1);
    Sys [7] ++;
    if (MibAddObj (Prf, Sys, 9, MibGetStr, NULL, MibSetStr, &MibSck.SckNme))
        return (-1);
    Sys [7] ++;
    if (MibAddObj (Prf, Sys, 9, MibGetStr, NULL, MibSetStr, &MibSck.SckLoc))
        return (-1);
    Sys [7] ++;
    if (MibAddObj (Prf, Sys, 9, MibGetSrv, NULL, NULL, NULL))
        return (-1);
    Sys [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckInPkts))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckOutPkts))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckInBadVersions))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &AutSck.SckBadCom))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &AutSck.SckBadUse))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckInASNParseErrs))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckInBadTypes))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckInTooBigs))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckInNoSuchNames))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckInBadValues))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckInReadOnlys))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckInGenErrs))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &MibSck.SckTotGet))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &MibSck.SckTotSet))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckInGetRequests))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckInGetNexts))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckInSetRequests))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckInGetResponses))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckInTraps))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckOutTooBigs))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckOutNoSuchNames))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckOutBadValues))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckOutReadOnlys))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckOutGenErrs))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckOutGetRequests))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckOutGetNexts))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckOutSetRequests))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckOutGetResponses))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetCnt, NULL, NULL, &PktSck.SckOutTraps))
        return (-1);
    Nmp [7] ++;
    if (MibAddObj (Prf, Nmp, 9, MibGetAut, NULL, MibSetAut, NULL))
        return (-1);
    return (0);
}

int
    MibIniAut
    (
        FILE * Cfg
    )
{
    int
        Lnr;
    aut_srv
        *Srv;
    char
        Str [80],
        Com [80],
        Adr [80],
        Msk [80];

    if (FFindSection (Cfg, "AUTHENTICATION") < 0)
        return (-1);
    Lnr = FGetNext (Cfg, Str, sizeof (Str), MIX);
    while (Lnr >= 0)    
    {    
        if (sscanf (Str, "Community %s", Com) != 1)    
        {    
            Lnr = FGetNext (Cfg, Str, sizeof (Str), MIX);    
            continue;    
        }    
        if ((Srv = AutAddSrv (Com, strlen (Com), NULL, NULL)) == NULL)
            return (-1);
        Lnr = FGetNext (Cfg, Str, sizeof (Str), MIX);
        while (Lnr >= 0)    
        {    
            if (sscanf (Str, "AddAddress %s %s", Adr, Msk) != 2)    
                break;    
            if (AutAddNet (Srv, inet_addr (Adr), inet_addr (Msk)))
                return (-1);
            Lnr = FGetNext (Cfg, Str, sizeof (Str), MIX);
        }    
    }    
    return (0);
}

int
    MibIniAgt
    (
        FILE *Cfg
    )
{
    char
        Prt [5],
        Oid [80],
        Trp [10],
        TrpPrt [5],
        TrpHst [80],
        TrpCom [80];
    int
        TrpEna;
    unsigned
        Oji [AGT_SZEOJI],
        OjiLen;

    if (FFindSection (Cfg, "AGENT") < 0)
    {
        ERR_ERR (ERR_MIB, ERR_CRITICAL, "Cannot find agent\n");
        return (-1);
    }
    if (FGetVar (Cfg, "OBJECTID", Oid, 80, UPC) < 0)
    {
        ERR_ERR (ERR_MIB, ERR_CRITICAL, "Cannot find objectidentifier\n");
        return (-1);
    }
    if (StrDecOid (Oid, Oji, &OjiLen) < 0)
    {
        ERR_ERR (ERR_MIB, ERR_CRITICAL, "Syntax error in objectidentifier\n");
        return (-1);
    }
    if (FGetVar (Cfg, "PORT", Prt, 5, UPC) < 0)
    {
        ERR_ERR (ERR_MIB, ERR_CRITICAL, "Cannot find port\n");
        return (-1);
    }
    if (FGetVar (Cfg, "TRAP", Trp, 10, UPC) < 0)
    {
        ERR_ERR (ERR_MIB, ERR_CRITICAL, "Cannot find trap\n");
        return (-1);
    }
    if (strcmp (Trp, "ENABLE") == 0)
    {
        TrpEna = 1;
        if (FGetVar (Cfg, "TRAPPORT", TrpPrt, 5, UPC) < 0)
        {
            ERR_ERR (ERR_MIB, ERR_CRITICAL, "Cannot find trapport\n");
            return (-1);
        }
        if (FGetVar (Cfg, "TRAPADDRESS", TrpHst, 80, UPC) < 0)
        {
            ERR_ERR (ERR_MIB, ERR_CRITICAL, "Cannot find trapaddress\n");
            return (-1);
        }
        if (FGetVar (Cfg, "TRAPCOMMUNITY", TrpCom, 80, MIX) < 0)
        {
            ERR_ERR (ERR_MIB, ERR_CRITICAL, "Cannot find trapcommunity\n");
            return (-1);
        }
    }
    else
        TrpEna = 0;
    if (AgtIni (atoi (Prt), Oji, OjiLen, TrpEna, atoi (TrpPrt), TrpHst, TrpCom, strlen (TrpCom)))
    {
        ERR_ERR (ERR_MIB, ERR_CRITICAL, "Cannot initialize agent\n");
        return (-1);
    }
    return (0);
}

int
    MibIniSys
    (
        FILE *Cfg
    )
{
    if (FFindSection (Cfg, "SYSTEM") < 0)
        return (-1);
    FGetVar (Cfg, "Description", MibSck.SckDsc, MIB_SZEOTS, MIX);
    FGetVar (Cfg, "Contact", MibSck.SckCtt, MIB_SZEOTS, MIX);
    FGetVar (Cfg, "Name", MibSck.SckNme, MIB_SZEOTS, MIX);
    FGetVar (Cfg, "Location", MibSck.SckLoc, MIB_SZEOTS, MIX);
    return (0);
}


/**************************************************************
** NAME:        MibIni        
** SYNOPSIS:    int
**                  MibIni
**                  (
**                      void
**                  )
** DESCRIPTION: Initializes MIB from configfile.
** RETURNS:     0 -->   no error
**              else    error code
**************************************************************/
int
    MibIni
    (
        void
    )
{
    FILE
        *Cfg;

    ERR_DEB (ERR_MIB, 5, "Start initializing\n");
    TitIni ();        
    if (MibIniMib () < 0)    
    {
        ERR_ERR (ERR_MIB, ERR_CRITICAL, "Cannot initialize mib\n");
        return (-1);
    }
    if ((Cfg = fopen (ConfigFile,"r")) == NULL)
    {
        ERR_ERR (ERR_MIB, ERR_CRITICAL, "Cannot open configfile\n");
        return (-1);
    }
    if (MibIniSys (Cfg) < 0)
    {
        ERR_ERR (ERR_MIB, ERR_CRITICAL, "Cannot initialize system\n");
        fclose (Cfg);
        return (-1);
    }
    if (MibIniAut (Cfg) < 0)    
    {
        ERR_ERR (ERR_MIB, ERR_CRITICAL, "Cannot initialize authentication\n");
        fclose (Cfg);
        return (-1);
    }
    if (MibIniAgt (Cfg) < 0)    
    {
        ERR_ERR (ERR_MIB, ERR_CRITICAL, "Cannot initialize agent\n");
        fclose (Cfg);
        return (-1);
    }
    fclose (Cfg);
    ERR_DEB (ERR_MIB, 5, "Initializing ready\n");
    return (0);    
}
