/*
  modes.c - Try and determine the modes of the graphics card.

  Jason Hood, 3 & 11 May, 2003.
*/

#include <stdio.h>
#include <dos.h>
#include <mem.h>
#include "vgamem.h"

#define PVERS "1.00"
#define PDATE "11 May, 2003"

#define PHELP "\
Modes by Jason Hood <jadoxa@yahoo.com.au>.\n\
Version "PVERS" ("PDATE"). Public Domain.\n\
http://tm.adoxa.cjb.net/\n\
\n\
MODES [t|g][b|v] >modes.txt\n\
\n\
t will display text modes;\n\
g will display graphics modes;\n\
b will display BIOS modes;\n\
v will display VESA modes.\n\
\n\
BIOS Notes: the info is an educated guess;\n\
	    number of colors is not determined;\n\
	    modes marked with '*' are set as Mode 03;\n\
	    a beep will sound every 10 modes, twice for the last;\n\
	    the monitor might appreciate being turned off.\
"


#define TEXT 1
#define GRFX 2
#define BIOS 4
#define VESA 8
int select = 0;

void write_BIOS_modes( void );
void write_VESA_modes( void );
void write_mode_info( Word );

int main( int argc, char* argv[] )
{
  if (argc > 1)
  {
    while (argv[1][0])
    {
      switch (argv[1][0])
      {
	case 't': select |= TEXT; break;
	case 'g': select |= GRFX; break;
	case 'b': select |= BIOS; break;
	case 'v': select |= VESA; break;
	default:
	  puts( PHELP );
	  return 0;
      }
      ++argv[1];
    }
  }
  if (!(select & (TEXT | GRFX))) select |= TEXT | GRFX;
  if (!(select & (BIOS | VESA))) select |= BIOS | VESA;

  if (select & BIOS) write_BIOS_modes();
  if (select & VESA)
  {
    if (select & BIOS) putchar( '\n' );
    write_VESA_modes();
  }

  return 0;
}


void write_BIOS_modes( void )
{
  int mode, type, scan;

  for (mode = 8; mode < 0x80; ++mode)
  {
    video->mode = 0;
    asm {
	mov ax,mode
	int 0x10
    }
    if (video->mode)
    {
      outp( 0x3ce, 6 ); 	// Graphics Miscellaneous Register
      type = inp( 0x3cf );
      if ((!(type & 1) && (select & TEXT)) ||
	   ((type & 1) && (select & GRFX)))
      {
	scan = (video->rows + 1) * video->font;
	// The scan lines and font don't necessarily divide evenly,
	// so make a guess for the real scan lines.
	if	(scan >= 580 && scan < 600) scan = 600;
	else if (scan >= 280 && scan < 300) scan = 300;
	printf( "Mode %02x%c: %3d x %d %s(%d) : %4d x %4d : %s%s\n",
		mode, (video->mode == 3) ? '*' : ' ',
		video->cols, video->rows + 1,
		(video->font < 10) ? " " : "", video->font,
		video->cols << 3, scan,
		(type & 1) ? "Graphics" : "Text",
		(video->port != 0x3d4) ? " (Mono)" : "" );
      }
    }
    if (mode % 10 == 7)
      asm {
	mov al,7
	int 0x29
      }
  }
  asm {
	mov ax,3
	int 0x10
	mov al,7
	int 0x29
  }
}


typedef unsigned long Long;

typedef struct
{
  Byte VESASignature[4];
  Word VESAVersion;
  Long OEMStringPtr;
  Long Capabilities;
  Long VideoModePtr;
  Byte stuff[238];
} VESA_INFO;

typedef struct
{
  Word attr;
  Byte stuff1[16];
  Word width;
  Word height;
  Byte cwidth;
  Byte cheight;
  Byte planes;
  Byte colors;
  Byte stuff2[230];
} MODE_INFO;


void write_VESA_modes( void )
{
  union REGS r;
  VESA_INFO  vesa;
  short far* mode;

  r.x.ax = 0x4f00;		// Get VESA information
     _ES = _SS;
  r.x.di = (unsigned)&vesa;
  int86( 0x10, &r, &r );
  if (r.h.ah != 0 || memcmp( vesa.VESASignature, "VESA", 4 ) != 0)
  {
    puts("Unable to retrieve VESA information." );
    return;
  }

  mode = (short far *)vesa.VideoModePtr;
  while (*mode != -1)
  {
    write_mode_info( *mode );
    ++mode;
  }
}


void write_mode_info( Word mode )
{
  union REGS r;
  MODE_INFO  mi;
  int wid,  hyt;
  int cols, rows;

  if (mode < 0x100) return;	// Ignore non-VESA modes.

  r.x.ax = 0x4f01;
     _ES = _SS;
  r.x.di = (unsigned)&mi;
  r.x.cx = mode;
  int86( 0x10, &r, &r );
  if (r.h.ah != 0 || !(mi.attr & 0x01)) return; // Mode's no good.

  if (mi.attr & 0x10)
  {
    if (!(select & GRFX)) return;
    // Translate pixel dimensions to characters.
    wid  = mi.width;
    hyt  = mi.height;
    cols = wid / mi.cwidth;
    rows = hyt / mi.cheight;
  }
  else
  {
    if (!(select & TEXT)) return;
    // Translate character dimensions to pixels.
    cols = mi.width;
    rows = mi.height;
    wid  = cols * mi.cwidth;
    hyt  = rows * mi.cheight;
  }
  printf( "Mode %x : %3d x %d %s(%d x %2d) : %4d x %4d x %2d : %s%s\n",
	  mode,
	  cols, rows,
	  (mi.cwidth < 10) ? " " : "", mi.cwidth, mi.cheight,
	  wid, hyt, mi.colors,
	  (mi.attr & 0x10) ? "Graphics" : "Text",
	  (mi.attr & 0x08) ? "" : " (Mono)" );
}
