/*
  tm.c - Set various text mode resolutions; display current.

  Jason Hood, 24 September to 3 October, 1999.

  Assumes a colour VGA.

  Credits from the Assembly Snippets, version 4.0 (asnip40a.zip):
    480 scan lines and 90-column mode by Tenie Remmel (set480.asm, 90wide.asm);
    6-line font was uncredited (font58.asm).

  Assembly code assumes DS == SS and CLD.


  The number of rows is determined by font size and scan lines:

				    F O N T  S I Z E
		ͻ
		 16  15  14  13  12  11  10   9   8 	7   6 
	S ͹
	C  200  12  13  14  15  16  18  20  22  25  28  33 
	A Ķ
	N  350  21  23  25  26  29  31  35  38  43  50  58 
	L Ķ
	I  400  25  26  28  30  33  36  40  44  50  57  66 
	N Ķ
	E  480  30  32  34  36  40  43  48  53  60  68  80 
	S ͼ

				      scan lines
			       rows = 
				      font size


  VESA text modes (from Ralf Brown's Interrupt List):
    108h	 80x60
    109h	132x25
    10Ah	132x43
    10Bh	132x50
    10Ch	132x60

  991125: save the screen when using a font file.
  000215: option to write fonts to file.
*/

#define VERSION "1.02"
#define DATE	"15 February, 2000"

#define USE_ASM 1

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <io.h>
#if !USE_ASM
#include <mem.h>
#endif

#include "vgamem.h"

extern Byte font12[], font6[];

void  set_cols( int );
void  set_scan( int );
void  set_font( int, char* );
Byte* get_BIOS_font( int );
Byte* get_font( int );
Byte* expand_font( int, int, Byte* );
void  write_font( int );

#if !USE_ASM
void set_curs( int, int, int );
#endif
void screen_cpy( Word*, int, int, int, int, int );

void help( char* );


#define MODE  1
#define DIMS  2		// Display flags
#define COLS  4
#define ROWS  8
#define FONT 16
#define PAGE 32
#define BRDR 64


int main( int argc, char* argv[] )
{
  char	 opt;
  int	 num;
  int	 quiet = 0;
  int	 save = 0;
  int	 display = 0;
  int	 cols = -1, rows = -1;
  int	 mode = -1, font =  0;
  int	 lines = 0, page =  0;
  int	 attr = -1, brdr = -1;
  char*  fontfile = NULL;
  int    wfont = -1;
  char*  end;
  Word	 screen;
  Word*  buffer;
  int	 ocols, orows, ofont, opage;
  int	 max_pages;
  Cursor cur;
  int  	 rc = 0;
  int	 j, k;

  ocols = video->cols;
  orows = video->rows + 1;
  ofont = video->font;
  opage = video->page;
  max_pages = 0x8000u / video->size;
  if (max_pages > 8) max_pages = 8;
  cur	 = video->curs[opage];
  screen = video->offset;

  if (argc == 1) help( argv[0] );
  else
  {
    for (j = 1; j < argc; ++j)
    {
      for (k = 0; argv[j][k]; )
      {
	opt = argv[j][k++];
	if ((opt == '-' || opt == '/') && argv[j][k]) opt = argv[j][k++];
	opt = tolower( opt );
	num = -1;
	if (!(opt == 'd' || opt == 'q' || opt == '?'))
	{
	  if (isdigit( opt )) opt = 'c', --k;
	  else if (argv[j][k] == 0 && j+1 < argc) ++j, k = 0;
	  if (isdigit( argv[j][k] ) || (opt == 'm' && isxdigit( argv[j][k] )))
	  {
	    num = (int)strtol( argv[j] + k, &end, (opt == 'm') ? 16 :
						  (opt == 's') ?  0 : 10 );
	    k = end - argv[j];
	  }
	  else if (opt == 'f')
	  {
	    if (argv[j][k] != '-' && argv[j][k] != '/' && argv[j][k])
	    {
	      fontfile = argv[j] + k;
	      break;
	    }
	  }
	  else if (opt == 'x' || opt == ',')
	  {
	    fprintf( stderr, "Option `%c' requires a number.\n", opt );
	    return -1;
	  }
	}
	switch (opt)
	{
	  case '?': help( argv[0] ); return 0;

	  case 'q': quiet = 1; break;

	  case 's': if (num < 0) save = 1;
		    else attr = num;
		    break;

	  case 'd': display |= DIMS; break;

	  case 'c': if (num < 0) display |= COLS, rc = ocols;
		    else cols = num;
		    break;

	  case 'x': // fall through (num is always valid)
	  case 'r': if (num < 0) display |= ROWS, rc = orows;
		    else rows = num;
		    break;

	  case 'm': if (num < 0) display |= MODE, rc = video->mode;
		    else mode = num;
		    break;

	  case ',': if (num < 1 || num > 4)
		    {
		      fprintf( stderr, "Unsupported lines: %d.\n", num );
		      return -1;
		    }
		    else lines = num;
		    break;

	  case 'f': if (num < 0) display |= FONT, rc = ofont;
		    else font = num;
		    break;

	  case 'p': if (num <= 0)
		    {
		      display |= PAGE;
		      rc = (num < 0) ? opage : max_pages;
		    }
		    else page = num;
		    break;

	  case 'b': if (num < 0) display |= BRDR, rc = video->palette & 0x0f;
		    else brdr = num;
		    break;

	  case 'w': if (num <= 0) wfont = 0;
		    else if (num >= 6 && num <= 16) wfont = num;
		    else
		    {
		      fprintf( stderr, "Unsupported font height: %d.\n", num );
		      return -1;
		    }
		    break;

	  default:  fprintf( stderr, "Unknown option: %c.\n", opt );
		    return -1;
	}
      }
    }
    if (lines && mode < 0 && rows < 0)
    {
      if (cols < 0) cols = ocols;
      if (!font && !fontfile) font = ofont;
    }

    if ((save && attr >= 0) || (font && fontfile) ||
	(rows >= 0 && (font || fontfile)) ||
	(mode >= 0 && (cols >= 0 || rows >= 0)) ||
	(display && (cols >= 0 || rows >= 0 || mode >= 0 ||
		     font || fontfile || page || attr >= 0)) ||
	((display & DIMS) && (quiet || (display & (COLS | ROWS | FONT)))))
    {
      fputs( "Mutually exclusive options in effect.\n", stderr );
      return -1;
    }
    if (fontfile && access( fontfile, 0 ))
    {
      fprintf( stderr, "Font \"%s\" does not exist.\n", fontfile );
      return -1;
    }
    if (font && (font < 6 || font > 16))
    {
      fprintf( stderr, "Unsupported font height: %d.\n", font );
      return -1;
    }
    if (font || fontfile)
    {
      if (mode < 0 && cols < 0 && (opage != 0 || page > 1))
      {
	fputs( "Font can only be changed in page 1.\n", stderr );
	return -1;
      }
    }

    if (display)
    {
      if ((display & MODE) && !quiet)
	printf( "Mode %02X.\n", video->mode );

      if (display & DIMS)
	printf( "%d x %d (%d-line font).\n", ocols, orows, ofont );

      if (!quiet)
      {
	if (display & COLS) printf( "%d columns.\n", ocols );
	if (display & ROWS) printf( "%d rows.\n", orows );
	if (display & FONT) printf( "%d-line font.\n", ofont );
	if (display & PAGE) printf( "Page %d (of %d).\n", opage+1, max_pages );
	if (display & BRDR)
	{
	  static char* color[8] = { "Black", "Blue",    "Green", "Cyan",
				    "Red",   "Magenta", "Brown", "Grey" };
	  static char* intense[2] = { "", " (intense)" };
	  int col = video->palette & 0x0f;
	  int brt = col >> 3;
	  col &= 7;
	  printf( "%s%s border.\n", color[col], intense[brt] );
	}
      }
    }

    else
    {
      --lines;
      --page;
      if ((font || fontfile) && mode < 0 && cols < 0 && attr < 0 && page < 0)
	save = 1;
      if (save)
      {
	int len = video->size;
	buffer = malloc( len );
#if USE_ASM
	asm {
	  push ds
	  push si
	  push di
	  mov  si,screen
	  mov  di,buffer
	  mov  cx,len
	  shr  cx,1
	  push ds
	  pop  es
	  push 0xb800
	  pop  ds
	  rep  movsw
	  pop  di
	  pop  si
	  pop  ds
	}
#else
	movedata( 0xb800, screen, _DS, (unsigned)buffer, len );
#endif
      }

      if (mode >= 0)
      {
	if (lines >= 0 && lines <= 2) set_scan( lines );
	asm {
	  mov  ax,mode
	  or   ah,ah
	  jz   SetMode
	  mov  bx,0x4f02
	  xchg ax,bx
	}
	SetMode: asm int 0x10;
	if (_AH != 0)
	{
	  if (!quiet)
	    fprintf( stderr, "VESA mode %04X not supported.\n", mode );
	  return -1;
	}
	if (lines == 3)
	{
	  set_scan( lines );
	  if (!font) font = 16;
	}
      }

      else if (cols >= 0 || rows >= 0)
      {
	if (cols <= 0) cols = ocols;
	if (rows <= 0) rows = orows;
	if (lines < 0) lines = 2 - (rows == 43);

	if (!(cols == 40 || cols == 80 || cols == 90))
	{
	  fprintf( stderr, "Unsupported columns: %d.\n", cols );
	  return -1;
	}
	if (!font && !fontfile)
	{
	  switch (rows)
	  {
	    case 12: lines = 0; font = 16; break;
	    case 13: lines = 0; font = 15; break;
	    case 14: lines = 0; font = 14; break;
	    case 15: lines = 0; font = 13; break;
	    case 16: lines = 0; font = 12; break;
	    case 18: lines = 0; font = 11; break;
	    case 20: lines = 0; font = 10; break;
	    case 21: lines = 1; font = 16; break;
	    case 22: lines = 0; font =	9; break;
	    case 23: lines = 1; font = 15; break;

	    case 25:	  if (lines == 2) font = 16;
		     else if (lines == 1) font = 14;
		     else if (lines == 0) font =  8;
		     break;

	    case 26:	  if (lines == 2) font = 16;
		     else if (lines == 1) font = 13;
		     break;

	    case 28:	  if (lines == 2) font = 14;
		     else if (lines == 0) font =  7;
		     break;

	    case 29: lines = 1; font = 12; break;

	    case 30:	  if (lines == 2) font = 13;
		     else if (lines == 3) font = 16;
		     break;

	    case 31: lines = 1; font = 11; break;
	    case 32: lines = 3; font = 15; break;

	    case 33:	  if (lines == 2) font = 12;
		     else if (lines == 0) font =  6;
		     break;

	    case 34: lines = 3; font = 14; break;
	    case 35: lines = 1; font = 10; break;

	    case 36:	  if (lines == 2) font = 11;
		     else if (lines == 3) font = 13;
		     break;

	    case 38: lines = 1; font =	9; break;

	    case 40:	  if (lines == 2) font = 10;
		     else if (lines == 3) font = 12;
		     break;

	    case 43:	  if (lines == 1) font =  8;
		     else if (lines == 3) font = 11;
		     break;

	    case 44: lines = 2; font =	9; break;
	    case 48: lines = 3; font = 10; break;

	    case 50:	  if (lines == 2) font =  8;
		     else if (lines == 1) font =  7;
		     break;

	    case 53: lines = 3; font =	9; break;
	    case 57: lines = 2; font =	7; break;
	    case 58: lines = 1; font =	6; break;
	    case 60: lines = 3; font =	8; break;
	    case 66: lines = 2; font =	6; break;
	    case 68: lines = 3; font =	7; break;
	    case 80: lines = 3; font =	6; break;

	    default: fprintf( stderr, "Unsupported rows: %d.\n", rows );
		     return -1;
	  }
	  if (!font)
	  {
	    fprintf( stderr, "Unsupported lines: %d.\n", lines+1 );
	    return -1;
	  }
	}
	if (lines == 3)
	{
	  set_cols( cols );
	  set_scan( lines );
	}
	else
	{
	  set_scan( lines );
	  set_cols( cols );
	}
      }

      if (font || fontfile)
      {
	asm mov ax,0x0500;
	asm int 0x10;
	set_font( font, fontfile );
      }
      max_pages = 0x8000u / video->size;
      if (max_pages > 8) max_pages = 8;
      if (page >= 0)
      {
	if (page < max_pages) asm {
	  mov ah,5
	  mov al,byte ptr page
	  int 0x10
	}
	else
	{
	  fprintf( stderr, "Only %d page%s available: %d\n",
		   max_pages, "s" + (max_pages == 1), page+1 );
	  return -1;
	}
      }

      if (save)
      {
	cols = video->cols;
	rows = video->rows + 1;
	page = video->page;

	lines = (cur.y < rows) ? 0 : cur.y + 1 - rows;
	screen_cpy( buffer, ocols, orows, cols, rows, lines );

#if USE_ASM
	asm {
	  mov al,byte ptr cols
	  dec ax
	  mov dl,byte ptr cur.x
	  cmp dl,al
	  jle XOkay
	  mov dl,al
	}
	XOkay: asm {
	  mov al,byte ptr rows
	  dec ax
	  mov dh,byte ptr cur.y
	  cmp dh,al
	  jle YOkay
	  mov dh,al
	}
	YOkay: asm {
	  mov ah,2
	  mov bh,byte ptr page
	  int 0x10
	}
#else
	if (cur.x > cols-1) cur.x = cols-1;
	if (cur.y > rows-1) cur.y = rows-1;
	set_curs( cur.x, cur.y, page );
#endif
      }
      else if (attr >= 0)
      {
	if (mode >= 0 || cols >= 0) asm {	// Blank entire video memory
	  push 0xb800
	  pop  es
	  xor  di,di
	  mov  cx,0x4000
	  mov  al,' '
	  mov  ah,byte ptr attr
	  rep  stosw
	}
	else asm {				// Blank only current page
	  push di
	  xor  ax,ax
	  mov  es,ax
	  mov  bh,es:[0x0462] // Page
	  mov  di,es:[0x044e] // Offset
	  mov  cx,es:[0x044c] // Size
	  shr  cx,1
	  push 0xb800
	  pop  es
	  mov  al,' '
	  mov  ah,byte ptr attr
	  rep  stosw
	  mov  ah,2
	  xor  dx,dx
	  int  0x10
	  pop  di
	}
      }
    }
    if (brdr >= 0) asm {
      mov ah,0x0b
      mov bh,0
      mov bl,byte ptr brdr
      int 0x10
    }
  }

  if (wfont >= 0) write_font( wfont );

  return rc;
}


void set_cols( int cols )
{
  switch (cols)
  {
    case 40: cols = 1; break;
    case 80: cols = 3; break;
    case 90:
    {
      static unsigned VGAdata[] =
      {
	0x67, 			// Clocking mode
	0x0101,0x0033,		// SC, AC data
	0x0E11,0x6B00,0x5901, 	// CRTC data
	0x5A02,0x8E03,0x5F04,
	0x8D05,0x8E11,0x2D13
      };
      asm {
	mov ax,3
	int 0x10
	mov si,offset VGAdata
	mov dx,0x3C2
	outsb
	inc si
	mov dl,0xC4
	outsw
	mov dl,0xDA
	in  al,dx
	mov dl,0xC0
	outsb
	outsb
	mov dl,0xD4
	mov cx,9
	rep outsw

	push ds
	xor  ax,ax
	mov  ds,ax
	mov  [byte ptr 0x44A],90
	pop  ds
      }
    }
    // fall through

    default: cols = 0;
  }
  asm {
    mov ax,cols
    or	ax,ax
    jz	GetOut
    int 0x10
  }
GetOut:
}


void set_scan( int lines )
{
  if (lines == 3)
  {
    static unsigned CRTCdata[] =
    {
      0x0C11,	// De-protect regs
      0x0B06,	// Vertical Total
      0x3E07,	// Vertical Overflow 1
      0x4F09,	// Vertical Overflow 2
      0xEA10,	// Vertical Retrace Start
      0x8C11,	// Vertical Retrace End & Re-protect
      0xDF12,	// Vertical Displayed
      0xE715,	// Vertical Blanking Start
      0x0416	// Vertical Blanking End
    };

    asm {
      mov dx,0x3C4		// Sequencer Port
      mov ax,0x100		// Synchronous reset
      out dx,ax 		// Send command
      mov dx,0x3CC		// Misc Output Read Port
      in  al,dx			// Read byte
      or  al,0xC0		// Set 480 scanlines
      mov dx,0x3C2		// Misc Output Write Port
      out dx,al 		// Send command
      mov dx,0x3D4		// CRT Controller Port
      mov si,offset CRTCdata	// Offset of CRTC data
      mov cx,9			// 9 words
      rep outsw 		// Send data
      mov dx,0x3C4		// Sequencer Port
      mov ax,0x300		// Restart Sequencer
      out dx,ax 		// Send command
    }
  }
  else asm {
    mov ah,0x12
    mov al,byte ptr lines
    mov bl,0x30
    int 0x10
  }
}


void set_font( int font, char* file )
{
  FILE* fp;
  Byte* buffer = NULL;
  long	len;

  if (file != NULL && *file != '\0')
  {
    fp = fopen( file, "rb" );
    len = filelength( fileno( fp ) );
    if (len < 256 || len > 8192 || (Byte)len != 0)
    {
      fprintf( stderr, "\"%s\" is not a valid font file.\n", file );
      exit( -1 );
    }
    font = (int)len >> 8;
    buffer = malloc( 256 * font );
    fread( buffer, font, 256, fp );
    fclose( fp );
  }
  else switch (font)
  {
    case 16: font   = 0x14; break;
    case 14: font   = 0x11; break;
    case  8: font   = 0x12; break;
    default: buffer = get_font( font );
	     if (buffer == NULL) font = 0;
  }
  asm {
    push bp
    mov  al,byte ptr font
    or	 al,al
    jz	 GetOut
    mov  bp,buffer
    or	 bp,bp
    jz	 BIOSFont

    push ds
    pop  es
    mov  bh,al
    xor  dx,dx
    mov  cx,256
    mov  al,0x10
  }
  BIOSFont: asm {
    mov  ah,0x11
    mov  bl,0
    int  0x10
  }
  GetOut: asm pop bp;

  if (buffer != NULL && font != 6 && font != 12) free( buffer );
}


Byte* get_font( int font )
{
  Byte* buffer;

  switch (font)
  {
    case 16: buffer = get_BIOS_font( 16 ); break;
    case 15: buffer = expand_font( 14, 1, get_BIOS_font( 14 ) ); break;
    case 14: buffer = get_BIOS_font( 14 ); break;
    case 13: buffer = expand_font( 12, 1, font12 ); break;
    case 12: buffer = font12; break;
    case 11: buffer = expand_font(  8, 3, get_BIOS_font( 8 ) ); break;
    case 10: buffer = expand_font(  8, 2, get_BIOS_font( 8 ) ); break;
    case  9: buffer = expand_font(  8, 1, get_BIOS_font( 8 ) ); break;
    case  8: buffer = get_BIOS_font( 8 ); break;
    case  7: buffer = expand_font(  6, 1, font6 ); break;
    case  6: buffer = font6; break;
    default: buffer = NULL;
  }
  return buffer;
}


// font should only be 8, 14 or 16. Any other value will be bad.
Byte* get_BIOS_font( int font )
{
  static Byte store[16 * 256];

  asm {
    push ds
    push si
    push di
    mov  al,byte ptr font
    cmp  al,14
    mov  bh,2
    je	 GetFont
    mov  bh,3
    jb	 GetFont
    mov  bh,6
  }
  GetFont: asm {
    push bp
    mov  ax,0x1130
    int  0x10
    mov  si,bp
    pop  bp
    push es
    pop  ds
    push ss
    pop  es
    mov  di,offset store

    mov  cx,font	// My Rage IIc returns the same pointer for 14- and 16-
    cmp  cl,14		// line fonts. 112 is the start of character 7 for 16,
    jne  FontOkay	// and character 8 for 14. Character 7 should start
    cmp  [si+112],ch	// with a blank line, character 8 with a solid.
    jnz	 FontOkay	// It's a solid line, so it really is the 14-line font.
    mov  ax,256
  }
  Loop: asm {
    inc  si
    mov  cl,7
    rep  movsw
    inc  si
    dec  ax
    jnz  Loop
    jmp  short Done
  }
  FontOkay: asm {
    shr  cl,1
    xchg ch,cl
    rep  movsw
  }
  Done: asm {
    pop  di
    pop  si
    pop  ds
  }

  return store;
}


// Increase the size of a font by adding a line at the bottom, a line at the
// top and bottom, or a line at the top and two at the bottom.
Byte* expand_font( int size, int add, Byte* font )
{
  Byte* buffer;
#if !USE_ASM
  int j, top, bot, last = size - 1;
  Byte* start;
#endif

  buffer = malloc( (size + add) * 256 );

#if USE_ASM
    asm {
      push si
      push di
      push ds
      pop  es
      mov  di,buffer
      mov  si,font
      mov  bx,size
    } _DX = add;        // DH = 0, DL = add
    Loop: asm {
      xor  ax,ax
      cmp  dh,8
      je   Graphic
      cmp  dh,10
      je   Graphic
      cmp  dh,0xb0
      jb   NotGraphic
      cmp  dh,0xe0
      jb   Graphic
      cmp  dh,244
      je   BotOnly
      cmp  dh,245
      jne  NotGraphic
      mov  al,[si+0]
      jmp  short NotGraphic
    }
    Graphic: asm mov al,[si+0]
    BotOnly: asm {
      mov  ah,[si+bx-1]
      cmp  dh,0xb2
      ja   NotGraphic
      xchg al,ah		// Assume the top and bottom of 8 and 10
    }				// are equal (should be solid lines)
    NotGraphic: asm {
      cmp  dl,2
      jb   SkipTop
      stosb
    }
    SkipTop: asm {
      mov  cx,bx
      rep  movsb
      xchg al,ah
      stosb
      jbe  SkipThird
      cmp  dh,0xb0
      jb   Keep
      cmp  dh,0xb2
      ja   Keep
      xchg al,ah
    }
    Keep: asm stosb
    SkipThird: asm {
      inc  dh
      jnz  Loop
      pop  di
      pop  si
    }
#else
  start = buffer;
  for (j = 0; j < 256; ++j)
  {
    top = bot = 0;
    if (j == 8 || j == 10 ||	      // Inverse of 7 and 9
	(j >= 179 && j <= 223))       // Graphics characters
    {
      top = font[0];
      bot = font[last];
    }
    else if (j >= 176 && j <= 178)    // Stipples
    {
      top = font[last];
      bot = font[0];
    }
    else if (j == 244) bot = font[last]; // Top half of integral
    else if (j == 245) top = font[0];	 // Bottom half of integral

    if (add > 1) *(buffer++) = top;
    memcpy( buffer, font, size );
    font   += size;
    buffer += size;
    *(buffer++) = bot;
    if (add == 3)
    {
      if (j >= 176 && j <= 178) bot = top;
      *(buffer++) = bot;
    }
  }
  buffer = start;
#endif

  return buffer;
}


void write_font( int font )
{
  Byte* buffer;
  FILE* file;
  char  name[] = "tmfont.f??";
  int   start, end;

  if (font == 0) start = 6, end = 16;
  else start = end = font;
  for (font = start; font <= end; ++font)
  {
    buffer = get_font( font );
    sprintf( name+8, "%d", font );
    file = fopen( name, "wb" );
    fwrite( buffer, font, 256, file );
    fclose( file );
  }
}


#if !USE_ASM
void set_curs( int x, int y, int page )
{
  asm {
    mov ah,2
    mov bh,byte ptr page
    mov dl,byte ptr x
    mov dh,byte ptr y
    int 0x10
  }
}
#endif


void screen_cpy( Word* buffer, int wb, int hb, int ws, int hs, int r )
{
  Word* screen = (Word*)video->offset;
  Word	left;
  int	w = (ws > wb) ? wb : ws;
  int	h = (hs > hb) ? hb : hs;
  Word	attr = ' ' | (buffer[wb * hb - 1] & 0xff00);
#if !USE_ASM
  Word  blank[(90 - 40)];
  int	j;
#endif

#if USE_ASM
  asm {
    push si
    push di
    mov  si,buffer
    mov  ax,r
    mov  bx,wb
    mul  bx
    shl  ax,1
    add  si,ax			// SI = (Byte*)buffer + r * wb * 2
    push 0xb800
    pop  es
    mov  di,screen		// DI = screen
    mov  ax,w
    sub  bx,ax
    shl  bx,1			// BX = (wb - w) * 2
    mov  dx,ws
    sub  dx,ax			// DX = ws - w
  }
  Loop: asm {
    mov  cx,w
    rep  movsw
    mov  ax,attr
    mov  cx,dx
    rep  stosw
    add  si,bx
    dec  h
    jnz  Loop
    mov  screen,di		// screen = DI
    pop  di
    pop	 si
  }
#else
  for (j = 0; j < ws - wb; ++j) blank[j] = attr;
  w *= 2;
  j *= 2; // j = (ws > wb) ? (ws - wb) * 2 : 0;
  buffer += r * wb;
  for (; h > 0; --h)
  {
    movedata( _DS, (unsigned)buffer, 0xb800, (unsigned)screen, w );
    if (j) movedata( _DS, (unsigned)blank, 0xb800, (unsigned)(screen + wb), j );
    buffer += wb;
    screen += ws;
  }
#endif
  left = video->size - ((unsigned)screen - video->offset);
  asm {
    push di
    push 0xb800
    pop  es
    mov  di,screen
    mov  ax,attr
    mov  cx,left
    shr  cx,1
    rep  stosw
    pop  di
  }
}


void help( char* prog )
{
  int len = -1;
  while (prog[++len]) ;

  printf(
  "Text Mode by Jason Hood <jadoxa@yahoo.com.au>.\n"
  "Version "VERSION" ("DATE"). Freeware.\n"
  "http://tm.adoxa.cjb.net/\n"
  "\n"
  "Usage: %s <cols>[x<rows>] -m[<mode>] ,<lines>\n"
  "       %*c -c[<cols>] -r[<rows>] -f[<height|file>]\n"
  "       %*c -p[<page>] -s[<attr>] -b[<border>] -d\n"
  "       %*c -w[<height>]\n"
  "\n"
  "cols  can be one of 40, 80 or 90\n"
  "rows  ranges from 12 to 80\n"
  "-m    uses BIOS to set a mode (in hexadecimal)\n"
  "lines sets the scan lines to use: 1 to 4 = 200, 350, 400 or 480\n"
  "-f    sets the font height (6 to 16) or loads a font from file\n"
  "-p    will select a page (1 to, at most, 8)\n"
  "-s    will save screen contents or clear screen to attr\n"
  "-b    will set the border/frame/overscan color\n"
  "-d    displays current columns, rows and font height\n"
  "-w    will write all fonts or a particular height to file\n"
  "\n"
  "Missing values will display the current value, unless -q is added.\n"
  "Errorlevel is set to the last used (not the last displayed).\n"
  , prog, len, ' ', len, ' ', len, ' ' );
}
