/* --------------------------------- oplane.c ------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Object description: shape and behaviour.
 * Various planes (then see oclassic.c, obasic.c, oxplane.c).
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fly.h"
#include "plane.h"
#include "hud.h"

/* obasic.c */
extern void	FAR dynamics_basic (OBJECT *p, int interval);

/* oclassic.c */
extern void	FAR dynamics_classic (OBJECT *p, int interval);

/* oxplane.c */
extern void	FAR dynamics_xplane (OBJECT *p, int interval);

/* autop.c */
extern void	FAR SetKillCorrection (OBJECT *p, OBJECT *target, VECT R,
	VECT A, int *tti);
extern void	FAR dynamics_auto (OBJECT *p, int interval);

static void (FAR *flight_models[])(OBJECT *p, int interval) = {
	dynamics_basic,
	dynamics_classic,
	dynamics_xplane
};

#define	RADAR_RANGE	10000
#define	MINSHOOT	(2000L*VONE)
#define	PDAMAGE		10

static struct e_parms *parms = 0;

static SHAPE shape_plane = {
	0,
	0,
	SH_BEHIT|SH_G,
	10000*1000L,	/* weight */
	0		/* drag */
};

static struct e_parms * FAR
get_plane (char *pname)
{
	struct e_parms	*parm;
	int	i, lineno;
	long	t;
	FILE	*pfile;
	char	line[256];

	if (!pname)
		return (parms);		/* use default */

	for (parm = parms; parm; parm = parm->next)
		if (!stricmp (parm->name, pname))
			return (parm);

	strcpy (st.filename, st.fdir);
	strcat (st.filename, DIRSEP);
	strcat (st.filename, pname);
	strcat (st.filename, ".prm");
	if (!(pfile = fopen (st.filename, RTMODE))) {
		LogPrintf ("missing parms file: %s\n", st.filename);
		return (0);
	}

	if (!NEW (parm)) {
		LogPrintf ("no memory for parms: %s\n", st.filename);
		return (0);
	}

	lineno = 0;

	strncpy (parm->name, pname, sizeof (parm->name));

#define READPARM(c,p)	break; \
		case c: parm->p = t

	for (i = 0; i >= 0; ++i) {
		do {
			fgets (line, sizeof (line), pfile);
			if (ferror (pfile)) {
				LogPrintf ("%s %d: plane read error\n",
					st.filename, lineno);
				goto badret;
			}
			++lineno;
		} while ('\n' == line[0] || '#' == line[0]);

		if (1 != sscanf (line, "%ld", &t)) {
			LogPrintf ("%s %d: bad plane data\n",
				st.filename, lineno);
			goto badret;
		}

		switch (i) {
		default:
			/* should never happen! */
		READPARM ( 0, wing_area);
		READPARM ( 1, wing_span);
		READPARM ( 2, weight);
		READPARM ( 3, mil_thrust);
		READPARM ( 4, mil_sfc);
		READPARM ( 5, ab_thrust);
		READPARM ( 6, ab_sfc);
		READPARM ( 7, liftoff_speed);
		READPARM ( 8, max_lift);
		READPARM ( 9, min_lift);
		READPARM (10, pitch_rate);
		READPARM (11, roll_rate);
		READPARM (12, max_flaps);
		READPARM (13, max_spoilers);
		READPARM (14, fuel_capacity);
		READPARM (15, efficiency_factor);
		READPARM (16, height);
		READPARM (17, gpitch);
		READPARM (18, ceiling);
		READPARM (19, stores[0]);
		READPARM (20, stores[1]);
		READPARM (21, stores[2]);
		READPARM (22, stores[3]);
		READPARM (23, stores[4]);

		READPARM (24, BRAKE_MU);	/* friction: brakes applied */
		READPARM (25, WHEEL_MU);	/* friction: freewheeling */
		READPARM (26, Aoffset);		/* Wing rigging ang vs. plane */
		READPARM (27, Toffset);		/* Tail rigging ang vs. wing */
		READPARM (28, Tvol);		/* tail volume */
		READPARM (29, FEff);		/* degs eff alpha/degs flaps */
		READPARM (30, Cl0);		/* alpha where Cl=0 */
		READPARM (31, MAXLIFT);		/* max Cl for wing foil */
		READPARM (32, Cdp0);		/* parasitic: profile */
		READPARM (33, Cds);		/* parasitic: speed brakes */
		READPARM (34, Cdg);		/* parasitic: gear */
		READPARM (35, CdMK82);		/* parasitic: each MK82 */

		READPARM (36, MaxElevators);
		READPARM (37, MaxAilerons);

		READPARM (38, FXrudder);	/* rudder sideforce */
		READPARM (39, FXvx);		/* vx damping */

		READPARM (40, MXelevators);	/* elevators effectiveness */
		READPARM (41, MXdx);		/* pitch damping */
		READPARM (42, MXtail);		/* stabilizer induced pitch */
		READPARM (43, MXlift);		/* lift induced pitch */
		READPARM (44, MXdrag);		/* drag induced pitch */
		READPARM (45, MXthrust);	/* thrust induced pitch */

		READPARM (46, MYailerons);	/* aileron effectiveness */
		READPARM (47, MYdy);		/* roll damping */
		READPARM (48, MYvx);		/* dihedral effect */

		READPARM (49, MZrudder);	/* rudder effectiveness */
		READPARM (50, MZdz);		/* yaw damping */
		READPARM (51, MZvx);		/* weathercock stability */
		READPARM (52, MZailerons);	/* ailerons induced yaw */
		READPARM (53, MZdy);		/* roll induced yaw */

		READPARM (54, hudtype);

		READPARM (55, opt[0]);		/* options */
		READPARM (56, opt[1]);		/* options */
		READPARM (57, opt[2]);		/* options */
		READPARM (58, opt[3]);		/* options */
		READPARM (59, opt[4]);		/* options */
		READPARM (60, opt[5]);		/* options */
		READPARM (61, opt[6]);		/* options */
		READPARM (62, opt[7]);		/* options */
		READPARM (63, opt[8]);		/* options */
		READPARM (64, opt[9]);		/* options */

		READPARM (65, PIDthrottle.Kp);
		READPARM (66, PIDthrottle.Iband);
		READPARM (67, PIDthrottle.Ki);
		READPARM (68, PIDthrottle.Dband);
		READPARM (69, PIDthrottle.Kd);
		READPARM (70, PIDthrottle.factor);
		READPARM (71, PIDthrottle.range);

		READPARM (72, PIDpitch.Kp);
		READPARM (73, PIDpitch.Iband);
		READPARM (74, PIDpitch.Ki);
		READPARM (75, PIDpitch.Dband);
		READPARM (76, PIDpitch.Kd);
		READPARM (77, PIDpitch.factor);
		READPARM (78, PIDpitch.range);

		READPARM (79, PIDroll.Kp);
		READPARM (80, PIDroll.Iband);
		READPARM (81, PIDroll.Ki);
		READPARM (82, PIDroll.Dband);
		READPARM (83, PIDroll.Kd);
		READPARM (84, PIDroll.factor);
		READPARM (85, PIDroll.range);

			i = -10;		/* force termination */
			break;
		}
	}

	fclose (pfile);

	if (parms) {
		parm->next = parms->next;	/* keep first as default */
		parms->next = parm;
	} else {
		parm->next = 0;
		parms = parm;
	}

	if ((sizeof(flight_models)/sizeof(*flight_models)) < parm->opt[0])
		parm->opt[0] = 0;		/* model=basic */

	if (0 == parm->opt[1])
		parm->opt[1] = 1;		/* response=stiff */

	if (0 == parm->opt[3]) {
		if (1 == parm->opt[0])		/* classic */
			parm->opt[3] = 4;	/* turn rate=average */
		else if (0 == parm->opt[0])	/* basic */
			parm->opt[3] = 8;	/* lift rate=standard */
	}

	LogPrintf ("plane: %s\n", st.filename);
	LogPrintf (" model %d\n", parm->opt[0]);

	return (parm);

badret:
	DEL (parm);
	fclose (pfile);
	return (0);
}

static int FAR
init_plane (BODY *b)
{
	if (shape_read (&shape_plane, "plane.vxx"))
		return (1);
	parms = 0;
	return (0);
}

static void FAR
term_plane (BODY *b)
{
	struct e_parms	*p, *pp;

	shape_free (shape_plane.v);

	for (p = parms; p; p = pp) {
		pp = p->next;
		DEL (p);
	}
	parms = 0;
}

extern void FAR
supply (OBJECT *p)
{
	E_PLANE	*e;

	e = EE(p);

	memcpy (e->stores, e->parms->stores, sizeof (e->stores));
	e->fuel = e->parms->fuel_capacity*100L;	/* full tank */
	p->damage = PDAMAGE;
}

static void FAR
free_plane (OBJECT *p)
{
	if (ET_PLANE == p->e_type) {
		DEL0 (EE(p)->PIDthrottle);
		DEL0 (EE(p)->PIDpitch);
		DEL0 (EE(p)->PIDroll);
		DEL0 (EE(p));
		p->e_type = 0;
	}
}

static int FAR
create_plane (OBJECT *p)
{
	E_PLANE	*e;

	if (!NEW (e))
		return (1);
	p->e_type = ET_PLANE;
	EE(p) = e;

	if (!NEW (e->PIDthrottle) ||
	    !NEW (e->PIDpitch) ||
	    !NEW (e->PIDroll)) {
	    	free_plane (p);
	    	return (1);
	}

	e->parms = get_plane (st.options);
	if (!e->parms) {
	    	free_plane (p);
		return (1);
	}

	memcpy (e->PIDthrottle, &e->parms->PIDthrottle, sizeof (PID));
	memcpy (e->PIDpitch,    &e->parms->PIDpitch,    sizeof (PID));
	memcpy (e->PIDroll,     &e->parms->PIDroll,     sizeof (PID));

	e->flags |= PF_ONGROUND;

	e->hud |= HUD_ON;
	e->hud1 |= e->parms->hudtype;

	supply (p);
	hud_setup (p);

	e->equip |= EQ_GEAR | EQ_GNDBRAKE;
	e->radar = 3*R_MODE;

	p->a[X] = e->parms->gpitch;
	Mident (p->T);
	Mrotx (p->T, p->a[X]);
	p->color = st.white;
	p->time = FOREVER;
	p->damaging = 5;
	p->flags |= F_VISIBLE | F_EXPORTED | F_MAINT | F_KEEPNAV;

	CP->eyez = fmul (e->parms->height, 5000)*VONE;	/* ft -> m */

	p->misc[1] =  149*60+10;
	p->misc[3] = -(35*60+20);
	return (0);
}

static void FAR
delete_plane (OBJECT *p)
{
	free_plane (p);
	p->owner = 0;
	p->ownerid = 0;
}

extern void FAR
CCnote (OBJECT *p, char *note)
{
	if (CC == p)
		MsgPrintf (50, note);
}

extern void FAR
CCland (OBJECT *p)
{
	if (CC == p) {
		CCnote (p, "touchdown");
		if (st.quiet)
			Snd->List (TnLand, 0);
	}
}

extern void FAR
CCfly (OBJECT *p)
{
	if (CC == p) {
		CCnote (p, "takeoff");
		if (st.quiet)
			Snd->List (TnFly, 0);
	}
}

extern void FAR
dampen (short *old, int new, int factor)
{
	int	diff;

	diff = new - *old;
	if (diff > factor || diff < -factor)
		diff /= factor;
	else if (diff > 0)
		diff = 1;
	else if (diff < 0)
		diff = -1;
	*old += diff;
}

static void FAR
ShootCue (OBJECT *p, int interval, OBJECT *target)
{
	VECT	V, R, RR;
	ANGLE	perr, err;
	int	t, limit, dist;
	LVECT	IP;

	if (WE_MK82 == EX->weapon) {
		BombSpeed (p, V);
		BombIP (p->R, V, target->R[Z], IP);
		if (est_dist (target->R, IP) < (Uint)SH(target)->extent)
			EX->radar |= R_SHOOT;
		return;
	}

	SetKillCorrection (p, target, R, V /* dummy */, &t /*dummy */);

	R[X] = (int)((target->R[X] + R[X] - p->R[X])/VONE);
	R[Y] = (int)((target->R[Y] + R[Y] - p->R[Y])/VONE);
	R[Z] = (int)((target->R[Z] + R[Z] - p->R[Z])/VONE);
	Mxpose (p->T);
	VMmul (RR, R, p->T);
	Mxpose (p->T);

	if (RR[Y] > 0) {
		dist = ihypot3d (RR);

		if (dist < MINSHOOT/VONE) {
			t = ihypot2d (RR[Z], RR[X]);
			perr = ATAN (t, RR[Y]);
			limit =  SH(target)->extent;
			err = ATAN (limit/4, RR[Y]) + BULLETSCATTER/2;
			if (perr < err)
				EX->radar |= R_SHOOT;
		}
	}
}

extern void FAR
do_radar (OBJECT *p, int interval)
{
	int	radar, i, dist, x, t, fref, shifty;
	OBJECT	*pp, *target;
	VECT	R, RR;
	MAT	T;

	radar = EX->radar;

	if (!(radar & R_ON) || !EX->weapon) {
		EX->target = 0;
		return;
	}
/*
 * target selection
*/

#define SEL3D	(D90/54)		/* 3.3 degrees wide circle */
#define SEL5DX	(D90/34)		/* 5.3 degrees wide */
#define SEL5DY	(D90/3*2)		/* 60 degrees high */

	if (!(radar & R_LOCK) ||		/* have target? */
	    !(target = EX->target) || target->id != EX->tid ||
	    (target->flags & (F_HIT|F_STEALTH)) ||
	    est_dist (p->R, target->R) > RADAR_RANGE) {

		target = 0;			/* no, seek new one */
		dist = RADAR_RANGE;

		if (radar & R_SELECT3) {
			fref = SEL3D;
			shifty = 0;
		} else if (radar & R_SELECT20) {
			fref = DEG2ANG (EX->hudarea);
			shifty = fmul (EX->misc[17], fref);
		} else if (radar & R_SELECT5) {
			fref = -1;
			shifty = fmul (EX->misc[17], DEG2ANG (EX->hudarea));
		} else {
			fref = 0;
			shifty = 0;
		}

		Mcopy (T, p->T);
		Mxpose (T);
		Mrotx (T, shifty);

		for (pp = CO; pp; pp = pp->next) {
			if (!(pp->shflags & SH_BEHIT)
			    || (pp->flags & (F_HIT|F_STEALTH))
			    || ((radar & R_INTELCC) && !(pp->flags & F_CC))
			    || pp == p)
				continue;
			if (WE_MK82 == EX->weapon && pp->R[Z] != 0L)
				continue;
			x = est_dist (p->R, pp->R);
			if (fref && x < dist) {
				R[X] = (int)((pp->R[X]-p->R[X])/VONE);
				R[Y] = (int)((pp->R[Y]-p->R[Y])/VONE);
				R[Z] = (int)((pp->R[Z]-p->R[Z])/VONE);
				VMmul (RR, R, T);
				if (RR[Y] > 0) {
					if (fref < 0) {
						t = ATAN (RR[X], RR[Y]);
						if (iabs (t) > SEL5DX)
							continue;
						t = ATAN (RR[Z], RR[Y]);
						if (t < 0 || t > SEL5DY)
							continue;
					} else {
						t = ihypot2d (RR[X], RR[Z]);
						if (ATAN (t, RR[Y]) > fref)
							continue;
					}
				} else
					continue;
			}
			if (x < dist) {
				dist = x;
				target = pp;
			}
		}
		if (!target) {				/* no new target */
			EX->target = 0;
			target = 0;
			return;
		}
	}
/*
 * target tracking data
*/
	if (EX->target != target) {			/* new target */
		EX->target = target;
		EX->tid = target->id;
		Vcopy (EX->tvnew, target->V);
		Vcopy (EX->tvold, target->V);
		EX->tvhist[0][X] = target->V[X]/TVHIST;
		EX->tvhist[0][Y] = target->V[Y]/TVHIST;
		EX->tvhist[0][Z] = target->V[Z]/TVHIST;
		for (i = 1; i < TVHIST; ++i)
			Vcopy (EX->tvhist[i], EX->tvhist[0]);
		EX->itvhist = 0;
		NEWTGT(p) = 11;		/* odd will show on first frame */

		if (EX->PIDthrottle) {
			EX->PIDthrottle->I = 0L;
			EX->PIDthrottle->Pprev = 0L;
		}
		if (EX->PIDpitch) {
			EX->PIDpitch->I = 0L;
			EX->PIDpitch->Pprev = 0L;
		}
		if (EX->PIDroll) {
			EX->PIDroll->I = 0L;
			EX->PIDroll->Pprev = 0L;
		}

		if (st.quiet)
			Snd->List (TnNotice, 0);
	} else {
		Vcopy (EX->tvold, EX->tvnew);
		i = EX->itvhist;
		EX->itvhist = (i+1) % TVHIST;
		EX->tvnew[X] -= EX->tvhist[i][X];
		EX->tvnew[X] += (EX->tvhist[i][X] = target->V[X]/TVHIST);
		EX->tvnew[Y] -= EX->tvhist[i][Y];
		EX->tvnew[Y] += (EX->tvhist[i][Y] = target->V[Y]/TVHIST);
		EX->tvnew[Z] -= EX->tvhist[i][Z];
		EX->tvnew[Z] += (EX->tvhist[i][Z] = target->V[Z]/TVHIST);
	}
	ShootCue (p, interval, target);
}

extern void FAR
eject (OBJECT *obj)
{
	OBJECT	*p;

	st.owner = obj;
	if ((p = create_object (O_CHUTE, 1))) {
		obj->gpflags &= ~GPF_PILOT;	/* indicate no one home */
		obj->flags |= F_MOD;
		if (obj == CC) {
			p->gpflags |= GPF_PILOT;	/* mark the chute */
			MsgWPrintf (50, "ejected");
			if (!(obj->flags & F_HIT))
				obj->flags |= F_HIT|F_MOD;
		}
		if (obj == CV) {
			save_viewport (CV);
			CV = p;
			get_viewport (CV);
		}
	}
}

extern int FAR
on_runway (OBJECT *p)
/*
 * Check if our object (assuming z=0) is inside any of the polygons on
 * any runway (assuming poly.z=0).
 */
{
	OBJECT	*w;
	long	lx, ly;

	for (w = CL; w; w = w->next) {
		if (w->name != O_RUNWAY)
			continue;
		lx = p->R[X] - w->R[X];
		ly = p->R[Y] - w->R[Y];
#ifdef ACM_RUNWAY
		if (lx < -25*VONE || lx > 25*VONE ||
		    ly < 0*VONE || ly > 2000*VONE)
			return (1);
#else
#define	W	(64*VONE)
#define	L	(1750*VONE)
#define	S	(250*VONE)
		if (labs (lx) < W) {
			if (ly < L && ly > -S)
				return (1);
		} else if (labs (ly) < W) {
			if (lx < L && lx > -S)
				return (1);
		}
#undef	W
#undef	L
#undef	S
#endif
	}
	return (0);
}

#if 0
extern int FAR
old_on_runway (OBJECT *p)
/*
 * Check if our object (assuming z=0) is inside any of the polygons on
 * any runway (assuming poly.z=0).
 */
{
	OBJECT	*w;
	VERTEX	*v;
	SHAPE	*shape;
	int	r[2];
	int	count, t1, t2;
	int	rt, rb, rr, rl, dir;
	int	xxxx, yyyy, oooo;

	shape = st.bodies[O_RUNWAY]->shape;

	for (w = CO; w; w = w->next) {
		if (w->name != O_RUNWAY)
			continue;
		/* r[] = translate p->R[] to runway coords */
		r[X] = (p->R[X] - w->R[X])/VONE;
		r[Y] = (p->R[Y] - w->R[Y])/VONE;
		if (iabs(r[X]) > shape->extent || iabs(r[Y]) > shape->extent)
			return (0);
		if (r[X] > 0) {
			rr = r[X];
			rl = 0;
		} else {
			rl = r[X];
			rr = 0;
		}
		if (r[Y] > 0) {
			rt = r[Y];
			rb = 0;
		} else {
			rb = r[Y];
			rt = 0;
		}
		dir = ((r[X] > 0) == (r[Y] > 0));
		for (v = shape->v;; ++v, yyyy = xxxx) {
			if (v->flags == 0 && (count%2 == 0))	/* end poly */
				return (1);
			xxxx = 0;
			if (v[0].V[X] > rr)
				xxxx |= 0x01;
			if (v[0].V[X] > rl)
				xxxx |= 0x02;
			if (v[0].V[Y] > rt)
				xxxx |= 0x04;
			if (v[0].V[Y] > rb)
				xxxx |= 0x08;

			if (v->flags == V_MOVE) {	/* new poly */
				count = 0;
				continue;
			}
			/* check if r[] crosses v[-1],v[0] */
			if (xxxx&yyyy)				/* reject */
				continue;
			oooo = xxxx|yyyy;
			if (oooo == 0x03 || oooo == 0x0c) {	/* accept */
				++count;
				continue;
			}
			if (dir) {
				if (oooo == 0x09 || oooo == 0x06) /* rej */
					continue;
			} else {
				if (oooo == 0x05 || oooo == 0x0a) /* rej */
					continue;
			}

			t1 = v[-1].V[Y] - muldiv (v[-1].V[X], r[Y], r[X]);
			t2 = muldiv (v[0].V[X]-v[-1].V[X], r[Y], r[X]) -
					v[0].V[Y]-v[-1].V[Y];
			if (t1 < 0 && t2 < 0) {
				t1 = -t1;
				t2 = -t2;
			}
			if (t1 < 0 || t2 < 0 || t1 > t2)
				continue;
		}
	}
	return (0);
}
#endif

extern void FAR
shoot (OBJECT *p, int weapon, int n, int seed, int interval)
{
	OBJECT	*w;
	int	i;

	st.owner = p;
	srand (seed);

    	if (WE_MK82 == weapon) {
		st.owner = p;
		for (i = 0; i < n;  ++i) {
			if (!(w = create_object (O_MK82, 1)))
				break;
			if (ET_PLANE == p->e_type)
				--EE(p)->stores[WE_MK82-1];
			else
				++p->misc[0];
		}
	} else if (WE_M61 == weapon) {
		for (i = 0; i < n;  ++i) {
			if (!(w = create_object (O_M61, 1)))
				break;
			if (ET_PLANE == p->e_type) {
				if (!(EE(p)->stores[WE_M61-1]-- % 8))
					w->flags |= F_VISIBLE;
			} else {
				if (!(p->misc[0]++ % 8))
					w->flags |= F_VISIBLE;
			}
		}
	} else
		i = 0;
	if (i && (p->flags & F_EXPORTED))
		remote_shoot (p, weapon, i, seed, interval);
}

extern int FAR
dynamics_input (OBJECT *p, int interval)
{
	POINTER	*ptr;
	int	i, t;
	OBJECT	*s;

	if (!(ptr = p->pointer) || ET_PLANE != p->e_type)
		return (1);

	if (p->flags & F_HIT) {
		body_dynamics (p, interval);
		return (1);
	}

	if (p->damage < PDAMAGE && (st.flags1 & SF_SMOKE)) {
		for (i = 2*TADJ(PDAMAGE-p->damage); i-- > 0;)
			if ((s = create_object (O_SMOKE, 1))) {
				s->time = 3*TIMEPSEC;
				t = rand () % interval;
				s->R[X] -= muldiv (p->V[X], t, 1000);
				s->R[Y] -= muldiv (p->V[Y], t, 1000);
				s->R[Z] -= muldiv (p->V[Z], t, 1000);
			}
	}

	if (!(p->gpflags & GPF_PILOT))
		return (0);

	do_radar (p, interval);

	if ((EX->flags & PF_AUTO) && !(EX->flags & PF_CHASE))
		dynamics_auto (p, interval);
	else {
		if ((*ptr->control->Read)(ptr)) {
			Gr->SetTextPos (5,1);
			MsgEPrintf (10, "pointer lost!");
			Snd->Beep (440*3, 500);
			return (1);
		}
		if (EX->flags & (PF_CHASE|PF_KILL))
			dynamics_auto (p, interval);
	}

	/*
	 * pointer usage:
	 * a[0] right/left turn
	 * a[1] up/down nose
	 *
	 * b[0]  power up
	 * b[1]  power down
	 * b[2]  level
	 * b[3]  origin
	 * b[4]  power 0
	 * b[5]  fire
	 * b[6]  rudder left
	 * b[7]  rudder right
	 * b[8]  stable
	 * b[9]  rudder center
	 * b[10] flaps
	 * b[11] spoilers
	 * b[12] reset roll
	 * b[13] air brakes
	 * b[14] gnd brakes
	 * b[15] gear
	 * b[16] power 100
	 * b[17] after burner
	 * b[18] release lock
	 *
	 * opt[0] flight model
	 * opt[1] response
	 * opt[2] linear controls
	 * opt[3] turn speed (classic)
	 */

	if (ptr->b[5]) {
		if (WE_MK82 == EE(p)->weapon) {
			if (EX->misc[11] + ptr->b[5] > 4)
				i = 4 - EX->misc[11];
			else
				i = ptr->b[5];
			if (i)
				shoot (p, WE_MK82, i, rand(), interval);
		} else if (WE_M61  == EE(p)->weapon)
			shoot (p, WE_M61, 1+interval/(1000/(3000/60)), rand(),
				interval);
	}

	if (iabs (ptr->a[0] - ptr->l[0]) >= 4)
		ptr->l[0] = ptr->a[0];
	if (iabs (ptr->a[1] - ptr->l[1]) >= 4)
		ptr->l[1] = ptr->a[1];

#define	JOY(n)	((EP->opt[2] || (EX->flags & (PF_AUTO|PF_CHASE))) \
			? ptr->l[n] : lin2log (ptr->l[n]))
	dampen (&EX->ailerons, JOY (0), EP->opt[1]);
	dampen (&EX->elevators, JOY (1), EP->opt[1]);
#undef JOY

	if (ptr->b[4]) {
		EX->throttle = 0;
		EX->afterburner = 0;
	}
	if (ptr->b[16]) {
		EX->throttle = 100;
		EX->afterburner = 0;
	}
	for (;ptr->b[0] > 0; --ptr->b[0]) {
		if (EX->afterburner) {
			EX->afterburner += 4;
			if (EX->afterburner > 100)
				EX->afterburner = 100;
		} else {
			EX->throttle += 1;
			if (EX->throttle > 100)
				EX->throttle = 100;
		}
	}
	for (;ptr->b[1] > 0; --ptr->b[1]) {
		if (EX->afterburner) {
			EX->afterburner -= 4;
			if (EX->afterburner < 0)
				EX->afterburner = 0;
		} else {
			EX->throttle -= 1;
			if (EX->throttle < -100)
				EX->throttle = -100;
		}
	}
	for (;ptr->b[17] > 0; --ptr->b[17]) {
		if (EX->afterburner) {
			EX->afterburner += 20;
			if (EX->afterburner > 100)
				EX->afterburner = 100;
		} else {
			EX->throttle = 100;
			EX->afterburner = 20;
		}
	}

	EX->rudder += (ptr->b[6] - ptr->b[7])*10;
	if (EX->rudder > 70)
		EX->rudder = 70;
	else if (EX->rudder < -70)
		EX->rudder = -70;
	if (ptr->b[9])
		EX->rudder = 0;

	if (ptr->b[12]) {
		p->a[Y] = 0;
	}
	if (ptr->b[2]) {
		p->a[X]  = 0;
		p->a[Y]  = 0;
		p->a[Z]  = 0;
		p->da[X] = 0;
		p->da[Y] = 0;
		p->da[Z] = 0;
		Mident (p->T);
		(*ptr->control->Center)(ptr);
	}

	if (ptr->b[3]) {
		p->R[X] = 0;
		p->R[Y] = 0;
		p->R[Z] = 0;
		EX->throttle = 0;
	}
	if (ptr->b[8])
		(*ptr->control->Center)(ptr);

	while (ptr->b[13] > 0) {
		EX->equip ^= EQ_AIRBRAKE;
		--ptr->b[13];
	}
	while (ptr->b[14] > 0) {
		EX->equip ^= EQ_GNDBRAKE;
		--ptr->b[14];
	}
	if (ptr->b[15] > 0) {
		if (EX->flags & PF_ONGROUND) {
			MsgWPrintf (10, "Gear locked");
		} else {
			while (ptr->b[15] > 0) {
				if (p == CC && st.quiet)
					Snd->List (TnGear, 0);
				EX->equip ^= EQ_GEAR;
				--ptr->b[15];
			}
			if (CC == p)
				MsgPrintf (50, "Gear %s",
					EX->equip&EQ_GEAR ? "down" : "up");
		}
	}

	EX->flaps += 10 * ptr->b[10];
	if (EX->flaps > EP->max_flaps)
		EX->flaps = EP->max_flaps;
	else if (EX->flaps < 0)
		EX->flaps = 0;

	EX->spoilers += 10 * ptr->b[11];
	if (EX->spoilers > EP->max_spoilers)
		EX->spoilers = EP->max_spoilers;
	else if (EX->spoilers < 0)
		EX->spoilers = 0;

	if (ptr->b[18])
		EX->target = 0;
	memset (ptr->b, 0, sizeof (ptr->b));	/* clear all buttons */

	if ((EX->equip & EQ_AIRBRAKE) && EX->airbrake < 100) {
		EX->airbrake += TADJ (100);
		if (EX->airbrake > 100)
			EX->airbrake = 100;
	} else if (!(EX->equip & EQ_AIRBRAKE) && EX->airbrake > 0) {
		EX->airbrake -= TADJ (100);
		if (EX->airbrake < 0)
			EX->airbrake = 0;
	}
	return (0);
}

static void FAR
dynamics_plane (OBJECT *p, int interval)
{
	EX->radar &= ~R_SHOOT;

	(*flight_models[EP->opt[0]]) (p, interval);
}

BODY BoPlane = {
	0,
	0,
	"PLANE",
	&shape_plane,
	init_plane,
	term_plane,
	create_plane,
	delete_plane,
	dynamics_plane,
	body_hit
};
