/*
    CGAEX: CGA video exerciser
    Copyright (C) 2003  John Elliott

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include 	"vga.h"
#include 	<conio.h>
#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>
#include	<sys.h>
#include	<dos.h>
#include	"testcard.h"
#include	"1512gfx.h"

#ifndef SEEK_SET	/* Hi-Tech for some reason don't define it */
#define SEEK_SET 0
#define SEEK_CUR 1
#define SEEK_END 2
#endif

#define MAXOPT 25
void cga_putchar(gfx_word y, gfx_word x, char c);


GFX_MENULINE mainlines[MAXOPT] = 
			{	{"Help", 'H'},
				{"Mode 0", '0'},
				{"Mode 1", '1'},
				{"Mode 2", '2'},
				{"Mode 3", '3'},
				{"Mode 4", '4'},
				{"Mode 5", '5'},
				{"Mode 6", '6'},
				{"160x100", '7'},
                                {"Exit", 'E'},
			};

GFX_MENU mainmenu = {10, 0, "CGA Exerciser", mainlines}; 
char *bbar = "CGA test utility";

static gfx_word oldcursor;
extern gfx_byte gfx_cga_pal;

static int is_pc1512;

void tracebuf(char c, char *s)
{
	char buf[41];
	char *t;
	
	sprintf(buf, "%c %-38.38s", c, s);
	while (t = strchr(buf, '\n')) *t = '_';

	cputs(buf);
	getch();
}

gfx_byte setupscr(void)
{
	gfx_byte mode;
	
	mode = gfx_gmode();
	if (gfx_force_mda)
	{
		oldcursor = gfx_getcursor();
		gfx_setcursor(oldcursor | 0x2000);
	} 
	gfx_setbackground(VGA_GREY87);
	gfx_cls();
	gfx_bbar(bbar);
	return mode;
}                                

void askspace(char *t)
{
	gfx_x = 0; 
	gfx_y = 24;
	gfx_putstr("Press SPACE to continue");
	getch();
	if (t)
	{
		gfx_setbackground(VGA_GREY87);
		gfx_cls();
		gfx_bbar(t);
	}
	gfx_setforeground(VGA_BLACK);
	gfx_setbackground(VGA_WHITE);
}

void showtext(char *s, char *t)
{
	gfx_word x = 0, y = 0;
	int scrw = (gfx_force_mda) ? 80 : 40;
	
	gfx_setbackground(VGA_GREY87);
	gfx_cls();
	gfx_bbar(t);

	gfx_setforeground(VGA_BLACK);
	gfx_setbackground(VGA_GREY87);
	while (*s)
	{
		if (s[0] == 0) break;
		if (s[0] == '\n' || s[0] == '\r' || x == scrw)
		{
			x = 0;
			++y;
			if (y == 22)
			{
				askspace(t);
				y = 0;
			}
		}
		if (s[0] >= ' ') 
		{
			gfx_putchar(y, x, s[0]);
			++x;
		}
		++s;
	}
	askspace(NULL);
}

void show_abuf(char *s, int ci)
{
	char bbuf[161];
	int scrw = (gfx_force_mda) ? 80 : 40;
	
	gfx_x = 0;
	gfx_y = 23;
	gfx_setforeground(VGA_BLACK);
	gfx_setbackground(VGA_GREY87);
	
	if (strlen(s) >= scrw)
	{
		sprintf(bbuf, "%-*.*s", 2*scrw, 2*scrw, s);
		gfx_putstr(bbuf);
		gfx_setbackground(VGA_DBLUE);
		gfx_setforeground(VGA_WHITE);
		if (ci < scrw) gfx_putchar(23, ci,     bbuf[ci]);
		else	       gfx_putchar(24, ci-scrw,bbuf[ci]);
	}
	else
	{
		sprintf(bbuf, "%-*.*s%-*.*s", scrw,scrw," ", scrw,scrw,s);
		gfx_putstr(bbuf);
		gfx_setbackground(VGA_DBLUE);
		gfx_setforeground(VGA_WHITE);
		gfx_putchar(24, ci, bbuf[ci + scrw]);
	}
	gfx_setforeground(VGA_BLACK);
	gfx_setbackground(VGA_GREY87);
	
}

void txt_putchar(gfx_word y, gfx_word x, unsigned w, unsigned char c, unsigned char attr)
{
	gfx_byte far *mem = MK_FP(0xB800, 2 * w * y + 2 * x);
	mem[0] = c;
	mem[1] = attr;
}

void txt_putstr(gfx_word y, gfx_word x, unsigned w, char *s, unsigned char attr)
{
	while (*s)
	{
		txt_putchar(y,x,w,*s,attr);
		++s;
		++x;
		if (x >= w) 
		{ 
			x = 0; 
			++y;
		}
	}
}


void cga_putstr(gfx_word y, gfx_word x, char *s)
{
	while (*s)
	{
		cga_putchar(y,x,*s);
		++s;
		++x;
		if (x >= 40) 
		{ 
			x = 0; 
			++y;
		}
	}
}

void putchar_100(gfx_word y, gfx_word x, unsigned char c, unsigned char attr)
{
	unsigned char far *scr = MK_FP(0xB800, 160*8*y + 8*x);
	unsigned char *data = gfx_font + 8 * c;
	unsigned char bm;

	for (y = 0; y < 8; y++)
	{
		bm = data[y];
		for (x = 0; x < 4; x++)
		{
			*scr = 222;
			++scr;
			*scr = 0;
			if (bm & 0x80) *scr |= (attr << 4); 
			else	       *scr |= (attr & 0xF0);
			if (bm & 0x40) *scr |= (attr & 0xF);
			else	       *scr |= (attr >> 4);
			++scr;
			bm = bm << 2;
		}
		scr += 152;
	}	
	
}

void putstr_100(gfx_word y, gfx_word x, char *s, unsigned char attr)
{
	while (*s)
	{
		putchar_100(y,x,*s, attr);
		++s;
		++x;
		if (x >= 20) 
		{ 
			x = 0; 
			++y;
		}
	}
}


/* TO DO: Write a fast asm version of this */
void putplane(const unsigned char far *plane)
{
	unsigned char far *vram = MK_FP(0xB800, 320);
	int n;

	for (n = 0; n < 16384; n++) vram[n] = plane[n];
}



void gshowstat()
{
	char buf[41];
	unsigned char far *vidctl = MK_FP(0x40, 0x65);

	gfx_setbackground(VGA_BLACK);
	gfx_setforeground(VGA_WHITE);
			    //1...5...10...15...20...25...30...35...40
	sprintf(buf, "Control reg=%02x Colour reg=%02x", vidctl[0], vidctl[1]);
	cga_putstr(0, 0, buf);
}


void showstat()
{
	char buf[41];
	unsigned char far *vidctl = MK_FP(0x40, 0x65);

			    //1...5...10...15...20...25...30...35...40
	sprintf(buf, "Control reg=%02x Colour reg=%02x", vidctl[0], vidctl[1]);
	txt_putstr(0, 0, 40, buf, 0x07);
}


void showstat100()
{
	char buf[21];
	unsigned char far *vidctl = MK_FP(0x40, 0x65);

	sprintf(buf, "Control reg=%02x", vidctl[0]);
	putstr_100(0, 0, buf, 0x07);
	sprintf(buf, "Colour reg= %02x", vidctl[1]);
	putstr_100(1, 0, buf, 0x07);
}



void help(void)
{
		//1...5...10...15...20...25...30...35...40
	showtext("CGAEX allows various modes of a CGA (or\n"
		 "compatible) graphics adaptor to be\n"
		 "tested.\n"
		 "To use CGAEX, choose a mode from the\n"
		 "menu. The mode will be selected and a\n"
		 "test pattern displayed. The values of\n"
		 "the CGA mode control register and \n"
		 "colour select register will also be\n"
		 "shown.\n"
		 "To leave a test screen, press ESC.\n", "CGA Exerciser");
	showtext("You can alter the mode control register\n"
		 "and colour select register, a bit at a\n"
		 "time. Use 0-7 to flip bits in the mode\n"
		 "control register, and a-h to flip bits\n"
		 "in the colour select register.\n"
		 "Bits are:\n"
		 "   Mode control   Colour select\n"
		 "7     unused       unused\n"
		 "6     unused       unused\n"
		 "5     blink        Palette\n"
		 "4     640x200      FG intensity\n"
		 "3     enable CGA   Intensity\n"
		 "2     b/w          Red\n"
                 "1     graphics     Green\n"
                 "0     80 cols      Blue\n", "CGA Exerciser"); 
}

void onfunc(char c)
{
	unsigned char mask;
	unsigned char far *vidctl = MK_FP(0x40, 0x65);
	
	if (c >= '0' && c <= '7')
	{
		mask = 1 << (c - '0');
		vidctl[0] ^= mask;
		outp(0x3D8, vidctl[0]);
	}
	if (c >= 'a' && c <= 'h')
	{
		mask = 1 << (c - 'a');
		vidctl[1] ^= mask;
		outp(0x3D9, vidctl[1]);
	}
}

void extext(gfx_byte mode)
{
	unsigned w;
	int y,x;
	char b[3], c;
	gfx_byte m = gfx_getmode();

	gfx_setmode(mode);
	if (mode < 2)  w = 40; else w = 80;
	for (y = 0; y < 16; y++)
		for (x = 0; x < 16; x++)
		{
			sprintf(b, "%02x", y*16+x);
			txt_putstr(y+6, 2*x, w, b, y*16+x);
		}

	showstat();
	while ((c = getch()) != 0x1B) 
	{
		onfunc(c);
		showstat();
	}
	gfx_setmode(m);
	setupscr();
}

#define SETREG(a,b) outp(0x3D4, a); outp(0x3D5, b)

void ex100()
{
	int y;
	char c;
	gfx_byte m = gfx_getmode();
	unsigned char far *vram = MK_FP(0xB800, 2560);
	unsigned const char *bitmap = testc160;

	gfx_setmode(3);		/* Start in CO80 */
	SETREG(1,  80);		/* 80 cols (should be so already) */
	SETREG(4, 127);		/* Vertical total 127 */
	SETREG(6, 100);		/* Vertical displayed 100 */
	SETREG(7, 112);		/* Vertical sync 112 */
	SETREG(9,   1);		/* Max raster address */

	for (y = 0; y < 16000; y++) vram[y] = bitmap[y];

	onfunc('5');	/* Disable blink */
	showstat100();
	while ((c = getch()) != 0x1B) 
	{
		onfunc(c);
		showstat100();
	}
	gfx_setmode(m);
	setupscr();

}

void exgfx320(unsigned char mode)
{
	char c;
	unsigned const char *bitmap;
	gfx_byte m = gfx_getmode();

	gfx_setmode(mode);
	bitmap = testc320;
	putplane(bitmap);
	gshowstat();
	while ((c = getch()) != 0x1B)
	{
		onfunc(c);
		gshowstat();
	}
	gfx_setmode(m);
	setupscr();
}




void exgfx640()
{
	char c;
	unsigned const char *bitmap;
	gfx_byte m = gfx_getmode();

	gfx_setmode(6);
	bitmap = testc640;

	putplane(bitmap);	
	gshowstat();
	while ((c = getch()) != 0x1B)
	{
		onfunc(c);
		gshowstat();
	}
	gfx_setmode(m);
	setupscr();
	
}


void exgfx1512()
{
	char c;
	unsigned const char *bitmap;

	gfx_mode();
	bitmap = testc640;

	gfx_wplane(1); putplane(testc1512b);
	gfx_wplane(2); putplane(testc1512g);
	gfx_wplane(4); putplane(testc1512r);
	gfx_wplane(8); putplane(testc1512i);
	gfx_wplane(15);
	gshowstat();
	while ((c = getch()) != 0x1B)
	{
		onfunc(c);
		gshowstat();
	}
	gfx_off();
	setupscr();
}





int  main(int argc, char **argv)
{
	gfx_byte mode;
	gfx_word n;
	gfx_word doing;

	is_pc1512 = gfx_init();
	for (n = 1; n < argc; n++)
	{
		if      (!stricmp(argv[n], "/CGA")) gfx_force_cga = 1;
		else if (!stricmp(argv[n], "/CGA0")) { gfx_force_cga = 1; gfx_cga_pal = 0;}
		else if (!stricmp(argv[n], "/CGA1")) { gfx_force_cga = 1; gfx_cga_pal = 1;}
		else if (!stricmp(argv[n], "/CGA2")) { gfx_force_cga = 1; gfx_cga_pal = 2;}
		else if (!stricmp(argv[n], "/MDA")) gfx_force_mda = 1;
		else 
		{
			fprintf(stderr, "Unrecognised option: %s\n", argv[n]);
			return 1;
		}
	}
	
	mode = setupscr();
	gfx_bbar(bbar);
	doing = 1;
	while (doing)
	{	
		n = gfx_domenu(&mainmenu);
		switch(n)
		{
			case 0: help(); continue;
			case 1: case 2: case 3: case 4: extext(n-1); continue;
			case 5: case 6: 		exgfx320(n-1);  continue;
			case 7:	if (is_pc1512) exgfx1512(); 
				else           exgfx640();  
				continue;
			case 8: ex100(); continue;
		}
		break;
	}
	if (gfx_force_mda) gfx_setcursor(oldcursor);
	gfx_setmode(mode);
	return 0;
}

