;=======================================================
;
; Music Quest Programmer's ToolKit
;
; Written for use with small, medium, compact, and large memory models
;
; Requires Turbo Assembler or Microsoft Assember 5.0 or later
;
; Supports
;   Turbo Assembler
;   Microsoft Assembler
;   Turbo C
;   Quick C and Microsoft C
;   Turbo Pascal
;   Quick Basic
;
; Copyright 1987, 1990 by Music Quest, Inc.
; All rights reserved
;
; PR227 10/29/89 - disable when manipulating PIC
; pm	7/30/91 -  restore PIC status on mcc_irq success
;
;=======================================================
;
; Check for valid language OR model specification
;
        DOSSEG
        IFDEF   _SMALL_
        .model  small
        ENDIF
        IFDEF   _MEDIUM_
        .model  medium
        ENDIF
        IFDEF   _COMPACT_
        .model  compact
        ENDIF
        IFDEF   _LARGE_
        .model  large
        ENDIF
        IFDEF   _QBASIC_
        .model  medium
        ENDIF

        IFNDEF  @codesize
        %out    ***ERROR***  Valid language or model name not specified
        .ERR
        END

	ENDIF

        IFDEF   _QBASIC_
;=======================================
; External references
;=======================================
        extrn   _coproc_slih:far
        ENDIF

        .data
;=======================================================
; The buffer size can be changed for specific use
;=======================================================
bsize   equ     1024                    ;MCC receive data buffer size
;=======================================
; Call parameter definitions
;=======================================
        if      @codesize GE 1
p1      equ     6                       ;parameter 1 medium, large, huge
p2      equ     8                       ;parameter 2 medium, large, huge
p3      equ     10                      ;parameter 3 medium, large, huge
        else
p1      equ     4                       ;parameter 1 small model
p2      equ     6                       ;parameter 2 small model
p3      equ     8                       ;parameter 3 small model
        endif
;=======================================================
; I/O definitions
;=======================================================
mccdata dw      0330H                   ;mcc data port
mccstat dw      0331H                   ;mcc status port
mcccmd  dw      0331H                   ;mcc command port
rxrdy   equ     80H                     ;mcc data ready, when 0
txrdy   equ     40H                     ;mcc ready to receive, when 0
$reset  equ     0FFh                    ;reset mcc
$ver    equ     0ACh                    ;request version
$uart   equ     03Fh                    ;mcc into UART mode
$ack    equ     0FEh                    ;mcc acknowledge
piccmd  equ     20h                     ;8259 Priority Interrupt Controller
picmsk  equ     21h                     ;8259 Priority Interrupt Controller
;=======================================================
;       data
;=======================================================
; MCC receive buffer
bhead   dw      0                       ;buffer empty pointer
btail   dw      0                       ;buffer fill  pointer
rca     dw      0                       ;count of received chars available
;=======================================================
; Interrupt control
;=======================================================
install         db      0               ;interrupt handler installed
imask           db      0               ;MCC interrupt mask
oldpic		db	?		;old pic interrupt mask
il_base         equ     8               ;int number for IRQ0
midi_level      db      il_base+2       ;midi interrupt level (IRQ2 default)
intip           dw      0               ;saved intertupt vector (offset)
intcs           dw      0               ;(segment)
                if      @codesize GE 1
slih_rtn        dd     _mcc_noslih      ;default second level int handler
                else
slih_rtn        dw     _mcc_noslih      ;default second level int handler
                endif
;=======================================================
        .data?
bfbeg   db      bsize dup(?)            ;buffer for receive data
bfend   equ     bfbeg+bsize             ;buffer end address
;=======================================================
        .code
;=======================================================
; mcccommand
;
; Send a command (synchronously) to the MCC
;
; Inputs:
;   ah = command
; Outputs:
;   ax = return code
;       0 = operation failed, command not sent
;       1 = command sent
;=======================================================
        public  mcccommand
mcccommand proc
        pushf                   ;save interrupt enable status
;=======================================================
; Wait for MCC to become ready to receive the command
;=======================================================
        mov     dx,mccstat              ;status reg
        mov     cx,-1                   ;loop spin count
mpc1:
        in      al,dx                   ;get status
        test    al,txrdy                ;MCC ready to receive?
        jz      mpc11                   ;it's ready
        loop    mpc1                    ;not ready
        sub     ax,ax                   ;time out, operation failed
        jmp     short mpc9              ;return
;=======================================================
; Send the command
;=======================================================
mpc11:
        cli
        mov     al,ah
        mov     dx,mcccmd
        out     dx,al                   ;send command to mcc
        mov     cx,-1                   ;loop spin count
;=======================================================
; Wait for the command acknowledge to arrive
;=======================================================
mpc2:
        mov     dx,mccstat              ;status reg
        in      al,dx                   ;get status
        test    al,rxrdy                ;in bound byte available?
        jnz     mpc4                    ;no
        mov     dx,mccdata              ;yes, read it
        in      al,dx
        cmp     al,$ack                 ;is it an acknowledge?
        je      mpc3                    ;yes
;=======================================================
; Handle interim bytes via current SLIH
;=======================================================
        push    cx                      ;no
        IF      @codesize
        call    dword ptr slih_rtn      ;pass control to slih
        ELSE
        call    word ptr slih_rtn       ;pass control to slih
        ENDIF
        pop     cx
mpc4:
        loop    mpc2                    ;limit amount of spin, preventing a hang
        sti
        sub     ax,ax                   ;error return code, ack not received
        jmp     short mpc9
mpc3:
        mov     ax,1            ;good return code
;=======================================================
; Exit with return code in AX
;=======================================================
mpc9:
        popf
        ret
mcccommand endp
;=======================================================
; _mcc_command
;
; _mcc_command sends a command to the MCC.
;
; C Model:
;       rc=mcc_command(command)
;               int command;
;               int rc;
;               0 = command not sent
;               1 = command sent
;=======================================================
        public  _mcc_command
_mcc_command proc
        push    bp
        mov     bp,sp
        push    di
        push    si
        mov     ah,[bp+p1]
        call    mcccommand
        pop     si
        pop     di
        pop     bp
        ret
_mcc_command endp
;=======================================================
; _mcc_reset
;
; _mcc_reset sends a reset to the MCC, thus returning the
; MCC to its power-on state.
;
; C Model:
;       _mcc_reset()
;
; Assembler Model:
;       call    _mcc_reset
;=======================================================
        public  _mcc_reset
_mcc_reset proc
        mov     ah,0FFh                 ;reset command code
        call    mcccommand              ;send to MCC
        push    ax                      ;save return code
        mov     dx,mccdata              ;clear data-in reg
        in      al,dx
        pop     ax                      ;restore return code
        ret
_mcc_reset endp
;=======================================================
; _mcc_get
;
; _mcc_get returns a character from the MCC receive
; buffer, provided one is available.
;
; C Model:
;       value=mcc_get();
;             int value;
;             value >= 0 is a valid byte available
;             value == -1 means no byte available
;
; Assembler Model:
;       call    _mcc_get
;       ax == return code or byte
;
;=======================================================
        public  _mcc_get
_mcc_get proc
;=======================================================
; Check for available byte
;=======================================================
        mov     ax,rca                  ;count of chars available
        or      ax,ax
        jnz     mdig0                   ;one ready
        mov     ax,0ffffh               ;set ax to -1 to indicate no char
        jmp     SHORT mdigx
;=======================================================
; Take next byte from receive buffer
;=======================================================
mdig0:
        cli                             ;disable interrupts
        dec     rca                     ;decrement count of chars avail
        mov     bx,bhead                ;buffer empty pointer into bx
        mov     cl,[bx]                 ;get character from buffer
        inc     bx                      ;bump buffer empty pointer
        cmp     bx,offset DGROUP:bfend  ;test for end of buffer
        jb      mdig1                   ;no, continue
        mov     bx,offset DGROUP:bfbeg  ;yes, reset buffer empty pointer
mdig1:
        mov     bhead,bx                ;update buffer empty pointer
;
        sub     ax,ax                   ;make sure ax is a character
        mov     al,cl                   ;return the char
        sti                             ;re-enable interrupt
;=======================================================
; Exit with byte/return code in AX
;=======================================================
mdigx:
        ret
_mcc_get endp
;=======================================================
; _mcc_put
;
; mdiput sends one data byte to the MCC
;
; C Model:
;       rc=mcc_put(byte);
;            int byte;
;                    data byte to be sent to the MCC
;            int rc;
;                    return code
;                    0 = MCC not ready to receive
;                    1 = byte sent
;
; Assembler Model:
;       push    byte
;       call    _mcc_put
;       pop     cx
;       ax == return code
;
;=======================================================
        public  _mcc_put
_mcc_put proc
        push    bp
        mov     bp,sp
;=======================================================
; Wait for MCC to post "ready to receive"
;=======================================================
        mov     dx,mccstat              ;status reg
        mov     cx,-1                   ;loop limiter
mp1:
        in      al,dx                   ;get status
        test    al,txrdy                ;ready?
        jz      mp2                     ;yes
        loop    mp1                     ;no
;
        sub     ax,ax                   ;rc=0, not ready
        jmp     SHORT mpxt
;=======================================================
; Send byte to MCC
;=======================================================
mp2:
        mov     dx,mccdata              ;data out reg
        mov     al,[bp+p1]              ;byte to be sent
        out     dx,al                   ;send byte
        mov     ax,1                    ;rc=1, byte sent
;=======================================================
; Exit with return code in AX
;=======================================================
mpxt:
        pop     bp
        ret
_mcc_put endp
;=======================================================
; _mcc_flush
;
; _mcc_flush clears the receive data buffer of its
; current contents.
;
; C model:
;       mcc_flush()
;
; Assembler Model:
;       call    _mcc_flush
;
;=======================================================
        public  _mcc_flush
_mcc_flush proc
        cli                             ;disable interrupts
        mov     ax,offset DGROUP:bfbeg  ;offset of buffer beginning
        mov     bhead,ax                ;reset buffer empty pointer
        mov     btail,ax                ;reset buffer fill  pointer
        mov     rca,0                   ;reset received count
        sti                             ;re-enable interrupts
        ret
_mcc_flush endp
;=======================================================
; _mcc_close
;
; _mcc_close removes the MCC interrupt handler from the
; system.  This MUST be done before program exit/termination.
;
; C Model:
;       mcc_close();
;
; Assembler Model
;       call    _mcc_close
;
;=======================================================
        public  _mcc_close
_mcc_close proc
        push    bp
;=======================================================
; Ignore the call if MCC is not currently installed
;=======================================================
        cmp     install,1               ;installed?
        jne     mdixx                   ;no
;=======================================================
; Reset MCC interrupt at the PIC
;=======================================================
        cli                             ;PR227
	mov	al,oldpic		;get old 8259 int mask
        out     picmsk,al               ;and restore it to pic
        sti                             ;PR227
;=======================================================
; Return MCC to power-on state
;=======================================================
        mov     ah,$reset               ;reset mcc
        call    mcccommand
;=======================================================
; Remove MCC interrupt vector
;=======================================================
        push    ds                      ;reset midi interrupt vector
        mov     ah,25h
        mov     al,midi_level
        mov     dx,intip
        mov     ds,intcs
        int     21h
        pop     ds
;=======================================================
; Exit
;=======================================================
mdixx:
        mov     install,0               ;now un-installed
        pop     bp
        ret
_mcc_close endp
;=======================================================
; _mcc_close_nr
;
; _mcc_close_nr removes the MCC interrupt handler from the
; system.  This MUST be done before program exit/termination.
; Unlike mcc_close, the interface is not reset to power on state
;
; C Model:
;       mcc_close_nr();
;
; Assembler Model
;       call    _mcc_close_nr
;
;=======================================================
        public  _mcc_close_nr
_mcc_close_nr proc
        push    bp
;=======================================================
; Ignore the call if MCC is not currently installed
;=======================================================
        cmp     install,1               ;installed?
        jne     mdixxr                  ;no
;=======================================================
; Reset MCC interrupt at the PIC
;=======================================================
        cli                             ;PR227
	mov	al,oldpic		;get old 8259 int mask
        out     picmsk,al               ;and restore it to pic
        sti                             ;PR227
;=======================================================
; Remove MCC interrupt vector
;=======================================================
        push    ds                      ;reset midi interrupt vector
        mov     ah,25h
        mov     al,midi_level
        mov     dx,intip
        mov     ds,intcs
        int     21h
        pop     ds
;=======================================================
; Exit
;=======================================================
mdixxr:
        mov     install,0               ;now un-installed
        pop     bp
        ret
_mcc_close_nr endp
;=======================================================
; mcc_flih
;
; MCC first level interrupt handler
;
; Called by _mcc_flih.
; Recieves interrupt from MCC.
; Reads data-in register.
; Passes data byte to second level interrupt handler
;
;=======================================================
        public  mcc_flih
mcc_flih proc    far
        push    ds                      ;save current machine state
        push    es
        push    bp
        push    di
        push    si
        push    dx
        push    cx
        push    bx
        push    ax
        mov     ax,seg DGROUP           ;set ds to DGROUP's data segment
        mov     ds,ax
;=======================================================
; Test to see if this is an MCC interrupt
;=======================================================
        mov     dx,mccstat              ;status ret
        in      al,dx                   ;read status
        test    al,rxrdy                ;data-in from mcc?
        jz      intrx                   ;yes
;=======================================================
; Pass interrupt on
;=======================================================
        pop     ax                      ;restore saved registers
        pop     bx
        pop     cx
        pop     dx
        pop     si
        pop     di
        pop     bp
        pop     es
        pop     ds
        jmp     dword ptr intip         ;pass interrupt on
;=======================================================
; MCC interrupt
;=======================================================
intrx:
        mov     dx,mccdata              ;read inbound char
        in      al,dx
;=======================================================
; Pass data byte to the current SLIH
;=======================================================
        IF      @codesize
        call    dword ptr slih_rtn      ;pass control to slih
        ELSE
        call    word ptr slih_rtn       ;pass control to slih
        ENDIF
;=======================================================
; Send and end-of-interrupt command to the PIC
;=======================================================
        mov     al,00100000b            ;EOI for 8259
        out     piccmd,al               ;reset 8259
;=======================================================
; Restore machine state and exit
;=======================================================
        pop     ax                      ;restore saved registers
        pop     bx
        pop     cx
        pop     dx
        pop     si
        pop     di
        pop     bp
        pop     es
        pop     ds
        iret                            ;return from interrupt
mcc_flih endp
;=======================================================
; _mcc_noslih
;
; _mcc_noslih is the default second level interrupt
; handler.  It throws away all MIDI int data.
;
; Inputs:
;   ax - received char from FLIH
; Outputs:
;   None.
; Notes:
;   The SLIH is called with interrupts disabled.
;=======================================================
        public  _mcc_noslih
_mcc_noslih proc
        ret
_mcc_noslih endp
;=======================================================
; _mcc_receive
;
; _mcc_receive is the generic second level interrupt
; handler.  If the user does not writee his own SLIH,
; he can use this one.  It merely places the received data
; byte into the receive buffer, in a FIFO fashion.
; Overrun conditions ARE NOT checked.
;
; Inputs:
;   ax - received char from FLIH
; Outputs:
;   None.
; Notes:
;   The SLIH is called with interrupts disabled.
;=======================================================
        public  _mcc_receive
_mcc_receive proc
        mov     bx,btail                ;buffer fill pointer
        mov     [bx],al                 ;store character in buffer
        inc     bx                      ;bump pointer
        cmp     bx,offset DGROUP:bfend  ;test for end of buffer
        jb      asin1                   ;no, continue
        mov     bx,offset DGROUP:bfbeg  ;offset of buffer start
asin1:
        mov     btail,bx                ;update buffer fill pointer
        inc     rca                     ;bump count of chars available
        ret
_mcc_receive endp
;=======================================================
; _set_slih
;
; _set_slih allows the user to set up his own second
; level interrupt handler.  The user's SLIH should be
; modelled after the _mcc_receive SLIH.
;
; C Model:
;       set_slih(&routine)
;               int (=routine)();
;               The address of the user's SLIH.
; Assembler Model:
;       mov     ax,offset routine
;       push    ax
;       call    _set_slih
;       pop     cx
;=======================================================
        public  _set_slih
_set_slih proc
        push    bp
        mov     bp,sp
        cli
        mov     ax,[bp+p1]
        mov     word ptr [slih_rtn],ax
        IF      @codesize
        mov     ax,[bp+p2]
        mov     word ptr [slih_rtn+2],ax
        ENDIF
        sti
        pop     bp
        ret
_set_slih endp
        IFDEF   _QBASIC_
;=======================================================
; Quick Basic SLIH Assignements
;=======================================================
;=======================================================
; _mcc_set_noslih
;
; _mcc_set_noslih establishes mcc_noslih as the current
; interrupt handler.
;
; Model:
;       mccsetnoslih
;=======================================================
        public  _mcc_set_noslih
_mcc_set_noslih proc  far
        push    bp
        mov     bp,sp
        cli
        mov     word ptr [slih_rtn],offset _mcc_noslih
        mov     word ptr [slih_rtn+2],seg _mcc_noslih
        sti
        pop     bp
        ret
_mcc_set_noslih endp
;=======================================================
; _mcc_set_receiveslih
;
; _mcc_set_receiveslih establishes mcc_receive as the current
; interrupt handler.
;
; Model:
;       mccsetreceiveslih
;=======================================================
        public  _mcc_set_receiveslih
_mcc_set_receiveslih proc  far
        push    bp
        mov     bp,sp
        cli
        mov     word ptr [slih_rtn],offset _mcc_receive
        mov     word ptr [slih_rtn+2],seg _mcc_receive
        sti
        pop     bp
        ret
_mcc_set_receiveslih endp
;=======================================================
; _mcc_set_coprocslih
;
; _mcc_set_coprocslih establishes _coproc_slih as the current
; interrupt handler.
;
; Model:
;       mccsetcoprocslih
;=======================================================
        public  _mcc_set_coprocslih
_mcc_set_coprocslih proc  far
        push    bp
        mov     bp,sp
        cli
        mov     word ptr [slih_rtn],offset _coproc_slih
        mov     word ptr [slih_rtn+2],seg _coproc_slih
        sti
        pop     bp
        ret
_mcc_set_coprocslih endp
        ENDIF
;=======================================================
; _mcc_open
;
; _mcc_open is called to initialize the
; MCC.  It must be call once, before
; accessing other MCC toolkit functions.
;
; C Model:
;       rc=mcc_open(address,intlevel)
;            int address;
;                MCC base address (usually 330)
;            int intlevel;
;                MCC interrupt
;                level (0-7), usually 2
;            int rc;
;                return code
;                0 = MCC not initialized, not installed
;                1 = MCC initialized
;
; Assembler Model:
;       push    intlevel
;       push    address
;       call    _mcc_open
;       add     sp,4
;       ax == return code
;
;=======================================================
        public  _mcc_open
_mcc_open proc
        push    bp
        mov     bp,sp
        cmp     install,1               ;already installed?
        jne     mdii                    ;no
        jmp     SHORT mdi9              ;yes
;=======================================================
; Generate MCC register addresses
;=======================================================
mdii:
        mov     ax,[bp+p1]              ;MCC base address
        mov     mccdata,ax              ;base = data reg
        inc     ax                      ;base + 1 = status/command reg
        mov     mccstat,ax
        mov     mcccmd,ax
;=======================================================
; Generate interrupt number and PIC mask
;=======================================================
        mov     al,[bp+p2]              ;int level
        add     al,il_base
        mov     midi_level,al
        mov     cl,[bp+p2]              ;compute interrupt mask
        mov     al,1                    ;mask = (1 << intlevel) ^ 0xFF
        shl     al,cl
        not     al
        mov     imask,al
;=======================================================
; Initialize receive buffer pointers
;=======================================================
        mov     ax,offset DGROUP:bfbeg  ;get offset of receive buffer
        mov     bhead,ax                ;initialize buffer empty pointer
        mov     btail,ax                ;initialize buffer fill pointer
;=======================================================
; Reset MCC to know state, verifying MCC is present
;=======================================================
;	call    _mcc_reset              ;force reset MCC, to known state
;	mov     ah,$reset               ;reset mcc to power-on state
;=======================================================
; Checkpoint the current interrupt vector contents
;=======================================================
        push    es
        mov     ah,35h                  ;get vector
        mov     al,midi_level           ;midi port interrupt level
        int     21h                     ;get current vector
        mov     intip,bx                ;save it
        mov     intcs,es
        pop     es
;=======================================================
; Establish MCC interrupt vector
;=======================================================
        push    ds                      ;set up midi interrupt vector
        mov     dx,offset mcc_flih
        mov     ah,25h
        mov     al,midi_level
        push    cs
        pop     ds
        int     21h
        pop     ds
;=======================================================
; Enable MCC interrupt level in PIC
;=======================================================
	cli
        in      al,picmsk               ;read 8259 interrupt mask
	mov	oldpic,al		;save the value
        and     al,imask                ;enable the midi interrupt
        out     picmsk,al               ;output the new mask
        mov     ax,1                    ;rc=1
	sti
;=======================================================
; Verify that the MCC is at the specified interrupt level
;=======================================================
	mov     install,1               ;assume successful install
;        IF      @codesize
;        mov     word ptr [slih_rtn],offset _mcc_receive ;reset slih routine
;        mov     word ptr [slih_rtn+2],seg _mcc_receive
;        ELSE
;        mov     slih_rtn,offset _mcc_receive ;reset slih routine
;        ENDIF
;        mov     ah,$ver                 ;get MCC version
;        call    mcccommand              ;send command
;        mov     cx,-1                   ;retry count
;mdi8:
;        push    cx
;        call    _mcc_get                ;get response
;        pop     cx
;        cmp     al,0FFh                 ;good response?
;        jne     mdi9                    ;yes
;        loop    mdi8
;        call    _mcc_close              ;remove interrupt handler/de-install
;=======================================================
; Exit with return code in AX
;=======================================================
mdi9:
        IF      @codesize
        mov     word ptr [slih_rtn],offset _mcc_noslih ;reset slih to default
        mov     word ptr [slih_rtn+2],seg _mcc_noslih
        ELSE
        mov     slih_rtn,offset _mcc_noslih ;reset slih to default
        ENDIF
        sub     ah,ah
        mov     al,install              ;return installed status
        pop     bp
        ret
_mcc_open endp
;=======================================================
; _mcc_irq
;
; _mcc_irq is called to automatically determine
; the IRQ being used by the MCC/MQX/PCMC
;
; C Model:
;       irq=mcc_irq(address)
;            int address;
;                MCC/MQX/PCMC base address (usually 330)
;            int irq;
;                IRQ for card
;                0 = MCC/MQX/PCMC not found
;                >0 = IRQ number
;
; Assembler Model:
;       push    address
;       call    _mcc_irq
;       add     sp,2
;       ax == irq number
;
;=======================================================
        public  _mcc_irq
_mcc_irq proc
        push    bp
        mov     bp,sp
        push    di
;=======================================================
; We'll make several attempts at finding the IRQ
;=======================================================
        mov     di,5
mirq1:
	cli
	in      al,picmsk               ;get current PIC int mask
        mov     oldpic,al               ;save it
        and     al,01010011b            ;isolate possible irqs
        out     picmsk,al               ;set PIC to possible irqs only
        mov     dx,[bp+p1]              ;get base i/o address
        in      al,dx                   ;clear data port
        inc     dx                      ;make it the cmd port
        mov     al,$reset
        out     dx,al                   ;send a reset
        mov     cx,-1                   ;give interface some time
mirq2:	loop    mirq2

        mov     al,0Ah
        out     piccmd,al               ;set up PIC to read IRR
	mov     cx,-1                   ;we'll look this many times
mirq3:
        in      al,piccmd               ;read irr
        and     al,10101100b            ;isolate possible irq bits
        jnz     mirq31                  ;something is set
        loop    mirq3                   ;try again
        jmp     short mirq5             ;nothing found
;=======================================================
; At least one IRQ bit was set.  Make sure it's just one!
;=======================================================
mirq31:
	push	ax
	mov	al,20h			;send eoi to 8259
	out	piccmd,al
	pop	ax

        sub     ah,ah                   ;count number of IRQ bits set
        mov     cx,8
mirq35:
        ror     al,1
        adc     ah,0
        loop    mirq35
        cmp     ah,1                    ;more than one bit?
        je      mirq7                   ;no, we're done
;=======================================================
; Failure to find an irq, try again
;=======================================================
mirq5:
	mov     al,oldpic               ;get the old PIC mask
	out     picmsk,al               ;restore it
	sti
        dec     di                      ;retry the whole thing
        jnz     mirq1
;=======================================================
; Failure to find an irq, retries exhausted
;=======================================================
        sub     ax,ax
        jmp     short mirq99
;=======================================================
; IRQ found - decode mask to irq number
;=======================================================
mirq7:
        mov     ah,al                   ;save the irq mask
        mov     al,oldpic               ;get the old PIC mask
	out     picmsk,al               ;restore it
        sti
        sub     al,al                   ;initialize the result
mirq71:
        ror     ah,1                    ;is this irq bit set
        jc      mirq8                   ;yes, break
        inc     al                      ;bump to next irq
        cmp     al,7                    ;all bits checked?
        jle     mirq71                  ;no
        sub     al,al                   ;not found in mask???
mirq8:
        sub     ah,ah                   ;form irq number in ax
;
mirq99:
        push    ax
        dec     dx                      ;back to data port
        in      al,dx                   ;clear ackn
        pop     ax
        pop     di
        pop     bp
        ret
_mcc_irq endp
;=======================================================
        END
