/*===================================*/
/*                                   */
/* Music Quest Programmer's ToolKit  */
/* Sequencer - Record/playback       */
/*                                   */
/*===================================*/
/*                                   */
/* Copyright (c) 1988                */
/* Music Quest, Inc.                 */
/*                                   */
/*===================================*/
#include "dos.h"
#include "bios.h"
#include "\ptk\asm\mcc.h"


static unsigned char ptrack[8192];      /* record/play track */
static int recflag = 0;                 /* 1=recorded  */
static int tracksize;                   /* how much recorded */
static unsigned char *trkp;             /* play back track pointer to next byte */
static int cmdl;                        /* data length for MIDI running status */

/*===================================*/
/*                                   */
/* Determine MIDI data length        */
/*                                   */
/*===================================*/

mddl(cmd)
  int cmd;
{
  static int midi_dl[8] = {2,2,2,2,1,1,2,0};    /* by MIDI msg type */

  return(midi_dl[(cmd & 0x70) >> 4]);           /* calc length by msg type */
}

/*===================================*/
/*                                   */
/* Set up MCC for record/play        */
/*                                   */
/*===================================*/

setup_mcc()
{
  int rc;

  mcc_reset();
  rc=mcc_command(DISABLE_MEASUREND);    /* measure end off */
  rc=mcc_command(INT_CLOCK_96);         /* 96 PPB */
  rc=mcc_command(DISABLE_CONDUCTOR);    /* conductor off */
  rc=mcc_command(SET_TEMPO);            /* set tempo */
  mcc_put(66);                          /* 66 beats/min */
  rc=mcc_command(DISABLE_ALL_THRU);     /* disable THRU mode */
  rc=mcc_command(DISABLE_THRU);
  rc=mcc_command(ENABLE_BENDER_PC);     /* pitch bend to host */
  rc=mcc_command(CHANNELS_1);           /* enable all channels */
  mcc_put(0xFF);
  rc=mcc_command(CHANNELS_9);
  mcc_put(0xFF);
  rc=mcc_command(SET_METRO);            /* set tics/beat */
  mcc_put(24);                          /* 1/4 note = beat */
  rc=mcc_command(SET_BPM);              /* set beats/measure */
  mcc_put(4);                           /* 4 beats/measure */
  rc=mcc_command(METRO_OFF);            /* metronome off */
  rc=mcc_command(SET_CLOCK_PC);         /* set clock to PC frequency */
  mcc_put(16);                          /* every 4 clock ticks */
  mclk_init(4);                         /* set up clock services */
  set_slih(coproc_slih);                /* set up co-processor interrupt handler */
}

/*===================================*/
/*                                   */
/* Record a track                    */
/*                                   */
/*===================================*/

record_track()
{
  int rc;
  struct SREGS sr;                      /* used to get DS value */
  int beat_efw;                         /* beat counter event flag word */
  int beats;                            /* beat counter */
  int trqx;                             /* our TRQ's handle */

  segread(&sr);                         /* get DS */
  rec_init(sr.ds,(unsigned)ptrack,8192); /* set up track recorder */
  setup_mcc();

  puts("Record MIDI Data\n\n");
  puts("Press any key to stop recording....");

  beat_efw=0;                           /* set up a running beat counter */
  beats=1;
  printf("\n\nBeat %-3d",beats);
  trqx=set_trq(&beat_efw,96);           /* 96 ticks = 1 beat = 1/4 note */

  mclk_start();                         /* start clock to PC messages */

  rc=mcc_command(START_REC);            /* start recording */
  rc=mcc_command(METRO_ON);             /* metronome on for recording */

  while (!_bios_keybrd(_KEYBRD_READY))  /* wait for exit key */
    {
      if (beat_efw)                     /* beat counter update */
        {
          clear_efw(&beat_efw);         /* reset EFW */
          set_trq_time(trqx,96);        /* reset beat TRQ */
          beats++;
          printf("\b\b\b%-3d",beats);   /* show new beat counter */
        }
      if (rec_overflow())
        {
          puts("\n\nRecording track full...recording stopped");
          puts("Press any key to return to ToolKit menu");
          break;
        }
    }
  _bios_keybrd(_KEYBRD_READ);           /* clear exit key */

  rc=mcc_command(STOP_REC);             /* stop recording */

  while (!recordend_efw());             /* wait for end of track received */

  rc=mcc_command(METRO_OFF);            /* metronome off */
  mclk_stop();                          /* turn off clock interrupts */
  end_trq(trqx);                        /* release our beat counter TRQ */
  recflag=1;

  tracksize=rec_bytes();
  printf("\n\n%d bytes recorded\n",tracksize);
  puts("Press any key to continue");
  _bios_keybrd(_KEYBRD_READ);
}

/*===================================*/
/*                                   */
/* Playback recorded track           */
/*                                   */
/*===================================*/

play_track()
{
  int rc;
  int eotflag;
  int beat_efw;                         /* beat counter event flag word */
  int beats;                            /* beat counter */
  int trqx;                             /* our TRQ's handle */

  if (recflag)                          /* only if valid track */
    {
      puts("Track Play Back\n\n");

      trkp=&ptrack[0];                  /* address of first track byte */
      cmdl=2;                           /* length for default running status */

      setup_mcc();                      /* set up co-processor interrupt handler */

      beat_efw=0;                       /* set up a running beat counter */
      beats=1;
      printf("\n\nBeat %-3d",beats);
      trqx=set_trq(&beat_efw,96);       /* 96 ticks = 1 beat = 1/4 note */

      rc=mcc_command(ACTIVE_TRACKS);    /* activate track 1 */
      mcc_put(0x01);
      rc=mcc_command(CLEAR_PLAY_TRACKS); /* clear play counters */
      mclk_start();                     /* start clock-to-PC messages */
      rc=mcc_command(START_PLAY);       /* start play */

      /* This is the body of play back */

      eotflag=0;                        /* end of track flag */
      while (!_bios_keybrd(_KEYBRD_READY) && (!eotflag)) /* loop until key or end of track */
        {
          tdr();                        /* respond to TDRs */
          if (beat_efw)                 /* beat counter update */
            {
              clear_efw(&beat_efw);     /* reset EFW */
              set_trq_time(trqx,96);    /* reset beat TRQ */
              beats++;
              printf("\b\b\b%-3d",beats); /* show new beat counter */
            }
          eotflag=playend_efw();        /* test for track end */
        }

      /* Stopping play back is an involved process, as the following shows */

      rc=clock_efw();                   /* must wait to next MIDI clock */
      while (!clock_efw())
        tdr();                          /* if a TDR occurs, answer it */
      while (!mcc_command(STOP_PLAY))   /* stop play (issued just after clock) */
        tdr();                          /* a TDR is the likely cause for failure */
      while (!clock_efw())              /* stop actually occurs at next clock */
        tdr();                          /* if a TDR occurs, answer it */

      mclk_stop();                      /* turn off clock interrupts */
      end_trq(trqx);                    /* release our beat counter TRQ */

      if (eotflag)                      /* end of track */
        {
          puts("\n\nEnd of recorded MIDI data");
          puts("Press any key to continue");
          _bios_keybrd(_KEYBRD_READ);
        }
      else
        _bios_keybrd(_KEYBRD_READ);     /* clear exit key */
    }
  else
    {
      puts("\nNo data recorded");
      puts("Press any key to continue");
      _bios_keybrd(_KEYBRD_READ);
    }
}

/*===================================*/
/*                                   */
/* Respond to TDR by sending next    */
/* track event                       */
/*                                   */
/* Note that variable tdrp and cmdl  */
/* were set up by the play back      */
/* routine.                          */
/*                                   */
/*===================================*/

tdr()
{
  unsigned char tbyte;
  unsigned i;

        if (track_efw(0))               /* TDR received on track 1 */
          {
            tbyte=*trkp++;              /* get next track byte (timing byte) */
            if (tbyte == 0xF8)          /* timer overflow event */
              mcc_put(tbyte);
            else                        /* time stamped event */
              {
                mcc_put(tbyte);         /* send new timing byte */
                tbyte=*trkp++;          /* get next track byte (status or data) */
                if (tbyte & 0x80)       /* new running status */
                  {
                    mcc_put(tbyte);     /* send new status byte */
                    cmdl=mddl(tbyte);   /* new length */
                    for(i=0; i < cmdl; i++) /* send data */
                      {
                        tbyte=*trkp++;  /* get next track byte */
                        mcc_put(tbyte);
                      }
                  }
                else                    /* continuing status */
                  {
                    mcc_put(tbyte);     /* send first byte of message */
                    if (cmdl == 2)      /* send second byte */
                      {
                        tbyte=*trkp++;  /* get next track byte */
                        mcc_put(tbyte);
                      }
                  }
              }
          }
}
