/************************************************************************
 *                                                                      *
 * manipul8.c - Manipulates password attempts saved in RESTORE.PAN or   *
 *              allows viewing of PASSWORD.NDS files.                   *
 *                                                                      *
 * Programmer - Simple Nomad - Nomad Mobile Research Centre             *
 *                                                                      *
 * 6/18/97    - Initial revision.                                       *
 *                                                                      *
 *----------------------------------------------------------------------*
 *                                                                      *
 * 6/29/97    - Added the option to create a RESTORE.PAN file for crypto*
 *                                                                      *
 ************************************************************************/

/*
 * Note - This program is to be used for the "seeding" of the RESTORE.PAN
 *        file. It will also allow you to look at all fields stored in
 *        PASSWORD.NDS.
 */

/*
 * Includes
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/*
 * Typedefs
 */
typedef unsigned char uint8;
typedef unsigned int  uint16;
typedef unsigned long uint32;

/*
 * Globals
 */
#define TRUE 1
#define FALSE 0
FILE *fPassword;

/*
 * Struct for PASSWORD.NDS records and a global pointer
 */
typedef struct password
{
	uint32          selfOffset;	/* Offset in PASSWORD.NDS. If this is
					   the first record, it is 0x00000000
					   followed by 0x0000014e for the
					   second record, etc. */
	uint32		id;		/* Object ID from ENTRY */
	uint32		parentID;	/* Parent ID */
	uint32		objectID;	/* Object ID from Private Key */
	uint32		pwlen;		/* Password length of user account */
	uint8		hash[16];	/* One-way hash */
	uint8           userOU[40];     /* OU of User */
	uint8		userCN[258];    /* User common name */
} PASSWORD; /* size=334 */

PASSWORD pPassword;


/*
 * Dump unicode to screen. I dislike unicode a lot, but this routine
 * skips the 0x00's and only prints the parts that matter.
 */
void printUnicodeName(char *name, int j)
{
   int i;
   for (i=0;i<j;i++)
   {
     if (name[i]!=0) putchar(name[i]);
   }
}

/*
 * This routine counts the number of PASSWORD.NDS records and returns the
 * value. calc.quot-1 is returned to account for the "zero" record.
 */
long int countPasswordRecords(void)
{
   uint32 k;
   ldiv_t calc;
   fseek(fPassword,334,SEEK_END);
   k=ftell(fPassword);
   calc=ldiv(k,334);
   return(calc.quot-1);
}

/*
 * This routine looks for a particular user. It is passed the total number
 * of users to look through and who to look for. If found, TRUE is returned.
 * FALSE is returned if the user was not found.
 */
int findEntryInPasswordFile(long int j, char *account)
{
   FILE *fPassword;
   int FOUND,i,k;
   FOUND=FALSE;

   fPassword=fopen("PASSWORD.NDS","rb");
   if (fPassword==NULL)
   {
     printf("Unable to open PASSWORD.NDS\n");
     exit(1);
   }

   while(j!=0)
   {
     fread(&pPassword,334,1,fPassword);
     FOUND=TRUE;
     for (i=0;i<strlen(account);i++)
     {
       k=i*2+6;
       if (account[i]!=pPassword.userCN[k]) FOUND=FALSE;
     }
     if (FOUND==TRUE)
       break;
     j--;
   }
   fclose(fPassword);
   return(FOUND);
}

/*
 * Print some usage info....
 */
void printHelp(int j)
{
   printf("USAGE: manipul8 <seed> <option>\n\n");
   printf("  seed is password attempt to insert into RESTORE.PAN file.\n");
   printf("  -h           prints this HELP file.\n");
   printf("  -u <account> create a RESTORE.PAN file for USER\n");
   printf("  -v           VIEW the contents of a PASSWORD.NDS file\n\n");
   printf("EXAMPLES:\n");
   printf("  manipul8 -h\n");
   printf("  manipul8 -v\n");
   printf("  manipul8 TESTSEED\n");
   printf("  manipul8 TESTSEED -u Admin\n\n");
   exit(j);
}

/*
 * Main prog...
 */

void main(int argc, char **argv)
{
   FILE *restoreTemp;
   char *newTry;
   int i,t,k,x,view,create,FOUND;
   char *c,*account;
   long int j;

   account=calloc(sizeof(account),1);
   newTry=calloc(sizeof(newTry),1);

/* Say hello... */
   printf("MANIPUL8 - seeding of RESTORE.PAN file plus PASSWORD.NDS viewing\n\n");
   printf("Simple Nomad - thegnome@nmrc.org\n");
   printf("http://www.nmrc.org\n");
   printf("1997 (c) Nomad Mobile Research Centre\n\n");

   view=FALSE;
   if (argc<2) printHelp(1);

   /* process command line switches, if any... */
   for (i=1 ; i<argc ; i++)
   {
      if (argv[i][0]=='-')
	 switch(argv[i][1])
	 {
	    case 'h':
	    case 'H':
	       printHelp(0);
	    case 'u':
	    case 'U':
	       if ((i+1>argc) || argv[i+1][0]=='-')
	       {
		  printf("No argument given for option -u\n");
		  exit(1);
	       }
	       sprintf(account,"%s",argv[i+1]);
               create=TRUE;
	       break;
	    case 'v':
	    case 'V':
	       view=TRUE; break;
	    default:
	       printf("Invalid option: %s\n", argv[i]);
	       exit(1);
	 }
   }

   sprintf(newTry, "%s", argv[1]);
   /* if viewing the PASSWORD.NDS file, dump 'em all... */
   if(view==TRUE)
   {
     fPassword=fopen("PASSWORD.NDS","rb");
     if (fPassword==NULL)
     {
       printf("Unable to open PASSWORD.NDS\n");
       exit(1);
     }
     j = countPasswordRecords();

     rewind(fPassword);
     while (j!=0) {
	j--;
	fread(&pPassword,334,1,fPassword);
	printUnicodeName(pPassword.userCN,258);
	printf(" ");
	printUnicodeName(pPassword.userOU,40);
	printf(" id - %08lx parentID - %08lx objectID - %08lx\n pwlen - %d hash - ",
pPassword.id,pPassword.parentID,pPassword.objectID,pPassword.pwlen);
	for (i=0;i<16;i++) printf("%02x",pPassword.hash[i]);
	printf("\n");
     }
     fflush(fPassword);
     fclose(fPassword);
     exit(0);
   }
   else
   {
     for (i=0;i<strlen(newTry);i++)
       newTry[i]=toupper(newTry[i]); /* convert lower case letters
                                        to upper case letters */
     if (create==TRUE)
     {
       fPassword=fopen("PASSWORD.NDS","rb");
       if (fPassword==NULL)
       {
         printf("Unable to open PASSWORD.NDS\n");
         exit(1);
       }
       j = countPasswordRecords();
       FOUND=findEntryInPasswordFile(j,account);
       if (FOUND==FALSE)
       {
         printf("User %s not found in PASSWORD.NDS\n",account);
         printHelp(1);
       }
       restoreTemp=fopen("RESTORE.PAN","w+b");
       if (restoreTemp==NULL)
       {
         printf("Unable to open RESTORE.PAN\n");
         exit(1);
       }
       fwrite(&pPassword,334,1,restoreTemp);
       if(pPassword.pwlen!=strlen(newTry))
       {
         printf("seed must be %d characters in length.\n",pPassword.pwlen);
         exit(1);
       }
       fputs(newTry,restoreTemp);
       fflush(restoreTemp);
       fclose(restoreTemp);
       printf("Created a RESTORE.PAN file containing %s and:\n",newTry);
       printUnicodeName(pPassword.userCN,258);
       printf(" ");
       printUnicodeName(pPassword.userOU,40);
       printf(" id - %08lx parentID - %08lx objectID - %08lx\n pwlen - %d hash - ",
pPassword.id,pPassword.parentID,pPassword.objectID,pPassword.pwlen);
       for (i=0;i<16;i++) printf("%02x",pPassword.hash[i]);
       printf("\n");
     }
     else
     {
       restoreTemp=fopen("RESTORE.PAN","r+b");
       if (restoreTemp==NULL)
       {
         printf("Unable to open RESTORE.PAN\n");
         exit(1);
       }
       fread(&pPassword,334,1,restoreTemp);
       if(pPassword.pwlen!=strlen(newTry))
       {
         printf("seed must be %d characters in length.\n",pPassword.pwlen);
         exit(1);
       }
       printUnicodeName(pPassword.userCN,258);
       printf(" ");
       printUnicodeName(pPassword.userOU,40);
       printf(" id - %08lx parentID - %08lx objectID - %08lx\n pwlen - %d hash - ",
pPassword.id,pPassword.parentID,pPassword.objectID,pPassword.pwlen);
       for (i=0;i<16;i++) printf("%02x",pPassword.hash[i]);
       printf("\n");
       printf("Last attempt - ");
       fgets(c,pPassword.pwlen,restoreTemp);
       printf("%s\nNew attempt - %s\n",c,newTry);
       fseek(restoreTemp,334,SEEK_SET);
       fputs(newTry,restoreTemp);
       fflush(restoreTemp);
       fclose(restoreTemp);
     }
   }
   printf("\n");
}



