/*  sound.h - Digital Sound Interface Kit V1.01a header file

    Copyright 1993,94 Carlos Hasan
*/

#ifndef __SOUND_H
#define __SOUND_H

#ifndef __LARGE__
#error Must use Large memory model for the sound system.
#endif

/* Misc Values */
#define MAXVOICES   16
#define MAXTRACKS   16
#define MAXSAMPLES  100
#define MAXORDERS   128
#define MINPERIOD   28
#define MAXPERIOD   6848
#define MIDCPERIOD  428
#define MIDCFREQ    8363L

/* Sound Cards */
#define ID_NONE     0
#define ID_SB       1
#define ID_SB201    2
#define ID_SBPRO    3
#define ID_SB16     4
#define ID_GUS      5

/* Types of RAM */
#define RAM_NONE    0
#define RAM_SYSTEM  1
#define RAM_CARD    2

/* Error Values */
#define ERR_OK      0
#define ERR_NORAM   1
#define ERR_NODRAM  2
#define ERR_NOFILE  3
#define ERR_FORMAT  4
#define ERR_ACCESS  5

/* Playing Status */
#define PS_STOPPED  0
#define PS_PLAYING  1
#define PS_PAUSED   2

/* Pattern Break Values */
#define PB_NONE     0
#define PB_BREAK    1
#define PB_JUMP     2

/* DSM File Block IDs */
#define ID_RIFF     0x46464952L
#define ID_DSMF     0x464D5344L
#define ID_SONG     0x474E4F53L
#define ID_INST     0x54534E49L
#define ID_PATT     0x54544150L

/* WAV File Block IDs */
#define ID_WAVE     0x45564157L
#define ID_FMT      0x20746D66L
#define ID_DATA     0x61746164L

/* Panning Values */
#define PAN_LEFT    0x00
#define PAN_RIGHT   0x80
#define PAN_DOLBY   0xA4

/* Inst Bit Flags */
#define INST_LOOPED 0x01
#define INST_SIGNED 0x02
#define INST_PACKED 0x04

/* Basic Data Types */
typedef unsigned char byte;
typedef unsigned short word;
typedef unsigned long dword;

/* RIFF Block Header Formats */
typedef struct {
    dword   ID;
    dword   Length;
    dword   FileType;
} DSMHeader;

typedef struct {
    dword   ID;
    dword   Length;
} DSMBlock;

/* DSM File Block Formats */
typedef struct {
    char    SongName[28];
    word    Version;
    word    Flags;
    dword   Pad;
    word    NumOrders;
    word    NumSamples;
    word    NumPatterns;
    word    NumChannels;
    byte    GlobalVolume;
    byte    MasterVolume;
    byte    InitTempo;
    byte    InitBPM;
    byte    ChanMap[MAXTRACKS];
    byte    Orders[MAXORDERS];
} DSMSong;

typedef struct {
    char    FileName[13];
    word    Flags;
    byte    Volume;
    dword   Length;
    dword   LoopStart;
    dword   LoopEnd;
    void    far *Address;
    word    MidCRate;
    word    Period;
    char    SampleName[28];
} DSMInst;

typedef struct {
    word    Length;
    byte    Data[1];
} DSMPatt;

/* WAV File Block Formats */
typedef struct {
    word    SampleFormat;
    word    NumChannels;
    dword   PlayRate;
    dword   BytesPerSec;
    word    Pad;
    word    BitsPerSample;
} DSMWave;

/* Internal DSM format */
typedef struct {
    DSMSong Song;
    DSMInst far *Inst[MAXSAMPLES];
    DSMPatt far *Patt[MAXORDERS];
} DSM;

/* Sound Card Configuration */
typedef struct {
    byte    CardID;
    byte    Flags;
    word    IOAddr;
    byte    IRQNum;
    byte    DRQNum;
    word    MixRate;
} DSMCard;

/* Internal Track structure */
typedef struct {
    word    NoteEvent;
    byte    VolumeEvent;
    byte    Note;
    byte    Sample;
    byte    Volume;
    word    Effect;
    word    Period;
    word    WantedPeriod;
    word    Rate;
    byte    VibrPos;
    byte    VibrParm;
    byte    TremPos;
    byte    TremParm;
    byte    ToneSpeed;
    byte    EQBar;
    word    ArpTable[3];
} Track;

/* Internal Music structure */
typedef struct {
    word    ActiveTracks;
    Track   Tracks[MAXTRACKS];
    byte    OrderPosition;
    byte    OrderLength;
    byte    PatternNumber;
    byte    PatternRow;
    byte    BreakFlag;
    byte    Tempo;
    byte    TempoCount;
    byte    BPM;
    word    CardStatus;
    word    PlayStatus;
    dword   SongPtr;
    byte    SyncMark;
} DSMMusicInfo;

/* Status Value */
extern int DSMStatus;

#ifdef __cplusplus
extern "C" {
#endif

/* Sound System API Routines */
DSM *DSMLoad(char *FileName, dword FileOffset);
void DSMFree(DSM *Module);
DSMInst *DSMLoadSample(char *FileName, dword FileOffset);
void DSMFreeSample(DSMInst *Inst);
int DSMLoadSetup(DSMCard *Card);
int DSMSaveSetup(DSMCard *Card);

int  DSMInit(DSMCard *Card);
void DSMDone(void);
void DSMPoll(void);
void DSMSetupVoices(word MaxVoices, word MasterVolume);
void DSMStopVoices(void);
int  DSMTypeOfRAM(void);
int  DSMAllocSampleData(DSMInst *Inst);
void DSMFreeSampleData(DSMInst *Inst);
void DSMPlaySample(word Voice, DSMInst *Inst);
void DSMStopSample(word Voice);
void DSMSetPeriod(word Voice, word Period);
void DSMSetVolume(word Voice, word Volume);
void DSMSetBalance(word Voice, word Balance);
int  DSMGetSampleStatus(word Voice);
void DSMSetMusicVolume(word Volume);
void DSMSetSoundVolume(word Volume);
void DSMPlayMusic(DSM *Module);
void DSMStopMusic(void);
int  DSMGetMusicStatus(void);
DSMMusicInfo *DSMGetMusicInfo(void);

#ifdef __cplusplus
}
#endif

#endif
