/******************************************************************************
* Cagd_lib.h - header file for the CAGD library.			      *
* This header is also the interface header to the world.		      *
*******************************************************************************
* Written by Gershon Elber, Mar. 90.					      *
******************************************************************************/

#ifndef CAGD_LIB_H
#define CAGD_LIB_H

#include <stdio.h>

#ifdef NO_VOID_PTR
#define VoidPtr		char *
#else
#define VoidPtr		void *
#endif /* NO_VOID_PTR */

#define CagdBType int      			            /* Boolean type. */
#ifdef __MSDOS__
typedef float CagdRType;
#define CAGD_FLOAT
#else
typedef double CagdRType;
#define CAGD_DOUBLE
#endif /* __MSDOS__ */

typedef void (*CagdPrintfFuncType)(char *Line);

typedef enum {
    CAGD_ERR_UNDEFINE_ERR,
    CAGD_ERR_180_ARC,
    CAGD_ERR_PT_OR_LEN_MISMATCH,
    CAGD_ERR_CRVS_INCOMPATIBLE,
    CAGD_ERR_RAT_LIN_NO_SUPPORT,
    CAGD_ERR_DIR_NOT_CONST_UV,
    CAGD_ERR_T_NOT_IN_CRV,
    CAGD_ERR_U_NOT_IN_SRF,
    CAGD_ERR_V_NOT_IN_SRF,
    CAGD_ERR_INDEX_NOT_IN_MESH,
    CAGD_ERR_NUM_KNOT_MISMATCH,
    CAGD_ERR_PARSER_STACK_OV,
    CAGD_ERR_KNOT_NOT_ORDERED,
    CAGD_ERR_UNDEF_CRV,
    CAGD_ERR_UNDEF_SRF,
    CAGD_ERR_UNSUPPORT_PT,
    CAGD_ERR_POWER_NO_SUPPORT,
    CAGD_ERR_ALLOC_ERR,
    CAGD_ERR_NOT_ENOUGH_MEM,
    CAGD_ERR_NOT_IMPLEMENTED,
    CAGD_ERR_WRONG_ORDER
} CagdFatalErrorType;

typedef enum {		/* Type of control point. The P-types are rationals. */
    CAGD_PT_E1_TYPE = 0,
    CAGD_PT_P1_TYPE,
    CAGD_PT_E2_TYPE,
    CAGD_PT_P2_TYPE,
    CAGD_PT_E3_TYPE,
    CAGD_PT_P3_TYPE,
    CAGD_PT_E4_TYPE,
    CAGD_PT_P4_TYPE,
    CAGD_PT_E5_TYPE,
    CAGD_PT_P5_TYPE
} CagdPointType;

#define CAGD_IS_RATIONAL_PT(PType)	(((int) PType) & 0x01)
#define CAGD_NUM_OF_PT_COORD(PType)	((((int) PType) >> 1) + 1)
#define CAGD_MAKE_PT_TYPE(IsRational, NumCoords) ((CagdPointType) \
						  ((((IsRational) ? -1 : -2) \
							+ (NumCoords << 1))))
#define CAGD_MAX_PT_SIZE		6	/* Rational P5 has 6 coords. */
#define CAGD_MAX_PT_COORD		5		       /* Without w. */

#define CAGD_IS_RATIONAL_CRV(Crv)	CAGD_IS_RATIONAL_PT(Crv -> PType)
#define CAGD_IS_RATIONAL_SRF(Srf)	CAGD_IS_RATIONAL_PT(Srf -> PType)

/* Bezier curves may be evaluated using a predefined cache. The cache must   */
/* be of size (FINENESS) which is power of 2 up to the maximum order below.  */
/* See CrvBzrSetCache routine below.					     */
#define CAGD_MAX_BEZIER_CACHE_ORDER	10
#define CAGD_MAX_BEZIER_CACHE_FINENESS	1024

typedef enum {
    CAGD_UNDEF_TYPE,
    CAGD_CBEZIER_TYPE,
    CAGD_CBSPLINE_TYPE,
    CAGD_CPOWER_TYPE,
    CAGD_SBEZIER_TYPE,
    CAGD_SBSPLINE_TYPE,
    CAGD_SPOWER_TYPE
} CagdGeomType;

typedef enum {
    CAGD_NO_DIR,
    CAGD_CONST_U_DIR,
    CAGD_CONST_V_DIR
} CagdSrfDirType;

typedef enum {
    CAGD_REG_POLY_PER_LIN,
    CAGD_ONE_POLY_PER_LIN,
    CAGD_ONE_POLY_PER_COLIN
} CagdLin2PolyType;

typedef struct CagdPtStruct {
    CagdRType Pt[3];
} CagdPtStruct;

typedef struct CagdCtlPtStruct {
    CagdPointType PtType;
    CagdRType Coords[CAGD_MAX_PT_SIZE];
} CagdCtlPtStruct;

typedef struct CagdVecStruct {
    CagdRType Vec[3];
} CagdVecStruct;

typedef struct CagdPlaneStruct {
    CagdRType Plane[4];
} CagdPlaneStruct;

typedef struct CagdMatStruct {
    CagdRType Mat[4][4];
} CagdMatStruct;

typedef struct CagdBBoxStruct {
    CagdRType Min[3];
    CagdRType Max[3];
} CagdBBoxStruct;

typedef struct CagdCrvStruct {
    struct CagdCrvStruct *Pnext;
    CagdGeomType GType;
    CagdPointType PType;
    int Length;            /* Number of control points (== order in Bezier). */
    int Order;	    /* Order of curve (only for Bspline, ignored in Bezier). */
    CagdRType *Points[CAGD_MAX_PT_SIZE];     /* Pointer on each axis vector. */
    CagdRType *KnotVector;
} CagdCrvStruct;

typedef struct CagdSrfStruct {
    struct CagdSrfStruct *Pnext;
    CagdGeomType GType;
    CagdPointType PType;
    int ULength, VLength;	 /* Mesh size in the tensor product surface. */
    int UOrder, VOrder;   /* Order in tensor product surface (Bspline only). */
    CagdRType *Points[CAGD_MAX_PT_SIZE];     /* Pointer on each axis vector. */
    CagdRType *UKnotVector, *VKnotVector;
} CagdSrfStruct;

typedef struct CagdPolygonStruct {
    struct CagdPolygonStruct *Pnext;
    CagdPtStruct Polygon[3];		      /* Polygon is always triangle. */
    CagdVecStruct Normal[3];
} CagdPolygonStruct;

typedef struct CagdPolylineStruct {
    struct CagdPolylineStruct *Pnext;
    CagdPtStruct *Polyline;    	 /* Polyline length is defined using Length. */
    int Length;
} CagdPolylineStruct;

#define CAGD_IS_BEZIER_CRV(Crv)		(Crv -> GType == CAGD_CBEZIER_TYPE)
#define CAGD_IS_BEZIER_SRF(Srf)		(Srf -> GType == CAGD_SBEZIER_TYPE)
#define CAGD_IS_BSPLINE_CRV(Crv)	(Crv -> GType == CAGD_CBSPLINE_TYPE)
#define CAGD_IS_BSPLINE_SRF(Srf)	(Srf -> GType == CAGD_SBSPLINE_TYPE)
#define CAGD_IS_POWER_CRV(Crv)		(Crv -> GType == CAGD_CPOWER_TYPE)
#define CAGD_IS_POWER_SRF(Srf)		(Srf -> GType == CAGD_SPOWER_TYPE)

/******************************************************************************
*		U -->			    The mesh is ordered raw after raw *
*       +-----------------------+	or the increments along U are 1 while *
*   V | |P0                 Pi-1|	the increment along V is full raw.    *
*     v	|Pi                P2i-1|	    To encapsulate it, NEXTU/V are    *
*	|			|	defined below.			      *
*	|Pn-i		    Pn-1|					      *
*	+-----------------------+					      *
******************************************************************************/
#define CAGD_NEXT_U(Srf)	(1)
#define CAGD_NEXT_V(Srf)	(Srf -> ULength)
#define CAGD_MESH_UV(Srf, i, j)	((i) + (Srf -> ULength) * (j))

/******************************************************************************
* Routines prototypes. Routines are prefixed as follows:		      *
* Cagd    - General routines such as dynamic memory handlers etc.	      *
* BzrCrv  - Bezier curves routines.					      *
* BzrSrf  - Bezier surface routines.					      *
* BspKnot - Bspline knot vector routines.				      *
* BspCrv  - Bspline curves routines.					      *
* BspSrf  - Bspline surface routines.					      *
* Cnvrt   - Conversion routines such as Bezier to Power basis.		      *
******************************************************************************/

#if defined(__cplusplus) || defined(c_plusplus)
extern "C" {
#endif

/******************************************************************************
* General routines of the Cagd library:					      *
******************************************************************************/
CagdCrvStruct *CagdCrvNew(CagdGeomType GType, CagdPointType PType, int Length);
CagdSrfStruct *CagdSrfNew(CagdGeomType GType, CagdPointType PType, int ULength,
								   int VLength);
CagdPolygonStruct *CagdPolygonNew(void);
CagdPolylineStruct *CagdPolylineNew(int Length);
CagdCrvStruct *CagdCrvCopy(CagdCrvStruct *Crv);
CagdSrfStruct *CagdSrfCopy(CagdSrfStruct *Srf);
CagdPolygonStruct *CagdPolygonCopy(CagdPolygonStruct *Poly);
CagdPolylineStruct *CagdPolylineCopy(CagdPolylineStruct *Poly);
void CagdCrvFree(CagdCrvStruct *Crv);
void CagdCrvFreeList(CagdCrvStruct *CrvList);
void CagdSrfFree(CagdSrfStruct *Srf);
void CagdSrfFreeList(CagdSrfStruct *SrfList);
void CagdPolylineFree(CagdPolylineStruct *Poly);
void CagdPolylineFreeList(CagdPolylineStruct *PolyList);
void CagdPolygonFree(CagdPolygonStruct *Poly);
void CagdPolygonFreeList(CagdPolygonStruct *PolyList);
void CagdCoerceToE2(CagdRType *E2Point, CagdRType *Points[CAGD_MAX_PT_SIZE],
					     int Index, CagdPointType PType);
void CagdCoerceToE3(CagdRType *E3Point, CagdRType *Points[CAGD_MAX_PT_SIZE],
					     int Index, CagdPointType PType);
void CagdCoerceToP2(CagdRType *P2Point, CagdRType *Points[CAGD_MAX_PT_SIZE],
					     int Index, CagdPointType PType);
void CagdCoerceToP3(CagdRType *P3Point, CagdRType *Points[CAGD_MAX_PT_SIZE],
					     int Index, CagdPointType PType);
void CagdCoercePointsTo(CagdRType *Points[], int Len,
			CagdPointType OldPType, CagdPointType NewPType);
CagdCrvStruct *CagdCoerceCrvTo(CagdCrvStruct *Crv, CagdPointType PType);
CagdSrfStruct *CagdCoerceSrfTo(CagdSrfStruct *Srf, CagdPointType PType);
CagdPointType CagdMergePointType(CagdPointType PType1, CagdPointType PType2);
VoidPtr CagdMalloc(unsigned size);
void CagdFree(VoidPtr p);
void CagdFatalError(CagdFatalErrorType ErrID);
char *CagdDescribeError(CagdFatalErrorType ErrID);
void CagdDbg(void *Obj);
void CagdSetCagdFprintf(CagdPrintfFuncType Func);
void CagdSetLinear2Poly(CagdLin2PolyType Lin2Poly);
void CagdMergeBBox(CagdBBoxStruct *DestBBox, CagdBBoxStruct *SrcBBox);

/******************************************************************************
* Matrix/Vector/Plane/Transformation routines:				      *
******************************************************************************/
CagdRType CagdDistanceTwoCtlPts(CagdRType **Points, int Index1, int Index2,
							CagdPointType PType);
CagdRType CagdFitPlaneThruCtlPts(CagdPlaneStruct *Plane, CagdPointType PType,
			         CagdRType **Points,
			         int Index1, int Index2, int Index3, int Index4);
CagdRType CagdDistPtPlane(CagdPlaneStruct *Plane, CagdRType **Points,
						      int Index, int MaxDim);

void CagdMultMatVec(CagdMatStruct *Mat, CagdRType *SrcVec, CagdRType *DstVec);
void CagdCrvMatTransform(CagdCrvStruct *Crv, CagdMatStruct *Mat);
void CagdSrfMatTransform(CagdSrfStruct *Srf, CagdMatStruct *Mat);
void CagdCrvTransform(CagdCrvStruct *Crv, CagdRType Translate[3],
							CagdRType Scale);
void CagdSrfTransform(CagdSrfStruct *Srf, CagdRType Translate[3],
							CagdRType Scale);

/******************************************************************************
* Routines to handle curves generically.				      *
******************************************************************************/
CagdRType CagdEstimateCrvColinearity(CagdCrvStruct *Crv);
CagdRType *CagdCrvEval(CagdCrvStruct *Crv, CagdRType t);
CagdCrvStruct *CagdCrvSubdivAtParam(CagdCrvStruct *Crv, CagdRType t);
CagdCrvStruct *CagdCrvRegionFromCrv(CagdCrvStruct *Crv, CagdRType t1,
							CagdRType t2);
CagdCrvStruct *CagdCrvRefineAtParams(CagdCrvStruct *Crv, CagdBType Replace,
							 CagdRType *t, int n);
CagdVecStruct *CagdCrvTangent(CagdCrvStruct *Crv, CagdRType t);
CagdVecStruct *CagdCrvBiNormal(CagdCrvStruct *Crv, CagdRType t);
CagdVecStruct *CagdCrvNormal(CagdCrvStruct *Crv, CagdRType t);
CagdCrvStruct *CagdCrvOffset(CagdCrvStruct *Crv, CagdRType OffsetDist);
CagdCrvStruct *CagdCrvReverse(CagdCrvStruct *Crv);
CagdCrvStruct *CagdCrvDegreeRaise(CagdCrvStruct *Crv);
CagdCrvStruct *CagdMergeCrvCrv(CagdCrvStruct *Crv1, CagdCrvStruct *Crv2);
CagdCrvStruct *CagdMergeCrvPt(CagdCrvStruct *Crv, CagdPtStruct *Pt);
CagdCrvStruct *CagdMergePtCrv(CagdPtStruct *Pt, CagdCrvStruct *Crv);
CagdCrvStruct *CagdMergePtPt(CagdPtStruct *Pt1, CagdPtStruct *Pt2);
CagdPolylineStruct *CagdCrv2CtrlPoly(CagdCrvStruct *Crv);
CagdPolylineStruct *CagdCrv2Polyline(CagdCrvStruct *Crv, int SamplesPerCurve);
CagdBType CagdMakeCrvsCompatible(CagdCrvStruct **Crv1, CagdCrvStruct **Crv2,
				 CagdBType SameOrder, CagdBType SameKV);
CagdCrvStruct *CagdEditSingleCrvPt(CagdCrvStruct *Crv, CagdCtlPtStruct *CtlPt,
								   int Index);
CagdCrvStruct *CagdCrvReadFromFile(char *FileName, char **ErrStr, int *Line);
CagdCrvStruct *CagdCrvReadFromFile2(FILE *f, char **ErrStr, int *Line);
int CagdCrvWriteToFile(CagdCrvStruct *Crvs, char *FileName, int Indent,
						 char *Comment,	char **ErrStr);
int CagdCrvWriteToFile2(CagdCrvStruct *Crvs, FILE *f, int Indent, char *Comment,
								char **ErrStr);
void CagdCrvBBox(CagdCrvStruct *Crv, CagdBBoxStruct *BBox);

/******************************************************************************
* Routines to handle surfaces generically.				      *
******************************************************************************/
CagdRType CagdEstimateSrfPlanarity(CagdSrfStruct *Srf);
CagdRType *CagdSrfEval(CagdSrfStruct *Srf, CagdRType u, CagdRType v);
CagdCrvStruct *CagdCrvFromSrf(CagdSrfStruct *Srf, CagdRType t,
							CagdSrfDirType Dir);
CagdCrvStruct *CagdCrvFromMesh(CagdSrfStruct *Srf, int Index,
							CagdSrfDirType Dir);
void CagdCrvToMesh(CagdCrvStruct *Crv, int Index,
					CagdSrfDirType Dir, CagdSrfStruct *Srf);
CagdSrfStruct *CagdSrfSubdivAtParam(CagdSrfStruct *Srf, CagdRType t,
							    CagdSrfDirType Dir);
CagdSrfStruct *CagdSrfRegionFromSrf(CagdSrfStruct *Srf, CagdRType t1,
					     CagdRType t2, CagdSrfDirType Dir);
CagdSrfStruct *CagdSrfRefineAtParams(CagdSrfStruct *Srf, CagdSrfDirType Dir,
					CagdBType Replace, CagdRType *t, int n);
CagdVecStruct *CagdSrfTangent(CagdSrfStruct *Srf, CagdRType u, CagdRType v,
							    CagdSrfDirType Dir);
CagdVecStruct *CagdSrfNormal(CagdSrfStruct *Srf, CagdRType u, CagdRType v);
CagdSrfStruct *CagdSrfOffset(CagdSrfStruct *Srf, CagdRType OffsetDist);
CagdSrfStruct *CagdSrfDegreeRaise(CagdSrfStruct *Srf, CagdSrfDirType Dir);
CagdSrfStruct *CagdSrfReverse(CagdSrfStruct *Srf);
CagdSrfStruct *CagdSrfReverse2(CagdSrfStruct *Srf);
CagdPolylineStruct *CagdSrf2CtrlMesh(CagdSrfStruct *Srf);
CagdPolygonStruct *CagdSrf2Polygons(CagdSrfStruct *Srf, int FineNess,
			      CagdBType ComputeNormals, CagdBType FourPerFlat);
CagdPolylineStruct *CagdSrf2Polylines(CagdSrfStruct *Srf, int NumOfIsocurves,
							  int SamplesPerCurve);
CagdSrfStruct *CagdSrfReadFromFile(char *FileName, char **ErrStr, int *Line);
CagdSrfStruct *CagdSrfReadFromFile2(FILE *f, char **ErrStr, int *Line);
int CagdSrfWriteToFile(CagdSrfStruct *Srfs, char *FileName, int Indent,
						 char *Comment,	char **ErrStr);
int CagdSrfWriteToFile2(CagdSrfStruct *Srfs, FILE *f, int Indent, char *Comment,
								char **ErrStr);
CagdSrfStruct *CagdExtrudeSrf(CagdCrvStruct *Crv, CagdVecStruct *Vec);
CagdSrfStruct *CagdSurfaceRev(CagdCrvStruct *Crv);
CagdSrfStruct *CagdSweepSrf(CagdCrvStruct *CrossSection, CagdCrvStruct *Axis,
				    CagdCrvStruct *ScalingCrv, CagdRType Scale);
CagdSrfStruct *CagdBoolSumSrf(CagdCrvStruct *CrvLeft,
			      CagdCrvStruct *CrvRight,
			      CagdCrvStruct *CrvTop,
			      CagdCrvStruct *CrvBottom);
CagdSrfStruct *CagdRuledSrf(CagdCrvStruct *Crv1, CagdCrvStruct *Crv2,
			    int OtherOrder, int OtherLen);
CagdSrfStruct *CagdSrfFromCrvs(CagdCrvStruct *CrvList);
CagdSrfStruct *CagdEditSingleSrfPt(CagdSrfStruct *Srf, CagdCtlPtStruct *CtlPt,
						       int UIndex, int VIndex);
void CagdSrfBBox(CagdSrfStruct *Srf, CagdBBoxStruct *BBox);

/******************************************************************************
* Routines to handle Bezier curves.					      *
******************************************************************************/
CagdCrvStruct *BzrCrvReadFromFile(char *FileName, char **ErrStr, int *ErrLine);
CagdCrvStruct *BzrCrvReadFromFile2(FILE *f, CagdBType NameWasRead,
						  char **ErrStr, int *ErrLine);
int BzrCrvWriteToFile(CagdCrvStruct *Crvs, char *FileName, int Indent,
						 char *Comment,	char **ErrStr);
int BzrCrvWriteToFile2(CagdCrvStruct *Crvs, FILE *f, int Indent, char *Comment,
								char **ErrStr);
CagdCrvStruct *BzrCrvNew(int Length, CagdPointType PType);
CagdRType BzrCrvEvalVecAtParam(CagdRType *Vec, int VecInc, int Order,
								CagdRType t);
CagdRType *BzrCrvEvalAtParam(CagdCrvStruct *Crv, CagdRType t);
void BzrCrvSetCache(int FineNess, CagdBType EnableCache);
void BzrCrvEvalToPolyline(CagdCrvStruct *Crv, int FineNess,
							 CagdRType *Points[]);
CagdCrvStruct *BzrCrvCreateArc(CagdPtStruct *Start, CagdPtStruct *Center,
							     CagdPtStruct *End);
CagdCrvStruct *BzrCrvSubdivAtParam(CagdCrvStruct *Crv, CagdRType t);
CagdCrvStruct *BzrCrvDegreeRaise(CagdCrvStruct *Crv);
CagdCrvStruct *BzrCrvDerive(CagdCrvStruct *Crv);
CagdVecStruct *BzrCrvTangent(CagdCrvStruct *Crv, CagdRType t);
CagdVecStruct *BzrCrvBiNormal(CagdCrvStruct *Crv, CagdRType t);
CagdVecStruct *BzrCrvNormal(CagdCrvStruct *Crv, CagdRType t);
CagdCrvStruct *BzrCrvInterpolate(int Size, CagdRType *XVec, CagdRType *YVec,
							       CagdRType *ZVec);
CagdPolylineStruct *BzrCrv2Polyline(CagdCrvStruct *Crv, int SamplesPerCurve);

/******************************************************************************
* Routines to handle Bezier surfaces.					      *
******************************************************************************/
CagdSrfStruct *BzrSrfReadFromFile(char *FileName, char **ErrStr, int *ErrLine);
CagdSrfStruct *BzrSrfReadFromFile2(FILE *f, CagdBType NameWasRead,
						  char **ErrStr, int *ErrLine);
int BzrSrfWriteToFile(CagdSrfStruct *Srfs, char *FileName, int Indent,
						 char *Comment,	char **ErrStr);
int BzrSrfWriteToFile2(CagdSrfStruct *Srfs, FILE *f, int Indent, char *Comment,
								char **ErrStr);
CagdSrfStruct *BzrSrfNew(int ULength, int VLength, CagdPointType PType);
CagdRType *BzrSrfEvalAtParam(CagdSrfStruct *Srf, CagdRType u, CagdRType v);
CagdCrvStruct *BzrSrfCrvFromSrf(CagdSrfStruct *Srf, CagdRType t,
							CagdSrfDirType Dir);
CagdCrvStruct *BzrSrfCrvFromMesh(CagdSrfStruct *Srf, int Index,
							CagdSrfDirType Dir);
CagdSrfStruct *BzrSrfSubdivAtParam(CagdSrfStruct *Srf, CagdRType t,
							CagdSrfDirType Dir);
CagdSrfStruct *BzrSrfDegreeRaise(CagdSrfStruct *Srf, CagdSrfDirType Dir);
CagdSrfStruct *BzrSrfDerive(CagdSrfStruct *Srf, CagdSrfDirType Dir);
CagdVecStruct *BzrSrfTangent(CagdSrfStruct *Srf, CagdRType u, CagdRType v,
							 CagdSrfDirType Dir);
CagdVecStruct *BzrSrfNormal(CagdSrfStruct *Srf, CagdRType u, CagdRType v);
CagdPolygonStruct *BzrSrf2Polygons(CagdSrfStruct *Srf, int FineNess,
			 CagdBType ComputeNormals, CagdBType FourPerFlat);
CagdPolylineStruct *BzrSrf2Polylines(CagdSrfStruct *Srf, int NumOfIsocurves,
							int SamplesPerCurve);

/******************************************************************************
* Routines to handle Bspline knot vectors.				      *
******************************************************************************/
CagdBType BspKnotHasOpenEC(CagdRType *KnotVector, int Len, int Order);
CagdBType BspKnotParamInDomain(CagdRType *KnotVector, int Len, int Order,
								CagdRType t);
int BspKnotLastIndexLE(CagdRType *KnotVector, int Len, CagdRType t);
int BspKnotLastIndexL(CagdRType *KnotVector, int Len, CagdRType t);
int BspKnotFirstIndexG(CagdRType *KnotVector, int Len, CagdRType t);
CagdRType *BspKnotUniformFloat(int Len, int Order, CagdRType *KnotVector);
CagdRType *BspKnotUniformOpen(int Len, int Order, CagdRType *KnotVector);
CagdRType *BspKnotSubtrTwo(CagdRType *KnotVector1, int Len1,
			   CagdRType *KnotVector2, int Len2,
			   int *NewLen);
CagdRType *BspKnotMergeTwo(CagdRType *KnotVector1, int Len1,
			   CagdRType *KnotVector2, int Len2,
			   int Mult, int *NewLen);
CagdRType *BspKnotNodes(CagdRType *KnotVector, int Len, int Order);
CagdRType *BspKnotReverse(CagdRType *KnotVector, int Len);
void BspKnotAffineTrans(CagdRType *KnotVector, int Len,
					CagdRType Translate, CagdRType Scale);
CagdRType *BspKnotCopy(CagdRType *KnotVector, int Len);
CagdRType *BspKnotInsertOne(CagdRType *KnotVector, int Order, int Len,
								CagdRType t);
CagdRType *BspKnotInsertMult(CagdRType *KnotVector, int Order, int *Len,
						       CagdRType t, int Mult);
int BspKnotFindMult(CagdRType *KnotVector, int Order, int Len, CagdRType t);
CagdBType BspKnotC1Discont(CagdRType *KnotVector, int Order, int Length,
								CagdRType *t);
CagdRType *BspKnotAllC1Discont(CagdRType *KnotVector, int Order, int Length,
								       int *n);
CagdRType *BspKnotParamValues(CagdRType PMin, CagdRType PMax, int NumSamples,
			      CagdRType *C1Disconts, int NumC1Disconts);

/******************************************************************************
* Routines to handle Bspline curves.					      *
******************************************************************************/
CagdCrvStruct *BspCrvReadFromFile(char *FileName, char **ErrStr, int *ErrLine);
CagdCrvStruct *BspCrvReadFromFile2(FILE *f, CagdBType NameWasRead,
						  char **ErrStr, int *ErrLine);
int BspCrvWriteToFile(CagdCrvStruct *Crvs, char *FileName, int Indent,
						 char *Comment,	char **ErrStr);
int BspCrvWriteToFile2(CagdCrvStruct *Crvs, FILE *f, int Indent, char *Comment,
								char **ErrStr);
CagdCrvStruct *BspCrvNew(int Length, int Order, CagdPointType PType);
void BspCrvDomain(CagdCrvStruct *Crv, CagdRType *TMin, CagdRType *TMax);

CagdRType *BspCrvCoxDeBoorBasis(CagdRType *KnotVector, int Order, int Len,
						CagdRType t, int *IndexFirst);
CagdRType *BspCrvEvalCoxDeBoor(CagdCrvStruct *Crv, CagdRType t);
CagdRType BspCrvEvalVecAtParam(CagdRType *Vec, int VecInc,
			       CagdRType *KnotVector, int Order, int Len,
			       CagdRType t);
CagdRType *BspCrvEvalAtParam(CagdCrvStruct *Crv, CagdRType t);
CagdCrvStruct *BspCrvCreateCircle(CagdPtStruct *Center, CagdRType Radius);
CagdCrvStruct *BspCrvCreateUnitCircle(void);
int BspCrvEvalToPolyline(CagdCrvStruct *Crv, int FineNess, CagdRType *Points[]);
CagdCrvStruct *BspCrvKnotInsert(CagdCrvStruct *Crv, CagdRType t);
CagdCrvStruct *BspCrvKnotInsertNSame(CagdCrvStruct *Crv, CagdRType t, int n);
CagdCrvStruct *BspCrvKnotInsertNDiff(CagdCrvStruct *Crv, CagdBType Replace,
							   CagdRType *t, int n);
CagdCrvStruct *BspCrvSubdivAtParam(CagdCrvStruct *Crv, CagdRType t);
CagdCrvStruct *BspCrvDegreeRaise(CagdCrvStruct *Crv);
CagdCrvStruct *BspCrvDerive(CagdCrvStruct *Crv);
CagdVecStruct *BspCrvTangent(CagdCrvStruct *Crv, CagdRType t);
CagdVecStruct *BspCrvBiNormal(CagdCrvStruct *Crv, CagdRType t);
CagdVecStruct *BspCrvNormal(CagdCrvStruct *Crv, CagdRType t);
CagdPolylineStruct *BspCrv2Polyline(CagdCrvStruct *Crv, int SamplesPerCurve);

/******************************************************************************
* Routines to handle Bspline surfaces.					      *
******************************************************************************/
CagdSrfStruct *BspSrfReadFromFile(char *FileName, char **ErrStr, int *ErrLine);
CagdSrfStruct *BspSrfReadFromFile2(FILE *f, CagdBType NameWasRead,
						  char **ErrStr, int *ErrLine);
int BspSrfWriteToFile(CagdSrfStruct *Srfs, char *FileName, int Indent,
						 char *Comment,	char **ErrStr);
int BspSrfWriteToFile2(CagdSrfStruct *Srfs, FILE *f, int Indent, char *Comment,
								char **ErrStr);
CagdSrfStruct *BspSrfNew(int ULength, int VLength,
			 int UOrder, int VOrder, CagdPointType PType);
void BspSrfDomain(CagdSrfStruct *Srf, CagdRType *UMin, CagdRType *UMax,
				      CagdRType *VMin, CagdRType *VMax);
CagdRType *BspSrfEvalAtParam(CagdSrfStruct *Srf, CagdRType u, CagdRType v);
CagdCrvStruct *BspSrfCrvFromSrf(CagdSrfStruct *Srf, CagdRType t,
							CagdSrfDirType Dir);
CagdCrvStruct *BspSrfCrvFromMesh(CagdSrfStruct *Srf, int Index,
							CagdSrfDirType Dir);
CagdSrfStruct *BspSrfKnotInsert(CagdSrfStruct *BspSrf, CagdSrfDirType Dir,
								   CagdRType t);
CagdSrfStruct *BspSrfKnotInsertNSame(CagdSrfStruct *BspSrf, CagdSrfDirType Dir,
							    CagdRType t, int n);
CagdSrfStruct *BspSrfKnotInsertNDiff(CagdSrfStruct *BspSrf, CagdSrfDirType Dir,
					      int Replace, CagdRType *t, int n);
CagdSrfStruct *BspSrfSubdivAtParam(CagdSrfStruct *Srf, CagdRType t,
							CagdSrfDirType Dir);
CagdSrfStruct *BspSrfDegreeRaise(CagdSrfStruct *Srf, CagdSrfDirType Dir);
CagdSrfStruct *BspSrfDerive(CagdSrfStruct *Srf, CagdSrfDirType Dir);
CagdVecStruct *BspSrfTangent(CagdSrfStruct *Srf, CagdRType u, CagdRType v,
							 CagdSrfDirType Dir);
CagdVecStruct *BspSrfNormal(CagdSrfStruct *Srf, CagdRType u, CagdRType v);
CagdVecStruct *BspSrfMeshNormals(CagdSrfStruct *Srf, int UFineNess,
								int VFineNess);
CagdPolygonStruct *BspSrf2Polygons(CagdSrfStruct *Srf, int FineNess,
			 CagdBType ComputeNormals, CagdBType FourPerFlat);
CagdPolylineStruct *BspSrf2Polylines(CagdSrfStruct *Srf, int NumOfIsocurves,
							int SamplesPerCurve);

/******************************************************************************
* Routines to handle basis function conversions.			      *
******************************************************************************/
CagdCrvStruct *CnvrtPower2BezierCrv(CagdCrvStruct *Crv);
CagdCrvStruct *CnvrtBezier2PowerCrv(CagdCrvStruct *Crv);
CagdCrvStruct *CnvrtBspline2BezierCrv(CagdCrvStruct *Crv);
CagdCrvStruct *CnvrtBezier2BsplineCrv(CagdCrvStruct *Crv);

CagdSrfStruct *CnvrtBezier2BsplineSrf(CagdSrfStruct *Srf);

#if defined(__cplusplus) || defined(c_plusplus)
}
#endif

#endif /* CAGD_LIB_H */
