/* ***********************************************************************
   * 									 *
   *		Special Effects for z-Buffered Scene Creation		 *
   *									 *
   *		 Original Material by Christopher D. Watkins		 *
   *									 *
   *			     'C' conversion by				 *
   *                            Larry Sharp				 *
   *									 *
   ***********************************************************************

   JuliaSet	       - plot a Julia set (lighting)
   MapJuliaSetToSphere - map a Julia set on to a sphere (moon with dendrite)
   Ground	       - checkerboard ground
   Sky		       - blend a sky color from the horizon to the zenith
*/

void PutDitheredSphere(int xx, int yy, int r,
		       int LocX, int LocY,
		       Byte Color, Byte Intensity)
{
  float Xc, Yc, Zc;
  int Xp, Yp;
  float Theta, Phi;

  Theta=Radians(yy);
  Phi=Radians(xx);
  Xc=r*sin(Theta)*cos(Phi);
  Yc=r*sin(Theta)*sin(Phi);
  Zc=r*cos(Theta);
  MapCoordinates(Xc, Yc, Zc, &Xp, &Yp);
  PutPixel(Xp+LocX, Yp+LocY, Color, Intensity);
}

void PlotSphrPix(int LocX, int LocY, int x, int y, int r, Byte Color, Byte Intensity)
{
  x*=2;
  y*=2;
  PutDitheredSphere(x, y, r, LocX, LocY, Color, Intensity);
  PutDitheredSphere(x+1, y, r, LocX, LocY, Color, Intensity);
  PutDitheredSphere(x, y+1, r, LocX, LocY, Color, Intensity);
  PutDitheredSphere(x+1, y+1, r, LocX, LocY, Color, Intensity);
}

/* ***********************************************************************
   *									 *
   *			     Julia Set Images				 *
   *									 *
   ***********************************************************************

   JuliaSet            - plot a Julia set
   MapJuliaSetToSphere - map a Julia set on to a sphere
*/

int Iterate(float p, float q, float xo, float yo)
{
  int MaxIntens=MaxInten-5;
  float DivBndSqr=100.0;
  float xn, yn;
  int Iters, A;

  Iters=0;
A:
  xn=SqrFP(xo)-SqrFP(yo)+p;
  yn=2*xo*yo+q;
  xo=xn;
  yo=yn;
  ++Iters;
  if((SqrFP(xn)+SqrFP(yn))>DivBndSqr)
    return(Iters);
  if(Iters==MaxIntens)
    return(0);
  goto A;
}

Boolean SphrPix;

void Scan(float p, float q,
	  float xmin, float ymin,
	  float xmax, float ymax,
	  int LocX, int LocY, int Rad,
	  int ResX, int ResY, int Color)
{
  float dx, dy;
  int x, y;
  float xo, yo;
  int Inten;

  dx=(xmax-xmin)/(ResX-1);
  dy=(ymax-ymin)/(ResY-1);
  for(x=0; x<=((ResX-1) >> 1); x++)
  {
    for(y=0; y<ResY; y++)
    {
      xo=xmin+x*dx;
      yo=ymin+y*dy;
      Inten=Iterate(p, q, xo, yo);
      Inten+=5;
      if(SphrPix)
      {
	if(HalfScale)
	{
	  PlotSphrPix(LocX, 3*Res/8+LocY, x, y, Rad, Color, Inten);
	  PlotSphrPix(LocX, 3*Res/8+LocY, ResX-1-x, ResY-1-y, Rad, Color, Inten);
	}
	else
	{
	  PlotSphrPix(LocX, LocY, x, y, Rad, Color, Inten);
	  PlotSphrPix(LocX, LocY, ResX-1-x, ResY-1-y, Rad, Color, Inten);
	}
      }
      else
      {
	if(HalfScale)
	{
	  PutPixel(LocX+x, LocY+y, Color, Inten);
	  PutPixel(LocX+ResX-1-x, LocY+ResY-1-y, Color, Inten);
	}
	else
	{
	  PutPixel(LocX+x*2, LocY+y, Color, Inten);
	  PutPixel(LocX+x*2+1, LocY+y, Color, Inten);
	  PutPixel(LocX+(ResX-1-x)*2, LocY+ResY-1-y, Color, Inten);
	  PutPixel(LocX+(ResX-1-x)*2+1, LocY+ResY-1-y, Color, Inten);
	}
      }
    }
  }
}

void JuliaSet(float p, float q,
	      int LocX, int LocY,
	      int ResX, int ResY,
	      int Color)
{
  SphrPix=false;
  LocX+=MaxX/2-ResX;
  LocY+=MaxY/2-ResY;
  Scan(p, q, -1.4, -1.4, 1.4, 1.4, LocX, LocY, 0, ResX, ResY, Color);
}

void MapJuliaSetToSphere(float p, float q,
			 int LocX, int LocY,
			 int Rad,
			 int ResX, int ResY,
			 int Color)
{
  SphrPix=true;
  InitPlotting(180, 0);
  InitPerspective(false, 0, 0, 0, 0);
  Scan(p, q, -2.2, -2.2, 2.2, 2.2, LocX, LocY, Rad, ResX, ResY, Color);
}

Byte CalcSky(TDA Dir)
{
  float Small=1E-03;
  float HorInten=0.25;
  float ZenInten=1.00;
  float sin2, cos2;
  float x2, y2, z2;

  x2=SqrFP(Dir[0]);
  y2=SqrFP(Dir[1]);
  z2=SqrFP(Dir[2]);
  if(z2==0)
    z2=Small;
  sin2=z2/(x2+y2+z2);
  cos2=1.0-sin2;
  return(Round((cos2*HorInten+sin2*ZenInten)*MaxInten));
}

void Sky(int LocX, int LocY,
	 int ResX, int ResY,
	 Byte Color)
{
  int i, j;
  TDA Dir;
  TDIA Tmp, Eye;
  int HalfResX;
  int HalfResY;
  int EyeDist;
  int OffX;
  int OffY;
  Byte Inten;

  HalfResX=ResX/2;
  HalfResY=ResY/2;
  OffX=0;
  OffY=-HalfResY-HalfResY/2-HalfResY/4;
  EyeDist=-HalfResY;
  VecInt(OffX, OffY, EyeDist, Eye);
  for(i=0; i<ResX; i++)
  {
    for(j=0; j<ResY; j++)
    {
      VecInt(i-HalfResX, j-HalfResY, 0, Tmp);
      VecSubInt(Tmp, Eye, Dir);
      VecNormalize(Dir);
      Inten=CalcSky(Dir);
      if(HalfScale)
	PutPixel(LocX+i, LocY+j+HalfResY, Color, Inten);
      else
      {
	PutPixel(LocX+i*2, LocY+j*2, Color, Inten);
	PutPixel(LocX+i*2+1, LocY+j*2, Color, Inten);
	PutPixel(LocX+i*2, LocY+j*2+1, Color, Inten);
	PutPixel(LocX+i*2+1, LocY+j*2+1, Color, Inten);
      }
    }
  }
}
