; Advanced VIDEO DRIVER   Ver 1.1

; Supports:
;   Standard VGA - 320x200x256
;   All VESA modes  (requires VBE 2.0 BIOS with linear addr enabled)

include qlib.inc
include file.inc
include video.inc  ;globals
include string.inc
include private.inc

.data
align 4
;v_copy is optimized for modes where bpsl=x*bypp
; v_copy = v_copy1 when bpsl=x*bypp
; v_copy = v_copy2 when bpsl!=x*bypp    (slower)
v_copy dd v_copy2

v_linear dd 0  ;set to 0a0000h-program_base for VGA mode (13h)
vesa_addr dd 0   ;Holds linear mapped addr
v_buffer dd 0   ;where everything is put/get from
v_xd label dword      ;X
v_x dw 0,0
v_xbd label dword     ;X*bypp
v_xb dw 0,0
v_yd label dword      ;Y
v_y dw 0,0
v_bpsl dd 0           ;bytes/scan line
v_bppd label dword    ;bits/pixel
v_bpp dw 0,0
v_byppd label dword   ;bytes/pixel
v_bypp dw 0,0
v_copy_add dw 0,0
v_init db 0

.code
comment ~
; VESA  2.0 implementation    (Original found in the DOS32 PAL package)
; You'll need UNIVBE5.1a+ installed to use this... PUT IT IN YOUR AUTOEXEC.BAT!!!
;  and get it registered (one way or another)  All games will use it soon...
*****************************************************************************
Function:   CheckVbeMode
Expects:    Eax = VESA video mode number (bits 9..31 ignored).
                  linear memory mapping is always detected and used (or it
                   will fail)
Returns:  if successful;
             EAX = near pointer to video address
             Carry = 0
          if unsuccessful;
             EAX = -1     ;linear can never be -1 (unless your program is loaded @ 0a0001h)
             Carry = 1
             DL = error code.   1 = VBE not installed (or is not V2.0+)
                                2 = mode not supported.
                                3 = mode incompatible.
Description: Checks to see if the VBE video mode can be used. If so then
             will return a pointer to video mapping address.
Notes:    *   All registers are preserved.
          *   EAX will point to the linear frame buffer (near ptr).

*****************************************************************************
Function:   SetVbeMode
Expects:    Nothing.
Returns:    Nothing.
Description: Set video mode to the mode number specified to last successful
             call to CheckVbeMode.
Notes:       All registers are preserved.

*****************************************************************************
Function:   VbeFunc06

   Input:   BL   = 00h          Set Scan Line Length in Pixels
                 = 01h          Get Scan Line Length
                 = 02h          Set Scan Line Length in Bytes
                 = 03h          Get Maximum Scan Line Length
            CX   =         If BL=00h  Desired Width in Pixels
                           If BL=02h  Desired Width in Bytes
                           (Ignored for Get Functions)

Output:     AX   =         VBE Return Status
            BX   =         Bytes Per Scan Line
            CX   =         Actual Pixels Per Scan Line
                           (truncated to nearest complete pixel)
            DX   =         Maximum Number of Scan Lines
Description: VBE Function 06h - Set/Get Logical Scan Line Length

Notes:       directly calls INT 10h AX=4F06h;

*****************************************************************************
~
.code
align 4
v_setup proc private
  ;sets funcs based on v_bypp
  ; NOTE : the 24bit funtions will work in any BPP
  ; but the 8bit functions will only work in 256 color modes
  .if v_copy_add
     .if mouse_con
       mov v_copy,offset v_copy2m
     .else
       mov v_copy,offset v_copy2
     .endif
  .else
     .if mouse_con
       mov v_copy,offset v_copy1m
     .else
       mov v_copy,offset v_copy1
     .endif
  .endif
  ret
v_setup endp

;copys v_buffer to video RAM
align 4
v_copy1 proc private uses eax ecx edi esi ebx
  ;could be 24bit ;  This just assumes v_bpsl==v_xbd ; v_copy2 does not
  mov eax,v_xbd
  mul v_yd
  mov ecx,eax
  mov edi,v_linear
  mov esi,v_buffer
  mov al,cl
  shr ecx,2
  rep movsd
  and al,3
  .if al
    mov cl,al
    rep movsb
  .endif
  ret
v_copy1 endp

;copys v_buffer to video RAM
align 4
v_copy2 proc private uses eax ecx edi esi ebx
  ;must copy each scan line seperately! due to bytesperscanline != x*bypp
  mov ecx,v_xbd
  mov ebx,v_yd
  mov edi,v_linear
  mov esi,v_buffer
@@:
  mov al,cl
  shr ecx,2
  rep movsd
  and al,3
  .if al
    mov cl,al
    rep movsb
  .endif
  mov ecx,v_xbd
  add edi,dword ptr[v_copy_add]
  dec ebx
  jnz @b
  ret
v_copy2 endp

;These two copies for when the mouse ptr is enabled: It does tricky things
; so that the mouse ptr does not blink
v_copy1m proc private uses eax ecx edi esi ebx
  ;could be 24bit ;  This just assumes v_bpsl==v_xbd ; v_copy2 does not

  ;steps :
  ; 1 - hold mouse until done
  ; 2 - save temp area where mouse lies
  ; 3 - print mouse on v_buffer
  ; 4 - copy to VRAM
  ; 5 - undo all (clean up)
  ; Don't : cli thru the entire copy

  cli
    mov m_paws,1
    call _print
  sti

  mov eax,v_xbd
  mul v_yd
  mov ecx,eax
  mov edi,v_linear
  mov esi,v_buffer
  mov al,cl
  shr ecx,2
  rep movsd
  and al,3
  .if al
    mov cl,al
    rep movsb
  .endif

  cli
    callp erase,1
    mov m_paws,0
  sti

  ret
v_copy1m endp

;copys v_buffer to video RAM
align 4
v_copy2m proc private uses eax ecx edi esi ebx
  ;must copy each scan line seperately! due to bytesperscanline != x*bypp

  cli
    mov m_paws,1
    call _print
  sti

  mov ecx,v_xbd
  mov ebx,v_yd
  mov edi,v_linear
  mov esi,v_buffer
@@:
  mov al,cl
  shr ecx,2
  rep movsd
  and al,3
  .if al
    mov cl,al
    rep movsb
  .endif
  mov ecx,v_xbd
  add edi,dword ptr[v_copy_add]
  dec ebx
  jnz @b

  cli
    callp erase,1
    mov m_paws,0
  sti
  ret
v_copy2m endp

include vesa.inc

.data
align 4
PmodeSetStart           DD 0
file_buffer             DD 0
VideoModeNumber         DW 0
Total64Kblocks          DW 0
VBE_StarPerPixelFactor  DB 0
align 4
DOS_segs LABEL DWORD
Real_ES     DW  ?
Real_DS     DW  ?

.code

align 4
VbeSetStart             DD Offset SetStart_RealMode
Old_PhysBasePtr         DD -1

;--------------------------------------------------------
; Call real mode set display start bank function
; CX=pixel in scan line
; DX=scan linenumber
; Note: very very slow....
;----------------------------------------------------------
align 4
SetStart_RealMode PROC
        push   ebx
        xor    ebx,ebx
        mov    ax,04F07h
        int    10h
        pop    ebx
        Ret
SetStart_RealMode ENDP

;--------------------------------------------------------
; Call Protected mode set display start bank function
; ECX=pixel in scan line
; EDX=scan line number
;----------------------------------------------------------
align 4
SetStart_ProtectedMode PROC
        push   ebx
        xor    ebx,ebx
        imul   edx,[v_bpsl]   ;bytes/scan line
        add    edx,ecx
        mov    cl,[VBE_StarPerPixelFactor]
        shr    edx,cl
        mov    cx,dx
        shr    edx,16
        mov    ax,04F07h
        Call   [PmodeSetStart]
        pop    ebx
        Ret
SetStart_ProtectedMode ENDP


;------------------------------------------------------------
; Procudure to call a DOS interrupt.
;
; Expects the intrrupt number pushed on the stack.
;
;  e.g    push  10h
;         call  DOSinterrupt
;         jc   error
; Real mode ES and DS registers are passed via varibles Real_ES and Real_DS.
;
;
; As explained in the DOS32 documentaion, DOS32 initally handles all interrupts
; such that calling a protected mode interrupt will automatically call
; the same interrupt number in real mode.  However, using this method there
; is no way to specify the values of the real mode SEGMENT registers.
; Some of the VESA calls require ES:DI to point to buffers and so
; we need to use INT31h AX=300h service to call a real mode interrupt.
; The procedure below does exactly that...
;
;------------------------------------------------------------
align 4
DOSinterrupt PROC
        push    dword ptr 0             ; ignore  SS, SP
        lea     esp,[esp - 8]           ; ignore  CS, IP ,FS, GS
        push    [DOS_segs]              ; push DS and ES
        db 66h  ;to make it pushfw
        pushf;w
        pushad
        mov     edi,esp
        mov     ax,0300h
        xor     cx,cx
        movzx   Ebx,Byte Ptr [esp+36h]  ; Get int number from stack param
        int     31h                     ; Emulate Real Mode Interrupt
        popad
        db 66h  ;to make it popfw
        popf;w
        pop     [DOS_segs]              ; get DS and ES
        lea     esp,[esp+12]            ; Ignore SS,SP,CS,IP,FS,GS
        ret     4                       ; return ingnoring parameter
DOSinterrupt ENDP


;
;  VBE Function 06h - Set/Get Logical Scan Line Length
;
align 4
VbeFunc06 PROC
          mov  ax,4F06h
          int  10h
          cmp  ah,00h
          jne  @@J1
          mov  Word Ptr [v_bpsl],BX
@@J1:     ret
VbeFunc06 ENDP

align 4
checkvbemode PROC

; Some of the vesa bios functions require a transfer of data in the
; real mode address space. The best way to do this is by taking advantage
; of the 8Kb file buffer initally setup by DOS32. As explained in the API.DOC
; this 8Kb buffer is in the real mode address space and may be use to temporary
; hold information.

         pushad

         and eax,1ffh
         or eax,4000h  ;enable linear addressing
         mov    VideoModeNumber,ax

         mov    ax,0EE02h       ; GET REAL MODE SEGMENT OF FILE I/O BUFFER
         int    31h

         ; returns AX = real mode segment of 8Kb buffer.
         ;         EBX = program linear base address.

         mov    Real_ES,ax
         And    Eax,0FFFFh
         shl    Eax,4
         sub    eax,ebx
         Mov    file_buffer,eax     ; save near address of 8Kb file buffer.

     ;
     ;  GET VESA INFORMATION
     ;
         mov    ax,4F00h
         mov    di,0                ; (real mode) ES:DI -> 256 byte buffer
         push   10h
         call   DOSinterrupt
         Cmp    AX,004Fh
         jne    NoVESA

     ;
     ; Search video mode list for 640x480x256 ( VESA mode 101h )
     ; The vesa driver Func 4F00h fills in the buffer with the information.
     ; Offset 0Eh of this buffer contains a real mode SEG:OFS of the video
     ; mode list. This list consists of each supported VESA mode and
     ; terminates with 0FFFFh.
     ;
         mov    EBP,file_buffer                       ; save video mem size
         mov    dx,[ VbeInfoBlock.TotalMemory + EBP]
         mov    Total64Kblocks,dx
      ;ensure we have VBE 2.0 or better
         mov    dx,[ VbeInfoBlock.VbeVersion + EBP]
         .if dx<200h
           jmp NoVesa
         .endif

     ;
     ; Get the real mode far pointer of the video mode list and
     ; convert the real mode SEG:OFS address into a 32bit near pointer.
     ;
         Movzx edx,Word Ptr [ VbeInfoBlock.VideoModePtr + EBP + 2]
         shl   edx,4
         sub   edx,_base
         Movzx ebx,Word Ptr [ VbeInfoBlock.VideoModePtr + EBP ]
         add   edx,ebx                      ; EDX points to video mode list.
         mov   bx,VideoModeNumber
         and   bx,0111111111b               ; read bits 0..8
         xor eax,eax
Loop01:  Mov   ax,[Edx]                     ; Read video mode from list
         Cmp   Ax,0FFFFh
         je  @@ModeNotFound
         add   Edx,2
         cmp   Ax,Bx
         jne   Loop01

     ;
     ; Get VBE MODE information
     ; Note, the mode information block is also stored in the file buffer
     ;
        mov    ax,4F01h
        mov    cx,VideoModeNumber
        and    cx,0111111111b               ; read bits 0..8
        mov    di,0                         ; ES:DI -> 256 byte buffer
        push   10h
        call   DOSinterrupt
        Cmp    AX,004Fh
        jne   ModeNotGood

        Mov    EBP,file_buffer              ; EBP -> ModeInfoBlock

 ;setup X/Y/BPP values
   ; Get bytes per scan line for the protected mode SetStart function.
   ;this is also used by v_copy
   mov ax,[ ModeInfoBlock.BytesPerScanLine + EBP]
   mov word ptr[v_bpsl],ax

   movzx eax,word ptr[ModeInfoBlock.XResolution + ebp]
   movzx ebx,word ptr[ModeInfoBlock.YResolution + ebp]
   movzx ecx,byte ptr[ModeInfoBlock.BitsPerPixel + ebp]
   mov v_x,ax
   mov v_y,bx
   mov v_bpp,cx
   .if ecx<=8
     mov v_bypp,1
   .elseif ecx<=16
     mov v_bypp,2
   .elseif ecx<=24
     mov v_bypp,3
   .else
     mov v_bypp,4  ;32bit?
   .endif
   mul v_byppd
   mov v_xb,ax  ;X*bpp
   ;if v_xb!=bytesperscanline then v_copy must do things differently
   mov bx,word ptr[v_bpsl]
   sub bx,ax
   mov v_copy_add,bx    ;differece to add to edi after each scan line

 ; ******************************************************
 ; Setup linear memory mapping mode.
 ; ******************************************************
         Test   [ ModeInfoBlock.ModeAttributes + EBP],10000000b
         jz     ModeNotGood

     ; Calulate a near pointer to physical linear mapping address.
         
         Mov    ebx,[ ModeInfoBlock.PhysBasePtr + EBP]
         Mov    eax,[Old_PhysBasePtr]
         .if (eax==ebx) && vesa_addr   ;check if we already have the area mapped
           mov ebx,vesa_addr
           jmp @f
         .endif
         mov    [Old_PhysBasePtr],ebx
         mov    cx,bx
         shr    ebx,16              ;bx:cx = physical addr
         xor esi,esi
         mov    si,Total64Kblocks   ;si:di = size in bytes
         xor    edi,edi
         mov    ax,0800h            ; map physical memory
         int    31h
         jc     ModeNotGood
         shl    ebx,16              ;bx:cx = linear addr
         mov    bx,cx               ;ebx = linear address
         sub    ebx,_base           ;ebx = near address
@@:
         mov    v_linear,ebx
         mov    vesa_addr,ebx
         mov    [esp+4*7],ebx       ;save eax in stack (EAX return value)
    ;
    ; If 32bit VBE inteface is availible then use the bloody thing
    ;
        mov    ax,4F0Ah
        mov    BL,0                 ; return pmode interface
        push   10h
        call   DOSinterrupt
        Cmp    AX,004Fh
        jne    No32bitInterface

        movzx  Esi,Real_ES          ; convert ES:DI to 32bit near ptr
        shl    esi,4
        and    edi,0ffffh
        add    esi,edi
        sub    esi,_base            ; ESI -> protected mode table.

      ; Save Set display start code address.
        movzx  eax,Word Ptr [ESI+02]
        add    eax,esi
        mov    [PmodeSetStart],eax
        mov    [VbeSetStart],Offset SetStart_ProtectedMode

      ; adjust for plane boundary for 8 bit+ modes
        mov    [VBE_StarPerPixelFactor],0
        Cmp    [ ModeInfoBlock.BitsPerPixel + EBP],4
        je @@1
        mov    [VBE_StarPerPixelFactor],2
@@1:

No32bitInterface:
         call v_setup
         popad
         clc
         ret
;------------------------ Error messages ---------------------------------
ModeNotGood:
         popad
         mov dl,3
         jmp Abort1

@@ModeNotFound:
         popad
         mov dl,2
         jmp Abort1

NoVESA:
         popad
         mov dl,1

Abort1:
         mov eax,ERROR
         stc
         ret
checkvbemode ENDP ;$

align 4
SetVbeMode PROC
  ; Set VIDEO mode
  pushad
  xor eax,eax
  xor ebx,ebx
  xor ecx,ecx
  xor edx,edx
  mov bx,VideoModeNumber
  mov ax,4F02h
  int 10h
  mov v_init,1
  popad
  ret
SetVbeMode ENDP

align 4
v_pp proc private uses edi eax edx,x:dword,y:dword,col:dword
  mov edi,v_linear
  mov eax,y
  mul v_bpsl
  add edi,eax
  mov eax,x
  mul v_byppd
  add edi,eax

  .if v_bypp==1
    mov al,bptr[col]
    stosb
    ret
  .elseif v_bypp==2
    mov ax,wptr[col]
    stosw
    ret
  .endif
  mov al,bptr[col]
  stosb
  mov al,bptr[col+1]
  stosb
  mov al,bptr[col+2]
  stosb
  ret
v_pp endp

align 4
v_gp proc private uses edi edx,x:dword,y:dword
  mov edi,v_linear
  mov eax,y
  mul v_bpsl
  add edi,eax
  mov eax,x
  mul v_byppd
  add edi,eax
  .if v_bypp==1
    mov al,[edi]
    ret
  .elseif v_bypp==2
    mov ax,[edi]
    ret
  .else
    mov eax,[edi]   ;BUG : what if this reads one byte in invalid addr range.
    ret                  ; it won't
  .endif
v_gp endp

;The next 2 use v_buffer/v_xbd
align 4
v_pp2 proc private uses edi eax edx,x:dword,y:dword,col:dword
  mov edi,v_buffer
  mov eax,y
  mul v_xbd
  add edi,eax
  mov eax,x
  mul v_byppd
  add edi,eax

  .if v_bypp==1
    mov al,bptr[col]
    stosb
    ret
  .elseif v_bypp==2
    mov ax,wptr[col]
    stosw
    ret
  .endif
  mov al,bptr[col]
  stosb
  mov al,bptr[col+1]
  stosb
  mov al,bptr[col+2]
  stosb
  ret
v_pp2 endp

align 4
v_gp2 proc private uses edi edx,x:dword,y:dword
  mov edi,v_buffer
  mov eax,y
  mul v_xbd
  add edi,eax
  mov eax,x
  mul v_byppd
  add edi,eax
  .if v_bypp==1
    mov al,[edi]
    ret
  .elseif v_bypp==2
    mov ax,[edi]
    ret
  .else
    mov eax,[edi]   ;BUG : what if this reads one byte in invalid addr range.
    ret                  ; it won't
  .endif
v_gp2 endp

align 4
get proc,buf:dword,x1:dword,y1:dword,x2:dword,y2:dword
  local xl:dword,yl:dword
  pushad
  mov eax,x2
  sub eax,x1
  inc eax
  mov xl,eax
  mov eax,y2
  sub eax,y1
  inc eax
  mov yl,eax

  mov esi,v_buffer
  mov eax,y1
  mul v_xbd
  add esi,eax
  mov eax,x1
  mul v_bypp
  add esi,eax
  mov ebx,yl
  mov eax,xl
  mul v_byppd
  mov xl,eax
  mov ecx,eax
  mov eax,v_xbd
  sub eax,ecx
g_t2:
  mov ecx,xl
  rep movsb
  add esi,eax
  dec ebx
  jnz g_t2
  popad
  ret
get endp

;puts src into v_buffer (no clipping)
align 4
put proc,src:dword,x1:dword,y1:dword,x2:dword,y2:dword
  local xl:dword,yl:dword
  pushad
  mov eax,x2
  sub eax,x1
  inc eax
  mov xl,eax
  mov eax,y2
  sub eax,y1
  inc eax
  mov yl,eax

  mov edi,v_buffer
  mov esi,src
  mov eax,y1
  mul v_xbd
  add edi,eax
  mov eax,x1
  mul v_byppd
  add edi,eax
  mov ecx,yl
  mov eax,xl
  mul v_byppd
  mov xl,eax
  mov ebx,xl
  mov eax,v_xbd
  sub eax,ebx
  mov edx,ecx
g_t1:
  mov ecx,ebx
  rep movsb
  add edi,eax
  dec edx   ;
  jnz g_t1  ;loop  (not restricted to CX) and faster!
  popad
  ret
put endp

;puts to v_buffer except for color 0  (very nice but slower than put)
;no clipping
align 4
put0 proc,src:dword,x1:dword,y1:dword,x2:dword,y2:dword
  local xl:dword,yl:dword
  pushad
  mov eax,x2
  sub eax,x1
  inc eax
  mov xl,eax
  mov eax,y2
  sub eax,y1
  inc eax
  mov yl,eax

  mov edi,v_buffer
  mov esi,src
  mov eax,y1
  mul v_xbd
  add edi,eax
  mov eax,x1
  mul v_byppd
  add edi,eax
  mov eax,xl
  mul v_bypp
  mov xl,eax
  mov ecx,eax
  mov ebx,yl
  mov eax,v_xbd
  sub eax,ebx
put0_1:
  cmp byte ptr[esi],0
  jz put0_2
  movsb
  dec ecx      ;
  jnz put0_1  ;loop but faster!
  jmp put0_3
put0_2:
  inc esi
  inc edi
  dec ecx
  jnz put0_1  ;loop
put0_3:
  add edi,eax
  mov ecx,xl
  dec ebx
  jnz put0_1  ;loop
  popad
  ret
put0 endp

comment ^
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
Mouse functions
 by : Peter Quiring
 date: April 10,96
 updated: May 1,96

 Generic mouse handler. (for 24/16/8 bit modes)
 It uses v_pp & v_gp to put/get pixels to draw the mouse ptr.
 See the examples of how to use this...

 Will have to reoptimize some day for speed, and get ride of v_pp,g_pp

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!^

.data
m_paws db 0    ;special printing mouse ptr
m_pp dd offset v_pp   ;are switched to v_?p2 when using v_buffer
m_gp dd offset v_gp
maxsiz equ 64  ;maximum size of cursor (x and y) must be a power of 2
moved db 0
backbuf db (maxsiz*maxsiz) dup (0) ;temp buffer
  dd ?   ;very important!!
;backbuf2 db (maxsiz*maxsiz) dup (0) ;temp buffer2 (used during v_copy)
;  dd ?   ;very important!!
cbb dd offset backbuf   ;current backbuf
align 4
atexitbak dd 0
prevx dw 0
prevy dw 0
mask1 dw ?      ;temp for button masks
inited db 0
align 4
xpos dw 0
ypos dw 0
cur dd 0 ;cursor
callbak dd 0  ;callback routine
curx dw 0 ;size x (max=32)
cury dw 0 ;size y (max=32)
hpx dw 0 ;hot point x
hpy dw 0 ;hot point y
mouse_con dw 0 ;cursor on?
but dw 0 ;button states
minx dw 0
miny dw 0
maxx dw (319)  ;These are absolute min that maxx,maxy can be
maxy dw (199) 
mx dw 0  ;the x after ratio shift
my dw 0  ;the y after ratio shift
once db 0  ;set the 1st time mouse_hand is used
xratio db 1
yratio db 1

.code

; NOTE : all functions that do clipping are not complete clipping
;      : the image must still have at least 1 pixel on screen (like a mouse ptr)
;      : these clipping puts/gets are for mouse functions only!
;clipping
;this is used by the mouse functions

align 4
put0c proc private,x:word,xl:word,y:word,yl:word
  local sadd:word
  local oxl:dword
  
  movzx eax,xl
  mov oxl,eax
  mov sadd,0
  mov esi,cur   ;source is the cursor
;do clipping
  .if x&08000h
    neg x
    movzx eax,x
    sub xl,ax
    add sadd,ax
    mul v_byppd
    add esi,eax
    mov x,0
  .endif
  .if y&08000h
    neg y
    movzx eax,y
    sub yl,ax
    mul oxl
    mul v_byppd
    add esi,eax
    mov y,0
  .endif
  mov ax,x
  add ax,xl
  cmp ax,v_x
  jbe @f
  sub ax,v_x  ;ax=overflow
  sub xl,ax
  add sadd,ax
@@:
  mov ax,y
  add ax,yl
  cmp ax,v_y
  jbe @f
  sub ax,v_y
  sub yl,ax
@@:
  ;start calc's
  mov cx,xl
  mov ax,sadd
  mul v_bypp
  movzx edx,ax  ;=sadd
  mov bx,x
put0_1:
  cmp byte ptr[esi],0   ;BUG: gotta check all bytes! (test w/ masks?)
  jz @f
  callp m_pp,x,y,dword ptr[esi]  ;EAX gone...
@@:
  inc x
  add esi,v_byppd
  dec cx
  jnz put0_1
put0_3:
  add esi,edx
  mov cx,xl
  mov x,bx
  inc y
  dec yl
  jnz put0_1  ;loop
  ret
put0c endp

;put clipping
;used by mouse support
align 4
putc proc private,x:word,xl:word,y:word,yl:word
  local sadd:word
  local oxl:dword

  movzx eax,xl
  mov oxl,eax
  mov sadd,0
  mov esi,cbb
;do clipping
  .if x&08000h
    neg x
    movzx eax,x
    sub xl,ax
    add sadd,ax
    mul v_byppd
    add esi,eax
    mov x,0
  .endif
  .if y&08000h
    neg y
    movzx eax,y
    sub yl,ax
    mul oxl
    mul v_byppd
    add esi,eax
    mov y,0
  .endif
  mov ax,x
  add ax,xl
  cmp ax,v_x
  jbe @f
  sub ax,v_x
  sub xl,ax
  add sadd,ax
@@:
  mov ax,y
  add ax,yl
  cmp ax,v_y
  jbe @f
  sub ax,v_y
  sub yl,ax
@@:
  mov cx,xl
  mov ax,sadd
  mul v_bypp
  movzx edx,ax
  mov bx,x
putc_1:
  callp m_pp,x,y,dword ptr[esi]   ;EAX destroyed
  add esi,v_byppd
  inc x
  dec cx
  jnz putc_1
  mov x,bx
  inc y
  add esi,edx
  mov cx,xl
  dec yl
  jnz putc_1  ;loop
  ret
putc endp

;clipping!
;this is used by the mouse functions
align 4
getc proc private,x:word,xl:word,y:word,yl:word
  local dadd:word
  local oxl:dword

  movzx eax,xl
  mov oxl,eax
  mov dadd,0
  mov edi,cbb
;do clipping
  .if x&08000h
    neg x
    movzx eax,x
    sub xl,ax
    add dadd,ax
    mul v_byppd
    add edi,eax
    mov x,0
  .endif
  .if y&08000h
    neg y
    movzx eax,y
    sub yl,ax
    mul oxl
    mul v_byppd
    add edi,eax
    mov y,0
  .endif
  mov ax,x
  add ax,xl
  cmp ax,v_x
  jbe @f
  sub ax,v_x
  sub xl,ax
  add dadd,ax
@@:
  mov ax,y
  add ax,yl
  cmp ax,v_y
  jbe @f
  sub ax,v_y
  sub yl,ax
@@:
  mov cx,xl
  mov ax,dadd
  mul v_bypp
  movzx edx,ax
  mov bx,x
@@:
  callp m_gp,x,y
  mov [edi],eax
  add edi,v_byppd
  inc x
  dec cx
  jnz @b
  add edi,edx
  mov cx,xl
  mov x,bx
  inc y
  dec yl
  jnz @b  ;loop
  ret
getc endp

mouse_hand  proto

mouse_init proc
  pushad
  mov ax,0
  int 33h
  cmp ax,0
  jnz @f
  popad
  mov eax,ERROR
  ret
@@:
  .if inited
    popad
    mov eax,ERROR
    ret
  .endif
  mov inited,1

  callp alloc_rmcallback,offset mouse_hand
  ;ax=seg dx=offset

  push _es
  mov _es,ax
;  mov dx,dx  ;dx is used by int 33h
  mov cx,0ffh  ;all movement & buttons cause a callback
  mov ax,0ch
  int 33h    ;c0.asm has patched int 10h,33h to load es,ds in RM w/ _es,_ds
  pop _es

  ;setup self-uninit
  mov eax,atexit
  mov atexitbak,eax
  mov eax,offset mouse_uninit
  mov atexit,eax

  ;setup X/Y
  .if v_init
    mov ax,v_x
    dec ax
    mov maxx,ax
    mov ax,v_y
    dec ax
    mov maxy,ax
  .endif

  popad
  xor eax,eax
  ret
mouse_init endp

mouse_uninit proc ;will self-uninit during atexit procedure
  pushad
  mov eax,atexit
  .if eax==mouse_uninit
    mov eax,atexitbak
    mov atexit,eax
  .endif

  ;disable callback thingy
  mov ax,_es
  push ax
  mov ax,0
  mov _es,ax
  mov dx,0
  mov cx,0h  ;no movement & buttons cause a callback
  mov ax,0ch
  int 33h
  pop ax
  mov _es,ax

  mov inited,0
  popad
  ret
mouse_uninit endp

;mask for callbacks
; CX = : bit # : cause because...
;          0     cursor moved
;          1     left pressed
;          2     left released
;          3     right pressed
;          4     right released
;        5-15    not used
          ;5,6=middle button (if 3 button)

align 4
mouse_setcursor proc,_cur:dword,_curx:byte,_cury:byte,_hpx:byte,_hpy:byte
  ;ONLY called when mouse is off or from the callback
  pushad
  mov eax,_cur
  mov cur,eax
  movzx ax,_curx
  and ax,maxsiz-1
  jnz @f
  mov ax,maxsiz
@@:        
  mov curx,ax
  movzx ax,_cury
  and ax,maxsiz-1
  jnz @f
  mov ax,maxsiz
@@:
  mov cury,ax
  movzx ax,_hpx
  mov hpx,ax
  movzx ax,_hpy
  mov hpy,ax
  popad
  ret
mouse_setcursor endp

;prints mouse cursor
align 4
_print proc private
  local tx:word,ty:word
  
  mov ax,mx
  sub ax,hpx
  mov tx,ax
  mov ax,my
  sub ax,hpy
  mov ty,ax

  .if m_paws
    mov m_pp,v_pp2
    mov m_gp,v_gp2
    callp getc,tx,curx,ty,cury
    callp put0c,tx,curx,ty,cury
    mov m_pp,v_pp
    mov m_gp,v_gp
    callp put0c,tx,curx,ty,cury
  .else
    callp getc,tx,curx,ty,cury
    callp put0c,tx,curx,ty,cury
  .endif

  ret
_print endp

align 4
erase proc private,one:byte
  local tx:word,ty:word

  mov ax,mx
  sub ax,hpx
  mov tx,ax
  mov ax,my
  sub ax,hpy
  mov ty,ax

  .if m_paws
    mov m_pp,v_pp2
    mov m_gp,v_gp2
    callp putc,tx,curx,ty,cury
    mov m_pp,v_pp
    mov m_gp,v_gp
    .if !one
      callp putc,tx,curx,ty,cury
    .endif
  .else
    callp putc,tx,curx,ty,cury
  .endif
  ret
erase endp

align 4
mouse_setpos proc,x:dword,y:dword
  pushad
  push mouse_con
  call mouse_off
  mov eax,x
  mov cl,xratio
  sal eax,cl   ;signed shift
  mov xpos,ax
  mov eax,y
  mov cl,yratio
  sal eax,cl
  mov ypos,ax
  call update
  pop ax
  .if ax==1
    call mouse_on
  .endif
  popad
  ret
mouse_setpos endp

align 4
mouse_hand proc private
  ;entry : ax=mask w/ condtion that cause this call
  ;      : bx=buttons (bit 0=left,bit 1=right)
  ;      : cx=horz pos  (I ignore)
  ;      : dx=vert pos  (")
  ;      : si=horz counts (mickeys)
  ;      : di=vert " (")
  pushad 
  pushf
  push ds
  push es
  push fs
  push gs
  mov ds,cs:seldata
  mov es,seldata
  mov fs,seldata
  mov gs,selzero
  cld
  mov mask1,ax
  ; FIX : Ver 1.1 : Ignore ax cause Win95 does not return it properly!!!
  ;       instead use BX which is easier anyways!
  mov but,bx

  .if once
    mov ax,prevx
    mov prevx,si
    sub si,ax
    mov ax,prevy
    mov prevy,di
    sub di,ax
  .else
    mov prevx,si
    mov prevy,di
    xor si,si
    xor di,di
    mov once,1
  .endif

  test mouse_con,1
  jz @f
  .if (si!=0) || (di!=0)
    pushad
    callp erase,0
    popad
    mov moved,1 ;it moved!
  .else
@@:
    mov moved,0
  .endif

;change X/Y values
  add xpos,si
  add ypos,di
  call update

done:
  .if callbak
    mov ax,mx
    mov bx,my
    mov cx,but
    mov dx,mask1  ;you should not use this under Win95 cause it's garbage!!
    call [callbak]  ;this is good for changing the cursor shape for new areas
  .endif
  .if (mouse_con)&&(moved)
    call _print
  .endif 
exit_hand:
  pop gs
  pop fs
  pop es
  pop ds
  popf
  popad
  retf
mouse_hand endp

align 4
mouse_setspd proc,xr:byte,yr:byte  ; (1-4 , 1-4)
  pushad
  push mouse_con
  call mouse_off

  mov al,xr
  dec al
  and al,3
  ;al=0-3
  mov xratio,al

  mov al,yr
  dec al
  and al,3
  ;al=0-3
  mov yratio,al

  call update
  pop ax
  .if ax==1
    call mouse_on
  .endif

  popad
  ret
mouse_setspd endp

align 4
mouse_setwin proc,x1:dword,y1:dword,x2:dword,y2:dword
  pushad
  push mouse_con
  call mouse_off
;x
  mov eax,x1
  mov minx,ax
  mov eax,x2
  mov maxx,ax
;y
  mov eax,y1
  mov miny,ax
  mov eax,y2
  mov maxy,ax

  call update
  pop ax
  .if ax==1
    call mouse_on
  .endif
  popad
  ret
mouse_setwin endp

;update x/y with maxx/maxy and ratios
align 4
update proc private
  mov ax,xpos
  mov cl,xratio
  sar ax,cl
  cmp ax,minx
  jge @f
  mov ax,minx
  sal ax,cl
  mov xpos,ax
  sar ax,cl
@@:
  cmp ax,maxx
  jle @f
  mov ax,maxx
  sal ax,cl
  mov xpos,ax
  sar ax,cl
@@:
  mov mx,ax

  mov ax,ypos
  mov cl,yratio
  sar ax,cl
  cmp ax,miny
  jge @f
  mov ax,miny
  sal ax,cl
  mov ypos,ax
  sar ax,cl
@@:
  cmp ax,maxy
  jle @f
  mov ax,maxy
  sal ax,cl
  mov ypos,ax
  sar ax,cl
@@:
  mov my,ax

  ret
update endp

align 4
mouse_on proc
  pushad
  .if cur==0
    popad ;fixed
    ret   ;cursor not setup yet!!  (which means x/y =0 and will crash _print)
  .endif
  cli
  .if mouse_con==0
    call _print
    mov mouse_con,1
    call v_setup
  .endif
  sti
  popad
  ret
mouse_on endp

align 4
mouse_off proc
  pushad
  cli
  .if mouse_con==1
    mov mouse_con,0
    callp erase,0
    call v_setup
  .endif
  sti
  popad
  ret
mouse_off endp

align 4
mouse_setuser proc,func:dword
  push eax
  .if (func==NULLPROC)
    mov callbak,0
  .else
    mov eax,func    ;may be 0
    mov callbak,eax
  .endif
  pop eax
  ret  
mouse_setuser endp

;grafix text support

.data?
align 4
font dd 0
fontxd label dword
fontx dw 0,0
fontyd label dword
fonty dw 0,0
fontcharsiz dd 0 ;size of each char (x*y*bypp)

.code
align 4
gsetfnt proc uses ebx ecx edx esi,fbuf:dword
  mov esi,fbuf
  movzx eax,[esi].fnthead.x
  movzx ebx,[esi].fnthead.y
  movzx ecx,[esi].fnthead.bypp

  mov fontx,ax
  mov fonty,bx
  mul ebx
  mul ecx
  mov fontcharsiz,eax
  mov font,esi

  xor eax,eax
  ret
gsetfnt endp

ggetfnt proc
  mov eax,font
  ret
ggetfnt endp

fnttmp db (sizeof fnthead) dup (?)

;FONT files now have a header
align 4
gloadfnt proc,nam:dword
  local h:word
  pushad
  mov font,0
  callp open,nam,0
  cmp eax,ERROR
  jnz @f
  popad
  mov eax,ERROR
  ret
@@:
  mov h,ax
  callp read,ax,offset fnttmp,sizeof fnthead
  .if eax!=sizeof fnthead
bad:
    callp free,font
    callp close,h
    popad
    mov eax,ERROR
    ret
  .endif
  mov esi,offset fnttmp
  cmp dptr[esi],01a544e46h  ;FNT,27
  jnz bad
  cmp [esi].fnthead.flg,0
  jnz bad
  movzx eax,[esi].fnthead.x
  movzx ebx,[esi].fnthead.y
  movzx ecx,[esi].fnthead.bypp

  mov fontxd,eax
  mov fontyd,ebx

  mul ebx
  mul ecx
  mov fontcharsiz,eax
  mov ebx,256  ;# of chars
  mul ebx
  mov ebx,eax
  add eax,sizeof fnthead
  callp malloc,eax
  .if eax==ERROR
    jmp bad
  .endif
  mov font,eax
  mov esi,offset fnttmp
  mov edi,eax
  mov ecx,sizeof fnthead
  rep movsb

  mov ecx,eax
  add ecx,sizeof fnthead
  callp read,h,ecx,ebx
  cmp eax,ebx
  jnz bad
  callp close,h
  popad
  mov eax,font
  ret
gloadfnt endp

align 4
gfntcolor proc,_font:dword,col:byte
  ;changes color of font
  ;for 256 color fonts only!!
  pushad
  mov edi,_font
  movzx eax,[edi].fnthead.x
  movzx ebx,[edi].fnthead.y
  .if [edi].fnthead.bypp>1
    popad
    mov eax,ERROR
    ret
  .endif

  add edi,sizeof fnthead
  mul ebx
  mov ebx,256
  mul ebx
  mov ecx,eax
  mov al,col
@@:
  .if bptr[edi]
    stosb
  .else
    inc edi
  .endif
  dec ecx
  jnz @b
  popad
  xor eax,eax
  ret
gfntcolor endp

align 4
gputch proc,x:dword,y:dword,char:byte
  pushad
  mov edi,v_buffer
  movzx eax,char
  mul fontcharsiz
  mov esi,eax
  add esi,font
  add esi,sizeof fnthead
  mov eax,y
  mul v_xbd
  add edi,eax
  mov eax,x
  mul v_byppd
  add edi,eax
  mov ax,fontx
  mul v_bypp
  mov cx,ax
  mov ebx,dptr[fonty]
  mov edx,v_xbd
  sub dx,cx  ;edx=v_buffer add
top:
  push cx
gputch_1:
  cmp byte ptr[esi],0   ;BUG should check all bytes (not each individually)
  jz gputch_2
  movsb
  jmp gputch_3
gputch_2:
  inc esi
  inc edi
gputch_3:
  dec cx
  jnz gputch_1
  pop cx
  add edi,edx
  dec ebx
  jnz top
  popad
  ret
gputch endp

align 4
gprintxy proc,x:dword,y:dword,s:dword
  pushad
  jmp gputstr_1
gputstr_2:
  callp gputch,x,y,bl
  mov eax,dptr[fontx]
  add x,eax
  inc s
gputstr_1:
  mov eax,s
  mov bl,[eax]
  cmp bl,13
  jz enter1
  cmp bl,10
  jz ignore
  cmp bl,0
  jnz gputstr_2
  popad
  ret
enter1:
  inc s
  mov x,0
  mov eax,dptr[fonty]
  add y,eax
  jmp gputstr_1
ignore:
  inc s
  jmp gputstr_1
gprintxy endp

align 4
printf_max proc private,str1:dword
  ;calculate maximum RAM required for string when expanded.
  ;assume all %'s=32 bytes for now.  EASY WAY OUT
  ;I should calculate each one but that's too much
  ;works this way anyways
  ;FIXED:Assumed max size=12 (but is 32 bytes)
  mov esi,str1
  xor eax,eax
@@:
  cmp byte ptr[esi],0
  jz @f
  cmp byte ptr[esi],'%'
  jz @@per
  inc eax
  inc esi
  jmp @b
@@per:
  inc esi
  cmp byte ptr[esi],'%'
  jnz @@per2
  inc esi
  inc eax
  jmp @b
@@per2:
  add eax,32        ;maximum size ! (dword in binary form!!)
  jmp @b
@@:
  inc eax ;for the 0
  ret
printf_max endp

align 4
gprintf proc,x:dword,y:dword,str1:dword,argv:vararg
  local siz:dword
  pushad
  mov ebx,ebp
  add ebx,20 ;now pts to varargs : locals(4),EBP(4),ret(4),x.y.str1(12),vararg
  mov esi,str1
  call _printf_siz_
  mov siz,eax
  sub esp,eax
  mov ebx,ebp
  add ebx,20 ;now pts to varargs : locals(4),EBP(4),ret(4),x.y.str1(12),vararg
  mov esi,str1
  mov edi,esp
  call _sprintf_
  mov eax,esp
  callp gprintxy,x,y,eax
  mov eax,esp
  callp strlen,eax
  movzx ecx,fontx
  mul ecx
  add x,eax
  add esp,siz
  popad
  xor eax,eax
  ret
gprintf endp

align 4
waitvsync proc uses edx
  mov dx,3dah
@@:
  in al,dx
  test al,8
  jnz @b
@@:
  in al,dx
  test al,8
  jz @b
  xor eax,eax
  ret
waitvsync endp

;sets 'v_buffer' for all other procs
;THIS functions must be called before all others
; requires a ptr to a RAM block for the temp area (must be resx*resy*bpp)
; EG: 320*200*1 ;for 320x200x256
align 4
gset proc,o:dword
  mov eax,o
  mov v_buffer,eax
  xor eax,eax
  ret
gset endp

align 4
setpal proc uses esi ecx edx,pal:dword
  call waitvsync
  mov esi,pal
  mov ecx,256*3
  mov dx,03c8h
  mov al,0
  out dx,al
  inc dx
  rep outsb
  xor eax,eax
  ret
setpal endp

align 4
setcol proc uses edx,n:byte,r:byte,g:byte,b:byte
  call waitvsync
  mov dx,3c8h
  mov al,n
  out dx,al
  inc dx
  mov al,r
  out dx,al
  mov al,g
  out dx,al
  mov al,b
  out dx,al
  xor eax,eax
  ret
setcol endp

align 4
hline proc,x1:dword,y1:dword,x2:dword,col:dword
  local l:dword
  pushad
  mov eax,x2
  .if x1>eax
    xchg eax,x1
    mov x2,eax
  .endif
  sub eax,x1
  inc eax
  mov l,eax

  mov edi,v_buffer
  mov eax,y1
  mul v_xbd
  add edi,eax
  mov eax,x1
  mul v_byppd
  add edi,eax
  .if v_bypp==1
    mov ecx,l
    mov al,bptr[col]
    rep stosb
  .elseif v_bypp==2
    mov ecx,l
    mov ax,wptr[col]
    rep stosw
  .else
    mov eax,l
@@:
    mov ecx,v_byppd
    lea esi,col
    rep movsb
    dec eax
    jnz @b
  .endif
  popad
  ret
hline endp

align 4
vline proc,x1:dword,y1:dword,y2:dword,col:dword
  local l:dword
  pushad
  mov eax,y2
  .if y1>eax
    xchg eax,y1
    mov y2,eax
  .endif
  sub eax,y1
  inc eax
  mov l,eax

  mov edi,v_buffer
  mov eax,y1
  mul v_xbd
  add edi,eax
  mov eax,x1
  mul v_byppd
  add edi,eax
  mov eax,l
  mul v_byppd
  mov ebx,eax
@@:
  mov ecx,v_byppd
  lea esi,col
  rep movsb
  add edi,v_xbd
  sub edi,v_byppd
  dec ebx
  jnz @b
  popad
  ret
vline endp

align 4
gbox proc,x1:dword,y1:dword,x2:dword,y2:dword,col:dword
  callp hline,x1,y1,x2,col
  callp vline,x1,y1,y2,col
  callp vline,x2,y1,y2,col
  callp hline,x1,y2,x2,col
  ret
gbox endp

;Fixed : Ver 1.1    (gcls never worked before)
align 4
gcls proc uses eax edi ecx edx
  mov eax,v_bpsl
  mul v_yd
  mov edi,v_linear
  mov ecx,eax
  mov bl,al
  xor eax,eax
  shr ecx,2
  rep stosd
  and bl,3
  .if bl
    mov cl,bl
    rep stosb
  .endif
  ret
gcls endp

align 4
setgmode proc,mode:word
  pushad
  .if mouse_con
    call mouse_off
  .endif
  .if mode==13h
    mov v_xb,320
    mov v_x,320
    mov v_bpsl,320
    mov v_y,200
    mov v_bypp,1
    mov v_bpp,8
    mov eax,0a0000h
    sub eax,_base
    mov v_linear,eax
    mov v_copy_add,0
    mov v_init,1
    call v_setup
  .else
    mov v_init,0
  .endif
  mov ax,mode
  int 10h
  popad
  xor eax,eax
  ret
setgmode endp

align 4
gboxfill proc,x1:dword,y1:dword,x2:dword,y2:dword,col:dword
  local xl:dword,yl:dword
  pushad
  mov eax,x2
  .if x1>eax
    xchg eax,x1
    mov x2,eax
  .endif
  sub eax,x1
  inc eax
  mov xl,eax
  mov eax,y2
  .if y1>eax
    xchg eax,y1
    mov y2,eax
  .endif
  sub eax,y1
  inc eax
  mov yl,eax

  mov edi,v_buffer
  mov eax,y1
  mul v_xbd
  add edi,eax
  mov eax,x1
  mul v_byppd
  add edi,eax
  mov eax,xl
  mul v_byppd
  mov xl,eax
  mov eax,v_xbd
  sub eax,xl  ;delta add
  mov ebx,yl
top:
  mov edx,xl
@@:
  lea esi,col
  mov ecx,v_byppd
  rep movsb
  dec edx
  jnz @b
  add edi,eax
  dec ebx
  jnz top
  popad
  ret
gboxfill endp

end





;year!
