unit PCX;

{ Requires Turbo/Borland Pascal for DOS, version 6 or later. }

{$G+}  { Requires 286 or better for Pusha and Popa assembler instructions }
{$X+}

(*  {$DEFINE RegisteredVersion}  *)

(* ----------------------------------------------------------------------

                                Version 4.0

                             Copyright (c) 1994
                             by Peter Donnelly
                              Skookum Software
                              1301 Ryan Street
                         Victoria BC Canada V8T 4Y8

   ͸
     Permission is granted for the non-commercial distribution and       
     private use of this source code. This is shareware; if you use all  
     or portions of it in programs you distribute, or make any other     
     public use of it, you are expected to pay a modest registration     
     fee. Registered users will receive the latest version of the code,  
     including support for 256-color Super-VGA modes. Please see the     
     READ.ME file for details.                                           
   ;
  
   This is a unit to read .PCX files and put them in displayable form. The
   actual work of decoding the file and moving the data into memory is done
   in assembler and is very fast.

   The following display modes are supported for VGA cards:

          Mode      Resolution    Colors
          ~~~~      ~~~~~~~~~~    ~~~~~~
          $0D       320 x 200        16
          $0E       640 x 200        16
          $10       640 x 350        16
          $12       640 x 480        16
          $13       320 x 200       256

   The following modes are supported for Super-VGA cards with VESA BIOS:

       *  $100      640 x 400       256
       *  $101      640 x 480       256
          $102      800 x 600        16
       *  $103      800 x 600       256
       *  $105     1024 x 768       256
       *  $107     1280 x 1024      256

     ( * Supported by registered version only. )

   The 256-color modes are not supported for PCX files with no palette
   information, i.e. those produced by versions of Paintbrush earlier
   than 3.0.

   Note that the video mode is set directly through BIOS, never with the
   BGI. If you want to use any BGI functions after the PCX file is
   displayed, you'll have to rewrite the mode-setting routines (for modes
   supported by the BGI) to use InitGraph and SetGraphMode.

   References:
   ~~~~~~~~~~
   Richard F. Ferraro, "Programmer's Guide to the EGA and VGA Cards"
   (Addison-Wesley, 1988).

   Richard Wilton, "Programmer's Guide to PC Video Systems" Second
   Edition (Microsoft Press, 1994).

   "Technical Reference Manual [for Paintbrush]" (Zsoft, 1988). The
   information in this booklet is also found in a file distributed with
   at least some versions of Microsoft/PC Paintbrush.
   
   Further information on programming for VESA-compliant Super-VGA cards 
   can be obtained from:

          Video Electronics Standards Association
          2150 North First St., Suite 440
          San Jose CA 95131-2029
          Phone (408) 435-0333
*)

{ ======================================================================= }

INTERFACE

uses DOS, CRT;

CONST
        Wipe: boolean = true;          { Can be passed to SetMode }
        AutoSet = 0;                   { ditto }
        NumModes = 11;
        OurModes: array[1..NumModes] of word =
                  ($0D, $0E, $10, $12, $13, $100,
                   $101, $102, $103, $105, $107);


TYPE    RGBrec = record
                   RedVal, GreenVal, BlueVal: byte;
                 end;

        PCXHeaderRec = record
                         Signature: byte;
                         Version: byte;
                         Code: byte;
                         BitsPerPlane: byte;
                         XMin, YMin, XMax, YMax: word;
                         HRes, VRes: word;
                         Palette: array[0..15] of RGBRec;
                         Reserved: byte;
                         NumPlanes: byte;
                         BytesPerLine: word;
                         OtherStuff: array[69..128] of byte;
                       end;

        VESAInfoRec = record
                        Signature: array[0..3] of char;
                        Version: word;
                        OEMptr: pointer;
                        Capabilities: array[0..3] of byte;
                        ModePtr: pointer;
        { There are reports of some VESA BIOSes returning more than 256
          bytes from function 0, so this record is padded a bit. }
                        Reserved: array[0..256] of byte;
                      end;

        ModeInfoRec = record
                        Attributes: word;
                        WindowA_atts, windowB_atts: byte;
                        GranuleKb, WindowSize: word;
                        WindowAstart, windowBstart: word;
                        FunctionAddr: longint;
                        BytesPerLine: word;
                        OtherStuff: array[19..256] of byte;
                      end;

VAR     PCXFilename: pathstr;
        PCXHeader: PCXHeaderRec;
        ModeInfo: ModeInfoRec;
        FileError: word;
        RGBpal: array[0..15] of RGBrec;
        RGB256: array[0..255] of RGBrec;
        VESAInfo: VESAInfoRec;
        ErrorStr: string;

function OpenFile(PicFileName: pathstr; var PicFile: file): boolean;
procedure ReportError(error: word);
function DetectVESA: boolean;
function HardwareSupports(mode: word): boolean;
function WeSupport(mode: word): boolean;
procedure SetMode(Mode: word; Clear: boolean);
function GetActiveWindow: byte;
procedure GetModeInfo(Mode: word);
procedure SetColorRegisters(var palrec);
function BestMode(Header: PCXHeaderRec): word;
procedure Read16(var PicFile: file; Mode: word);
procedure ReadVGA256(var PicFile: file);
procedure ReadSVGA256(var PicFile: file; Mode: word);
procedure ReadIt(PicFileName: pathstr; Mode: word);

{========================================================================}

IMPLEMENTATION

CONST   BufferSize = 65521;   { Largest possible }

VAR
        Regs: registers;
        GranuleBytes: word;
        Scratch: pointer;
        ColumnCount: word;
        Plane: word;
        BytesPerLine: word;
        RepeatCount: byte;
        DataLength: word;
        Window: byte;
        WindowStep: word;
        VideoSeg, VideoOffs: word;
        LineEnd, PicWidth, ScreenWidth: integer;
        Skip: integer;


function OpenFile(PicFileName: pathstr; var PicFile: file): boolean;

begin
assign(PicFile, PicFileName);
{$I-} reset(PicFile, 1);
blockread(PicFile, PCXHeader, 128);  {$I+}
OpenFile:= IOresult = 0;
end;


procedure ReportError(error: word);

begin
case Error of
  1: ErrorStr:= 'Could not open file.';
  2: ErrorStr:= 'No palette information in file.';
  3: ErrorStr:= 'Picture is too wide for requested video mode.';
  4: ErrorStr:= 'Number of colors in file does not match selected mode.';
  5: ErrorStr:= 'Unsupported picture format.';
end;
end;


function Get256Palette(var TheFile: file; var PaletteStart: longint): boolean;

{ TheFile must be open. }

var    x: integer;
       PaletteFlag: byte;

begin
PaletteStart:= filesize(TheFile) - 769;

{ The last 769 btes of the file are palette information, starting with a
   one-byte flag. Each group of three bytes represents the RGB values of
   one of the color registers. We take the 6 most significant bits
   to bring the values within the range 0-63 expected by the registers. }

seek(TheFile, PaletteStart);
blockread(TheFile, PaletteFlag, 1);
if (PaletteFlag <> 12) or (PCXHeader.Version < 5) then
begin
  FileError:= 2;
  Get256Palette:= false;
  exit;
end;
blockread(TheFile, RGB256, 768);         { Get palette info. }
for x:= 0 to 255 do
with RGB256[x] do
begin
  RedVal:= RedVal shr 2;
  GreenVal:= GreenVal shr 2;
  BlueVal:= BlueVal shr 2;
end;
SetColorRegisters(RGB256);
Get256Palette:= true;
end;  { Get256Palette }

(* The above code is called before the image is displayed, to set the
   proper colors. The image is then displayed line by line as it is
   read from the file. On slower machines this may lead to an undesired
   effect. An alternative is to set the entire palette to black while
   the image is being written to video memory, then restore the palette
   to the image colors, so that the picture appears all at once.
   This is done as follows:

   var   blackpal: array[0..255] of RGBrec;

   ...
   fillchar(blackpal, 768, 0);
   SetColorRegisters(blackpal);
   { Now read the palette and decode the image, but don't call
     SetRegisters(RBG256) until the end. )
*)


procedure GetSkip(ScreenWidth: word; var Skip, LineEnd: integer);

{ Calculate how many pixels have to be skipped when advancing to the
  next line, so that files of less than screen width can be displayed. }

begin
LineEnd:= PCXHeader.BytesPerLine;      { Used as counter in assembler }
Skip:= ScreenWidth - LineEnd;
if Skip < 0 then FileError:= 3;        { Too wide }
end;


function BestMode(Header: PCXHeaderRec): word;

{ Attempts to match the mode to the originating format, but goes to a
  higher resolution if the image doesn't fit the screen. }

var   M: word;

procedure Try(Mode: word);

begin
if HardwareSupports(Mode) and WeSupport(Mode)
  then M:= Mode;
end;

function Fits: boolean;

begin
Fits:= (Header.XMax < Header.HRes) and (Header.YMax < Header.VRes);
end;

begin
if Header.NumPlanes = 1 then
begin
  M:= $13;
  if (Header.HRes > 320) or (not Fits) then Try($101);
  if (Header.HRes > 640) or (not Fits) then Try($103);
  if (Header.HRes > 800) or (not Fits) then Try($105);
  if (Header.HRes > 1024) or (not Fits) then Try($107);
end
else if Header.NumPlanes = 4 then
begin
  if Header.HRes <= 320 then M:= $0D else M:= $0E;
  if (Header.VRes > 200) or (not Fits) then Try($10);
  if (Header.VRes > 350) or (not Fits) then Try($12);
  if (Header.VRes > 480) or (not Fits) then Try($102);
end
else
begin
  FileError:= 5;
  M:= $FFFF;
end;
BestMode:= M;
end;


function WeSupport(Mode: word): boolean;

{ True if requested mode is supported by PCX.PAS }

var  x: integer;
     InThere: boolean;

begin
InThere:= false;
for x:= 1 to NumModes do
  if Mode = OurModes[x] then InThere:= true;
WeSupport:= InThere;
end;

{ -------------------------- BIOS calls --------------------------------- }

function DetectVESA: boolean;

var  Signature: string[4];
     IsVESA: boolean;

begin
IsVESA:= False;
Regs.AX:= $4F00;                { VESA Get SuperVGA Info function }
Regs.ES:= seg(VESAInfo);        { Info returns in VESAInfo record }
Regs.DI:= ofs(VESAInfo);
intr($10, regs);
if (Regs.AH = 0) then           { Function failed if AH <> 0 }
begin
  Signature[0]:= #4;
  Move(VESAInfo.Signature, Signature[1], 4);
  if Signature = 'VESA' then IsVESA:= true;
end;
DetectVESA:= IsVESA;
end;


function HardwareSupports(Mode: word): boolean;

{ VESA function $4F00 returns, among other things, a pointer to a list
  of the video modes supported. The list terminates in $FFFF. }

type  ModeList = array[0..255] of word;

var  Supported: boolean;
     Modes: ^ModeList;
     x: integer;

begin
Supported:= false;
if Mode >= $100 then
begin
  if DetectVESA then    { Fills info record }
  begin
    x:= 0;
    Modes:= VESAInfo.ModePtr;
    repeat
      if Modes^[x] = Mode then Supported:= true;
      inc(x);
    until Supported or (Modes^[x] = $FFFF) or (x = 256);
  end else Halt;        { if VESA not detected }
end
else Supported:= true;         { Assume VGA card }
HardwareSupports:= Supported;
end;


procedure SetMode(Mode: word; Clear: boolean);

begin
if Mode >= $100 then
{ --- VESA Super-VGA modes --- }
begin
  if not Clear then Mode:= Mode or $8000;
                             { Set bit 15 to preserve video memory }
  Regs.AX:= $4F02;
  Regs.BX:= Mode;
  intr($10, Regs);
end else
{ --- Standard VGA modes --- }
begin
  if not Clear then Mode:= Mode or $80;
                             { Set bit 7 to preserve video memory }
  Regs.AH:= 0;
  Regs.AL:= lo(Mode);
  intr($10, Regs);
end;
end;  { SetMode }


procedure SetColorRegisters(var PalRec);

{ We can't use the BGI's SetRGBPalette even for the modes supported by
  the BGI, because it won't work unless the BGI initializes the mode
  itself. }

{ PalRec is a string of 768 bytes containing the RGB data. }

begin
Regs.AH:= $10;               { BIOS color register function }
Regs.AL:= $12;               { Subfunction }
Regs.ES:= seg(PalRec);       { Address of palette info }
Regs.DX:= ofs(PalRec);
Regs.BX:= 0;                 { First register to change }
Regs.CX:= $100;              { Number of registers to change }
intr($10, Regs);             { Call BIOS }
end;


procedure SetPalette(var Palette);

{ Replaces the BGI SetAllPalette procedure. Palette is a 17-byte record
  of the contents of the 16 EGA/VGA palette registers plus the overscan
  register. }

begin
Regs.AH:= $10;
Regs.AL:= 2;
Regs.ES:= seg(Palette);
Regs.DX:= ofs(Palette);
intr($10, Regs);
end;

procedure GetModeInfo(mode: word);

{ Puts information on the selected VESA mode into the ModeInfo record. }

begin
Regs.AX:= $4f01;
Regs.CX:= Mode;
Regs.ES:= seg(ModeInfo);
Regs.DI:= ofs(ModeInfo);
intr($10, Regs);
end;


function GetActiveWindow: byte;

{ Double-checks that we're using the right window for addressing the
  video memory in VESA modes. }

begin
Regs.AX:= $4f05;
Regs.BH:= 1;
intr($10, Regs);
GetActiveWindow:= Regs.BL;
end;


{ -------------------------- VGA 16-color files ------------------------- }

procedure Decode16; assembler;

asm

(* Registers used:

   AL   data byte to be written to video
   AH   data bytes per scan line
   BX   end of input buffer
   CL   number of times data byte is to be written
   DL   current column in scan line
   ES   output segment
   DI   index into output buffer
   DS   preserved for access to Pascal variables
   SI   index into input buffer
   BP   current color plane
*)

push    bp

{ The first section is initialization done on each run through the
  input buffer. }

@startproc:
mov     bp, Plane           { plane in BP }
mov     di, VideoOffs       { index into video segment }
mov     ah, byte ptr BytesPerLine  { line length in AH }
mov     dx, ColumnCount     { column counter }
mov     bx, DataLength      { no. of bytes to read }
xor     cx, cx              { clean up CX for loop counter }
mov     cl, RepeatCount     { count in CX }
les     si, Scratch         { index into input buffer in SI }
  { ES  not significant here - we don't use LDS because we want to
    preserve DS }
  { We have to adjust datalength for comparison with SI. Pascal pointers
    are normalized, but the offset can still be 0 or 8. }
mov     es, VideoSeg        { video display segment }
add     bx, si
cld                         { clear DF for stosb }
cmp     cl, 0               { was last byte a count? }
jne     @multi_data         { yes, so next is data }
jmp     @getbyte            { no, so find out what next is }

{ The data in the .PCX file is organized by color plane, by line; that is,
  all the data for plane 0 for line 1, then for plane 1, line 1, etc.
  Writing the data to display memory is just a matter of masking out the
  other planes while one plane is being written to. This is done with the
  map mask register in the sequencer. All the other weird and wonderful
  registers in the EGA/VGA do just fine with their default settings, thank
  goodness. }

@writebyte:
stosb                       { AL into ES:DI, inc DI }
inc     dl                  { increment column }
cmp     dl, ah              { reached end of scanline? }
je      @doneline           { yes }
loop    @writebyte          { no, do another }
jmp     @getbyte            {   or get more data }
@doneline:
shl     bp, 1               { shift to next plane }
cmp     bp, 8               { done 4 planes? }
jle     @setindex           { no }
mov     bp, 1               { yes, reset plane to 1 }
add     di, Skip            { skip to start of next scanline }
mov     dx, ScreenWidth
add     LineEnd, dx
jmp     @setplane
@setindex:
sub     di, dx              { reset index to start of same line }
@setplane:
push    ax                  { save AX }
cli                         { no interrupts }
mov     ax, bp              { plane is 1, 2, 4, or 8 }
mov     dx, 3C5h            { sequencer data register }
out     dx, al              { mask out 3 planes }
sti                         { enable interrupts }
pop     ax                  { restore AX }
xor     dx, dx              { reset column count }
loop    @writebyte          { do it again, or fetch more data }

{ --- Loop through input buffer --- }

{ Here's how the data compression system works. Each byte is either image
  data or a count byte that tells how often the next byte is to be
  repeated. The byte is image data if it follows a count byte, or if
  either of the top 2 bits is clear. Otherwise it is a count byte, with
  the count derived from the lower 6 bits. }

@getbyte:                   { last byte was not a count }
cmp     si, bx              { end of input buffer? }
je      @exit               { yes, quit }
push    ds                  { save Pascal's DS }
push    si
lds     si, Scratch         { input segment into DS }
pop     si
lodsb                       { get a byte from DS:SI into AL, increment SI }
pop     ds                  { Restore Pascal's DS }
cmp     al, 192             { test high bits }
jb      @one_data           { not set, it's data to be written once }
 { It's a count byte: }
xor     al, 192             { get count from 6 low bits }
mov     cl, al              { store repeat count }
cmp     si, bx              { end of input buffer? }
je      @exit               { yes, quit }
@multi_data:
push    ds                  { save Pascal's DS }
push    si
lds     si, Scratch         { segment of input buffer into DS }
pop     si
lodsb                       { get a byte from DS:SI into AL, increment SI }
pop     ds                  { Restore Pascal's DS }
jmp     @writebyte          { write it CL times }
@one_data:
mov     cl, 1               { write byte once }
jmp     @writebyte

{ --- Finished with buffer --- }

@exit:
mov     Plane, bp           { save status for next run thru buffer }
mov     RepeatCount, cl
mov     ColumnCount, dx
mov     VideoOffs, di
pop     bp
end;  { asm }

{ --------------- Main procedure for VGA 16-color files ----------------- }

procedure Read16(var PicFile: file; Mode: word);

TYPE
        PaletteBytes = array[0..2] of byte;

VAR
        Entry, Gun, PCXCode: byte;
        PalRegs: array[0..16] of byte;

begin   { READ16 }
if Mode >= $100 then
begin
  GetModeInfo(Mode);
  ScreenWidth:= ModeInfo.BytesPerLine;
end
else if Mode = $0D then ScreenWidth:= 40
else ScreenWidth:= 80;
GetSkip(ScreenWidth, Skip, LineEnd);
if FileError <> 0 then exit;
if PCXHeader.NumPlanes <> 4 then
begin
  FileError:= 4;
  exit;
end;
VideoOffs:= 0;            { Index into video memory }
VideoSeg:= $A000;         { Segment of video memory }
port[$3C4]:= 2;           { Index to map mask register }
Plane:= 1;                { Initialize plane }
port[$3C5]:= Plane;       { Set sequencer to mask out other planes }

{ --- Decipher 16-color palette --- }

{  The palette information is stored in bytes 16-63 of the header. Each of
   the 16 palette slots is allotted 3 bytes - one for each primary color.
   Any of these bytes can have a value of 0-255. However, the VGA is
   capable only of 6-bit RGB values (making for 64x64x64 = 256K possible
   colors), so we take only the 6 most significant bits from each PCX
   color value.

   In 16-color modes, the VGA uses the 16 CGA/EGA palette registers.
   However, the actual color values (18 bits per slot) won't fit here,
   so the palette registers are used as pointers to 16 of the 256 color
   registers, which hold the RGB values.

   What we have to do is extract the RGB values from the PCX header, put
   them in the first 16 color registers, then set the palette to point to
   those registers. }

BytesPerLine:= PCXHeader.BytesPerLine;
for Entry:= 0 to 15 do
begin
  for Gun:= 0 to 2 do
  begin
    PCXCode:= PaletteBytes(PCXHeader.Palette[entry])[Gun];
    with RGBPal[Entry] do
    case gun of
      0: RedVal:= PCXCode shr 2;
      1: GreenVal:= PCXCode shr 2;
      2: BlueVal:= PCXCode shr 2;
    end;
  end;  { gun }
  PalRegs[Entry]:= Entry;
end;  { Entry }
PalRegs[16]:= 0;                       { overscan color }
SetColorRegisters(RGBPal);             { RGB values into registers 0-15 }
SetPalette(PalRegs);                   { point to registers 0-15 }

{ --- Read and decode the image data --- }

RepeatCount:= 0;                       { Initialize assembler vars. }
ColumnCount:= 0;
seek(PicFile, 128);
getmem(Scratch, BufferSize);           { Allocate scratchpad }
repeat
  blockread(PicFile, Scratch^, BufferSize, DataLength);
  Decode16;                           { Call assembler routine }
until eof(PicFile);
port[$3C5]:= $F;                       { Reset mask map }
freemem(Scratch,BufferSize);           { Discard scratchpad }
end;  { READ16 }

{ ------------------------- VGA 256-color files ------------------------- }

procedure Decode256; assembler;

(* Registers used:

   AL   data byte to be written to video
   AX   temporary storage of GranuleBytes
   BX   end of input buffer
   CL   number of times data byte is to be written
   DX   temporary storage of EndOfLine
   ES   segment of output buffer
   DI   index into output buffer
   DS   segment of input buffer
   SI   index into input buffer
*)

asm
les     si, Scratch         { index into input buffer in SI }
  { ES  not used here - we don't use LDS because we want to preserve DS }
mov     bx, DataLength      { end of input buffer }
add     bx, si              { adjust datalength - SI may not be 0 }
mov     es, VideoSeg        { base address of output window }
mov     di, VideoOffs       { index into window }
xor     cx, cx              { clean up loop counter }
mov     cl, RepeatCount     { restore count from last byte }
cld                         { clear DF }
cmp     cl, 0               { was last byte a count? }
jne     @multi_data         { yes, so next is data }

{ --- Loop through input buffer --- }

@getbyte:                   { last byte was not a count }
cmp     di, LineEnd         { reached end of line? }
jb      @NoLineEnd          { no }
add     di, Skip
mov     dx, ScreenWidth
add     LineEnd, dx
@NoLineEnd:
cmp     si, bx              { end of input buffer? }
je      @exit               { yes, quit }
push    ds                  { save Pascal's data segment }
push    si                  { Get segment of input buffer into DS }
lds     si, Scratch         {   while preserving SI }
pop     si
lodsb                       { get byte from DS:SI into AL, increment SI }
pop     ds                  { restore Pascal's DS }
cmp     al, 192             { test high bits }
jb      @one_data           { not set, not a count }
{ It's a count byte }
xor     al, 192             { get count from 6 low bits }
mov     cl, al              { store repeat count }
cmp     si, bx              { end of input buffer? }
je      @exit               { yes, quit }
@multi_data:
push    ds                  { save Pascal's data segment }
push    si
lds     si, Scratch
pop     si
lodsb                       { get byte from DS:SI into AL, increment SI }
pop     ds                  { restore Pascal's DS }
rep     stosb               { write byte CX times }
jmp     @getbyte
@one_data:
stosb                       { byte into video }
jmp     @getbyte

{ --- Finished with buffer --- }

@exit:
mov     VideoOffs, di     { save status for next run thru buffer }
mov     RepeatCount, cl
end;  { asm }

{ --------------- Main procedure for VGA 256-color files --------------- }

procedure ReadVGA256(var PicFile: file);

var     TotalRead: longint;
        PaletteStart: longint;

begin

{ --- Set palette  --- }
if not Get256Palette(PicFile, PaletteStart) then exit;
ScreenWidth:= 320;
GetSkip(ScreenWidth, Skip, LineEnd);
if FileError <> 0 then exit;
if PCXHeader.NumPlanes <> 1 then
begin
  FileError:= 4;
  exit;
end;

{ --- Read image data --- }
seek(PicFile, 128);
TotalRead:= 128;
repeatcount:= 0;                           { Initialize assembler vars. }
VideoOffs:= 0;
VideoSeg:= $A000;
getmem(Scratch, BufferSize);                { Allocate scratchpad }
repeat
  blockread(PicFile, Scratch^, BufferSize, DataLength);
  inc(TotalRead, DataLength);
  if (TotalRead > PaletteStart) then
      dec(DataLength, TotalRead - PaletteStart);
  Decode256;
until (eof(PicFile)) or (TotalRead>= PaletteStart);
end;  { ReadVGA256 }

{ ------------------------- SVGA 256-color files ------------------------ }

{$IFDEF RegisteredVersion}
  {$I SVGA256.PAS}
{$ENDIF}

{$IFNDEF Registered}

procedure ReadSVGA256(var PicFile: file; Mode: word);

begin
  SetMode(3, Wipe);
  Writeln('Support for this video mode is available only to registered');
  Writeln('users of PCX.PAS. Please see READ.ME for details.');
  Writeln;
end;

{$ENDIF}

{ -------------------------- Main Procedure ----------------------------- }

procedure ReadIt(PicFileName: pathstr; Mode: word);

var  PCXfile: file;

begin
FileError:= 0;
if not OpenFile(PicFileName, PCXFile) then           { Gets PCX header }
begin
  FileError:= 1;
  exit;
end;
{ Trap CGA files }
if (PCXHeader.BitsPerPlane < 8) and (PCXHeader.NumPlanes = 1) then
begin
  FileError:= 5;
  close(PCXFile);
  exit;
end;
if Mode = AutoSet then Mode:= BestMode(PCXHeader);
SetMode(Mode, Wipe);
case Mode of
  $0D, $0E, $10, $12, $102: Read16(PCXFile, Mode);
  $13: ReadVGA256(PCXFile);
  $100, $101, $103, $105, $107: ReadSVGA256(PCXFile, Mode);
end;
close(PCXFile);
end;

{ ---------------------- Implementation section ------------------------- }

BEGIN
END.

