#include <avr/wdt.h>
#include <avr/io.h>
#include <avr/pgmspace.h>
#include <avr/eeprom.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include <stdbool.h>
#include <stdint.h>
#include "chars.h"

//each packet begins with a command byte, followed by data
#define CMD_RESET			0x00
#define CMD_FLIP			0x01	//flip backbuffer to frontbuffer
#define CMD_SET_ADDR		0x02	//followed by one byte - desired i2c addr in lower 7 bits
#define CMD_SET_ADDR_UID	0x03	//followed by a valid uid, only device with that uid sets its address to the following byte
#define CMD_GET_UID			0x04	//get device uid - collisions are expected :) - 16 bytes long
#define CMD_DRAW_BMP		0x05	//followed by 5 bytes in char format
#define CMD_DRAW_CHAR		0x06	//followed by one byte - the char to draw
#define CMD_GET_VER			0x07	//read hw ver (1 byte), fw ver(1 byte)
#define CMD_FW_UPDATE		0x08	//followed by update chunk (struct FwUpdatePacket)
#define CMD_SET_FLAGS		0x09	//followed by a single u8, low bit means flip H, next bit means flipV, dot doesnt flip obviously. value persists in eeprom
#define CMD_GET_FLAGS		0x0A	//read flags byte

#define UID_LEN				16

#define NUM_EEPROM_CHARS	25
#define NUM_COLS			6

#define FLAG_FLIP_H			0x01
#define FLAG_FLIP_V			0x02

#define MY_HW_VER			1
#define MY_FW_VER			2


struct FwUpdatePacket {
	uint16_t addr;
	uint8_t data[PROGMEM_PAGE_SIZE];
};

struct EeData {
	uint8_t i2cAddr;						//0x00 or 0xFF if not valid, must be 0x01..0x7F
	uint8_t flags;							//0x01 - flipH, 0x02 - flipV
};

#define EE ((struct EeData*)EEPROM_START)

//LED state
static uint8_t mData[NUM_COLS] = {0};	//in our order

//i2c state
static uint8_t mPacket[68];
static uint8_t mBackBuffer[NUM_COLS];	//in human format


#define mPacketLen		GPIOR0
#define mCurCol			GPIOR1
#define mIsGeneralCall	GPIOR2

//row to bi tmappings
static const uint8_t mRowToBit[] = {0x02, 0x04, 0x08, 0x80, 0x40, 0x10, 0x20};


ISR(__vector_lvl_1)		//we only use level 1 for LEDs
{
	static const uint8_t col2b[] = {0x08, 0x20, 0x00, 0x10, 0x00, 0x00};
	static const uint8_t col2c[] = {0x00, 0x00, 0x02, 0x00, 0x04, 0x00};
	static const uint8_t col2cInv[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x08};
	uint8_t val, rowB = 0, rowC = 0, rowCinv = 0, col = mCurCol;
	
	TCB0.INTFLAGS = 1;
	
	//get next row data val and column selector value
	rowB = col2b[col];
	rowC = col2c[col];
	rowCinv = col2cInv[col];
	val = mData[col];
	
	//turn off all cols
	//cols 1..5 are driven by FETs and thus inverted, col6 isnt
	PORTB.OUTSET = 0x38;
	PORTC.OUTSET = 0x06;
	PORTC.OUTCLR = 0x08;
	
	//set new rows data using only hw-atomic ops
	PORTA.OUTSET = 0xFE;
	PORTA.OUTCLR = val &~ (uint8_t)1	/* be extra careful with updi pin */;
	
	//turn on cur row
	PORTB.OUTCLR = rowB;
	PORTC.OUTCLR = rowC;
	PORTC.OUTSET = rowCinv;

	//advance to next col
	if (++col == NUM_COLS)		//also indicates last col and this compiles to smaller code (do not ask!)
		col = 0;
	mCurCol = col;
}

static void nvmWait(void)
{
	while (NVMCTRL.STATUS & NVMCTRL_FBUSY_bm);
}

static void nvmCommand(uint8_t cmd)
{
	CCP = CCP_SPM_gc;
	NVMCTRL.CTRLA = cmd;
	nvmWait();
}

static void nvmClearBuffer(void)
{
	nvmCommand(4);
}

static void nvmEraseWrite(void)
{
	nvmCommand(3);
}

static void eeWrite(void* dst_, const void* src_, uint16_t len)
{
	uintptr_t d = (uintptr_t)dst_;
	uintptr_t s = (uintptr_t)src_;
	
	nvmClearBuffer();
	while (len--)
	{
		*(volatile uint8_t*)d++ = *(const uint8_t*)s++;
		if ((d % EEPROM_PAGE_SIZE) == 1)
			nvmEraseWrite();
	}
	if ((d % EEPROM_PAGE_SIZE) != 1)
		nvmEraseWrite();
}

static void flashWrite(uintptr_t dst, const void *src_)	//write a page (PROGMEM_PAGE_SIZE bytes)
{
	const uint8_t *src = (const uint8_t*)src_;
	uint8_t i;
	
	nvmClearBuffer();
	for (i = 0; i < PROGMEM_PAGE_SIZE; i++)
		*(volatile uint8_t*)dst++ = *src++;
	nvmEraseWrite();
}

static void eeFwUpdate(const struct FwUpdatePacket* updt)
{
	flashWrite(updt->addr, updt->data);	//chip security makes sure we cannot write where w eshouldnt
}

static void eeSetSelfAddr(uint8_t addr)
{
	eeWrite(&EE->i2cAddr, &addr, 1);
}

static void eeSetFlags(uint8_t flags)
{
	eeWrite(&EE->flags, &flags, 1);
}

static void setFlags(uint8_t flags)
{
	eeSetFlags(flags);	//read from EE on use, so no need for ram backing
}

static void setSelfAddr(uint8_t addr)
{
	TWI0.SADDR = (addr << 1) + 1;
	eeSetSelfAddr(addr);
}

static void drawChar(uint8_t ch)
{
	const uint8_t *data = NULL;
	uint8_t c, r, v, m, t;
	
	//clear backbuffer
	for (c = 0; c < NUM_COLS; c++)
		mBackBuffer[c] = 0;
	
	if (ch >= FIRST_CHAR && (ch - (uint8_t)FIRST_CHAR < (uint8_t)NUM_CHARS))	//search rom
		data = mChars + (uint16_t)(uint8_t)(ch - FIRST_CHAR) * (uint16_t)(uint8_t)SIZE_OF_CHAR_IMG;
	else																		//nope
		return;
	
	for (c = 0; c < SIZE_OF_CHAR_IMG; c++) {
		
		v = *data++;
		t = 0;
		
		if (!c && (v & 0x80))	//dot
			mBackBuffer[NUM_COLS - 1] = 1;
		
		for (r = 0, m = 1; r < 7; r++, v >>= 1, m <<= 1) {
			if (v & 1)
				t |= m;
		}
		
		mBackBuffer[c] = t;
	}
}

static uint8_t toOurRowOrder(uint8_t data)
{
	uint8_t bit, idx, ret;
	
	for (ret = 0, bit = 1, idx = 0; !(bit & 0x80); bit <<= 1, idx++)
	{
		if (data & bit)
			ret |= mRowToBit[(EE->flags & FLAG_FLIP_V) ? (6 - idx) : idx];
	}
	
	return ret;
}

int main()
{
	//enable compact vector table, vectors in app space (not bl) set LED interrupt to be high prio, turn on ints
	{
		CCP = CCP_IOREG_gc;
		CPUINT.CTRLA = CPUINT_CVT_bm;
		CPUINT.LVL1VEC = TCB0_INT_vect_num;
		asm volatile("sei":::"memory");
	}
	
	//clock up to 20MHz
	{
		CCP = CCP_IOREG_gc;
		CLKCTRL.MCLKCTRLB = 0;
	}
	
	//gpio config
	{
		//LEDN: A1..7
		//LEDP: B3..5 C1..3
		//I2C: B0..1
		
		//set outputs
		PORTA.DIRSET = 0xFE;
		PORTB.DIRSET = 0x3B;
		PORTC.DIRSET = 0x0E;
		
		//leds off
		PORTA.OUTSET = 0xFE;
		PORTB.OUTSET = 0x38;
		PORTC.OUTSET = 0x06;
		PORTC.OUTCLR = 0x08;
	}
	
	//display timer
	{
		TCB0.CTRLA = 0;			//off
		TCB0.CTRLB = 0;			//periodic int mode
		TCB0.INTFLAGS = 1;
		TCB0.INTCTRL = 1;		//int on
		TCB0.CNT = 0;
		TCB0.CCMP = 27776;		//just over 720 ints per second (120 FPS on display)
		TCB0.CTRLA = 1;			//on at full speed
	}
	
	//i2c
	{
		TWI0.SADDR = (EE->i2cAddr << 1) | 0;	//general call enabled
		TWI0.SADDRMASK = 0x01;					//match zero or addr in slave address
		TWI0.CTRLA = 0x0C;						//conservative timing
		TWI0.MCTRLA = 0;						//master TWI subunit is off
		TWI0.SCTRLB = 0;						//no action for now
		TWI0.SSTATUS = 0xff;					//clear all statusses
		TWI0.SCTRLA = 0b11100001;				//on, with proper settings
	}
	
	while(1);	//gStop gcc warning...
}

ISR(__vector_lvl_0)	//i2c
{
	uint8_t sta, i, reply = 3 /* default */;
	
	sta = TWI0.SSTATUS;
	
	if (sta & TWI_APIF_bm) {			//address or stop (we never enabled stop so just address)
		
		if (sta & TWI_AP_bm) {			//address byte RXed
			
			mIsGeneralCall = (TWI0.SDATA == 0);
			if (sta & TWI_DIR_bm) {
				
				mPacketLen = 0;
				//send ack
			}
			else {
				
				mPacketLen = 0;
				//send ack and wait for next byte
			}
		}
		else if (sta & TWI_COLL_bm) {	//collision?
			
			TWI0.SSTATUS = TWI_COLL_bm;
			reply = 2;			//done
		}
		else {							//stop?
			
			if (!(sta & TWI_DIR_bm)) {	//write completed? {
			
				if (mPacketLen) {
					switch (mPacket[0]) {	//command packet
						
						case CMD_RESET:
							if (mPacketLen != 1)
								break;
							CCP = CCP_IOREG_gc;
							RSTCTRL.SWRR = 1;
							break;
						
						case CMD_FLIP:
							if (mPacketLen != 1)
								break;
							for (i = 0; i < NUM_COLS - 1; i++) {	//dot is never flipped
							
								uint8_t destCol = (EE->flags & FLAG_FLIP_H) ? (NUM_COLS - 2 - i) : i;
								mData[destCol] = toOurRowOrder(mBackBuffer[i]);
							}
							//dot column needs no flips
							mData[NUM_COLS - 1] = mBackBuffer[NUM_COLS - 1] ? mRowToBit[0] : 0;
							break;
						
						case CMD_SET_ADDR:
							if (mPacketLen != 2 || !(i = mPacket[1]) || i > 0x7f)
								break;
							setSelfAddr(i);
							break;
						
						case CMD_SET_ADDR_UID:
							if (mPacketLen != 2 + UID_LEN)
								break;
							//see if for me
							for (i = 0; i < 10; i++)
								if (mPacket[i + 1] != ((volatile uint8_t*)&SIGROW.SERNUM0)[i])
									break;
							if (i != 10)
								break;
							for (;i < 16; i++)
								if (mPacket[i + 1])
									break;
							if (i != 16)
								break;
							i = mPacket[1 + UID_LEN];
							if (!i || i > 0x7f)
								break;
							setSelfAddr(i);
							break;
						
						case CMD_GET_UID:
							if (mPacketLen != 1)
								break;
							
							if (EE->i2cAddr > 0 && EE->i2cAddr < 0x7f && mIsGeneralCall) {
								//if we have an address assigned and this was a general call,
								// send all 0xFF to for sure lose arbitration
								
								for (i = 0; i < 16; i++)
									mPacket[i] = 0xff;
							}
							else {
								for (i = 0; i < 10; i++)
									mPacket[i] = ((volatile uint8_t*)&SIGROW.SERNUM0)[i];
								for (; i < 16; i++)
									mPacket[i] = 0;
							}
							break;
						
						case CMD_DRAW_BMP:
							if (mPacketLen != 1 + NUM_COLS)
								break;
							for (i = 0; i < NUM_COLS; i++)
								mBackBuffer[i] = mPacket[i + 1];
							break;
						
						case CMD_DRAW_CHAR:
							if (mPacketLen != 2)
								break;
							drawChar(mPacket[1]);
							break;
						
						case CMD_GET_VER:
							if (mPacketLen != 1)
								break;
							mPacket[0] = MY_HW_VER;
							mPacket[1] = MY_FW_VER;
							break;
						
						case CMD_FW_UPDATE:
							if (mPacketLen != 1 + sizeof(struct FwUpdatePacket))
								break;
							eeFwUpdate((const struct FwUpdatePacket*)(mPacket + 1));
							break;
						
						case CMD_SET_FLAGS:
							if (mPacketLen != 2)
								break;
							setFlags(mPacket[1]);
							break;
						
						case CMD_GET_FLAGS:
							if (mPacketLen != 1)
								break;
							mPacket[0] = EE->flags;
							break;
					}
				}
			}
			reply = 2;
		}
	}
	if (sta & TWI_DIF_bm) {				//data
		
		if (sta & TWI_DIR_bm) {			//read proceeding
			
			TWI0.SDATA = mPacket[mPacketLen++];
			if (mPacketLen == sizeof(mPacket))
				mPacketLen = 0;
			//ack
		}
		else {							//write processing
			if (mPacketLen < sizeof(mPacket))
				mPacket[mPacketLen++] = TWI0.SDATA;
			
			if (mPacketLen > sizeof(mPacket))
				reply = 7;	//NAK if too much data
		}
	}
	
	TWI0.SCTRLB = reply;
}











