/*
 (c) 2014 Dmitry Grinberg - http://dmitry.gr - me@dmitry.gr
  Free for non-commercial use only
*/

#include <avr/io.h>
#include <util/delay.h>
#include "SD.h"

#define FLAG_TIMEOUT		0x80
#define FLAG_PARAM_ERR		0x40
#define FLAG_ADDR_ERR		0x20
#define FLAG_ERZ_SEQ_ERR	0x10
#define FLAG_CMD_CRC_ERR	0x08
#define FLAG_ILLEGAL_CMD	0x04
#define FLAG_ERZ_RST		0x02
#define FLAG_IN_IDLE_MODE	0x01


/*
	this is a very simplified SD/MMC diver [which is adapted from PowerSDHC sources :) ]
	it supports reads in stream mode. i removed support for block reads/writes and SDHC/MMCplus support since it is not needed.
	also removed is ability to read card size since it is also unused (it saves us 300 instructions)

*/

static Boolean gFast = false, gSD = false;

static void sdClockSpeed(Boolean fast){
	
	gFast = fast;
}

static void sdChipSelect(Boolean active) 
{
	if (active)
		PORTB &=~ (1 << 3);
	else
		PORTB |= 1 << 3;
}

uint8_t sdSpiByteFast(uint8_t byte)
{
	uint8_t v1, v2;
	
	USISR = 1 << 6;
	USIDR = byte;
	v1 = 0x11;
	v2 = v1 | 0x02;
		
	USICR = v1;
	USICR = v2;
	USICR = v1;
	USICR = v2;
	USICR = v1;
	USICR = v2;
	USICR = v1;
	USICR = v2;
	USICR = v1;
	USICR = v2;
	USICR = v1;
	USICR = v2;
	USICR = v1;
	USICR = v2;
	USICR = v1;
	USICR = v2;
	
	return USIDR;
}

static uint8_t sdSpiByte(uint8_t byte)
{
	uint8_t i, v1, v2;
	
	if (gFast)
		return sdSpiByteFast(byte);
	
	USISR = 1 << 6;
	USIDR = byte;
	v1 = 0x11;
	v2 = v1 | 0x02;
		
	for(i = 0; i < 8; i++) {
		_delay_us(2);
		USICR = v1;
		_delay_us(2);
		USICR = v2;
	}
	return USIDR;
}

static void sdSpiInit(void){

	DDRB = (DDRB &~ (1 << 0)) | (1 << 1) | (1 << 2) | (1 << 3);
	PORTB |= 1 << 0;	//pullup
}

static uint8_t sdCrc7(uint8_t* chr,uint8_t cnt,uint8_t crc){

	uint8_t i, a;
	uint8_t Data;

	for(a = 0; a < cnt; a++){
		
		Data = chr[a];
		
		for(i = 0; i < 8; i++){
			
			crc <<= 1;

			if( (Data & 0x80) ^ (crc & 0x80) ) crc ^= 0x09;
			
			Data <<= 1;
		}
	}
	
	return crc & 0x7F;
}

static inline void sdPrvSendCmd(uint8_t cmd, uint32_t param){
	
	uint8_t send[6];
	
	send[0] = cmd | 0x40;
	send[1] = param >> 24;
	send[2] = param >> 16;
	send[3] = param >> 8;
	send[4] = param;
	send[5] = (sdCrc7(send, 5, 0) << 1) | 1;
	
	for(cmd = 0; cmd < sizeof(send); cmd++){
		sdSpiByte(send[cmd]);
	}
}

static inline uint8_t sdPrvReadResp(void){	//if return has FLAG_TIMEOUT bit_ set, we timed out
	
	uint8_t v, i = 0;
	
	do{		//our max wait time is 128 byte clocks (1024 clock ticks)
		
		v = sdSpiByte(0xFF);
		
	}while(i++ < 128 && (v == 0xFF));
	
	
	return v;
}

static uint8_t sdPrvSimpleCommand(uint8_t cmd, uint32_t param, Boolean cmdDone){	//do a command, return R1 reply
	
	uint8_t ret;
	
	sdChipSelect(true);
	sdSpiByte(0xFF);
	sdPrvSendCmd(cmd, param);
	ret = sdPrvReadResp();
	if(cmdDone) sdChipSelect(false);
	
	return ret;
}

static uint8_t sdPrvACMD(uint8_t cmd, uint32_t param){
	
	uint8_t ret;
	
	ret = sdPrvSimpleCommand(55, 0, true);
	if(ret & FLAG_TIMEOUT) return ret;
	if(ret & FLAG_ILLEGAL_CMD) return ret;
	
	return sdPrvSimpleCommand(cmd, param, true);
}

Boolean sdPrvCardInit(void){
	
	uint16_t time = 0;
	uint8_t resp;
	uint32_t param = 0;
	
	resp = sdPrvSimpleCommand(55, 0, true);			//see if this is SD or MMC
	if(resp & FLAG_TIMEOUT) return false;
	gSD = !(resp & FLAG_ILLEGAL_CMD);
		
	while(time++ < 500UL){	//retry 500 times
	
		resp = gSD ? sdPrvACMD(41, param) : sdPrvSimpleCommand(1, param, true);
		
		if(resp & FLAG_TIMEOUT)	//init fail
			break;
		
		if((param & 0x00200000UL) && !(resp & FLAG_IN_IDLE_MODE))	//init done
			return true;
			
		param |= 0x00200000UL;

	}

	return false;
}

Boolean sdInit(){
	
	uint8_t v, tries = 0;
	
	
	sdSpiInit();

	sdClockSpeed(false);
	sdChipSelect(false);
	for(v = 0; v < 20; v++) sdSpiByte(0xFF);	//lots of clocks with CS not asserted to give card time to init
	
	//with CS tied low, we get here with clock sync a bit weird, so we need to re-sync it, we do so here, since we know for sure what the valid RESP for CMD0 is
	do{
		v = sdPrvSimpleCommand(0, 0, true);
		//resync usage makes this bad, so i comment it out: if(v & FLAG_TIMEOUT) return false;
		tries++;
		if(tries > 30) return false;
	}while(v != 0x01);

	if(!sdPrvCardInit())return false;
	
	v = sdPrvSimpleCommand(16, 512, true);		//sec sector size
	if(v) return false;

	v = sdPrvSimpleCommand(59, 0, true);		//crc off
	if(v) return false;

	sdClockSpeed(true);
	
	return true;
}







//stream mode read

static uint8_t sdPrvSimpleCommandFast(uint8_t cmd, uint32_t param, Boolean cmdDone){	//do a command, return R1 reply
	
	uint8_t ret, i = 0;
	
	sdChipSelect(true);
	sdSpiByteFast(0xFF);
	sdSpiByteFast(cmd | 0x40);
	sdSpiByteFast(param >> 24);
	sdSpiByteFast(param >> 16);
	sdSpiByteFast(param >> 8);
	sdSpiByteFast(param);
	sdSpiByteFast(0); // no crc
	
	do{		//our max wait time is 128 byte clocks (1024 clock ticks)
		
		ret = sdSpiByteFast(0xFF);
		
	}while(i++ < 128 && (ret == 0xFF));
	
	if(cmdDone) sdChipSelect(false);
	
	return ret;
}

Boolean sdReadStart(uint32_t sec){

	uint8_t v;

	v = sdPrvSimpleCommandFast(18, sec << 9, false);
	if(v & FLAG_TIMEOUT) {return false; }

	do{
		v = sdSpiByteFast(0xFF);
	}while(v == 0xFF);
	if(v != 0xFE) {return false;}

	return true;
}

void sdNextSec(){

	uint8_t v;

	sdSpiByteFast(0xFF);	//skip crc
	sdSpiByteFast(0xFF);

	do{
		v = sdSpiByteFast(0xFF);
	}while(v == 0xFF);
}

void sdSecReadStop(){

	//cancel read
	sdPrvSimpleCommandFast(12, 0, true);
}
