#include <string.h>
#include <stdint.h>
#include "printf.h"
#include "entry.h"
#include "load.h"
#include "util.h"
#include "vfs.h"



struct ElfHeader {
	uint8_t		e_ident[16];
	uint16_t	e_type;
	uint16_t	e_machine;
	uint32_t	e_version;
	uint32_t	e_entry;
	uint32_t	e_phoff;		// -> program header table offset
	uint32_t	e_shoff;		// -> ElfSection[]
	uint32_t	e_flags;
	uint16_t	e_ehsize;		//this header's size in bytes
	uint16_t	e_phentsize;	//size of program header table entries
	uint16_t	e_phnum;		//number of entries in program header table
	uint16_t	e_shentsize;	//sizeof(ElfSection) (size of entry of section table
	uint16_t	e_shnum;		//num sections in section table
	uint16_t	e_shstrndx;		//section header table index of the entry associated with the section name string table. 
} __attribute__((packed));

#define EI_MAG0         0               /* e_ident[] indexes */
#define EI_MAG1         1
#define EI_MAG2         2
#define EI_MAG3         3
#define EI_CLASS        4
#define EI_DATA         5
#define EI_VERSION      6
#define EI_PAD          7

#define ELFMAG0         0x7F
#define ELFMAG1         'E'
#define ELFMAG2         'L'
#define ELFMAG3         'F'

#define ELFCLASS32      1
#define ELFCLASS64      2

#define ELFDATA2LSB     1
#define ELFDATA2MSB     2

#define ET_EXEC   		2

#define EM_MIPS         8 
#define EM_MIPS_RS4_BE  10

#define EV_CURRENT      1


struct ElfProgHdr{
	uint32_t	p_type;
	uint32_t	p_offset;
	uint32_t	p_vaddr;
	uint32_t	p_paddr;
	uint32_t	p_filesz;
	uint32_t	p_memsz;
	uint32_t	p_flags;
	uint32_t	p_align;
} __attribute__((packed));


#define PHF_EXEC		0x1
#define PHF_WRITE		0x2
#define PHF_READ		0x4

#define PT_NULL			0x0
#define PT_LOAD			0x1
#define PT_DYNAMIC		0x2
#define PT_INTERP		0x3
#define PT_NOTE			0x4
#define PT_SHLIB		0x5
#define PT_PHDR			0x6

bool isValidKernelName(const char *name)
{
	return !strcasecmp(name, "vmlinux");
}

void loadOS(struct VfsFile *f, uint32_t fileSize, const struct DecPromVectors *vecs)
{
	static const char *argv[] = {
			"vmlinux"
			"unused",
			"earlyprintk=prom0",
			"console=ttyS0",
			"root=/dev/pvd3",
			"rootfstype=ext4",
			"rw",
			"lpj=31232",	//as calibrated by linux when running without this param
			"init=/sbin/uMIPSinit",
//			"init=/bin/sh",
	};
	struct ElfProgHdr phdr;
	struct ElfHeader ehdr;
	uint_fast16_t i;
	
	
	if (sizeof(ehdr) != vfsFileRead(f, &ehdr, sizeof(ehdr))) {
		
		pr("cannot read elf header\r\n");
		return;
	}
		
	if (ehdr.e_ident[EI_MAG0] != ELFMAG0 || ehdr.e_ident[EI_MAG1] != ELFMAG1 || ehdr.e_ident[EI_MAG2] != ELFMAG2 ||
			ehdr.e_ident[EI_MAG3] != ELFMAG3 || ehdr.e_ehsize < sizeof(struct ElfHeader) || ehdr.e_phentsize < sizeof(struct ElfProgHdr)) {
		
		pr("not a valid elf file\r\n");
		return;
	}

	if (ehdr.e_ident[EI_CLASS] != ELFCLASS32 || ehdr.e_ident[EI_DATA] != ELFDATA2LSB || ehdr.e_ident[EI_VERSION] != EV_CURRENT ||
			ehdr.e_type != ET_EXEC || (ehdr.e_machine != EM_MIPS && ehdr.e_machine != EM_MIPS_RS4_BE)) {
		
		pr("Only v1 MIPS LE32 executable elf files supported\r\n");
		return;
	}
		
	pr("Entrypoint VA: 0x%08x\r\n", ehdr.e_entry);
		
	for (i = 0; i < ehdr.e_phnum; i++) {
		
		if (!vfsFileSeek(f, ehdr.e_phoff + i * ehdr.e_phentsize) || sizeof(phdr) != vfsFileRead(f, &phdr, sizeof(phdr))) {
			
			pr("cannot read program header %u\r\n", i);
			return;
		}
		
		if (phdr.p_type != PT_LOAD)
			continue;
		
		if (phdr.p_filesz > phdr.p_memsz) {
			
			pr("This load command is impossible\r\n");
			return;
		}
		
		pr("File 0x%08x + 0x%08x\r\n loads to 0x%08x + 0x%08x...\r\n", phdr.p_offset, phdr.p_filesz, phdr.p_paddr, phdr.p_memsz);
		
		if (phdr.p_paddr < MIN_SAFE_ADDR) {
			pr("Cannot load this low\r\n");
			return;
		}
		
		if (!vfsFileSeek(f, phdr.p_offset)) {
			pr("cannot seek to source data\r\n");
			return;
		}
		if (!copyWithProgress((void*)phdr.p_paddr, f, phdr.p_filesz)) {
			pr("cannot copy data\r\n");
			return;
		}
		//XXX: seems that linux clears its own bss, so we can not bother...

		//pr("clearing BSS...\r\n");
		//zeroWithProgress((void*)(phdr.p_paddr + phdr.p_filesz), phdr.p_memsz - phdr.p_filesz);
		pr("Kernel Loaded\r\n");
	}
	callWithGpVal(sizeof(argv) / sizeof(*argv), argv, 0x30464354, vecs, 0, ehdr.e_entry);
}
