#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "cpu.h"
#include "soc.h"
#include "vPSRAM.h"

#define VERBOSE				0

enum PsramState {
	PsramIdle,
	PsramRxingReadAddr,
	PsramRxingWriteAddr,
	PsramRxingData,
	PsramTxingData,
	PsramWeirdState,
};

struct VPSRAM {

	struct VSPI *vspi;

	uint64_t ticksAtSelection;
	uint64_t totalSelectionsCount, totalSelectionsCountP0, totalSelectedTicks;
	uint32_t maxSelectedDuration;

	enum PsramState state;
	uint32_t totalSize;

	uint32_t addr;
	uint8_t haveAddrBytes;
	uint8_t curCmd;

	char *name;

	uint8_t dataBuf[];
};


static uint8_t vpsramPrvSpiProvideByteF(void *userData)
{
	struct VPSRAM *psram = (struct VPSRAM *)userData;
	uint8_t ret = 0;

	if (psram->state == PsramTxingData) {
		
		ret = psram->dataBuf[psram->addr];
		if (VERBOSE)
			fprintf(stderr, " %s [0x%06x] -> 0x%02x\n", psram->name, psram->addr, ret);
		if (++psram->addr == psram->totalSize)
			psram->addr = 0;
	}

	//fprintf(stderr, "PSRAM: gave byte 0x%02x\n", ret);

	return ret;
}

static void vpsramPrvSpiAcceptByteF(void *userData, uint8_t byte)
{
	struct VPSRAM *psram = (struct VPSRAM *)userData;

	//fprintf(stderr, "PSRAM: got byte 0x%02x\n", byte);

	switch (psram->state) {
		case PsramIdle:	//command
			switch (byte) {
				case 0x02:		//write
					psram->state = PsramRxingWriteAddr;
					psram->haveAddrBytes = 0;
					psram->addr = 0;
					break;

				case 0x03:		//read:
					psram->state = PsramRxingReadAddr;
					psram->haveAddrBytes = 0;
					psram->addr = 0;
					break;
				
				default:
					psram->state = PsramWeirdState;
					break;
			}
			break;

		case PsramRxingReadAddr:
			psram->addr = psram->addr * 256 + byte;
			if (++psram->haveAddrBytes == 3) {
				psram->addr %= psram->totalSize;
				psram->state = PsramTxingData;
			}
			break;

		case PsramRxingWriteAddr:
			psram->addr = psram->addr * 256 + byte;
			if (++psram->haveAddrBytes == 3) {
				psram->addr %= psram->totalSize;
				psram->state = PsramRxingData;
			}
			break;

		case PsramRxingData:
			if (VERBOSE)
				fprintf(stderr, " %s [0x%06x] <- 0x%02x\n", psram->name, psram->addr, byte);
			psram->dataBuf[psram->addr] = byte;
			if (++psram->addr == psram->totalSize)
				psram->addr = 0;
			break;

		default:
			break;
	}
}

static void vpsramPrvSpiSelectionChanged(void *userData, bool selected)
{
	struct VPSRAM *psram = (struct VPSRAM *)userData;

	if (selected) {
		psram->ticksAtSelection = cpuGetCy();
		psram->state = PsramIdle;

		if (VERBOSE)
			fprintf(stderr, "===== %s :selected =====\n", psram->name);

	}
	else {
		uint32_t dur = cpuGetCy() - psram->ticksAtSelection;

		if (dur > psram->maxSelectedDuration)
			psram->maxSelectedDuration = dur;
		psram->totalSelectedTicks += dur;
		psram->totalSelectionsCount++;

		if (!socGetRomPage())
			psram->totalSelectionsCountP0++;

		if (VERBOSE)
			fprintf(stderr, "==== %s: deselected ====\n", psram->name);
	}
}

struct VPSRAM* vpsramInit(const char *spiName, uint32_t size)
{
	struct VPSRAM *psram = calloc(1, sizeof(struct VPSRAM) + size + strlen(spiName) + 1);

	if (psram) {
		psram->totalSize = size;
		psram->vspi = vspiInit(spiName, SpiMode0);
		psram->name = (char*)psram->dataBuf + size;
		strcpy(psram->name, spiName);
		
		if (psram->vspi) {

			vspiDeviceRegister(psram->vspi, vpsramPrvSpiProvideByteF, vpsramPrvSpiAcceptByteF, vpsramPrvSpiSelectionChanged, psram);
			return psram;
		}
		free(psram);
		psram = NULL;
	}

	return psram;
}

struct VSPI* vpsramGetVSPI(struct VPSRAM* psram)
{
	return psram->vspi;
}

void vpsramDeinit(struct VPSRAM* psram)
{
	vspiDestroy(psram->vspi);
	free(psram);
}

void vpsramGetStats(struct VPSRAM* psram, struct PsramStats *stats)
{
	stats->numTimesSelected = psram->totalSelectionsCount;
	stats->numCyclesSelected = psram->totalSelectedTicks;
	stats->longestSelectedDuration = psram->maxSelectedDuration;
	stats->numTimesSelectedPage0 = psram->totalSelectionsCountP0;
}






