/******************************************************************************
 * Copyright (c) 2004 palmOne, Inc. or its subsidiaries.
 * All rights reserved.
 *****************************************************************************/
/**
 * @ingroup PmKeyLib
 */

/**
 * @file  PmKeyLib.h
 * @brief System Library that exports key-related APIs.
 *
 * This library is based on key APIs from HsExtensions and was broken apart from HsExtensions
 * so the APIs could be ported to other platforms.
 *
 */

/*
 * $Id: //device/handheld/dev/sdk/2.1/incs/68k/libraries/pmkeylib/PmKeyLib.h#5 $
 *
 *****************************************************************************/

#ifndef __PMKEYLIB_H__
#define __PMKEYLIB_H__


/********************************************************************
 * Prototypes
 ********************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

/// Standard open library function
///
/// @param refNum: IN: library reference number (returned by SysLibFind() / SysLibLoad() )
/// @retval Err error code.
Err
PmKeyLibOpen (UInt16 refNum)
			SYS_TRAP (kPmKeyLibTrapOpen);

/// Standard close library function
///
/// @param refNum: IN: library reference number (returned by SysLibFind() / SysLibLoad() )
/// @retval Err error code.
Err
PmKeyLibClose (UInt16 refNum)
			SYS_TRAP (kPmKeyLibTrapClose);

/// Get the current state of specified keys
///
/// @param refNum:		IN:  library reference number (returned by SysLibFind() / SysLibLoad() )
/// @param count:		IN:  number of elements in keyCodes[] and pressed[]
/// @param keyCodes:	IN:  array of keyCodes to test
/// @param pressed:		OUT: Each element is set to true if the key specified by the
///                          corresponding element in keyCodes[] is pressed, and set
///                          to false if it is not pressed.
///                          It is acceptable to pass NULL if only the return value is of interest.
/// @retval UInt16 count of keys that are pressed.
UInt16
PmKeyKeysPressed (UInt16 refNum, UInt16 count, const UInt16 keyCodes[], Boolean pressed[])
			SYS_TRAP (kPmKeyLibTrapKeysPressed);

/// Stop a key from sending any more auto repeat events until
/// it is released.
///
/// @param refNum:		IN: library reference number (returned by SysLibFind() / SysLibLoad() )
/// @param keyCode:		IN: key to stop
/// @retval Err errNone if the keyCode is the last key pressed and it is still held.
Err
PmKeyStop (UInt16 refNum, UInt16 keyCode)
			SYS_TRAP (kPmKeyLibTrapStop);

/// Enable/disable a key from generating key events.
/// This extends HALKeySetMask().
///
/// @param refNum:	IN: library reference number (returned by SysLibFind() / SysLibLoad() )
/// @param keyCode:	IN: the key to enable/disable
/// @param enabled:	IN: true to enable, false to disable
/// @retval Boolean true if previously enabled
Boolean
PmKeyEnableKey (UInt16 refNum, UInt16 keyCode, Boolean enabled)
			SYS_TRAP (kPmKeyLibTrapEnableKey);

/// Determine which key generates a certain character
/// This is the complement of HALKeyKeyCodeToChrCode()
///
/// @param refNum:	IN: library reference number (returned by SysLibFind() / SysLibLoad() )
/// @param chrCode:	IN: the character to search for
/// @retval UInt16 the keycode returned.
UInt16
PmKeyChrCodeToKeyCode (UInt16 refNum, UInt16 chrCode)
			SYS_TRAP (kPmKeyLibTrapChrCodeToKeyCode);

/// Translate a key code to the character generated by that key.
/// Each key can generate several different characters, depending
/// on which modifier keys are held when it is pressed. The caller
/// needs to specify which modified state of the key should be used
/// to select the appropriate character.
/// This is the complement of HALKeyChrCodeToKeyCode().
///
/// @param refNum:			IN:  library reference number (returned by SysLibFind() / SysLibLoad() )
/// @param keyCode:			IN:  the key to lookup
/// @param modifiersIn:		IN:  Modifier mask indicating which character
///                   		     should be returned
/// @param chrP:			OUT: The character produced by the key
/// @param modifiersOutP:	OUT: The keyMask bits that must be set if a keyDown event is
///                              formed with the character. (Not including the bits
///                              set in modifiersIn)
/// @retval None.
void
PmKeyKeyCodeToChrCode (UInt16 refNum, UInt16 keyCode, UInt16 modifiersIn,
					   UInt16* chrP, UInt16* modifiersOutP)
			SYS_TRAP (kPmKeyLibTrapKeyCodeToChrCode);

/// Determine if an event came from the physical keyboard
///
/// @param refNum: IN: library reference number (returned by SysLibFind() / SysLibLoad() )
/// @param eventP: IN: the event to check
/// @retval Boolean true if the event came from the keyboard
Boolean
PmKeyEventIsFromKeyboard (UInt16 refNum, EventPtr eventP)
			SYS_TRAP (kPmKeyLibTrapEventIsFromKeyboard);

/// Get any palmOne specific key attributes that aren't
/// appropriate to be accessed through a feature. (e.g.
/// dynamic information about device state)
///
///	The following code sample retrieves the keyguard state (active or
///	inactive) and disables it accordingly.
/// @code
///     UInt32 keyguard;
///     PmKeyAttrGet (pmKeyAttrKeyboardLocked, 0, &keyguard);
///     if (keyguard)
///     {
///         keyguard = 0;
///         PmKeyAttrSet (pmKeyAttrKeyboardLocked, 0, &keyguard);
///     }
/// @endcode
///
/// @param refNum:	IN:  library reference number (returned by SysLibFind() / SysLibLoad() )
/// @param attr:	IN:  the attribute being retrieved @see PmKeyAttrEnum
/// @param flags:	IN:  depends on attr (currently implemented attributes only take 0 for this argument)
/// @param valueP:	OUT: cast depending on attr (refer to PmKeyAttrEnum for data type this argument should point to)
/// @retval Err error code. 0 if no error, hsErrNotSupported otherwise.
Err
PmKeyAttrGet (UInt16 refNum, UInt16 attr /*PmKeyAttrEnum*/, UInt32 flags, UInt32* valueP)
			SYS_TRAP (kPmKeyLibTrapAttrGet);

/// Set palmOne specific attributes
///
/// @param attr:	IN: the attribute being set
/// @param flags:	IN: depends on attr (currently implemented attributes only take 0 for this argument)
/// @param value:	IN: cast depending on attr (refert to PmKeyAttrEnum for data type this argument should point to)
/// @retval Err error code. 0 if no error, hsErrNotSupported otherwise.
Err
PmKeyAttrSet (UInt16 refNum, UInt16 attr /*PmKeyAttrEnum*/, UInt32 flags, UInt32 value)
			SYS_TRAP (kPmKeyLibTrapAttrSet);

#ifdef __cplusplus
}
#endif

#endif  // __PMKEYLIB_H__
