#include "util.h"

#define BLITTER_CAN_DO_THREE_QUARTERS_SCALING


static int32_t scaleCoordStdToNative(int32_t coord, const struct PalmDrawState *ds, bool roundUpIfUpscaling)
{
	return scaleCoord(scaleCoord(coord, ds->scaling.preGarnet.standardToActive, roundUpIfUpscaling), ds->scaling.preGarnet.activeToNative, roundUpIfUpscaling);
}

static void WinPaintRoundedRectangleFrameGuts(struct PalmWindow *w, const struct RectangleType *rectP, int32_t normalWidth, int32_t cornerDiam, int32_t shadowWidth)
{
	struct RectangleType clipRect, drawRect, initialClip;
	struct RectangleType rect = *rectP;
	struct PalmBitmapV3fat canvasBmp;
	struct PalmCanvas canvas;
	
	WinPrvInitCanvas(w, &canvasBmp, &canvas);
	RctInsetRectangle(&rect, -normalWidth);
	rect.topLeft.x += w->bounds.topLeft.x;
	rect.topLeft.y += w->bounds.topLeft.y;
	
	if (cornerDiam == -1) {
		HALDraw_Rectangle(&canvas, &rect, 2, 1);
		normalWidth--;
		cornerDiam = 0;
		RctInsetRectangle(&rect, 1);
	}
	
	if (normalWidth > 0)
		HALDraw_Rectangle(&canvas, &rect, cornerDiam, normalWidth);
	
	if (shadowWidth) {
		
		initialClip = canvas.clippingRect;
		
		clipRect = rect;
		drawRect = rect;
		
		drawRect.topLeft.x += shadowWidth;
		drawRect.topLeft.y += shadowWidth;
		
		clipRect.topLeft.x += rect.extent.x - normalWidth;
		shadowWidth += normalWidth;
		clipRect.extent.x = shadowWidth;
		clipRect.extent.y -= normalWidth + cornerDiam;
		
		RctGetIntersection(&initialClip, &clipRect, &canvas.clippingRect);		//they call WinCostrainClip
		HALDraw_Rectangle(&canvas, &drawRect, cornerDiam, shadowWidth);
		
		clipRect = rect;
		clipRect.topLeft.y += rect.extent.y - normalWidth;
		clipRect.extent.x -= normalWidth + cornerDiam;
		clipRect.extent.y = shadowWidth;
		RctGetIntersection(&initialClip, &clipRect, &canvas.clippingRect);		//they call WinCostrainClip
		HALDraw_Rectangle(&canvas, &drawRect, cornerDiam, shadowWidth);
		
		clipRect = rect;
		clipRect.topLeft.x += rect.extent.x - normalWidth - cornerDiam;
		clipRect.topLeft.y += rect.extent.y - normalWidth - cornerDiam;
		clipRect.extent.x = shadowWidth + cornerDiam;
		clipRect.extent.y = shadowWidth + cornerDiam;
		RctGetIntersection(&initialClip, &clipRect, &canvas.clippingRect);		//they call WinCostrainClip
		HALDraw_Rectangle(&canvas, &drawRect, cornerDiam, shadowWidth);
	}
}

static void WinPaintRectangleFrameGuts(struct PalmWindow *w, uint16_t frameType, const struct RectangleType *rectP)
{
	int32_t width, cornerDiam, shadowWidth;	//in native coords
	union PalmFrameFlags frm = {.frameType = frameType};
	struct PalmDrawState *ds = w->drawState;
	struct RectangleType rect = *rectP;
	
	width = scaleCoordStdToNative(frm.width, ds, true);
	cornerDiam = scaleCoordStdToNative(frm.cornerDiam, ds, true);
	shadowWidth = scaleCoordStdToNative(frm.shadowWidth, ds, true);
	
	scaleRect(&rect, ds->scaling.preGarnet.activeToNative);					//scale frame rect from active to native coord system
	
	WinPaintRoundedRectangleFrameGuts(w, &rect, width, frm.threeD ? -1 : cornerDiam, frm.threeD ? 0 : shadowWidth);
}

static void WinRectangleFrameOpWithBackup(uint16_t frame, const struct RectangleType *rect, enum PalmWinDrawOperation drawMode, enum PalmPatternType pattern)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	enum PalmWinDrawOperation bcpDrawMode;
	enum PalmPatternType bcpPattern;
	
	bcpDrawMode = ds->drawMode;
	bcpPattern = ds->pattern;
	ds->drawMode = drawMode;
	ds->pattern = pattern;
	WinPaintRectangleFrameGuts(w, frame, rect);
	ds->drawMode = bcpDrawMode;
	ds->pattern = bcpPattern;
}

static void pWinDrawRectangleFrame(uint16_t frame, const struct RectangleType *rect)
{
	struct Globals *g = globalsGet();
	
	//ui hacks
	if (frame == WinFrameLE_roundFrame && g->ctl && g->uiHacksEnabled.replaceOneRectangleFrame && rect->topLeft.x == g->ctl->bounds.topLeft.x && rect->topLeft.y == g->ctl->bounds.topLeft.y && rect->extent.x == g->ctl->bounds.extent.x && rect->extent.y == g->ctl->bounds.extent.y && WinGetCoordinateSystem() == kCoordinatesStandard) {
		
		struct RectangleType rect = g->ctl->bounds;
		CustomPatternType oldPattern;
		WinDrawOperation oldDrawOp;
		
		WinSetCoordinateSystem(kCoordinatesNative);
		WinScaleRectangle(&rect);
		oldDrawOp = WinSetDrawMode(winPaint);
		WinGetPattern(&oldPattern);
		WinSetPatternType(blackPattern);
		
		WinPaintRoundedRectangleFrame(&rect, 1, 4, 0);
		
		WinSetPattern(&oldPattern);
		WinSetDrawMode(oldDrawOp);
		WinSetCoordinateSystem(kCoordinatesStandard);
		
		g->uiHacksEnabled.replaceOneRectangleFrame = false;
	}
	
	WinRectangleFrameOpWithBackup(frame, rect, palmWinDrawOpWinPaint, palmPatternBlackPattern);
}
DEF_BOOT_PATCH(pWinDrawRectangleFrame, 0xB34);

static void pWinEraseRectangleFrame(uint16_t frame, const struct RectangleType *rect)
{
	WinRectangleFrameOpWithBackup(frame, rect, palmWinDrawOpWinMask, palmPatternBlackPattern);
}
DEF_BOOT_PATCH(pWinEraseRectangleFrame, 0xB54);

static void pWinInvertRectangleFrame(uint16_t frame, const struct RectangleType *rect)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	uint8_t foreColorBcpI = ds->foreColorIdx;
	struct PalmClutEntry foreColorBcp = ds->foreColorRGB;
	
	WinSetForeColor(BmpGetBitDepth(WinGetBitmap((WinHandle)w)) - 1);
	WinRectangleFrameOpWithBackup(frame, rect, palmWinDrawOpWinInvert, palmPatternBlackPattern);
	
	ds->foreColorRGB = foreColorBcp;
	ds->foreColorIdx = foreColorBcpI;
}
DEF_BOOT_PATCH(pWinInvertRectangleFrame, 0xBBC);

static void pWinPaintRectangleFrame(uint16_t frame, const struct RectangleType *rect)
{
	WinPaintRectangleFrameGuts((struct PalmWindow*)WinGetDrawWindow(), frame, rect);
}
DEF_BOOT_PATCH(pWinPaintRectangleFrame, 0xBE8);

static void pWinPaintRoundedRectangleFrame(const struct RectangleType *rP, Coord width, Coord cornerRadius, Coord shadowWidth)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	struct RectangleType rect = *rP;
	
	scaleRect(&rect, ds->scaling.preGarnet.activeToNative);					//scale frame rect from active to native coord system
	WinPaintRoundedRectangleFrameGuts(w, &rect, scaleCoord(width, ds->scaling.preGarnet.activeToNative, false), scaleCoord(cornerRadius, ds->scaling.preGarnet.activeToNative, false), scaleCoord(shadowWidth, ds->scaling.preGarnet.activeToNative, false));
}
DEF_BOOT_PATCH(pWinPaintRoundedRectangleFrame, 0xCBC);

static void pWinDrawGrayRectangleFrame(union PalmFrameFlags frame, const struct RectangleType *rP)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PointType pt = rP->topLeft;
	uint32_t patt = 0x55AA55AA;
	struct PalmDrawState *ds;
	
	WinPushDrawState();
	ds = w->drawState;	//must be done after WinPushDrawState() ince that changes the pointer
	
	WinWindowToDisplayPt(&pt.x, &pt.y);
	pt.x = scaleCoord(pt.x, ds->scaling.preGarnet.activeToStandard, false);
	pt.y = scaleCoord(pt.y, ds->scaling.preGarnet.activeToStandard, false);
	
	if (BmpGetDensity(WinGetBitmap((WinHandle)w)) == kDensityOneAndAHalf) {
		
		if (((pt.x % 4) ^ (pt.y % 4)) & 2)		//do not ask
			patt = ~patt;
	}
	else if ((pt.x + pt.y) & 1)
		patt = ~patt;
	
	ds->patternData.data32[0] = patt;
	ds->patternData.data32[1] = patt;
	ds->drawMode = palmWinDrawOpWinPaint;
	ds->pattern = palmPatternCustomPattern;
	WinPaintRectangleFrameGuts(w, frame.frameType, rP);
	WinPopDrawState();
}
DEF_BOOT_PATCH(pWinDrawGrayRectangleFrame, 0xB20);

static void pWinDrawWindowFrame(void)
{
	//we must use WinSetDrawWindow since otherwise w->drawState is NULL, non-current-draw-window windows do not necessarily have drawState
	struct PalmWindow *dispW = (struct PalmWindow*)WinGetDisplayWindow(), *w = (struct PalmWindow*)WinSetDrawWindow((WinHandle)dispW);
	uint32_t density = BmpGetDensity(w->bmp);
	struct PalmDrawState *ds = w->drawState;
	int32_t width, cornerDiam, shadowWidth;
	enum PalmWinDrawOperation bcpDrawMode;
	union PalmFrameFlags frm = w->frm;
	enum PalmPatternType bcpPattern;
	
	bcpDrawMode = ds->drawMode;
	bcpPattern = ds->pattern;
	ds->drawMode = palmWinDrawOpWinPaint;
	ds->pattern = palmPatternBlackPattern;
	
	if (frm.frameType == WinFrameLE_dialogFrame && density == kDensityDouble)
		WinPaintRoundedRectangleFrameGuts(dispW, &w->bounds, 4, 7, 0);
	else if (frm.frameType == WinFrameLE_dialogFrame && density == kDensityQuadruple)
		WinPaintRoundedRectangleFrameGuts(dispW, &w->bounds, 8, 14, 0);
	else if (frm.frameType == WinFrameLE_dialogFrame && density == kDensityOneAndAHalf)
		WinPaintRoundedRectangleFrameGuts(dispW, &w->bounds, 1, 2, 1);
	else if (frm.frameType == WinFrameLE_dialogFrame && density == kDensityTriple)
		WinPaintRoundedRectangleFrameGuts(dispW, &w->bounds, 6, 9, 0);
	else {
		
		width = scaleCoordStdToNative(frm.width, ds, true);
		cornerDiam = scaleCoordStdToNative(frm.cornerDiam, ds, true);
		shadowWidth = scaleCoordStdToNative(frm.shadowWidth, ds, true);
		
		if (frm.threeD) {
			
			shadowWidth = 0;
			cornerDiam = -1;
		}
		WinPaintRoundedRectangleFrameGuts(dispW, &w->bounds, width, cornerDiam, shadowWidth);
	}
	ds->drawMode = bcpDrawMode;
	ds->pattern = bcpPattern;

	WinSetDrawWindow((WinHandle)w);
}
DEF_BOOT_PATCH(pWinDrawWindowFrame, 0xB3C);

static bool PrvDrawCharsCharVerifyFunc(uint16_t wch)
{
	return TxtCharIsValid(wch);
}

static void PrvDrawChars(const char *chars, int16_t len, Coord x, Coord y, bool replaceDrawMode, enum PalmWinDrawOperation drawMode, enum PalmPatternType pattern)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	enum PalmWinDrawOperation bcpDrawMode;
	struct PalmFontMap *fontMap = NULL;
	struct PalmBitmapV3fat canvasBmp;
	enum PalmPatternType bcpPattern;
	struct PalmCanvas canvas;
	struct PalmFont *font;
	bool bcpUseFloor;
	
	bcpUseFloor = ds->flags.useFloor;
	bcpDrawMode = ds->drawMode;
	bcpPattern = ds->pattern;
	if (replaceDrawMode)
		ds->drawMode = drawMode;
	ds->pattern = pattern;
	
	WinPrvInitCanvas(w, &canvasBmp, &canvas);
	
	font = (struct PalmFont*)FntGetFontPtr();
	if ((font->metrics.fontType & FONT_MAP_MASK) == FONT_MAP_MASK) {
		//FntGetFontPtr() might return a pointer to a font map and not a font
		//this is strange but normal. Then we need to find a pointer
		//to the proper font table to pass along with the map
		
		fontMap = (struct PalmFontMap*)font;
		
		//yes this is correct, when we have a font map, font is actually a font table pointer
		font = (struct PalmFont*)getFontTableForFontMap(font->metrics.fontType);
	}
	
	if (ds->underlineMode != palmUnderlineModeNoUnderline && canvasBmp.density == kDensityOneAndAHalf && (y & 1) && !(FntLineHeight() & 1))
		ds->flags.useFloor = 1;
	
	if (ds->coordinateSystem != kCoordinatesNative) {
		
		x = scaleCoord(x, ds->scaling.preGarnet.activeToNative, false);
		y = scaleCoord(y, ds->scaling.preGarnet.activeToNative, false);
	}
	
	HALDraw_Chars(&canvas, x + w->bounds.topLeft.x, y + w->bounds.topLeft.y, chars, len, font, fontMap, (PalmDrawCharsVerifyFunc)&PrvDrawCharsCharVerifyFunc/* it works, i promise */);
	
	ds->drawMode = bcpDrawMode;
	ds->pattern = bcpPattern;
	ds->flags.useFloor = bcpUseFloor;
}

static void pWinDrawChar(WChar chr, Coord x, Coord y)
{
	char str[8];
	
	PrvDrawChars(str, TxtSetNextChar(str, 0, chr), x, y, true, palmWinDrawOpWinPaint, palmPatternBlackPattern);
}
DEF_BOOT_PATCH(pWinDrawChar, 0xB14);

static void pWinPaintChar(WChar chr, Coord x, Coord y)
{
	char str[8];
	
	PrvDrawChars(str, TxtSetNextChar(str, 0, chr), x, y, false, 0, palmPatternBlackPattern);
}
DEF_BOOT_PATCH(pWinPaintChar, 0xBCC);

static void pWinDrawChars(const char *chars, int16_t len, Coord x, Coord y)
{
	PrvDrawChars(chars, len, x, y, false, palmWinDrawOpWinPaint, palmPatternBlackPattern);
}
DEF_BOOT_PATCH(pWinDrawChars, 0xB18);

static void pWinEraseChars(const char *chars, int16_t len, Coord x, Coord y)
{
	PrvDrawChars(chars, len, x, y, false, palmWinDrawOpWinMask, palmPatternBlackPattern);
}
DEF_BOOT_PATCH(pWinEraseChars, 0xB44);

static void pWinPaintChars(const char *chars, int16_t len, Coord x, Coord y)
{
	PrvDrawChars(chars, len, x, y, false, 0, palmPatternBlackPattern);
}
DEF_BOOT_PATCH(pWinPaintChars, 0xBD0);

static void pWinInvertChars(const char *chars, int16_t len, Coord x, Coord y)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	uint8_t oldForeColor = ds->foreColorIdx;
	struct PalmClutEntry oldForeRGB = ds->foreColorRGB;
	
	WinSetForeColor(BmpGetBitDepth(WinGetBitmap((WinHandle)w)) - 1);
	PrvDrawChars(chars, len, x, y, true, palmWinDrawOpWinInvert, palmPatternBlackPattern);
	ds->foreColorIdx = oldForeColor;
	ds->foreColorRGB = oldForeRGB;
}
DEF_BOOT_PATCH(pWinInvertChars, 0xBAC);

static void pWinDrawInvertedChars(const char *chars, int16_t len, Coord x, Coord y)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	uint8_t oldForeColor = ds->foreColorIdx;
	uint8_t oldBackColor = ds->backColorIdx;
	uint8_t oldTextColor = ds->textColorIdx;
	struct PalmClutEntry oldForeRGB = ds->foreColorRGB;
	struct PalmClutEntry oldBackRGB = ds->backColorRGB;
	struct PalmClutEntry oldTextRGB = ds->textColorRGB;
	
	WinSetForeColor(oldBackColor);
	WinSetBackColor(oldTextColor);
	WinSetTextColor(oldBackColor);
	
	PrvDrawChars(chars, len, x, y, true, palmWinDrawOpWinPaint, palmPatternBlackPattern);
	
	ds->foreColorIdx = oldForeColor;
	ds->backColorIdx = oldBackColor;
	ds->textColorIdx = oldTextColor;
	ds->foreColorRGB = oldForeRGB;
	ds->backColorRGB = oldBackRGB;
	ds->textColorRGB = oldTextRGB;
}
DEF_BOOT_PATCH(pWinDrawInvertedChars, 0xB24);

static void PrvDrawLine(Coord x1, Coord y1, Coord x2, Coord y2, bool replaceOps, enum PalmWinDrawOperation drawMode, enum PalmPatternType pattern)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	enum PalmWinDrawOperation bcpDrawMode;
	struct PalmBitmapV3fat canvasBmp;
	enum PalmPatternType bcpPattern;
	struct PalmCanvas canvas;
	int16_t penWidth = 1;
	

	if (ds->coordinateSystem != kCoordinatesNative) {
		
		uint32_t scaleFactor = ds->scaling.preGarnet.activeToNative;
		
		penWidth = scaleFactor / SCALE_UNITY;
		if (penWidth < 1)
			penWidth = 1;
		
		x1 = scaleCoord(x1, scaleFactor, false);
		y1 = scaleCoord(y1, scaleFactor, false);
		x2 = scaleCoord(x2, scaleFactor, false);
		y2 = scaleCoord(y2, scaleFactor, false);
	}
	
	x1 += w->bounds.topLeft.x;
	x2 += w->bounds.topLeft.x;
	y1 += w->bounds.topLeft.y;
	y2 += w->bounds.topLeft.y;
	
	bcpDrawMode = ds->drawMode;
	bcpPattern = ds->pattern;
	if (replaceOps) {
		ds->drawMode = drawMode;
		ds->pattern = pattern;
	}
	
	WinPrvInitCanvas(w, &canvasBmp, &canvas);
	HALDraw_Line(&canvas, x1, y1, x2, y2, penWidth);
	
	ds->drawMode = bcpDrawMode;
	ds->pattern = bcpPattern;
}

static void pWinDrawGrayLine(Coord x1, Coord y1, Coord x2, Coord y2)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	union PalmCustomPattType oldPatt;
	enum PalmPatternType patt;
	
	oldPatt = ds->patternData;
	if (y1 != y2 && x2 - x1 != y2 - y1)
		patt = palmPatternGrayPattern;
	else {
		
		patt = palmPatternCustomPattern;
		ds->patternData.data32[0] = 0xAAAAAAAA;
		ds->patternData.data32[1] = 0xAAAAAAAA;
	}
	
	PrvDrawLine(x1, y1, x2, y2, true, palmWinDrawOpWinPaint, patt);
	ds->patternData = oldPatt;
}
DEF_BOOT_PATCH(pWinDrawGrayLine, 0xB1C);

static void pWinEraseLine(Coord x1, Coord y1, Coord x2, Coord y2)
{
	PrvDrawLine(x1, y1, x2, y2, true, palmWinDrawOpWinMask, palmPatternBlackPattern);
}
DEF_BOOT_PATCH(pWinEraseLine, 0xB48);

static void pWinDrawLine(Coord x1, Coord y1, Coord x2, Coord y2)
{
	PrvDrawLine(x1, y1, x2, y2, true, palmWinDrawOpWinPaint, palmPatternBlackPattern);
}
DEF_BOOT_PATCH(pWinDrawLine, 0xB28);

static void pWinPaintLine(Coord x1, Coord y1, Coord x2, Coord y2)
{
	PrvDrawLine(x1, y1, x2, y2, false, 0, 0);
}
DEF_BOOT_PATCH(pWinPaintLine, 0xBD4);

static void pWinInvertLine(Coord x1, Coord y1, Coord x2, Coord y2)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	uint8_t oldForeColor = ds->foreColorIdx;
	struct PalmClutEntry oldForeRGB = ds->foreColorRGB;
	
	WinSetForeColor(BmpGetBitDepth(WinGetBitmap((WinHandle)w)) - 1);
	PrvDrawLine(x1, y1, x2, y2, true, palmWinDrawOpWinInvert, palmPatternBlackPattern);
	ds->foreColorIdx = oldForeColor;
	ds->foreColorRGB = oldForeRGB;
}
DEF_BOOT_PATCH(pWinInvertLine, 0xBB0);

static void pWinFillLine(Coord x1, Coord y1, Coord x2, Coord y2)
{
	PrvDrawLine(x1, y1, x2, y2, true, palmWinDrawOpWinPaint, palmPatternCustomPattern);
}
DEF_BOOT_PATCH(pWinFillLine, 0xB5C);

static void pWinPaintLines(uint16_t numLines, const WinLineType *lines)
{
	while(numLines--) {
		PrvDrawLine(lines->x1, lines->y1, lines->x2, lines->y2, false, 0, 0);
		lines++;
	}
}
DEF_BOOT_PATCH(pWinPaintLines, 0xBD8);

static int32_t PrvGetPixel(Coord xv, Coord yv, bool asIndex)	//negative for error, else xrgb8888 or index
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmBitmapV3 *winBmp = (struct PalmBitmapV3*)w->bmp;
	struct PalmDrawState *ds = w->drawState;
	struct PalmClutEntry oldForeColor;
	struct PalmBitmapV3fat fatBmp;
	uint32_t ret, r, g, b;
	uint8_t oldForeIdx;
	bool requestIndex;
	int32_t x, y;
	
	x = scaleCoord(xv, ds->scaling.preGarnet.activeToNative, false) + w->bounds.topLeft.x;
	y = scaleCoord(yv, ds->scaling.preGarnet.activeToNative, false) + w->bounds.topLeft.y;
	
	if (!RctPtInRectangle(x, y, &w->bounds))
		return -1;
	
	BmpPrvConvertBitmap((BitmapType*)winBmp, &fatBmp);
	
	requestIndex = asIndex || !winBmp->directColor;
	ret = HALDraw_GetPixel(&fatBmp, x, y, requestIndex);
	
	if (asIndex)							//if we were asked for index, we definitely requested it and got it -> just return it
		return (uint8_t)ret;
	
	if (requestIndex) {
	
		//we were asked for a rgb value, but got an index. sadly there is no way to simply lookup an index's
		// rgb value. WinIndexToRGB does not do the same as what WinGetPixel does internally, so we use this trick
		
		//save state
		oldForeColor = ds->foreColorRGB;
		oldForeIdx = ds->foreColorIdx;
		
		//does a lookup for us
		WinSetForeColor(ret);
		
		//grab results
		r = ds->foreColorRGB.r;
		g = ds->foreColorRGB.g;
		b = ds->foreColorRGB.b;
		
		//restore state
		ds->foreColorRGB = oldForeColor;
		ds->foreColorIdx = oldForeIdx;
	}
	else {	//we got rgb565
		
		r = (ret >> 8) & 0xf8;
		g = (ret >> 3) & 0xfc;
		b = (ret << 3) & 0xf8;
	}
	
	//export as xrgb8888
	return (r << 16) | (g << 8) | b;
}

static uint8_t pWinGetPixel(Coord x, Coord y)
{
	int32_t ret = PrvGetPixel(x, y, true);
	
	if (ret < 0)
		return 0;
	
	return ret;
}
DEF_BOOT_PATCH(pWinGetPixel, 0xB90);

static Err pWinGetPixelRGB(Coord x, Coord y, RGBColorType *rgbP)
{
	int32_t ret = PrvGetPixel(x, y, false);
	
	if (ret < 0)
		return sysErrParamErr;
	
	rgbP->r = ret >> 16;
	rgbP->g = ret >> 8;
	rgbP->b = ret >> 0;
	
	return errNone;
}
DEF_BOOT_PATCH(pWinGetPixelRGB, 0xB94);

static void PrvDrawRectangle(const struct RectangleType *rP, uint16_t cornerDiam, bool replaceOps, enum PalmWinDrawOperation drawMode, enum PalmPatternType pattern)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	enum PalmWinDrawOperation bcpDrawMode;
	struct PalmBitmapV3fat canvasBmp;
	enum PalmPatternType bcpPattern;
	struct RectangleType r = *rP;
	struct PalmCanvas canvas;
	
	if (!rP->extent.x || !rP->extent.y)
		return;
	
	scaleRect(&r, ds->scaling.preGarnet.activeToNative);
	
	r.topLeft.x += w->bounds.topLeft.x;
	r.topLeft.y += w->bounds.topLeft.y;
	
	if (ds->coordinateSystem != kCoordinatesNative)
		cornerDiam = scaleCoord(cornerDiam, ds->scaling.preGarnet.activeToNative, false);
	
	bcpDrawMode = ds->drawMode;
	bcpPattern = ds->pattern;
	if (replaceOps) {
		ds->drawMode = drawMode;
		ds->pattern = pattern;
	}
	
	WinPrvInitCanvas(w, &canvasBmp, &canvas);
	HALDraw_Rectangle(&canvas, &r, cornerDiam, 0);
	
	ds->drawMode = bcpDrawMode;
	ds->pattern = bcpPattern;
}

static void pWinPaintRectangle(const struct RectangleType *rP, uint16_t cornerDiam)
{
	PrvDrawRectangle(rP, cornerDiam, false, 0, 0);
}
DEF_BOOT_PATCH(pWinPaintRectangle, 0xBE4);

static void pWinDrawRectangle(const struct RectangleType *rP, uint16_t cornerDiam)
{
	PrvDrawRectangle(rP, cornerDiam, true, palmWinDrawOpWinPaint, palmPatternBlackPattern);
}
DEF_BOOT_PATCH(pWinDrawRectangle, 0xB30);

static void pWinEraseRectangle(const struct RectangleType *rP, uint16_t cornerDiam)
{
	struct Globals *g = globalsGet();
	
	//ui hacks
	if (g->ctl && g->uiHacksEnabled.replaceOneEraseRectangle) {
		
		struct RectangleType rect;
		
		WinGetClip(&rect);
		if (rect.topLeft.x == rP->topLeft.x && rect.topLeft.y == rP->topLeft.y && rect.extent.x == rP->extent.x && rect.extent.y == rP->extent.y) {
			
			WinSetCoordinateSystem(kCoordinatesNative);
			WinScaleRectangle(&rect);
			PrvDrawRectangle(&rect, 4, true, palmWinDrawOpWinMask, palmPatternBlackPattern);	//erase as requested
			WinSetCoordinateSystem(kCoordinatesStandard);
			
			g->uiHacksEnabled.replaceOneEraseRectangle = false;
			
			return;
		}
	}
	
	PrvDrawRectangle(rP, cornerDiam, true, palmWinDrawOpWinMask, palmPatternBlackPattern);
}
DEF_BOOT_PATCH(pWinEraseRectangle, 0xB50);

static void pWinInvertRectangle(const struct RectangleType *rP, uint16_t cornerDiam)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	uint8_t oldForeColor = ds->foreColorIdx;
	struct PalmClutEntry oldForeRGB = ds->foreColorRGB;
	
	WinSetForeColor(BmpGetBitDepth(WinGetBitmap((WinHandle)w)) - 1);
	PrvDrawRectangle(rP, cornerDiam, true, palmWinDrawOpWinInvert, palmPatternBlackPattern);
	ds->foreColorIdx = oldForeColor;
	ds->foreColorRGB = oldForeRGB;
}
DEF_BOOT_PATCH(pWinInvertRectangle, 0xBB8);

static void pWinFillRectangle(const struct RectangleType *rP, uint16_t cornerDiam)
{
	PrvDrawRectangle(rP, cornerDiam, true, palmWinDrawOpWinPaint, palmPatternCustomPattern);
}
DEF_BOOT_PATCH(pWinFillRectangle, 0xB60);

static void pWinPaintPixels(uint16_t numPoints, struct PointType *pts)
{
	#define NUM_PTS_LOCAL	20
	
	struct PalmWindow *w = (struct PalmWindow*)WinGetDrawWindow();
	struct PointType *ptsToDraw = pts, ptsLocal[NUM_PTS_LOCAL];
	struct PalmDrawState *ds = w->drawState;
	struct PalmBitmapV3fat canvasBmp;
	struct PalmCanvas canvas;
	int16_t penWidth = 1;
	
	
	if (ds->coordinateSystem || w->bounds.topLeft.x || w->bounds.topLeft.y) {
		
		uint32_t i, scaleFactor = ds->scaling.preGarnet.activeToNative;
		
		penWidth = scaleFactor / SCALE_UNITY;
		if (penWidth < 1)
			penWidth = 1;
		
		if (numPoints <= NUM_PTS_LOCAL)
			ptsToDraw = ptsLocal;
		else {
			ptsToDraw = MemChunkNew(0, sizeof(struct PointType) * numPoints, 0x1200);
			if (!ptsToDraw)
				ErrDisplayFileLineMsg(__FILE__, __LINE__, "WinPaintPixelsPatch allocation failed");
		}
		
		for (i = 0; i < numPoints; i++) {
			
			ptsToDraw[i].x = scaleCoord(pts[i].x, scaleFactor, false);
			ptsToDraw[i].y = scaleCoord(pts[i].y, scaleFactor, false);
		}
	}
	
	WinPrvInitCanvas(w, &canvasBmp, &canvas);
	HALDraw_Pixels(&canvas, numPoints, ptsToDraw, penWidth);
	if (ptsToDraw != pts && ptsToDraw != ptsLocal)
		MemChunkFree(ptsToDraw);
}
DEF_BOOT_PATCH(pWinPaintPixels, 0xBE0);

static void WinDisplayToWindowOrBackPt(Coord *xP, Coord* yP, int32_t mulBy)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	
	if (w->bounds.topLeft.x)
		*xP = scaleCoord(scaleCoord(*xP, ds->scaling.preGarnet.activeToNative, false) + mulBy * w->bounds.topLeft.x, ds->scaling.preGarnet.nativeToActive, false);
	if (w->bounds.topLeft.y)
		*yP = scaleCoord(scaleCoord(*yP, ds->scaling.preGarnet.activeToNative, false) + mulBy * w->bounds.topLeft.y, ds->scaling.preGarnet.nativeToActive, false);
}

static void pWinDisplayToWindowPt(Coord *xP, Coord* yP)
{
	WinDisplayToWindowOrBackPt(xP, yP, -1);
}
DEF_BOOT_PATCH(pWinDisplayToWindowPt, 0xB0C);

static void pWinWindowToDisplayPt(Coord *xP, Coord* yP)
{
	WinDisplayToWindowOrBackPt(xP, yP, 1);
}
DEF_BOOT_PATCH(pWinWindowToDisplayPt, 0xC60);

static Coord WinScaleOrUnscaleCoord(Coord coord, bool ceiling, bool scale)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	
	return scaleCoord(coord, scale ? ds->scaling.preGarnet.standardToActive : ds->scaling.preGarnet.activeToStandard, ceiling);
}

static Coord pWinScaleCoord(Coord coord, bool ceiling)
{
	return WinScaleOrUnscaleCoord(coord, ceiling, true);
}
DEF_BOOT_PATCH(pWinScaleCoord, 0xCB4);

static Coord pWinUnscaleCoord(Coord coord, bool ceiling)
{
	return WinScaleOrUnscaleCoord(coord, ceiling, false);
}
DEF_BOOT_PATCH(pWinUnscaleCoord, 0xCB8);

static void WinScaleOrUnscalePoint(struct PointType *pP, bool ceiling, bool scale)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	uint32_t scaleFactor = scale ? ds->scaling.preGarnet.standardToActive : ds->scaling.preGarnet.activeToStandard;
	
	pP->x = scaleCoord(pP->x, scaleFactor, ceiling);
	pP->y = scaleCoord(pP->y, scaleFactor, ceiling);
}

static void pWinScalePoint(struct PointType *pP, bool ceiling)
{
	WinScaleOrUnscalePoint(pP, ceiling, true);
}
DEF_BOOT_PATCH(pWinScalePoint, 0xC9C);

static void pWinUnscalePoint(struct PointType *pP, bool ceiling)
{
	WinScaleOrUnscalePoint(pP, ceiling, false);
}
DEF_BOOT_PATCH(pWinUnscalePoint, 0xCA8);

static void WinScaleOrUnscaleRectangle(struct RectangleType* rectP, bool scale)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	
	scaleRect(rectP, scale ? ds->scaling.preGarnet.standardToActive : ds->scaling.preGarnet.activeToStandard);
}

static void pWinScaleRectangle(struct RectangleType* rectP)
{
	WinScaleOrUnscaleRectangle(rectP, true);
}
DEF_BOOT_PATCH(pWinScaleRectangle, 0xCA0);

static void pWinUnscaleRectangle(struct RectangleType* rectP)
{
	WinScaleOrUnscaleRectangle(rectP, false);
}
DEF_BOOT_PATCH(pWinUnscaleRectangle, 0xCAC);

static void pWinGetWindowFrameRect(struct PalmWindow *win, struct RectangleType *r)
{
	uint32_t density = BmpGetDensity(WinGetBitmap(WinGetDisplayWindow()));
	struct PalmDrawState *curDs = UIGetGlobalsPtr()->curDrawState;
	union PalmFrameFlags frm = win->frm;
	int32_t width, shadowSz;
	
	*r = win->bounds;
	
	if (frm.frameType == WinFrameLE_dialogFrame && density == kDensityDouble) {
		width = 4;
		shadowSz = 0;
	}
	else if (frm.frameType == WinFrameLE_dialogFrame && density == kDensityQuadruple) {
		width = 8;
		shadowSz = 0;
	}
	else if (frm.frameType == WinFrameLE_popupFrame && density == kDensityOneAndAHalf) {
		width = 1;
		shadowSz = 1;
	}
	else if (frm.frameType == WinFrameLE_popupFrame && density == kDensityTriple) {
		width = 6;
		shadowSz = 0;
	}
	else {
		
		width = scaleCoordStdToNative(frm.width, curDs, true);
		shadowSz = scaleCoordStdToNative(frm.shadowWidth, curDs, true);
		
		if (frm.threeD)
			shadowSz = 0;
	}
	
	RctInsetRectangle(r, -width);
	r->extent.x += shadowSz;
	r->extent.y += shadowSz;
	
	scaleRect(r, curDs->scaling.preGarnet.nativeToActive);
}
DEF_BOOT_PATCH(pWinGetWindowFrameRect, 0xBA0);

static void pWinGetDrawWindowBounds(struct RectangleType *r)
{
	struct PalmWindow *w = (struct PalmWindow*)WinGetDrawWindow();
	
	*r = w->bounds;
	scaleRect(r, w->drawState->scaling.preGarnet.nativeToActive);
}
DEF_BOOT_PATCH(pWinGetDrawWindowBounds, 0xB7C);

static void pWinGetBounds(struct PalmWindow *w, struct RectangleType *r)
{
	struct PalmDrawState *curDs = UIGetGlobalsPtr()->curDrawState;
	uint32_t scale;
	
	*r = w->bounds;
	if (curDs->coordinateSystem == kCoordinatesNative)
		scale = BmpGetDensity(WinGetBitmap(WinGetDrawWindow())) * SCALE_UNITY / BmpGetDensity(w->bmp);
	else
		scale = curDs->coordinateSystem * SCALE_UNITY / BmpGetDensity(w->bmp);

	scaleRect(r, scale);
}
DEF_BOOT_PATCH(pWinGetBounds, 0xB98);

static void pWinSetBounds(struct PalmWindow *w, const struct RectangleType *rP)
{
	WinHandle prevDrawWin = WinSetDrawWindow((WinHandle)w);
	
	w->bounds = *rP;
	scaleRect(&w->bounds, w->drawState->scaling.preGarnet.activeToNative);
	WinResetClip();
	
	WinSetDrawWindow(prevDrawWin);
}
DEF_BOOT_PATCH(pWinSetBounds, 0xC58);

static void pWinGetDisplayExtent(Coord *xP, Coord *yP)
{
	struct PalmWindow *dispWin = (struct PalmWindow*)WinGetDisplayWindow();
	struct PalmDrawState *curDs = UIGetGlobalsPtr()->curDrawState;
	uint32_t scaleFactor;
	
	if (curDs->coordinateSystem == kCoordinatesNative)
		scaleFactor = BmpGetDensity(WinGetBitmap(WinGetDrawWindow()));
	else
		scaleFactor = curDs->coordinateSystem;
		
	scaleFactor = scaleFactor * SCALE_UNITY / BmpGetDensity(WinGetBitmap((WinHandle)dispWin));
	
	if (xP)
		*xP = scaleCoord(dispWin->bounds.extent.x, scaleFactor, false);
	if (yP)
		*yP = scaleCoord(dispWin->bounds.extent.y, scaleFactor, false);
}
DEF_BOOT_PATCH(pWinGetDisplayExtent, 0xB70);

static void pWinGetWindowExtent(Coord *xP, Coord *yP)
{
	struct PalmWindow *w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	
	if (xP)
		*xP = scaleCoord(w->bounds.extent.x, ds->scaling.preGarnet.nativeToActive, false);
	if (yP)
		*yP = scaleCoord(w->bounds.extent.y, ds->scaling.preGarnet.nativeToActive, false);
}
DEF_BOOT_PATCH(pWinGetWindowExtent, 0xB9C);

static void pWinGetClip(struct RectangleType *r)
{
	struct PalmWindow *w = (struct PalmWindow*)WinGetDrawWindow();
	
	r->topLeft.x = w->clipBounds.left;
	r->topLeft.y = w->clipBounds.top;
	r->extent.x = w->clipBounds.right + 1 - w->clipBounds.left;
	r->extent.y = w->clipBounds.bottom + 1 - w->clipBounds.top;
	
	scaleRect(r, w->drawState->scaling.preGarnet.nativeToActive);
}
DEF_BOOT_PATCH(pWinGetClip, 0xB6C);

static void pWinSetClip(const struct RectangleType *rP)
{
	struct PalmWindow *w = (struct PalmWindow*)WinGetDrawWindow();
	struct RectangleType r = *rP;
	Coord left, top, bottom, right;
	
	WinResetClip();
	scaleRect(&r, w->drawState->scaling.preGarnet.activeToNative);
	
	//convert to abs vals
	left = r.topLeft.x;
	top = r.topLeft.y;
	right = r.topLeft.x + r.extent.x - 1;
	bottom = r.topLeft.y + r.extent.y - 1;
	
	//intersect with current clip
	if (left < w->clipBounds.left)
		left = w->clipBounds.left;
	if (top < w->clipBounds.top)
		top = w->clipBounds.top;
	if (right > w->clipBounds.right)
		right = w->clipBounds.right;
	if (bottom > w->clipBounds.bottom)
		bottom = w->clipBounds.bottom;
	
	//make sure that width and height are not negative (zero is allowd, represented as left = right - 1)
	if (right < left - 1)
		right = left - 1;
	if (bottom < top - 1)
		bottom = top - 1;
	
	//write
	w->clipBounds.left = left;
	w->clipBounds.top = top;
	w->clipBounds.right = right;
	w->clipBounds.bottom = bottom;
}
DEF_BOOT_PATCH(pWinSetClip, 0xC2C);

static void pWinClipRectangle(struct RectangleType *r)
{
	int32_t left, top, right, bottom;
	struct PalmWindow *w = (struct PalmWindow*)WinGetDrawWindow();
	
	scaleRect(r, w->drawState->scaling.preGarnet.activeToNative);
	left = r->topLeft.x;
	top = r->topLeft.y;
	right = r->extent.x + left - 1;
	bottom = top + r->extent.y - 1;
	
	if (left < w->clipBounds.left)
		left = w->clipBounds.left;
	if (top < w->clipBounds.top)
		top = w->clipBounds.top;
	if (right > w->clipBounds.right)
		right = w->clipBounds.right;
	if (bottom > w->clipBounds.bottom)
		bottom = w->clipBounds.bottom;
	
	if (right - left < -1)
		right = left - 1;
	if (bottom - top < -1)
		bottom = top - 1;
	
	r->topLeft.x = left;
	r->topLeft.y = top;
	r->extent.x = right + 1 - left;
	r->extent.y = bottom + 1 - top;
	
	scaleRect(r, w->drawState->scaling.preGarnet.nativeToActive);
}
DEF_BOOT_PATCH(pWinClipRectangle, 0xAF0);

static struct BitmapType* BmpSkipDensitySeparatorIfAny(struct BitmapType* bmp)//if bmp points to the density separator, skip it. will not work if nothign follows separator (weird case)
{
	struct BitmapType* next = BmpGetNextBitmapAnyDensity(bmp);
	
	return (BmpGetBits(bmp) == BmpGetBits(next)) ? next : bmp;
}

static struct BitmapType* WinFindOptimalBitmap(struct BitmapType *src)	//returns pointer
{
	struct BitmapType *t, *bmpList[5][5] = {{0,},}, *drawWinBmp = WinGetBitmap(WinGetDrawWindow()), *srcOrig = src;
	uint32_t j, preferredDensity, preferredDepth = BmpGetBitDepth(drawWinBmp);
	int32_t i;
	
	//sort out what density we'd like
	preferredDensity = ((struct PalmWindow*)WinGetDrawWindow())->drawState->flags.unscaledBitmaps ? kDensityLow : BmpGetDensity(drawWinBmp);
	
	//find all the densities that we have, and get a pointer to the list head for each
	for (t = src; t; t = BmpGetNextBitmapAnyDensity(t)) {
		
		uint32_t density = BmpGetDensity(t);
		uint32_t depth = BmpGetBitDepth(t);
		
		//it might be a high-density separator. if so, skip it
		t = BmpSkipDensitySeparatorIfAny(t);
	
		//quick shortcut: pick the perfectly matched bitmap if it exists
		if (depth == BmpGetBitDepth(drawWinBmp) && density == preferredDensity)
			return t;
		
		//convert to indices, skip unsupported ones
		switch (density) {
			case kDensityLow:			density = 0;	break;
			case kDensityOneAndAHalf:	density = 1;	break;
			case kDensityDouble:		density = 2;	break;
			case kDensityTriple:		density = 3;	break;
			case kDensityQuadruple:		density = 4;	break;
			default:	continue;
		}
		switch (depth) {
			case 1:		depth = 0;	break;
			case 2:		depth = 1;	break;
			case 4:		depth = 2;	break;
			case 8:		depth = 3;	break;
			case 16:	depth = 4;	break;
			default:	continue;
		}

		if (bmpList[density][depth])
			SysFatalAlert("Bitmap has duplicate entries!");
		
		bmpList[density][depth] = t;
	}
	
	src = NULL;
	if (preferredDensity == kDensityLow) {
		//low density dst prefers low density bmp. so check low density listing to
		// see if we get a match first, if not, then check 2x and only then 1.5x
		// search order is exactly as in the docs. See "Palm OS Programmers Companion,
		// Volume I" section "High-Density Bitmap Families"
		static const uint8_t densitiesToCheck[] = {0, 2, 1};	//we never support downscaling 3x and 4x images
		
		for (j = 0; j < sizeof(densitiesToCheck) / sizeof(*densitiesToCheck) && !src; j++) {
			
			struct BitmapType **perDepthList = bmpList[densitiesToCheck[j]];
			
			switch (preferredDepth) {
				case 1:
					for (i = 0; i < 5 && !src; i++)
						src = perDepthList[i];
					break;
				
				case 2:
					src = perDepthList[1];
					for (i = 0; i < 5 && !src; i++)
						src = perDepthList[i];
					break;
				
				case 4:
					for (i = 2; i >= 0 && !src; i--)
						src = perDepthList[i];
					for (i = 3; i < 5 && !src; i++)
						src = perDepthList[i];
					break;
				
				case 8:
					for (i = 3; i < 5 && !src; i++)
						src = perDepthList[i];
					for (i = 2; i >= 0 && !src; i--)
						src = perDepthList[i];
					break;
				
				case 16:
					for (i = 4; i >= 0 && !src; i--)
						src = perDepthList[i];
					break;
			}
		}
	}
	else if (preferredDensity == kDensityOneAndAHalf || preferredDensity == kDensityDouble || preferredDensity == kDensityTriple || preferredDensity == kDensityQuadruple) {
		static const int8_t densitiesToCheckDD[] = {2, 0, -1};				//upscaling from 1.5x to 2x is never supported
		static const int8_t densitiesToCheck4xD[] = {4, 2, 0, -1};			//prefer integer upscaling
	#ifdef BLITTER_CAN_DO_THREE_QUARTERS_SCALING
		//palm's blitter CANNOT downscale from 2x to 1.5x density! ours can. do not enable unless using rePalm blitter or you'll see graphical corruption
		static const int8_t densitiesToCheck3xD[] = {3, 1, 0, 2, 4, -1};	//prefer integer upscaling
		static const int8_t densitiesToCheck15[] = {1, 0, 2, -1};
	#else
		static const int8_t densitiesToCheck3xD[] = {3, 1, 0, 2, -1};		//prefer integer upscaling
		static const int8_t densitiesToCheck15[] = {1, 0, -1};				//downscaling from 2x to 1.5x is never supported, but might as well try...
	#endif
	
		const int8_t *densitiesToCheck = NULL;
		
		switch (preferredDensity) {
			case kDensityOneAndAHalf:	densitiesToCheck = densitiesToCheck15;	break;
			case kDensityDouble:		densitiesToCheck = densitiesToCheckDD;	break;
			case kDensityTriple:		densitiesToCheck = densitiesToCheck3xD;	break;
			case kDensityQuadruple:		densitiesToCheck = densitiesToCheck4xD;	break;
			default:
				preferredDepth = -1;	//make search fail for sure
				break;
		}
		
		switch (preferredDepth) {
			case 1:
				for (j = 0; densitiesToCheck[j] >= 0 && !src; j++) {
					for (i = 0; i < 3 && !src; i++)
						src = bmpList[densitiesToCheck[j]][i];
				}
				for (j = 0; densitiesToCheck[j] >= 0 && !src; j++) {
					for (i = 3; i < 5 && !src; i++)
						src = bmpList[densitiesToCheck[j]][i];
				}
				break;
			
			case 2:
				for (j = 0; densitiesToCheck[j] >= 0 && !src; j++) {
					src = bmpList[densitiesToCheck[j]][1];
					for (i = 0; i < 3 && !src; i++)
						src = bmpList[densitiesToCheck[j]][i];
				}
				for (j = 0; densitiesToCheck[j] >= 0 && !src; j++) {
					for (i = 3; i < 5 && !src; i++)
						src = bmpList[densitiesToCheck[j]][i];
				}
				break;
		
			case 4:
				for (j = 0; densitiesToCheck[j] >= 0 && !src; j++) {
					for (i = 2; i >= 0 && !src; i--)
						src = bmpList[densitiesToCheck[j]][i];
				}
				for (j = 0; densitiesToCheck[j] >= 0 && !src; j++) {
					for (i = 3; i < 5 && !src; i++)
						src = bmpList[densitiesToCheck[j]][i];
				}
				break;
		
			case 8:
				for (j = 0; densitiesToCheck[j] >= 0 && !src; j++) {
					for (i = 3; i < 5 && !src; i++)
						src = bmpList[densitiesToCheck[j]][i];
				}
				for (j = 0; densitiesToCheck[j] >= 0 && !src; j++) {
					for (i = 2; i >= 0 && !src; i--)
						src = bmpList[densitiesToCheck[j]][i];
				}
				break;
			
			case 16:
				for (j = 0; densitiesToCheck[j] >= 0 && !src; j++) {
					for (i = 4; i >= 0 && !src; i--)
						src = bmpList[densitiesToCheck[j]][i];
				}
				break;
		}
	}
	
	if (!src) {
		logt("Failed to find optimal bitmap to draw for (%uppi, %ubpp)\nCANDIDATES: \n", preferredDensity, preferredDepth);
		
		for (t = BmpSkipDensitySeparatorIfAny(srcOrig); t; t = BmpGetNextBitmapAnyDensity(t)) {
			Coord w, h;
			
			BmpGetDimensions(t, &w, &h, NULL);
			logt("\t %u x %u %ubpp, %uppi\n", w, h,BmpGetBitDepth(t), BmpGetDensity(t));
		}
	}
	
	return src;
}

static struct BitmapType* WinPickOptimalBitmap(void* dstPtr, struct BitmapType *src)
//returns pointer, possibly to dst
//we create a valid V3 copy to make sure that there is only one depth to choose from, forcing the OS to choose it!
{
	struct PalmBitmap *picked = (struct PalmBitmap*)WinFindOptimalBitmap(src);
	struct PalmBitmapV3 *dst = (struct PalmBitmapV3*)dstPtr; 
	void **bitsPtr = NULL, **clutPtr = NULL;
	
	if (!picked)
		return NULL;
	
	//convert to standalone bitmap
	MemSet(dst, sizeof(*dst), 0);
	dst->width = picked->width;
	dst->height = picked->height;
	dst->stride = picked->stride;
	
	switch (picked->version & 0x7F) {
		case 0:
			if ((dst->compressed = ((struct PalmBitmapV0*)picked)->compressed))
				dst->compressionType = PALM_BMP_COMPRES_TYPE_SCAN_LINE;
			dst->pixelSz = 1;
			dst->pixelFormat = PALM_BMP_PIXEL_FORMAT_INDEXED;
			dst->density = kDensityLow;
			bitsPtr = (void**)(dst + 1);
			break;
		
		case 1:
			if ((dst->compressed = ((struct PalmBitmapV1*)picked)->compressed))
				dst->compressionType = PALM_BMP_COMPRES_TYPE_SCAN_LINE;
			dst->pixelSz = ((struct PalmBitmapV1*)picked)->pixelSz;
			dst->pixelFormat = PALM_BMP_PIXEL_FORMAT_INDEXED;
			dst->density = kDensityLow;
			if (((struct PalmBitmapV1*)picked)->hasColorTable) {
				dst->hasColorTable = 1;
				dst->indirectColorTable = 1;
				clutPtr = (void**)(dst + 1);
				bitsPtr = clutPtr + 1;
			}
			else
				bitsPtr = (void**)(dst + 1);
			break;
		
		case 2:
			dst->compressed = ((struct PalmBitmapV2*)picked)->compressed;
			dst->compressionType = ((struct PalmBitmapV2*)picked)->compressionType;
			dst->directColor = ((struct PalmBitmapV2*)picked)->directColor;
			dst->hasTransparency = ((struct PalmBitmapV2*)picked)->hasTransparency;
			dst->pixelSz = ((struct PalmBitmapV2*)picked)->pixelSz;
			dst->pixelFormat = dst->directColor ? PALM_BMP_PIXEL_FORMAT_RGB565_BE : PALM_BMP_PIXEL_FORMAT_INDEXED;
			
			if (dst->directColor) {
				struct BitmapDirectInfoType *di = (struct BitmapDirectInfoType*)(((uint8_t*)picked) + BmpColortableSize((BitmapPtr)picked) + sizeof(struct PalmBitmapV2));
				uint32_t transColor = (((uint32_t)(di->transparentColor.r & 0xf8)) << 8) + 
										(((uint32_t)(di->transparentColor.g & 0xfc)) << 3) + 
										(((uint32_t)(di->transparentColor.b & 0xf8)) >> 3);
				
				if (dst->pixelFormat == PALM_BMP_PIXEL_FORMAT_RGB565_BE)
					transColor = __builtin_bswap16(transColor);
				
				dst->transparentValue = transColor; 
			}
			else
				dst->transparentValue = ((struct PalmBitmapV2*)picked)->transparentIndex;
			
			if (((struct PalmBitmapV2*)picked)->hasColorTable) {
				dst->hasColorTable = 1;
				dst->indirectColorTable = 1;
				clutPtr = (void**)(dst + 1);
				bitsPtr = clutPtr + 1;
			}
			else
				bitsPtr = (void**)(dst + 1);
			break;
		
		case 3:
			*dst = *(struct PalmBitmapV3*)picked;
			if (dst->hasColorTable) {
				dst->hasColorTable = 1;
				dst->indirectColorTable = 1;
				clutPtr = (void**)(dst + 1);
				bitsPtr = clutPtr + 1;
			}
			else
				bitsPtr = (void**)(dst + 1);
			break;
		
		default:
			SysFatalAlert("Found a weird bitmap version");
	}
	
	dst->version = 0x83;
	dst->indirect = true;
	dst->structSz = sizeof(*dst);
	if (clutPtr)
		*clutPtr = BmpGetColortable((BitmapPtr)picked);
	if (bitsPtr)
		*bitsPtr = BmpGetBits((BitmapPtr)picked);
	
	return (struct BitmapType*)picked;
}

static void PerformDrawingUsingNativeCoords(BitmapPtr bmp, struct PalmWindow *dstWin, const struct RectangleType *r1P, const struct AbsRectType *r2, Coord dstX, Coord dstY, enum PalmWinDrawOperation mode)
{
	struct PalmBitmapV3fat canvasBmpFat, srcBmpFat;
	struct PalmDrawState *ds = dstWin->drawState;
	struct RectangleType r1, dstRect, r3;
	struct PointType dstTopLeft;
	struct PalmCanvas canvas;
	
	dstRect.topLeft.x = dstX + dstWin->bounds.topLeft.x;
	dstRect.topLeft.y = dstY + dstWin->bounds.topLeft.y;
	dstRect.extent = r1P->extent;
	dstTopLeft.x = dstRect.topLeft.x - r1P->topLeft.x;
	dstTopLeft.y = dstRect.topLeft.y - r1P->topLeft.y;
	
	r3.topLeft.x = dstWin->clipBounds.left + dstWin->bounds.topLeft.x;
	r3.topLeft.y = dstWin->clipBounds.top + dstWin->bounds.topLeft.y;
	r3.extent.x = dstWin->clipBounds.right - dstWin->clipBounds.left + 1;
	r3.extent.y = dstWin->clipBounds.bottom - dstWin->clipBounds.top + 1;
	RctGetIntersection(&dstRect, &r3, &dstRect);
	
	r3.topLeft.x = r2->left;
	r3.topLeft.y = r2->top;
	r3.extent.x = r2->right - r2->left + 1;
	r3.extent.y = r2->bottom - r2->top + 1;
	RctGetIntersection(r1P, &r3, &r1);
	
	r1.topLeft.x += dstTopLeft.x;
	r1.topLeft.y += dstTopLeft.y;
	RctGetIntersection(&dstRect, &r1, &dstRect);
	
	if (dstRect.extent.x > 0 && dstRect.extent.y > 0) {
	
		struct PalmBitmapV3 *dstBmp = (struct PalmBitmapV3*)dstWin->bmp;
		enum PalmWinDrawOperation bcpDrawMode = ds->drawMode;
		
		ds->drawMode = mode;
		WinPrvInitCanvas(dstWin, &canvasBmpFat, &canvas);
		BmpPrvConvertBitmap(bmp, &srcBmpFat);
		
		HALDraw_Bitmap(&canvas, &srcBmpFat, &dstRect, dstTopLeft.x, dstTopLeft.y);
		
		//drawing might have uncomprssed dst bitmap - check for that and if so, propagate changes to real bitmap 
		// sicne changes were only made in the fat version that lives on our stack
		if (dstBmp->compressed && !canvasBmpFat.compressed) {
			dstBmp->compressed = 0;
			dstBmp->compressionType = 0;
		}
		ds->drawMode = bcpDrawMode;
	}
}

static void PrvDrawBitmap(struct BitmapType *bmp, Coord x, Coord y, bool useCurDrawMode)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	struct WorstCaseBitmapSize placeholder;
	uint32_t bmpDensity, drawWinDensity;
	struct RectangleType bmpRect;
	struct AbsRectType r2;
	
	bmp = WinPickOptimalBitmap(&placeholder, bmp);
	if (!bmp)
		return;
	
	bmpRect.topLeft.x = 0;
	bmpRect.topLeft.y = 0;
	BmpGetDimensions(bmp, &bmpRect.extent.x, &bmpRect.extent.y, NULL);
	drawWinDensity = BmpGetDensity(w->bmp);
	bmpDensity = BmpGetDensity(bmp);
	
	if (drawWinDensity != bmpDensity && !ds->flags.unscaledBitmaps) {
		
		bmpRect.extent.x = drawWinDensity * bmpRect.extent.x / bmpDensity;
		bmpRect.extent.y = drawWinDensity * bmpRect.extent.y / bmpDensity;
	}
	
	r2.top = 0;
	r2.left = 0;
	r2.right = bmpRect.extent.x - 1;
	r2.bottom = bmpRect.extent.y - 1;
	
	x = scaleCoord(x, ds->scaling.preGarnet.activeToNative, false);
	y = scaleCoord(y, ds->scaling.preGarnet.activeToNative, false);
	PerformDrawingUsingNativeCoords(bmp, w, &bmpRect, &r2, x, y, useCurDrawMode ? ds->drawMode : palmWinDrawOpWinPaint);
}

static void pWinDrawBitmap(struct BitmapType *bitmapP, Coord x, Coord y)
{
	PrvDrawBitmap(bitmapP, x, y, false);
}
DEF_BOOT_PATCH(pWinDrawBitmap, 0xB10);

static void pWinPaintBitmap(struct BitmapType *bitmapP, Coord x, Coord y)
{
	PrvDrawBitmap(bitmapP, x, y, true);
}
DEF_BOOT_PATCH(pWinPaintBitmap, 0xBC8);

static void pWinCopyRectangle(struct PalmWindow *srcW, struct PalmWindow *dstW, const struct RectangleType *srcRect, Coord dstX, Coord dstY, enum PalmWinDrawOperation mode)
{
	uint32_t srcDensity, dstDensity, scaleFactor, curCoordSys = UIGetGlobalsPtr()->curDrawState->coordinateSystem;
	struct PalmWindow* drawW = (struct PalmWindow*)WinGetDrawWindow();
	struct RectangleType rect = *srcRect;
	struct AbsRectType srcClipCopy;
	struct PointType srcTopLeft;
	
	if (!srcW)
		srcW = drawW;
	if (!dstW)
		dstW = drawW;
	
	srcDensity = BmpGetDensity(WinGetBitmap((WinHandle)srcW));
	dstDensity = BmpGetDensity(WinGetBitmap((WinHandle)dstW));
	if (dstW == drawW)
		scaleFactor = dstW->drawState->scaling.preGarnet.activeToNative;
	else if (curCoordSys == dstDensity || curCoordSys == kCoordinatesNative)
		scaleFactor = SCALE_UNITY;
	else
		scaleFactor = SCALE_UNITY * dstDensity / curCoordSys;
	
	dstX = scaleCoord(dstX, scaleFactor, false);
	dstY = scaleCoord(dstY, scaleFactor, false);
	
	if (dstW->drawState->coordinateSystem == kCoordinatesNative)
		scaleFactor = SCALE_UNITY * dstDensity / srcDensity;

	if (!dstW->drawState->flags.unscaledBitmaps) {
		
		rect.topLeft.x = scaleCoord(rect.topLeft.x, scaleFactor, false);
		rect.topLeft.y = scaleCoord(rect.topLeft.y, scaleFactor, false);
		rect.extent.x = scaleCoord(rect.extent.x, scaleFactor, true);
		rect.extent.y = scaleCoord(rect.extent.y, scaleFactor, true);
	}
	srcClipCopy = srcW->clipBounds;
	srcTopLeft = srcW->bounds.topLeft;
	srcClipCopy.left += srcTopLeft.x;
	srcClipCopy.right += srcTopLeft.x;
	srcClipCopy.top += srcTopLeft.y;
	srcClipCopy.bottom += srcTopLeft.y;
	
	if (srcDensity != dstDensity) {
		
		scaleFactor = SCALE_UNITY * dstDensity / srcDensity;
		
		srcClipCopy.left = scaleCoord(srcClipCopy.left, scaleFactor, false);
		srcClipCopy.top = scaleCoord(srcClipCopy.top, scaleFactor, false);
		srcClipCopy.right = scaleCoord(srcClipCopy.right + 1, scaleFactor, false) - 1;
		srcClipCopy.bottom = scaleCoord(srcClipCopy.bottom + 1, scaleFactor, false) - 1;
		srcTopLeft.x = scaleCoord(srcTopLeft.x, scaleFactor, false);
		srcTopLeft.y = scaleCoord(srcTopLeft.y, scaleFactor, false);
	}
	
	rect.topLeft.x += srcTopLeft.x;
	rect.topLeft.y += srcTopLeft.y;
	
	PerformDrawingUsingNativeCoords(WinGetBitmap((WinHandle)srcW), dstW, &rect, &srcClipCopy, dstX, dstY, mode);
}
DEF_BOOT_PATCH(pWinCopyRectangle, 0xAF4);

static void pWinPaintTiledBitmap(struct BitmapType *bitmapP, const struct RectangleType *rectP)
{
	struct WorstCaseBitmapSize placeholder;
	struct BitmapType *bmp = WinPickOptimalBitmap(&placeholder, bitmapP);
	
	if (bmp) {
		struct PalmWindow *drawWin = (struct PalmWindow*)WinGetDrawWindow();
		struct BitmapType *drawWinBmp = WinGetBitmap((WinHandle)drawWin), *tmpBmp = NULL;
		struct RectangleType rectTodo, rectReq = *rectP;
		struct PalmDrawState *ds = drawWin->drawState;
		struct PalmBitmapV3fat fatBmp, canvasBmp;
		uint32_t bmpDensity, drawWinDensity;
		struct PalmCanvas canvas;
		WinHandle tmpWin = NULL;
		int32_t drawX, drawY;
		Coord bmpW, bmpH;
		Err err;
		
		bmpDensity = BmpGetDensity(bmp);
		drawWinDensity = BmpGetDensity(drawWinBmp);
		BmpGetDimensions(bmp, &bmpW, &bmpH, NULL);
		
		if (drawWinDensity != bmpDensity) {
			
			bmpW = drawWinDensity * bmpW / bmpDensity;
			bmpH = drawWinDensity * bmpH / bmpDensity;
		}
		
		tmpBmp = BmpCreateVersion3(bmpW, bmpH, BmpGetBitDepth(drawWinBmp), NULL, drawWinDensity, 0xffffffff, ((struct PalmBitmapV3*)drawWinBmp)->pixelFormat, &err);
		if (err == errNone) {
			
			tmpWin = WinCreateBitmapWindow(tmpBmp, &err);
			if (err != errNone)
				BmpDelete(tmpBmp);
			else {
				
				WinHandle prevDrawWin = WinSetDrawWindow(tmpWin);
				WinDrawBitmap(bmp, 0, 0);
				WinSetDrawWindow(prevDrawWin);
				bmp = tmpBmp;
			}
		}
		
		BmpPrvConvertBitmap(bmp, &fatBmp);
		WinPrvInitCanvasWithBitmap(drawWinBmp, &canvasBmp, ds, &canvas);
		scaleRect(&rectReq, ds->scaling.preGarnet.activeToNative);
		RctGetIntersection(&rectReq, &drawWin->bounds, &rectTodo);
		
		for (drawX = 0; drawX < rectTodo.extent.x; drawX += bmpW) {
			
			rectReq.topLeft.x = drawX + rectTodo.topLeft.x;
			rectReq.extent.x = rectTodo.extent.x - drawX;
			if (rectReq.extent.x > bmpW)
				rectReq.extent.x = bmpW;
			
			for (drawY = 0; drawY < rectTodo.extent.y; drawY += bmpH) {
				
				rectReq.topLeft.y = drawY + rectTodo.topLeft.y;
				rectReq.extent.y = rectTodo.extent.y - drawY;
				if (rectReq.extent.y > bmpH)
					rectReq.extent.y = bmpH;
			
				HALDraw_Bitmap(&canvas, &fatBmp, &rectReq, rectReq.topLeft.x, rectReq.topLeft.y);
			}
		}
		
		if (bmp == tmpBmp) {
			WinDeleteWindow(tmpWin, false);
			BmpDelete(tmpBmp);
		}
	}
}
DEF_BOOT_PATCH(pWinPaintTiledBitmap, 0xC94);

static struct PalmWindow* pWinCreateOffscreenWindow(Coord w, Coord h, WindowFormatType format, Err *errP)
{
	struct BitmapType *dispBmp = WinGetBitmap(WinGetDisplayWindow());
	struct BitmapType *bmp;
	struct PalmWindow *win;
	
	if (!errP)
		return NULL;
	
	*errP = errNone;
	
	if (format != nativeFormat)
		bmp = BmpCreate(w, h, BmpGetBitDepth(dispBmp), NULL, errP);
	else {
		uint32_t desiredCoordSys = UIGetGlobalsPtr()->curDrawState->coordinateSystem;
		uint32_t dispDensity = BmpGetDensity(dispBmp);
		uint32_t scale;
		
		if (desiredCoordSys == kCoordinatesNative)
			desiredCoordSys = BmpGetDensity(WinGetBitmap(WinGetDrawWindow()));
		
		scale = dispDensity * SCALE_UNITY / desiredCoordSys;
		
		w = scaleCoord(w, scale, false);
		h = scaleCoord(h, scale, false);
		
		bmp = BmpCreateVersion3(w, h, BmpGetBitDepth(dispBmp), NULL, dispDensity, 0xffffffff, ((struct PalmBitmapV3*)dispBmp)->pixelFormat, errP);
	}
	
	if (*errP != errNone)
		return NULL;
	
	win = (struct PalmWindow*)WinCreateBitmapWindow(bmp, errP);
	if (*errP != errNone) {
		BmpDelete(bmp);
		return NULL;
	}
	
	win->flags.format = format == genericFormat;
	win->flags.freeBitmap = true;
	
	return win;
}
DEF_BOOT_PATCH(pWinCreateOffscreenWindow, 0xAFC);

struct PalmWindow* pWinSaveBits(const struct RectangleType *rP, Err *errP)
{
	struct RectangleType r = *rP, prevClip;
	int32_t prevCoordsys = -1;
	struct PalmBitmapV3 *bmp;
	struct PalmWindow *win;
	
	scaleRect(&r, ((struct PalmWindow*)WinGetDrawWindow())->drawState->scaling.preGarnet.activeToNative);
	prevCoordsys = WinSetCoordinateSystem(kCoordinatesNative);
	WinGetClip(&prevClip);
	WinResetClip();
	
	win = (struct PalmWindow*)WinCreateOffscreenWindow(r.extent.x, r.extent.y, nativeFormat, errP);
	if (!win)
		return NULL;
	
	bmp = (struct PalmBitmapV3*)(win->bmp);
	bmp->compressed = true;
	bmp->compressionType = BitmapCompressionTypeScanLine;
	*(uint32_t*)BmpGetBits(win->bmp) = 0;	//compressed size is stored as a uint32 before the data in V3 bmps
	
	WinCopyRectangle(NULL, (WinHandle)win, &r, 0, 0, palmWinDrawOpWinPaint);
	WinSetClip(&prevClip);
	
	WinSetCoordinateSystem(prevCoordsys);
	
	if (bmp->compressed) {
		
		uint32_t compressedSz = *(uint32_t*)BmpGetBits(win->bmp);
		uint32_t dataSz, headerSz;
		
		if (compressedSz) {
			
			BmpGetSizes (win->bmp, &dataSz, &headerSz);
			MemPtrResize(win->bmp, dataSz + headerSz);
		}
	}
	
	return win;
}
DEF_BOOT_PATCH(pWinSaveBits, 0xC08);

static void pWinRestoreBits(WinHandle srcWin, Coord x, Coord y)
{
	struct RectangleType rect;
	
	if (!srcWin)
		return;
	if (WinGetCoordinateSystem() == kCoordinatesStandard) {
		
		uint32_t oldCoordsys = WinSetCoordinateSystem(kCoordinatesNative);
		
		WinGetBounds(srcWin, &rect);
		WinCopyRectangle(srcWin, NULL, &rect, WinScaleCoord(x, false), WinScaleCoord(y, false), palmWinDrawOpWinPaint);
		WinSetCoordinateSystem(oldCoordsys);
	}
	else {
		
		WinGetBounds(srcWin, &rect);
		WinCopyRectangle(srcWin, NULL, &rect, x, y, palmWinDrawOpWinPaint);
	}
	
	WinDeleteWindow(srcWin, false);
}
DEF_BOOT_PATCH(pWinRestoreBits, 0xC00);

static void pWinInitializeWindow(struct PalmWindow *win)
{
	struct PalmWindow *dispWin = (struct PalmWindow*)WinGetDisplayWindow();
	WinHandle oldDrawWin;
	
	win->oldStuff2 = dispWin->oldStuff2;
	win->oldStuff1[0] = 0;
	win->oldStuff1[1] = 0;
	win->bmp = dispWin->bmp;
	win->drawState = UIGetGlobalsPtr()->curDrawState;
	oldDrawWin = WinSetDrawWindow((WinHandle)dispWin);
	scaleRect(&win->bounds, win->drawState->scaling.preGarnet.activeToNative);
	WinSetDrawWindow((WinHandle)win);
	WinResetClip();
	WinSetDrawWindow(oldDrawWin);
}
DEF_BOOT_PATCH(pWinInitializeWindow, 0xBA8);



//fron now on: add support for 1.5 density feature set. requires a DAL that allows patching nonexistent functions
uint32_t pWinGetScalingMode(void)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	uint32_t ret = 0;
	
	if (ds->flags.unscaledBitmaps)
		ret |= 1;
	if (ds->flags.unscaledText)
		ret |= 2;
	if (ds->flags.unpaddedText)
		ret |= 4;
	
	return ret;
}
DEF_BOOT_PATCH(pWinGetScalingMode, 0xD5C);

uint32_t pWinSetScalingMode(uint32_t mode)
{
	struct PalmWindow* w = (struct PalmWindow*)WinGetDrawWindow();
	struct PalmDrawState *ds = w->drawState;
	uint32_t ret = 0;
	
	if (ds->flags.unscaledBitmaps)
		ret |= 1;
	ds->flags.unscaledBitmaps = !!(mode & 1);
	if (ds->flags.unscaledText)
		ret |= 2;
	ds->flags.unscaledText = !!(mode & 2);
	if (ds->flags.unpaddedText)
		ret |= 4;
	ds->flags.unpaddedText = !!(mode & 4);
	
	return ret;
}
DEF_BOOT_PATCH(pWinSetScalingMode, 0xD58);

