#include <string.h>
#include <stdio.h>
#include <boot.h>
#include <kal.h>
#include <dal.h>
#include "bitfield.h"
#include "nand.h"
#include "ftl.h"


//This FTL does not concern itself with read disturb. It could, but for the NAND i am currently dealing with, it is not necesary
//This could be added later, easily.

//we do write index page many times and do not use stock NAND's ECC for this. we expect it to fail. we can live with that
//it has: [struct IndexPageData] [struct BlockFormatInfo]  in that order
//those two are stored using our our ecc
//this does violte the "flash must be written in order" mandate, and we can avoid the violation by using page0 only for format info, but
//as long as this works, why not use it this way? :D


#ifdef EMBEDDED
	#include "printf.h"
	
	#define LOG(...)				do { logt(__VA_ARGS__); } while(0);
#else
	#define LOG(...)				do { fprintf(stderr, __VA_ARGS__); } while(0);
#endif

#define OVERCOMMIT					32		//out of 1024

#define OUR_MAGIC					0x6e416e44
#define OUR_VERSION					4

#define PAGES_PER_BLOCK				(1 << state->spec->pagesPerBlockShift)
#define BLOCKS_KEPT_FREE			4		//always. we bank on this many blocks not going bad in a row at once
#define ECC_CAP_BEFORE_REWRITE		65		//% of ecc capacity is ok to use, after that rewrite


struct FtlBlockInfo {
	uint32_t wearCount;
	uint16_t cleanPages;
	uint8_t isFormatted				: 1;
	uint8_t isBad					: 1;
};

struct FtlLifetimeStats {
	
	uint64_t lifetimeLogicalSecR;
	uint64_t lifetimeLogicalSecW;
	uint64_t lifetimeLogicalSecT;
	uint64_t lifetimeLogicalSecReloc;
	uint64_t lifetimeFlashBlockE;
	uint64_t lifetimeFlashPageW;
};

struct BlockFormatInfo {
	uint32_t wearCount;
	uint32_t magic;
};

struct IndexPageData {
	
	uint32_t magic;
	uint32_t version;
	
	uint64_t generation;
	struct FtlLifetimeStats lifetimeStats;
	uint32_t numLogicalSectors;		//so we can mount partitions with "extra" overpovisioned blocks
	uint32_t thisBlockWear;
	
	uint32_t indexData[];			//[FTL_PAGES_PER_BLOCK - 1]
	
	//crc16 here
};

struct FtlState {
	uint32_t mutex;
	
	const struct NandSpec *spec;
	struct NandFuncs *funcs;
	
	//things we'd rather not recalculate
	uint32_t numLogicalSectors;	
	uint32_t numPhysicalPages;		//sector map entry with this value means "none"
	uint8_t secMapBitsPerElem;		//to not have to recalculate it
	
	//current block
	bool haveOpenBlock;
	uint16_t curPage;
	uint32_t curBlock;
	
	//lifetime stats
	struct FtlLifetimeStats lifetimeStats;
	
	//misc state (pointers point to later in this same struct)
	uint64_t generation;
	uint32_t curFormattedBlocks;
	uint32_t curBadBlocks;
	
	struct FtlBlockInfo *blocks;	//[FTL_NUM_BLOCKS];
	uint32_t *secMap;				//bitfield of (FTL_MAP_BITS, FTL_NUM_SECTORS);
	struct IndexPageData *ipd;		//struct IndexPageData with enough space for a proper index for our kind of nand
	uint8_t *pageBuf;				//[NAND_PAGE_SIZE + NAND_SPARE_SIZE];
	uint8_t *spareBuf;				//[NAND_SPARE_SIZE] an extra buffer for just spare r/w
};

struct ReliableBoolT {				//defaults to TRUE
	uint8_t val;
};

struct ReliableBoolF {				//defaults to FALSE
	uint8_t val;
};

struct UneccedDataNormalPage {
	struct ReliableBoolF pageWritten;
	struct ReliableBoolF trimmedAway;
};

struct UneccedDataIndexPage {
	struct ReliableBoolF blockFormatted;
	struct ReliableBoolF blockOpened;
	struct ReliableBoolF blockClosed;
};

struct EccedDataNormalPage {
	uint32_t sectorNo;
};

struct EccedDataIndexPage {
	char unused;
};

static uint32_t ftlPrvSafeEncodingSize(uint32_t inSize);
static bool ftlPrvSafelyDecodeChunk(void *dstP, uint32_t dstLen, const void *srcP, uint32_t srcLen);
static uint32_t ftlPrvSafelyEncodeChunk(void *dstP, uint32_t dstLen, const void *srcP, uint32_t srcLen);


static bool ftlPrvReadBoolRaw(uint_fast8_t val)
{
	//popcount for uint8, this formulation compiles to slightly better code than the obvious way to code this
	uint_fast8_t t;
	
	t = val & 0x55; val = t + ((val - t) >> 1);
	t = val & 0x33; val = t + ((val - t) >> 2);
	val = (val & 0x0f) + (val >> 4);
	
	return val >= 4;
}

static inline bool ftlPrvReadBoolF(const struct ReliableBoolF *boolPtr)
{
	return !ftlPrvReadBoolRaw(boolPtr->val);
}

static inline bool ftlPrvReadBoolT(const struct ReliableBoolT *boolPtr)
{
	return ftlPrvReadBoolRaw(boolPtr->val);
}

static inline void ftlPrvWriteBoolF(struct ReliableBoolF *boolPtr, bool val)
{
	boolPtr->val = val ? 0x00 : 0xff;
}

static inline void ftlPrvWriteBoolT(struct ReliableBoolT *boolPtr, bool val)
{
	boolPtr->val = val ? 0xff : 0x00;
}

static uint_fast16_t ftlPrvCountSparseBytesAvail(const struct NandByteRange *range)
{
	uint_fast16_t ret = 0, now = 0;
	
	do {
		ret += now = range++->len;
	} while (now);
	
	return ret;
}

static uint32_t ftlPrvCollectSparseData(void *dstP, uint32_t len, const uint8_t *src, const struct NandByteRange *pieces)
{
	uint8_t *dst = (uint8_t*)dstP;
	uint32_t now, done, i;
	
	for (done = 0; done < len; done += now, pieces++) {
		
		now = pieces->len;
		
		if (!now)
			break;
		
		if (now > len - done)
			now = len - done;
		
		for (i = 0; i < now; i++)
			*dst++ = src[pieces->offset + i];
	}
	
	return done;
}

static uint32_t ftlPrvDistributeSparseData(uint8_t *dst, const struct NandByteRange *pieces, const void *srcP, uint32_t len)
{
	uint8_t *src = (uint8_t*)srcP;
	uint32_t now, done, i;
	
	for (done = 0; done < len; done += now, pieces++) {
		
		now = pieces->len;
		
		if (!now)
			break;
		
		if (now > len - done)
			now = len - done;
		
		for (i = 0; i < now; i++)
			dst[pieces->offset + i] = *src++;
	}
	
	return done;
}

static uint32_t ftlPrvGetIndexPageRawSize(const struct NandSpec *spec)
{
	uint32_t idexPgDataSz = 0;
	
	idexPgDataSz += sizeof(struct IndexPageData);								//header
	idexPgDataSz += sizeof(uint32_t) * ((1 << spec->pagesPerBlockShift) - 1);	//per-page info
	idexPgDataSz += sizeof(uint16_t);											//crc16
	
	return idexPgDataSz;
}

static MemHandle ftlPrvAllocState(const struct NandSpec *spec, struct NandFuncs *funcs)
{
	uint32_t size = sizeof(struct FtlState), dataBlocks, numLogicalSectors, numPhysicalPages, indexPageSzRoundedToWords, indexPlusFormatOnDiskSz;
	uint_fast8_t secMapBitsPerElem;
	struct FtlState *state;
	MemHandle stateH;
	
	//make sure we have enough blocks
	if (spec->nandBlocks <= spec->maxBadBlocks || spec->nandBlocks < BLOCKS_KEPT_FREE)
		return NULL;
	
	//verify index and format marker would fit
	indexPlusFormatOnDiskSz = ftlPrvSafeEncodingSize(ftlPrvGetIndexPageRawSize(spec)) + ftlPrvSafeEncodingSize(sizeof(struct BlockFormatInfo));
	if (spec->dataBytesPerPage < indexPlusFormatOnDiskSz)
		return NULL;
	
	//verify spare is big enough for our use
	secMapBitsPerElem = ftlPrvCountSparseBytesAvail(spec->eccedSpareBytes);
	if (secMapBitsPerElem < sizeof(struct EccedDataNormalPage) || secMapBitsPerElem < sizeof(struct EccedDataIndexPage))
		return false;
	
	secMapBitsPerElem = ftlPrvCountSparseBytesAvail(spec->uneccedSpareBytes);
	if (secMapBitsPerElem < sizeof(struct UneccedDataNormalPage) || secMapBitsPerElem < sizeof(struct UneccedDataIndexPage))
		return false;
	
	//calc how many blocks will store actual data (vs overcommit or wait-for-go-bad)
	dataBlocks = spec->nandBlocks - spec->maxBadBlocks;
	dataBlocks -= dataBlocks * 32ULL / 1024;									//overcommit
	
	if (dataBlocks > spec->nandBlocks - BLOCKS_KEPT_FREE)
		dataBlocks = spec->nandBlocks - BLOCKS_KEPT_FREE;
	
	//see how many sectors we'll fit into that
	numLogicalSectors = dataBlocks * ((1 << spec->pagesPerBlockShift) - 1);
	
	//other misc math
	numPhysicalPages = spec->nandBlocks << spec->pagesPerBlockShift;
	secMapBitsPerElem = 32 - __builtin_clz(numPhysicalPages + 1);				//numPhysicalPages is itself a valid value so we must account for it
	
	//per-block info
	size += sizeof(struct FtlBlockInfo) * spec->nandBlocks;
	
	//index page (and rounded to nearest word)
	indexPageSzRoundedToWords = (ftlPrvGetIndexPageRawSize(spec) + sizeof(uint32_t) - 1) / sizeof(uint32_t) * sizeof(uint32_t);
	size += indexPageSzRoundedToWords;
	
	//a page buffer
	size += spec->dataBytesPerPage + spec->spareAreaSize;
	
	//a spare buffer
	size += spec->spareAreaSize;
	
	//index info for currently-being-written block
	size += sizeof(uint32_t) * ((1 << spec->pagesPerBlockShift) - 1);
	
	//sector map
	size += sizeof(uint32_t) * BITFIELD_NWORDS(secMapBitsPerElem, numLogicalSectors);
	
	stateH = MemChunkNew(0, size, 0x1000);
	if (!stateH)
		return NULL;
	
	state = MemHandleLock(stateH);
	MemSet(state, size, 0);
	
	if (KALMutexCreate(&state->mutex, CREATE_4CC('n','a','n','d'))) {
		MemHandleUnlock(stateH);
		MemChunkFree(stateH);
		return NULL;
	}
	
	state->spec = spec;
	state->funcs = funcs;
	state->numLogicalSectors = numLogicalSectors;
	state->numPhysicalPages = numPhysicalPages;
	state->secMapBitsPerElem = secMapBitsPerElem;
	state->blocks = (struct FtlBlockInfo*)(state + 1);
	state->secMap = (uint32_t*)(state->blocks + spec->nandBlocks);
	state->ipd = (struct IndexPageData*)(state->secMap + BITFIELD_NWORDS(secMapBitsPerElem, numLogicalSectors));
	state->pageBuf = ((uint8_t*)state->ipd) + indexPageSzRoundedToWords;
	state->spareBuf = state->pageBuf + spec->dataBytesPerPage + spec->spareAreaSize;
	
	MemHandleUnlock(stateH);
	return stateH;
}

static int_fast8_t ftlPrvPageRead(struct FtlState *state, uint32_t pageNo, bool withEcc)				//to internal buffer
{
	logt(" NRf %u.%u\n", pageNo / 64, pageNo % 64);
	
	return state->funcs->NandPageReadPartial(state->funcs, pageNo, state->pageBuf, 0, state->spec->dataBytesPerPage + state->spec->spareAreaSize, withEcc);
}

static int_fast8_t ftlPrvPageReadSpare(struct FtlState *state, uint32_t pageNo, bool withEcc)			//spare spare buffer
{
	logt(" NRs %u.%u\n", pageNo / 64, pageNo % 64);
	
	return state->funcs->NandPageReadPartial(state->funcs, pageNo, state->spareBuf, state->spec->dataBytesPerPage, state->spec->spareAreaSize, withEcc);
}

static bool ftlPrvPageWritePartial(struct FtlState *state, uint32_t pageNo, const void *data, uint32_t ofst, uint32_t len, bool withEcc)
{
	logt(" NWp %u.%u\n", pageNo / 64, pageNo % 64);
	
	return state->funcs->NandPageWritePartial(state->funcs, pageNo, data, ofst, len, withEcc);
}

static bool ftlPrvPageWrite(struct FtlState *state, uint32_t pageNo, const void *data, bool withEcc)
{
	logt(" NWf %u.%u\n", pageNo / 64, pageNo % 64);
	
	return state->funcs->NandPageWritePartial(state->funcs, pageNo, data, 0, state->spec->dataBytesPerPage + state->spec->spareAreaSize, withEcc);
}

static bool ftlPrvBlockErase(struct FtlState *state, uint32_t firstPageNo)
{
	logt(" NE %u\n", firstPageNo / 64);
	
	return state->funcs->NandBlockErase(state->funcs, firstPageNo);
}

static bool ftlPrvBlkIsBad(struct FtlState *state, uint32_t blkNo)		//uses spare buffer
{
	uint_fast8_t i;
	
	(void)ftlPrvPageReadSpare(state, blkNo << state->spec->pagesPerBlockShift, false);
	
	//any non-0xFF bytes in the marker area mean "bad"
	for (i = 0; i < state->spec->badBlockMarker.len; i++) {
		
		if (state->spareBuf[state->spec->badBlockMarker.offset + i] != 0xff)
			return true;
	}
	
	return false;
}

static void ftlPrvSetMapping(struct FtlState *state, uint32_t logicalAddr, uint32_t physicalAddr)
{
	bitfieldWrite(state->secMap, state->secMapBitsPerElem, logicalAddr, physicalAddr);
}

static uint32_t ftlPrvGetMapping(struct FtlState *state, uint32_t logicalAddr)
{
	return bitfieldRead(state->secMap, state->secMapBitsPerElem, logicalAddr);
}

static uint16_t ftlPrvChecksumIndexPage(struct FtlState *state, const struct IndexPageData *ipd)
{
	return Crc16CalcBlock(ipd, ftlPrvGetIndexPageRawSize(state->spec) - sizeof(uint16_t), 0);
}

static uint16_t* ftlPrvGetChecksumLocForIndexPage(struct FtlState *state, const struct IndexPageData *ipd)
{
	return (uint16_t*)(((char*)ipd) + ftlPrvGetIndexPageRawSize(state->spec) - sizeof(uint16_t));
}

static bool ftlPrvMarkBlockBad(struct FtlState *state, uint32_t blkNo)
{
	MemSet(state->pageBuf + state->spec->dataBytesPerPage, state->spec->spareAreaSize, 0xff);
	MemSet(state->pageBuf + state->spec->dataBytesPerPage + state->spec->badBlockMarker.offset, state->spec->badBlockMarker.len, 0x00);
	
	return ftlPrvPageWritePartial(state, blkNo << state->spec->pagesPerBlockShift, state->pageBuf + state->spec->dataBytesPerPage, state->spec->dataBytesPerPage, state->spec->spareAreaSize, false);
}

static bool ftlPrvWritePageWithData(struct FtlState *state, uint32_t secNum)	//data is in the page buffer
{
	uint32_t physicalAddr = (state->curBlock << state->spec->pagesPerBlockShift) + state->curPage;
	struct UneccedDataNormalPage ued;
	struct EccedDataNormalPage ed;
	
	//prepare spare area
	MemSet(state->pageBuf + state->spec->dataBytesPerPage, state->spec->spareAreaSize, 0xff);
	ed.sectorNo = secNum;
	ftlPrvWriteBoolF(&ued.pageWritten, true);
	ftlPrvWriteBoolF(&ued.trimmedAway, false);
	(void)ftlPrvDistributeSparseData(state->pageBuf + state->spec->dataBytesPerPage, state->spec->uneccedSpareBytes, &ued, sizeof(ued));
	(void)ftlPrvDistributeSparseData(state->pageBuf + state->spec->dataBytesPerPage, state->spec->eccedSpareBytes, &ed, sizeof(ed));

	//account
	state->lifetimeStats.lifetimeFlashPageW++;
	state->ipd->indexData[state->curPage] = secNum;
	state->blocks[state->curBlock].cleanPages++;
	if (state->blocks[state->curBlock].cleanPages > PAGES_PER_BLOCK - 1) {
		LOG("too many clean pages allegedly in blk %u\n", state->curBlock);
		return false;
	}
	ftlPrvSetMapping(state, secNum, physicalAddr);
	state->curPage++;
	
	//write
	return ftlPrvPageWrite(state, physicalAddr, state->pageBuf, true);
}

static bool ftlPrvFormatErasedBlock(struct FtlState *state, uint32_t blkNo)	//called on an EMPTY block
{
	struct UneccedDataIndexPage ued;
	struct BlockFormatInfo bfi;
	
	MemSet(state->pageBuf, state->spec->dataBytesPerPage + state->spec->spareAreaSize, 0xff);
	
	bfi.wearCount = state->blocks[blkNo].wearCount;
	bfi.magic = OUR_MAGIC;
	
	(void)ftlPrvSafelyEncodeChunk(
		state->pageBuf + ftlPrvSafeEncodingSize(ftlPrvGetIndexPageRawSize(state->spec)),
		ftlPrvSafeEncodingSize(sizeof(bfi)), &bfi, sizeof(bfi));

	if (!blkNo) {
		
		unsigned char *from = (void*)&bfi, *to = state->pageBuf + ftlPrvSafeEncodingSize(ftlPrvGetIndexPageRawSize(state->spec));
		
		LOG("encoding for blk0: %02x %02x %02x %02x %02x %02x %02x %02x\n", from[0], from[1], from[2], from[3], from[4], from[5], from[6], from[7]);
		LOG("got: %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x %02x\n",
			to[0], to[1], to[2], to[3], to[4], to[5], to[6], to[7], to[8], to[9], to[10], to[11], to[12], to[13], to[14], to[15]);
		
		MemSet(&bfi, sizeof(bfi), 0);
		LOG("decodes: %u\n", ftlPrvSafelyDecodeChunk(&bfi, sizeof(bfi),  state->pageBuf + ftlPrvSafeEncodingSize(ftlPrvGetIndexPageRawSize(state->spec)), 16));
		LOG("recodes to: %02x %02x %02x %02x %02x %02x %02x %02x\n", from[0], from[1], from[2], from[3], from[4], from[5], from[6], from[7]);
	}


	ftlPrvWriteBoolF(&ued.blockFormatted, true);
	ftlPrvWriteBoolF(&ued.blockOpened, false);
	ftlPrvWriteBoolF(&ued.blockClosed, false);
	(void)ftlPrvDistributeSparseData(state->pageBuf + state->spec->dataBytesPerPage, state->spec->uneccedSpareBytes, &ued, sizeof(ued));

	state->lifetimeStats.lifetimeFlashPageW++;
	if (!ftlPrvPageWrite(state, ((blkNo + 1) << state->spec->pagesPerBlockShift) - 1, state->pageBuf, false)) {
		LOG("ftlPrvFormatErasedBlock: write failed\n");
		return false;
	}
	
	state->blocks[blkNo].isFormatted = true;
	
	return true;
}

static bool ftlPrvFormatBlockRaw(struct FtlState *state, uint32_t blkNo)		//data goes poof, block is erased, no checks
{
	state->lifetimeStats.lifetimeFlashBlockE++;
	state->blocks[blkNo].wearCount++;
	
	//erase can fail. we can live with it
	if (!ftlPrvBlockErase(state, blkNo << state->spec->pagesPerBlockShift)) {
		
		LOG("block %u went bad\n", blkNo);
		
		state->blocks[blkNo].isBad = true;
		state->curBadBlocks++;
		return ftlPrvMarkBlockBad(state, blkNo);	//yes we reutrn TRUE even though we did not format a block. our return code indicates an unrecoverable failure. a block going bad isn't
	}
	
	if (!ftlPrvFormatErasedBlock(state, blkNo))
		return false;
	
	state->curFormattedBlocks++;
	return true;
}

static bool ftlPrvFormatBlock(struct FtlState *state, uint32_t blkNo)		//data goes poof, block is erased
{
	//no bad blocks
	if (state->blocks[blkNo].isBad) {
		LOG(" ftlPrvFormatBlock: refusing on %u due to bad\n", blkNo);
		return false;
	}
	
	//refuse to do it to blocks with clean pages
	if (state->blocks[blkNo].cleanPages) {
		
		uint32_t i;
		
		LOG(" ftlPrvFormatBlock: refusing on %u due to %u clean pages\n", blkNo, state->blocks[blkNo].cleanPages);
		
		for (i = 0; i < state->numLogicalSectors; i++) {
			
			uint32_t phy = ftlPrvGetMapping(state, i);
			uint32_t blk = phy >> state->spec->pagesPerBlockShift;
			
			if (phy == state->numPhysicalPages)
				continue;
			
			if (blk == blkNo)
				LOG(" VA 0x%08x points to 0x%08x (page %u in this block)\n", i, phy, phy % PAGES_PER_BLOCK);
		}
		
		return false;
	}
	
	//maybe it already is formatted
	if (state->blocks[blkNo].isFormatted)
		return true;
	
	if (!ftlPrvFormatBlockRaw(state, blkNo)) {
		LOG(" ftlPrvFormatBlock: refusing on %u due to raw error\n", blkNo);
		return false;
	}
	
	return true;
}

static uint32_t ftlPrvPickVictimGcBlock(struct FtlState *state, uint_fast16_t maxCleanPages)		//pick block we'll GC from
{
	uint32_t blkNo, bestScore = 0, bestBlock = state->spec->nandBlocks;
	
	if (state->curFormattedBlocks <= BLOCKS_KEPT_FREE) {		//do not bother GCing if we have free blocks
		
		//we do prefer erased blocks as those have already accumulated the cost
		for (blkNo = 0; blkNo < state->spec->nandBlocks; blkNo++) {
			
			uint32_t thisScore = 0;
			
			//current block is never eligible
			if (blkNo == state->curBlock)
				continue;
			
			//our "cleanliness" estimate is usually only approximately accurate, so we lean on it only slightly
			if (state->blocks[blkNo].isBad || state->blocks[blkNo].isFormatted)
				continue;
			
			//limit as requested
			if (state->blocks[blkNo].cleanPages > maxCleanPages)
				continue;
			
			//more wear count is bad
			thisScore -= state->blocks[blkNo].wearCount * 16;
			
			//more clean pages is bad
			thisScore -= state->blocks[blkNo].cleanPages * 64 >> state->spec->pagesPerBlockShift;
			
			if (thisScore <= bestScore)
				continue;
			
			bestScore = thisScore;
			bestBlock = blkNo;
		}
	}
	
	return bestBlock;
}

static uint32_t ftlPrvPickVictimFormattedBlock(struct FtlState *state)		//pick block we'll GC to
{
	//easy here - just pick the least worn formatted block

	uint32_t blkNo, bestScore = 0xffffffff, bestBlock = state->spec->nandBlocks;
	
	for (blkNo = 0; blkNo < state->spec->nandBlocks; blkNo++) {
		if (state->blocks[blkNo].isBad || !state->blocks[blkNo].isFormatted)
			continue;
		
		if (state->blocks[blkNo].wearCount > bestScore)
			continue;
		
		bestScore = state->blocks[blkNo].wearCount;
		bestBlock = blkNo;
	}
	
	return bestBlock;
}

static bool ftlPrvMarkBlockAsOpened(struct FtlState *state, uint32_t blkNo)
{
	struct UneccedDataIndexPage ued;
	
	MemSet(state->pageBuf + state->spec->dataBytesPerPage, state->spec->spareAreaSize, 0xff);
	
	//set up spare area
	ftlPrvWriteBoolF(&ued.blockFormatted, true);
	ftlPrvWriteBoolF(&ued.blockOpened, true);
	ftlPrvWriteBoolF(&ued.blockClosed, false);
	(void)ftlPrvDistributeSparseData(state->pageBuf + state->spec->dataBytesPerPage, state->spec->uneccedSpareBytes, &ued, sizeof(ued));

	//write
	state->lifetimeStats.lifetimeFlashPageW++;
	state->blocks[blkNo].isFormatted = 0;
	state->curFormattedBlocks--;	//we about to make it NOT formatted
	
	return ftlPrvPageWritePartial(state, ((blkNo + 1) << state->spec->pagesPerBlockShift) - 1, state->pageBuf + state->spec->dataBytesPerPage, state->spec->dataBytesPerPage, state->spec->spareAreaSize, false);
}

static bool ftlPrvGcData(struct FtlState *state, uint32_t fromBlk)
{
	struct UneccedDataNormalPage ued;
	struct EccedDataNormalPage ed;
	uint_fast16_t i;
	
	for (i = 0; i < (uint_fast16_t)(PAGES_PER_BLOCK - 1); i++) {
		
		uint32_t logicalAddr, expectedPhysicalAddr, physicalAddr = (fromBlk << state->spec->pagesPerBlockShift) + i;
		int_fast8_t readRet;
		
		readRet = ftlPrvPageRead(state, physicalAddr, true);
		
		(void)ftlPrvCollectSparseData(&ued, sizeof(ued), state->pageBuf + state->spec->dataBytesPerPage, state->spec->uneccedSpareBytes);
		(void)ftlPrvCollectSparseData(&ed, sizeof(ed), state->pageBuf + state->spec->dataBytesPerPage, state->spec->eccedSpareBytes);
		
		if (!ftlPrvReadBoolF(&ued.pageWritten))
			continue;
		
		if (readRet < 0) {	//we possibly lost data
			LOG(" ftlPrvOpenSomeBlockLL: lost data on relocate\n");
			return false;
		}
					
		//sanity check
		logicalAddr = ed.sectorNo;
		if (logicalAddr >= state->numLogicalSectors) {
			LOG(" ftlPrvOpenSomeBlockLL: bad LA\n");
			return false;
		}
		expectedPhysicalAddr = ftlPrvGetMapping(state, logicalAddr);
		
		if (ftlPrvReadBoolF(&ued.trimmedAway)) {	//was trimmed, but we shoudl still check map
			
			/*
				index page is not updated when TRIM is done, this means that if this block HAD the most recent mapping for a VA,
				it was then trimmed, and the device was then remounted, the map will still point here. While we GC we'll correctly
				not move this garbage data, but our pointer will dangle. Thus we hceck VAs of ALL pages we read from the device
				during GC to se if they map back to this block. TRIMMED away data will THUS be accounted for
			*/
			
			if (expectedPhysicalAddr == physicalAddr) {	//this IS the case
				
				ftlPrvSetMapping(state, logicalAddr, state->numPhysicalPages);
				if (state->blocks[fromBlk].isFormatted) {
					
					LOG("GC: saw page pointing at formatted block\n");
					continue;
				}
				state->blocks[fromBlk].cleanPages--;	//would have been counted as such
			}
			
			continue;
		}
		
		if (expectedPhysicalAddr != physicalAddr) {
			LOG(" ftlPrvOpenSomeBlockLL: bad PA. mapping says PA 0x%08x's VA is recorded as 0x%08x, but map says that va is at PA 0x%08x, yet it is not marked as trimmed\n", 
				physicalAddr, logicalAddr, expectedPhysicalAddr);
			
			(void)ftlPrvPageRead(state, expectedPhysicalAddr, false);
			(void)ftlPrvCollectSparseData(&ued, sizeof(ued), state->pageBuf + state->spec->dataBytesPerPage, state->spec->uneccedSpareBytes);
			(void)ftlPrvCollectSparseData(&ed, sizeof(ed), state->pageBuf + state->spec->dataBytesPerPage, state->spec->eccedSpareBytes);
			
			LOG("PA 0x%08x's VA is 0x%08x, written: %u trimed: %u\n", expectedPhysicalAddr, ed.sectorNo, ftlPrvReadBoolF(&ued.pageWritten), ftlPrvReadBoolF(&ued.trimmedAway));
			
			return false;
		}
		
		//we have data to move
		if (!ftlPrvWritePageWithData(state, logicalAddr)) {
			LOG(" ftlPrvOpenSomeBlockLL: write err\n");
			return false;
		}
		
		//account
		state->blocks[fromBlk].cleanPages--;
		if (state->blocks[fromBlk].isFormatted) {
			LOG("GC: saw page pointing at formatted block\n");
			continue;
		}
		
		//we do NOT mark pages as trimmed here since we're about to erase the block anyways. let's save it some wear
	}
	
	//we moved all pages out of the from block, it can now be formatted
	if (!ftlPrvFormatBlock(state, fromBlk)) {
		LOG(" ftlPrvOpenSomeBlockLL: format err\n");
		return false;
	}
	
	return true;
}

static bool ftlPrvOpenSomeBlockLL(struct FtlState *state)	//might produce a block with NO free pages!
{
	uint32_t fromBlk = ftlPrvPickVictimGcBlock(state, PAGES_PER_BLOCK);
	uint32_t toBlk = ftlPrvPickVictimFormattedBlock(state);

	if (toBlk == state->spec->nandBlocks) {
		uint32_t i;
		
		LOG(" ftlPrvOpenSomeBlockLL: failed to find \"TO\" block\n");
		
		for (i = 0; i < state->spec->nandBlocks; i++) {
			
			if (state->blocks[i].isFormatted)
				LOG(" avail formatted block: %u\n", i);
		}
		
		return false;
	}
	
	if (!ftlPrvMarkBlockAsOpened(state, toBlk)) {
		LOG(" ftlPrvOpenSomeBlockLL: failed to open \"TO\" block\n");
		return false;
	}
	
	state->haveOpenBlock = true;
	state->curBlock = toBlk;
	state->curPage = 0;
	
	if (fromBlk != state->spec->nandBlocks) {	//we do not always have a block to GC from (consider a newly-formatted device)
		
		uint_fast16_t availPagesPages;
		
		if (!ftlPrvGcData(state, fromBlk))
			return false;
		
		//at this point we might have enough free pages to GC from another block. see if a suitable one exists
		
		availPagesPages = PAGES_PER_BLOCK - 1 - state->curPage;
		if (availPagesPages > 1) {
			
			uint32_t fromBlk2 = ftlPrvPickVictimGcBlock(state, availPagesPages - 1);
			
			if (fromBlk2 != state->spec->nandBlocks) {
				if (!ftlPrvGcData(state, fromBlk2))
					return false;
			}
		}
	}
	
	return true;
}

static bool ftlPrvCloseCurrentBlock(struct FtlState *state)
{
	struct IndexPageData *ipd = state->ipd;
	struct UneccedDataIndexPage ued;
	bool ret;
	
	//must have an open block and be pointing to the last page
	if (!state->haveOpenBlock || state->curPage != PAGES_PER_BLOCK - 1)
		return false;
	
	//we WILL write. be accurate and include that
	state->lifetimeStats.lifetimeFlashPageW++;
	
	//produce the data
	ipd->magic = OUR_MAGIC;
	ipd->version = OUR_VERSION;
	ipd->generation = ++state->generation;
	ipd->lifetimeStats = state->lifetimeStats;
	ipd->numLogicalSectors = state->numLogicalSectors;
	ipd->thisBlockWear = state->blocks[state->curBlock].wearCount;
	
	//checksum it
	*ftlPrvGetChecksumLocForIndexPage(state, ipd) = ftlPrvChecksumIndexPage(state, ipd);
	
	//encode it
	MemSet(state->pageBuf, state->spec->dataBytesPerPage + state->spec->spareAreaSize, 0xff);
	(void)ftlPrvSafelyEncodeChunk(state->pageBuf, state->spec->dataBytesPerPage, ipd, ftlPrvGetIndexPageRawSize(state->spec));

	//set up spare area
	ftlPrvWriteBoolF(&ued.blockFormatted, true);
	ftlPrvWriteBoolF(&ued.blockOpened, true);
	ftlPrvWriteBoolF(&ued.blockClosed, true);
	(void)ftlPrvDistributeSparseData(state->pageBuf + state->spec->dataBytesPerPage, state->spec->uneccedSpareBytes, &ued, sizeof(ued));

	//write
	ret = ftlPrvPageWrite(state, ((state->curBlock + 1) << state->spec->pagesPerBlockShift) - 1, state->pageBuf, false);
	
	//account
	state->haveOpenBlock = false;
	
	//done
	return ret;
}

static bool ftlPrvSync(struct FtlState *state, bool forceWrite)
{
	if (!state->haveOpenBlock)
		return true;
	
	//if the currently open block has NO pages written, just leave it be, unless we were forced to write
	if (!state->curPage && !forceWrite)
		return true;
	
	//else we'll fill the rest of the index with nothingness and write it
	while (state->curPage != PAGES_PER_BLOCK - 1)
		state->ipd->indexData[state->curPage++] = state->numLogicalSectors;
	
	return ftlPrvCloseCurrentBlock(state);
}

static bool ftlPrvOpenBlockIfNeeded(struct FtlState *state)	//will produce a block with at least one free page
{
	while (1) {
		
		if (!state->haveOpenBlock) {
			
			if (!ftlPrvOpenSomeBlockLL(state)) {
				LOG(" ftlPrvOpenBlockIfNeeded: failed to openLL\n");
				return false;
			}
		}
		//we have a block, but maybe it needs to be closed?
		//wear leveling aginst long-term-static data can cause this even immediately after a call to ftlPrvOpenSomeBlockLL()
		if (state->curPage != PAGES_PER_BLOCK - 1)
			return true;
		
		if (!ftlPrvCloseCurrentBlock(state)) {
			LOG(" ftlPrvOpenBlockIfNeeded: failed to close\n");
			return false;
		}
	}
}

//return true if this is the new newest mapping fo rthis logicl addr
static bool ftlPrvMountMaybeMapPage(struct FtlState *state, uint64_t *generations, uint64_t curGeneration, uint32_t physicalAddr, uint32_t logicalAddr)
{
	uint32_t competitorPage, competitorBlock;
	
	logt("  proposal to map %u -> %u.%u\n", logicalAddr, physicalAddr >> state->spec->pagesPerBlockShift, physicalAddr & ((1 <<  state->spec->pagesPerBlockShift) - 1));
	
	//invalid logical addresses get ignored
	if (logicalAddr < state->numLogicalSectors) {
	
		competitorPage = ftlPrvGetMapping(state, logicalAddr);
		competitorBlock = competitorPage >> state->spec->pagesPerBlockShift;
		
		if (competitorPage == state->numPhysicalPages || generations[competitorBlock] <= curGeneration) {		//later instance still wins, first encounter also wins
			
			logt("   mapped %u -> %u.%u\n", logicalAddr, physicalAddr >> state->spec->pagesPerBlockShift, physicalAddr & ((1 <<  state->spec->pagesPerBlockShift) - 1));
			ftlPrvSetMapping(state, logicalAddr, physicalAddr);
			return true;
		}
	}
	
	return false;
}

//we are guaranteed to be at the end of the mount process, no more blocks!
static bool ftlPrvMountBlockWithFullScan(struct FtlState *state, uint32_t blkNo, uint64_t *generations)
{
	uint_fast16_t pgNo, pagesValid = 0, pagesTrimmed = 0;
	uint64_t thisGeneration = ++state->generation;
	struct UneccedDataNormalPage ued;
	struct EccedDataNormalPage ed;
	
	//we'll produce an index as we go
	for (pgNo = 0; pgNo < (uint_fast16_t)(PAGES_PER_BLOCK - 1); pgNo++)
		state->ipd->indexData[pgNo] = state->numLogicalSectors;
	
	for (pgNo = 0; pgNo < (uint_fast16_t)(PAGES_PER_BLOCK - 1); pgNo++) {
		
		uint32_t logicalAddr, physicalAddr = (blkNo << state->spec->pagesPerBlockShift) + pgNo;
		int_fast8_t readRet;
		
		readRet = ftlPrvPageReadSpare(state, physicalAddr, true);
		(void)ftlPrvCollectSparseData(&ued, sizeof(ued), state->spareBuf, state->spec->uneccedSpareBytes);
		
		if (!ftlPrvReadBoolF(&ued.pageWritten))		//writes ARE in order
			break;
		
		if (readRet >= 0) {
			
			(void)ftlPrvCollectSparseData(&ed, sizeof(ed), state->spareBuf, state->spec->eccedSpareBytes);
			
			logicalAddr = ed.sectorNo;
			if (ftlPrvMountMaybeMapPage(state, generations, thisGeneration, physicalAddr, logicalAddr)) {
				
				if (ftlPrvReadBoolF(&ued.trimmedAway)) {
					
					ftlPrvSetMapping(state, logicalAddr, state->numPhysicalPages);
					pagesTrimmed++;
				}
				
				state->ipd->indexData[pgNo] = logicalAddr;
			}
			pagesValid++;
		}
	}
	
	//stats are missing this blocks's erasure (and maybe more blocks), programming, and maybe more things. try to improve that
	state->lifetimeStats.lifetimeLogicalSecW += pagesValid;		//this might be an overestimation
	state->lifetimeStats.lifetimeLogicalSecT += pagesTrimmed;	//this might be an underestimation
	state->lifetimeStats.lifetimeFlashBlockE++;
	state->lifetimeStats.lifetimeFlashPageW += pagesValid;
	
	state->haveOpenBlock = true;
	state->curBlock = blkNo;
	state->curPage = pgNo;
	
	//we could be in a situation where we need to close this block right away (we were interrupted just before closing it last boot)
	if (state->curPage == PAGES_PER_BLOCK - 1 && !ftlPrvCloseCurrentBlock(state))
		return false;
	
	return true;
}

static bool ftlPrvMountBlockWithIndexPage(struct FtlState *state, uint32_t blkNo, uint64_t *generations, bool readParams /* else verify */)
{
	struct IndexPageData *ipd = state->ipd;
	uint_fast16_t i;
	
	//we must be able to decode it
	if (!ftlPrvSafelyDecodeChunk(ipd, ftlPrvGetIndexPageRawSize(state->spec), state->pageBuf, state->spec->dataBytesPerPage))
		return false;
	
	//it must have a valid magic number and version
	if (ipd->magic != OUR_MAGIC || ipd->version != OUR_VERSION)
		return false;
	
	//crc must match
	if (*ftlPrvGetChecksumLocForIndexPage(state, ipd) != ftlPrvChecksumIndexPage(state, ipd))
		return false;
	
	if (readParams) {
		
		//our only param is num logical sectors and our current state should be an upper limit
		if (ipd->numLogicalSectors > state->numLogicalSectors)
			return false;
		
		state->numLogicalSectors = ipd->numLogicalSectors;
	}
	else {
		
		//verify
		if (ipd->numLogicalSectors != state->numLogicalSectors)
			return false;
	}
	
	
	//account for generation & lifetime stats
	generations[blkNo] = ipd->generation;
	if (state->generation < ipd->generation) {
		
		state->lifetimeStats = ipd->lifetimeStats;
		state->generation = ipd->generation;
	}
	state->blocks[blkNo].wearCount = ipd->thisBlockWear;
	
	//deal with each page
	for (i = 0; i < (uint_fast16_t)(PAGES_PER_BLOCK - 1); i++) {
		
		uint32_t physicalAddr = (blkNo << state->spec->pagesPerBlockShift) + i;
		uint32_t logicalAddr = ipd->indexData[i];
		
		ftlPrvMountMaybeMapPage(state, generations, ipd->generation, physicalAddr, logicalAddr);
	}
	
	return true;
}

static bool ftlPrvMountWithState(struct FtlState *state)
{
	uint32_t i, blkNo, prevUnclosedBlock, prevErasedBlock, numFormattedBlocks = 0, numBadBlocks = 0;
	bool haveUnclosedBlock = false, haveErasedBlock = false, haveClosedBlock = false;
	struct UneccedDataIndexPage ued;
	uint64_t *generations;
	bool ret = false;
	
	
	MemSet(state->blocks, sizeof(struct FtlBlockInfo) * state->spec->nandBlocks, 0);
	MemSet(state->secMap, sizeof(uint32_t) * BITFIELD_NWORDS(state->secMapBitsPerElem, state->numLogicalSectors), 0);
	MemSet(state->ipd, sizeof(struct IndexPageData), 0);	//just the header
	
	generations = MemChunkNew(0, sizeof(uint64_t) * state->spec->nandBlocks, 0x1200);
	if (!generations) {
		LOG("cannot allocate generation tracker\n");
		return false;
	}
	
	MemSet(generations, sizeof(uint64_t) * state->spec->nandBlocks, 0);
	
	LOG("mount...\n");
	
	//clear out the map
	for (i = 0; i < state->numLogicalSectors; i++)
		ftlPrvSetMapping(state, i, state->numPhysicalPages);
	
	//go block by block
	for (blkNo = 0; blkNo < state->spec->nandBlocks; blkNo++) {
		
		bool blockOpened, blockClosed, blockFormatted, validFormatInfo;
		struct BlockFormatInfo bfi;
		
		
		//bad?
		if (ftlPrvBlkIsBad(state, blkNo)) {
			
			numBadBlocks++;
			state->blocks[blkNo].isBad = true;
			continue;
		}
		
		//read the last page of this block - we have own ecc here so no need for flash's ECC
		(void)ftlPrvPageRead(state, ((blkNo + 1) << state->spec->pagesPerBlockShift) - 1, false);
		
		//grab OUR info
		(void)ftlPrvCollectSparseData(&ued, sizeof(ued), state->pageBuf + state->spec->dataBytesPerPage, state->spec->uneccedSpareBytes);
		blockFormatted = ftlPrvReadBoolF(&ued.blockFormatted);
		blockOpened = ftlPrvReadBoolF(&ued.blockOpened);
		blockClosed = ftlPrvReadBoolF(&ued.blockClosed);
		
		validFormatInfo = ftlPrvSafelyDecodeChunk(&bfi, sizeof(bfi), state->pageBuf + ftlPrvSafeEncodingSize(ftlPrvGetIndexPageRawSize(state->spec)), ftlPrvSafeEncodingSize(sizeof(bfi))) && bfi.magic == OUR_MAGIC;
		
		if (!blockFormatted) {	
			
			if (blockOpened || blockClosed)	{//invalid 
				LOG("invalid unformatted block %u\n", blkNo);
				goto out;
			}
			
			//we should never have more than one of these, and even then only if we were interrupted during an erase
			if (haveErasedBlock)
				goto out;
			
			haveErasedBlock = true;
			prevErasedBlock = blkNo;
		}
		else if (!blockOpened) {		//formatted block
			
			if (blockClosed) {			//invalid
				
				LOG("invalid closed unopened block %u\n", blkNo);
				goto out;
			}
			
			//formatted page with format info missing or undecodeable? bad
			if (!validFormatInfo) {
				
				LOG("invalid formatinfo on unopened blk %u\n", blkNo);
				goto out;
			}
			
			state->blocks[blkNo].wearCount = bfi.wearCount;
			state->blocks[blkNo].isFormatted = true;
			numFormattedBlocks++;
		}
		else if (!blockClosed) {		//we did not close it last session. at most one block should be like this
			
			if (haveUnclosedBlock) {	//this should not ever happen - more than one unclosed block
				LOG("too many unclosed blocks: %u %u\n", prevUnclosedBlock, blkNo);
				goto out;
			}
			
			if (!validFormatInfo) {
				
				LOG("invalid formatinfo on unclosed blk %u\n", blkNo);
				goto out;
			}
			
			state->blocks[blkNo].wearCount = bfi.wearCount;
			
			haveUnclosedBlock = true;
			prevUnclosedBlock = blkNo;
			
			//we'll process it later
		}
		else {		//complete block (formtted, opened, written, closed)
			
			//index page has own ecc so "readRet" does not matter
			if (!ftlPrvMountBlockWithIndexPage(state, blkNo, generations, !haveClosedBlock)) {
				
				LOG("mount with index on blk %u failed\n", blkNo);
				goto out;
			}
			
			haveClosedBlock = true;
		}
	}
	
	if (haveUnclosedBlock) {
		
		LOG("Scanning unclosed block %u\n", prevUnclosedBlock);
		if (!ftlPrvMountBlockWithFullScan(state, prevUnclosedBlock, generations)) {
			
			LOG("mount with scan faile don blk %u\n", blkNo);
			goto out;
		}
	}
	
	//any blocks that are fully erased by definition lack wear counts, but they should have been among the least worn of all the formatted blocks
	//so find the min wear of all formatted blocks and assign that to the erased block
	if (haveErasedBlock) {
		
		uint32_t minWear = 0xffffffff;
		
		for (blkNo = 0; blkNo < state->spec->nandBlocks; blkNo++) {
			
			if (state->blocks[blkNo].isBad)
				continue;
			
			if (blkNo == prevErasedBlock)
				continue;
			
			if (minWear > state->blocks[blkNo].wearCount)
				minWear = state->blocks[blkNo].wearCount;
		}
		
		LOG("For erased block %u, scanned others, decided on wear %u\n", prevErasedBlock, minWear);
		
		//we surely must have at least one formatted block!
		if (minWear == 0xffffffff)
			goto out;
		
		state->blocks[prevErasedBlock].wearCount = minWear;
		if (!ftlPrvFormatErasedBlock(state, prevErasedBlock)) {
			
			LOG("failed to format erzd block %u\n", haveErasedBlock)
			goto out;
		}
		numFormattedBlocks++;
	}
	state->curFormattedBlocks = numFormattedBlocks;
	state->curBadBlocks = numBadBlocks;
	
	//count up cleanliness stats
	for (i = 0; i < state->numLogicalSectors; i++) {
		
		uint32_t physicalAddr = ftlPrvGetMapping(state, i);
		uint32_t blkNo = physicalAddr >> state->spec->pagesPerBlockShift;
		
		if (physicalAddr != state->numPhysicalPages) {
			
			state->blocks[blkNo].cleanPages++;
			
			if (state->blocks[blkNo].cleanPages > PAGES_PER_BLOCK - 1) {
				
				LOG("Too many pages point to block %u\n", blkNo);
				return false;
			}
		}
	}
	
	
	LOG("mount finished\n");
	
	state->generation++;
	ret = true;
	
out:
	MemChunkFree(generations);
	return ret;
}

MemHandle ftlInit(const struct NandSpec *spec, struct NandFuncs *funcs)
{
	return ftlPrvAllocState(spec, funcs);
}

static bool ftlPrvMarkPageAsTrimmedAway(struct FtlState *state, uint32_t physicalAddr)
{
	uint32_t block = physicalAddr >> state->spec->pagesPerBlockShift;
	uint_fast16_t page = physicalAddr & (PAGES_PER_BLOCK - 1);
	struct UneccedDataNormalPage ud;
	
	if (page == (uint_fast16_t)(PAGES_PER_BLOCK - 1))	//index pages shouldnt be trimmed
		return false;
	
	//prepare the data
	MemSet(state->pageBuf + state->spec->dataBytesPerPage, state->spec->spareAreaSize, 0xff);
	ftlPrvWriteBoolF(&ud.pageWritten, true);	//it WAS written
	ftlPrvWriteBoolF(&ud.trimmedAway, true);	//it IS trimmed away
	(void)ftlPrvDistributeSparseData(state->pageBuf + state->spec->dataBytesPerPage, state->spec->uneccedSpareBytes, &ud, sizeof(ud));

	//account
	if (state->blocks[block].isFormatted) {
		LOG("trimming a page on a formatted block\n");
		return false;
	}
	state->blocks[block].cleanPages--;
	if (state->blocks[block].cleanPages == 65535) {
		LOG("TRIM: clean underflow\n");
		return false;
	}

	//this is a spare-only write, so we do not count it in lifetimeStats.lifetimeFlashPageW++
	return ftlPrvPageWritePartial(state, physicalAddr, state->pageBuf + state->spec->dataBytesPerPage, state->spec->dataBytesPerPage, state->spec->spareAreaSize, false);
}

static bool ftlPrvCanWrite(struct FtlState *state)
{
	return state->curBadBlocks < state->spec->maxBadBlocks;
}

static bool ftlPrvWrite(struct FtlState *state, uint32_t secNum, const void *buf)
{
	uint32_t physicalAddr;
	
	if (secNum >= state->numLogicalSectors)
		return false;
	
	if (!ftlPrvCanWrite(state)) {
		
		LOG("going read only due to bad blocks\n");
		return false;
	}
	
	state->lifetimeStats.lifetimeLogicalSecW++;
	
	if (!ftlPrvOpenBlockIfNeeded(state)) {
		LOG( "write: failed to open/close/etc block\n");
		return false;
	}
	
	physicalAddr = ftlPrvGetMapping(state, secNum);
	
	MemMove(state->pageBuf, buf, state->spec->dataBytesPerPage);
	if (!ftlPrvWritePageWithData(state, secNum)) {
		LOG(" write %u failed\n", secNum);
		return false;
	}
	
	if (physicalAddr != state->numPhysicalPages)
		(void)ftlPrvMarkPageAsTrimmedAway(state, physicalAddr);
	
	return true;
}

static bool ftlPrvLowLevelFormat(struct FtlState *state)	//we lose all state incl wear counts. this is ok since they should be near each other anyways
{
	uint32_t blkNo;
	
	state->curFormattedBlocks = 0;
	for (blkNo = 0; blkNo < state->spec->nandBlocks; blkNo++) {
		
		//bad?
		if (ftlPrvBlkIsBad(state, blkNo))
			continue;
		
		if (!ftlPrvFormatBlockRaw(state, blkNo))
			return false;
	}
	
	//we need to write at least one closed block
	return ftlPrvOpenBlockIfNeeded(state) && ftlPrvSync(state, true);
}

static bool ftlPrvTrim(struct FtlState *state, uint32_t secNum)
{
	uint32_t physicalAddr;
	
	if (secNum >= state->numLogicalSectors)
		return false;
	
	if (!ftlPrvCanWrite(state)) {
		
		LOG("going read only due to bad blocks\n");
		return false;
	}
	
	state->lifetimeStats.lifetimeLogicalSecT++;
	
	physicalAddr = ftlPrvGetMapping(state, secNum);
	
	if (physicalAddr == state->numPhysicalPages) {
		
		//already trimmed
		return true;
	}
	ftlPrvSetMapping(state, secNum, state->numPhysicalPages);
	return ftlPrvMarkPageAsTrimmedAway(state, physicalAddr);
}

static bool ftlPrvReadOnly(struct FtlState *state, uint32_t secNum, bool *isTrimmedP, bool *needRewriteP)		//read to internal buffer, do not rewrite despite need
{
	struct UneccedDataNormalPage ued;
	struct EccedDataNormalPage ed;
	uint32_t physicalAddr;
	int_fast8_t readRet;
	
	
	if (secNum >= state->numLogicalSectors)
		return false;
	
	physicalAddr = ftlPrvGetMapping(state, secNum);
	
	if (physicalAddr == state->numPhysicalPages) {
		
		*isTrimmedP = true;
		return true;
	}
	*isTrimmedP = false;
	
	readRet = ftlPrvPageRead(state, physicalAddr, true);
	(void)ftlPrvCollectSparseData(&ued, sizeof(ued), state->pageBuf + state->spec->dataBytesPerPage, state->spec->uneccedSpareBytes);
		
	if (!ftlPrvReadBoolF(&ued.pageWritten))		//map points to a page that has not been written?!
		return false;
	
	if (ftlPrvReadBoolF(&ued.trimmedAway)) {
		
		//we did not know. now we do, account for it
		ftlPrvSetMapping(state, secNum, state->numPhysicalPages);
		
		if (state->blocks[physicalAddr >> state->spec->pagesPerBlockShift].isFormatted) {
			
			LOG("READ: saw page pointing at formatted block\n");
			return false;
		}
		
		state->blocks[physicalAddr >> state->spec->pagesPerBlockShift].cleanPages--;
		if (state->blocks[physicalAddr >> state->spec->pagesPerBlockShift].cleanPages == 65535) {
			
			LOG("READ: clean underflow\n");
			return false;
		}
		
		*isTrimmedP = true;
		return true;
	}
	
	if (readRet < 0)	//data existed but we cnanot read it
		return false;
	
	(void)ftlPrvCollectSparseData(&ed, sizeof(ed), state->pageBuf + state->spec->dataBytesPerPage, state->spec->eccedSpareBytes);
	
	if (ed.sectorNo != secNum)	//BUG
		return false;
	
	*needRewriteP = readRet > ECC_CAP_BEFORE_REWRITE;
	
	return true;
}

static bool ftlPrvIsSecTrimmed(struct FtlState *state, uint32_t secNum)
{
	bool isTrimmed, needRewrite;	//we d onot rewrite in here as this func is not expoected to be used (except by tests)
	
	return ftlPrvReadOnly(state, secNum, &isTrimmed, &needRewrite) && isTrimmed;
}

static bool ftlPrvRead(struct FtlState *state, uint32_t secNum, void *buf)
{
	bool isTrimmed, needRewrite;
	
	if (secNum >= state->numLogicalSectors)
		return false;
	
	state->lifetimeStats.lifetimeLogicalSecR++;
	
	if (!ftlPrvReadOnly(state, secNum, &isTrimmed, &needRewrite))
		return false;
	
	if (isTrimmed) {
		
		MemSet(buf, state->spec->dataBytesPerPage, 0);
		return true;
	}

	MemMove(buf, state->pageBuf, state->spec->dataBytesPerPage);
	
	if (!needRewrite)
		return true;
	
	state->lifetimeStats.lifetimeLogicalSecReloc++;
	LOG("rewriting page %u due to ecc\n", secNum);
	return ftlPrvWrite(state, secNum, buf);
}

static uint64_t ftlPrvGetInfo(struct FtlState *state, enum FtlDataType which)
{
	uint64_t ret = 0, tmp;
	uint32_t blkNo, secNo;
	
	switch (which) {
		case FtlVersion:
			return OUR_VERSION;
		
		case FtlNandGeometry:
			return state->spec->dataBytesPerPage + (((uint32_t)PAGES_PER_BLOCK) << 16) + (((uint64_t)state->spec->nandBlocks) << 32);
		
		case FtlGeometry:
			return state->spec->dataBytesPerPage + (((uint64_t)state->numLogicalSectors) << 32);
		
		case FtlLifetimeSectorsRead:
			return state->lifetimeStats.lifetimeLogicalSecR;
		
		case FtlLifetimeSectorsWritten:
			return state->lifetimeStats.lifetimeLogicalSecW;
		
		case FtlLifetimeSectorsTrimmed:
			return state->lifetimeStats.lifetimeLogicalSecT;
		
		case FtlLifetimePagesWritten:
			return state->lifetimeStats.lifetimeFlashPageW;
		
		case FtlLifetimeBlocksErased:
			return state->lifetimeStats.lifetimeFlashBlockE;
		
		case FtlNumTrimmedSectors:
			for (secNo = 0; secNo < state->numLogicalSectors; secNo++) {
				if (ftlPrvGetMapping(state, secNo) == state->numPhysicalPages)
					ret++;
			}
			return ret;
		
		case FtlAverageBlockCleanliness:
		case FtlAverageBlockWear:
		case FtlNumBadBlocks:
		case FtlGetLifetimeLeft:
			break;
		
		case FtlLifetimeSectorsRelocated:
			return state->lifetimeStats.lifetimeLogicalSecReloc;
		
		default:
			return 0;
	}
	
	for (blkNo = 0; blkNo < state->spec->nandBlocks; blkNo++) {
		
		switch (which) {
			
			case FtlAverageBlockCleanliness:
				ret += state->blocks[blkNo].cleanPages;
				break;
				
			case FtlAverageBlockWear:
			case FtlGetLifetimeLeft:
				ret += state->blocks[blkNo].wearCount;
				break;
			
			case FtlNumBadBlocks:
				ret += state->blocks[blkNo].isBad;
				break;
			
			default:
				break;
		}
	}
	
	switch (which) {
			
		case FtlAverageBlockCleanliness:
			ret *= 1000000;
			ret = (ret + state->spec->nandBlocks * PAGES_PER_BLOCK / 2) / (state->spec->nandBlocks * PAGES_PER_BLOCK);
			break;
			
		case FtlAverageBlockWear:
			LOG("liftime erases: %llu, wear sum: %llu\n",  state->lifetimeStats.lifetimeFlashBlockE, ret);
			ret = (ret + state->spec->nandBlocks / 2) / state->spec->nandBlocks;
			break;
			
		case FtlGetLifetimeLeft:
		
			//calc lifetime in terms of wear count
			ret = (ret + state->spec->nandBlocks / 2) / state->spec->nandBlocks;
			if (ret >= state->spec->speccedErasesPerBlock)
				ret = 0;
			else 
				ret = ((state->spec->speccedErasesPerBlock - ret) * 1000000 + state->spec->speccedErasesPerBlock / 2) / state->spec->speccedErasesPerBlock;
			
			//see if bad blocks make it worse
			if (state->curBadBlocks > state->spec->maxBadBlocks)
				tmp = 0;
			else if (state->curBadBlocks <= state->spec->maxFactoryBadBlocks)
				tmp = 1000000;
			else
				tmp = (state->spec->maxBadBlocks - state->curBadBlocks + state->spec->maxFactoryBadBlocks) * 1000000ULL / (state->spec->maxBadBlocks - state->spec->maxFactoryBadBlocks);
			
			if (tmp < ret)
				ret = tmp;
			
			break;
		
		case FtlNumBadBlocks:
			break;
		
		default:
			break;
	}
	
	return ret;
}

bool ftlRead(MemHandle ftlH, uint32_t secNum, void* buf)
{
	struct FtlState *state = MemHandleLock(ftlH);
	bool ret;
	
	KALMutexReserve(state->mutex, -1);
	
	ret = ftlPrvRead(state, secNum, buf);
	
	KALMutexRelease(state->mutex);
	MemHandleUnlock(ftlH);
	
	return ret;
}

bool ftlIsSecTrimmed(MemHandle ftlH, uint32_t secNum)
{
	struct FtlState *state = MemHandleLock(ftlH);
	bool ret;
	
	KALMutexReserve(state->mutex, -1);
	
	ret = ftlPrvIsSecTrimmed(state, secNum);
	
	KALMutexRelease(state->mutex);
	MemHandleUnlock(ftlH);
	
	return ret;
}

bool ftlWrite(MemHandle ftlH, uint32_t secNum, const void* buf)
{
	struct FtlState *state = MemHandleLock(ftlH);
	bool ret;
	
	KALMutexReserve(state->mutex, -1);
	
	//our trim is reliable, so we cna do this
	ret = buf ? ftlPrvWrite(state, secNum, buf) : ftlPrvTrim(state, secNum);
	
	KALMutexRelease(state->mutex);
	MemHandleUnlock(ftlH);
	
	return ret;
}

bool ftlTrim(MemHandle ftlH, uint32_t secNum)
{
	struct FtlState *state = MemHandleLock(ftlH);
	bool ret;
	
	KALMutexReserve(state->mutex, -1);
	
	ret = ftlPrvTrim(state, secNum);
	
	KALMutexRelease(state->mutex);
	MemHandleUnlock(ftlH);
	
	return ret;
}


uint32_t ftlGetNumSectors(MemHandle ftlH)
{
	struct FtlState *state = MemHandleLock(ftlH);
	uint32_t ret;
	
	KALMutexReserve(state->mutex, -1);
	
	ret = state->numLogicalSectors;
	
	KALMutexRelease(state->mutex);
	MemHandleUnlock(ftlH);
	
	return ret;
}

uint32_t ftlGetSectorSize(MemHandle ftlH)
{
	struct FtlState *state = MemHandleLock(ftlH);
	uint32_t ret;
	
	KALMutexReserve(state->mutex, -1);
	
	ret = state->spec->dataBytesPerPage;
	
	KALMutexRelease(state->mutex);
	MemHandleUnlock(ftlH);
	
	return ret;
}

uint64_t ftlGetInfo(MemHandle ftlH, enum FtlDataType which)
{
	struct FtlState *state = MemHandleLock(ftlH);
	uint64_t ret;
	
	KALMutexReserve(state->mutex, -1);
	
	ret = ftlPrvGetInfo(state, which);
	
	KALMutexRelease(state->mutex);
	MemHandleUnlock(ftlH);
	
	return ret;
}

bool ftlMount(MemHandle ftlH)
{
	struct FtlState *state = MemHandleLock(ftlH);
	bool ret;
	
	KALMutexReserve(state->mutex, -1);
	
	ret = ftlPrvMountWithState(state);
	
	KALMutexRelease(state->mutex);
	MemHandleUnlock(ftlH);
	
	return ret;
}

bool ftlLowLevelFormat(MemHandle ftlH)
{
	struct FtlState *state = MemHandleLock(ftlH);
	bool ret;
	
	KALMutexReserve(state->mutex, -1);
	
	ret = ftlPrvLowLevelFormat(state);
	
	KALMutexRelease(state->mutex);
	MemHandleUnlock(ftlH);
	
	return ret;
}

bool ftlSleepNotify(MemHandle ftlH)
{
	struct FtlState *state = MemHandleLock(ftlH);
	bool ret;
	
	KALMutexReserve(state->mutex, -1);
	
	ret = ftlPrvSync(state, false);
	
	KALMutexRelease(state->mutex);
	MemHandleUnlock(ftlH);
	
	return ret;
}

void ftlUnmount(MemHandle ftlH, bool allowWrite)
{
	//if we cnanot write then we have nothing to do
	if (allowWrite)
		(void)ftlSleepNotify(ftlH);
}

void ftlFree(MemHandle ftlH)
{
	MemHandleFree(ftlH);	//self-contained
}








//4 bit -> 8 bit code allowing one bit correction. tables make it fast
static const uint8_t mHammingDecode[] = {
	0xff, 0x0a, 0x00, 0xff, 0x05, 0xff, 0xff, 0xff, 0xff, 0x08, 0x02, 0xff, 0xff, 0xff, 0xff, 0x0d,
	0xff, 0xff, 0x01, 0xff, 0x04, 0xff, 0xff, 0x0e, 0xff, 0x09, 0xff, 0xff, 0x06, 0xff, 0xff, 0x0c,
	0x0a, 0x0a, 0x02, 0x0a, 0xff, 0x0a, 0xff, 0x0e, 0x02, 0x0a, 0x02, 0x02, 0x06, 0xff, 0x02, 0xff,
	0xff, 0x0a, 0xff, 0x0e, 0x06, 0x0e, 0x0e, 0x0e, 0x06, 0xff, 0x02, 0xff, 0x06, 0x06, 0x06, 0x0e,
	0x00, 0x08, 0x00, 0x00, 0x04, 0xff, 0x00, 0xff, 0x08, 0x08, 0x00, 0x08, 0xff, 0x08, 0xff, 0x0c,
	0x04, 0xff, 0x00, 0xff, 0x04, 0x04, 0x04, 0x0c, 0xff, 0x08, 0xff, 0x0c, 0x04, 0x0c, 0x0c, 0x0c,
	0xff, 0x0a, 0x00, 0xff, 0xff, 0xff, 0xff, 0x0f, 0xff, 0x08, 0x02, 0xff, 0x07, 0xff, 0xff, 0xff,
	0xff, 0x0b, 0xff, 0xff, 0x04, 0xff, 0xff, 0x0e, 0xff, 0xff, 0x03, 0xff, 0x06, 0xff, 0xff, 0x0c,
	0x05, 0xff, 0x01, 0xff, 0x05, 0x05, 0x05, 0x0d, 0xff, 0x09, 0xff, 0x0d, 0x05, 0x0d, 0x0d, 0x0d,
	0x01, 0x09, 0x01, 0x01, 0x05, 0xff, 0x01, 0xff, 0x09, 0x09, 0x01, 0x09, 0xff, 0x09, 0xff, 0x0d,
	0xff, 0x0a, 0xff, 0xff, 0x05, 0xff, 0xff, 0x0f, 0xff, 0xff, 0x02, 0xff, 0x07, 0xff, 0xff, 0x0d,
	0xff, 0x0b, 0x01, 0xff, 0xff, 0xff, 0xff, 0x0e, 0xff, 0x09, 0x03, 0xff, 0x06, 0xff, 0xff, 0xff,
	0xff, 0xff, 0x00, 0xff, 0x05, 0xff, 0xff, 0x0f, 0xff, 0x08, 0xff, 0xff, 0x07, 0xff, 0xff, 0x0d,
	0xff, 0x0b, 0x01, 0xff, 0x04, 0xff, 0xff, 0xff, 0xff, 0x09, 0x03, 0xff, 0xff, 0xff, 0xff, 0x0c,
	0xff, 0x0b, 0xff, 0x0f, 0x07, 0x0f, 0x0f, 0x0f, 0x07, 0xff, 0x03, 0xff, 0x07, 0x07, 0x07, 0x0f,
	0x0b, 0x0b, 0x03, 0x0b, 0xff, 0x0b, 0xff, 0x0f, 0x03, 0x0b, 0x03, 0x03, 0x07, 0xff, 0x03, 0xff, 
};

static const uint8_t mHammingEncode[] = {
	 0x42, 0x92, 0x2a, 0xfa, 0x54, 0x84, 0x3c, 0xec, 0x49, 0x99, 0x21, 0xf1, 0x5f, 0x8f, 0x37, 0xe7, 
};


#define BLOCK_SIZE_DATA		8
#define BLOCK_SIZE_ECCED	16
#define ECC_PIECE_SIZE		1


static void ftlPrvSafelyEncodePiece(uint8_t *dst, uint_fast8_t src, uint_fast8_t bitOfst)
{
	uint_fast8_t i;
	
	for (i = 0; i < 2; i++, src <<= 4) {
		
		uint_fast8_t j, val = mHammingEncode[(src >> 4) & 0xf];
		
		for (j = 0; j < 8; j++, val <<= 1)
			*dst++ |= ((val & 0x80) >> 7) << bitOfst;
	}
}

static int_fast16_t ftlPrvSafelyDecodePiece(const uint8_t *src, uint_fast8_t bitOfst)		//negative on error
{
	uint_fast8_t i, hi, val;
	
	for (i = 0; i < 2; i++) {
		
		uint_fast8_t j;
		
		val = 0;
		for (j = 0; j < 8; j++)
			val = (val << 1) + ((*src++ >> bitOfst) & 1);
		
		val = mHammingDecode[val];
		
		if (val == 0xff)
			return -1;
		
		if (!i)
			hi = val << 4;
	}
	
	return (uint16_t)(uint8_t)(hi + val);
}

static bool ftlPrvSafelyDecodeChunk(void *dstP, uint32_t dstLen, const void *srcP, uint32_t srcLen)
{
	uint32_t blkNo, numBlocks = (dstLen + BLOCK_SIZE_DATA - 1) / BLOCK_SIZE_DATA;
	const uint8_t *src = (const uint8_t*)srcP;
	uint8_t subblk[BLOCK_SIZE_DATA] = {0, };
	uint8_t *dst = (uint8_t*)dstP;
	uint_fast8_t subBlkNo;
	
	//source MUST be proper size
	if (srcLen < numBlocks * BLOCK_SIZE_ECCED)
		return false;
	
	for (blkNo = 0; blkNo < numBlocks; blkNo++, src += BLOCK_SIZE_ECCED, dst += BLOCK_SIZE_DATA) {
		
		bool partial = (dstLen % BLOCK_SIZE_DATA) && (blkNo == numBlocks - 1);
		uint8_t *dstNow = partial ? subblk : dst;
		
		for (subBlkNo = 0; subBlkNo < BLOCK_SIZE_DATA / ECC_PIECE_SIZE; subBlkNo++) {
			
			int_fast16_t decRet = ftlPrvSafelyDecodePiece(src, subBlkNo);
			
			if (decRet < 0)
				return false;
			
			*dstNow++ = decRet;
		}
		if (partial)
			MemMove(dst, subblk, dstLen % BLOCK_SIZE_DATA);
	}
	
	return true;
}

static uint32_t ftlPrvSafelyEncodeChunk(void *dstP, uint32_t dstLen, const void *srcP, uint32_t srcLen)
{
	uint32_t blkNo, numBlocks = (srcLen + BLOCK_SIZE_DATA - 1) / BLOCK_SIZE_DATA;
	const uint8_t *src = ( const uint8_t*)srcP;
	uint8_t bufIn[BLOCK_SIZE_DATA];
	uint8_t *dst = (uint8_t*)dstP;
	uint_fast8_t subBlkNo;
	
	//make sure there is space
	if (dstLen < BLOCK_SIZE_ECCED * numBlocks)
		return 0;
	
	for (blkNo = 0; blkNo < numBlocks; blkNo++, src += BLOCK_SIZE_DATA, dst += BLOCK_SIZE_ECCED) {
		
		MemSet(dst, BLOCK_SIZE_ECCED, 0);
		
		//get data
		if ((srcLen % BLOCK_SIZE_DATA) && (blkNo == numBlocks - 1)) {
			MemSet(bufIn, BLOCK_SIZE_DATA, 0);
			MemMove(bufIn, src, srcLen % BLOCK_SIZE_DATA);
		}
		else {
			
			MemMove(bufIn, src, BLOCK_SIZE_DATA);
		}
		
		for (subBlkNo = 0; subBlkNo < BLOCK_SIZE_DATA / ECC_PIECE_SIZE; subBlkNo++)
			ftlPrvSafelyEncodePiece(dst, bufIn[subBlkNo], subBlkNo);
	}
	
	return BLOCK_SIZE_ECCED * numBlocks;
}

static uint32_t ftlPrvSafeEncodingSize(uint32_t inSize)
{
	return (inSize + BLOCK_SIZE_DATA - 1) / BLOCK_SIZE_DATA * BLOCK_SIZE_ECCED;
}



