#ifndef _ATOMIC_H_
#define _ATOMIC_H_

#include <stdint.h>
#include "irqs.h"

#ifdef CPU_CUSTOM_ATOMICS
#include "cpuAtomic.h"
#else
	
	static inline void atomicAdd8(volatile uint8_t *valP, uint8_t valToAdd)
	{
		
		#if (!defined(BUILD_FOR_THUMB_1) || defined(HAVE_v8M_BASE)) && !defined(BUILDING_FOR_BIG_ARM)
		
			uint32_t t, fail;
			
			do {
				asm volatile(
					".syntax unified		\n\t"
					"	ldrexb %0, [%2]		\n\t"
					"	adds   %0, #1		\n\t"
					"	strexb %1, %0, [%2]	\n\t"
					:"=&r"(t), "=&r"(fail)
					:"r"(valP)
					:"memory"
				);
			} while(fail);
		
		#else			//cortex or legacy
			
			irq_state_t sta = irqsAllOff();
			(*valP) += valToAdd;
			irqsRestoreState(sta);
		
		#endif
	}
	
	static inline void atomicLogicOp32(volatile uint32_t *valP, uint32_t andWith, uint32_t orrIn)
	{
		#if (!defined(BUILD_FOR_THUMB_1) || defined(HAVE_v8M_BASE)) && !defined(BUILDING_FOR_BIG_ARM)
		
			uint32_t newVal, fail;
			
			do {
				asm volatile(
					".syntax unified			\n\t"
					"	ldrex	%0, [%2]		\n\t"
					"	ands	%0, %3			\n\t"
					"	orrs	%0, %4			\n\t"
					"	strex	%1, %0, [%2]	\n\t"
					:"=&r"(newVal), "=r"(fail)
					:"r"(valP), "r"(andWith), "r"(orrIn)
					:"memory", "cc"
				);
				
			} while(fail);
		
		#else 			//cortex or legacy
			
			irq_state_t sta = irqsAllOff();
			*valP = ((*valP) & andWith) | orrIn;
			irqsRestoreState(sta);
	
		#endif
	}
	
	static inline void atomicSwap32(uint32_t *valOutP, uint32_t *valP, uint32_t newVal)	//*valOutP = *valP; *valP = newVal, ATOMICALLY
	{
		#if (!defined(BUILD_FOR_THUMB_1) || defined(HAVE_v8M_BASE)) && !defined(BUILDING_FOR_BIG_ARM)
		
			uint32_t storeFail, val;
			
			do {
				asm(
					"	ldrex %0, [%2]			\n\t"
					"	strex %1, %3, [%2]		\n\t"
					:"=&r"(val), "=&r"(storeFail)
					:"r"(valP), "r"(newVal)
					:"memory"
				);
			} while (storeFail);
			*valOutP = val;
		
		#else 			//cortex or legacy
			
			irq_state_t sta = irqsAllOff();
			*valOutP = *valP;
			*valP = newVal;
			irqsRestoreState(sta);
	
		#endif
	}
	
	static inline void atomicSwap8(uint8_t *valOutP, uint8_t *valP, uint8_t newVal)	//*valOutP = *valP; *valP = newVal, ATOMICALLY
	{
		#if (!defined(BUILD_FOR_THUMB_1) || defined(HAVE_v8M_BASE)) && !defined(BUILDING_FOR_BIG_ARM)
		
			uint32_t storeFail, val;
			
			do {
				asm(
					"	ldrexb %0, [%2]			\n\t"
					"	strexb %1, %3, [%2]		\n\t"
					:"=&r"(val), "=&r"(storeFail)
					:"r"(valP), "r"(newVal)
					:"memory"
				);
			} while (storeFail);
			*valOutP = val;
		
		#else 			//cortex or legacy
			
			irq_state_t sta = irqsAllOff();
			*valOutP = *valP;
			*valP = newVal;
			irqsRestoreState(sta);
	
		#endif
	}
	
	static inline bool atomicTestAndSet32(volatile uint32_t *valP, uint32_t expectedVal, uint32_t setVal)
	{
		#if (!defined(BUILD_FOR_THUMB_1) || defined(HAVE_v8M_BASE)) && !defined(BUILDING_FOR_BIG_ARM)
		
			uint32_t curVal, fail;
			
			asm volatile(
				".syntax unified			\n\t"
				"1:							\n\t"
				"	ldrex	%0, [%2]		\n\t"
				"	cmp		%0, %3			\n\t"
				"	bne		2f				\n\t"
				"	strex	%1, %4, [%2]	\n\t"
				"	cmp		%1, #0			\n\t"
				"	bne		1b				\n\t"
				"2:							\n\t"
				:"=&r"(curVal), "=&r"(fail)
				:"r"(valP), "r"(expectedVal), "r"(setVal)
				:"memory", "cc"
			);
	
			return curVal == expectedVal;
		
		#else			//cortex or legacy
			
			irq_state_t sta = irqsAllOff();
			bool ret = false;
			
			if (*valP == expectedVal) {
				*valP = setVal;
				ret = true;
			}
			
			irqsRestoreState(sta);
			
			return ret;
		#endif
	}
#endif


#endif
