#include <stdint.h>
#include <stdbool.h>
#include <string.h>
#include "printf.h"
#include "disasm.h"

/*
	%_$## - emit $ or any other char present there if bit ## is set
	%!_$## - emit $ or any other char present there if bit ## is clear
	%~$^## - emit $ or any other char present there if bit ## is set, else emit ^ or other char present there
	%aA1 - arm addr mode 1 op
	%aA2 - arm addr mode 2 op (T versions decoded as is not T)
	%aA3 - arm addr mode 3 op
	%r## - 4-bit reg no at bit ##
	%l## - 3-bit reg no at bit ##
	%L## - 4-bit reg no spit across 3 bits, 3 skipped, than one (like some T1 instrs)
	%C## - ARM CC with lowest bit at ## (two spaces for "AL")
	%c## - ARM CC with lowest bit at ## ("AL" for AL)
	%bX - ARM branch target of the BLX type
	%bB - ARM branch target of the B/BL type
	%bT1C - thumb1 conditional branch offset (i8 @ 0)
	%bT1U - thumb1 unconditional branch offset (i11 @ 0)
	%bT1CB - thumb1 CBZ/CBNZ branch target
	%bT2U - thumb2 unconditional branch (eg: BL/BLX/B.W)	I1 = NOT(J1 EOR S); I2 = NOT(J2 EOR S); imm32 = SignExtend(S:I1:I2:imm10:imm11:'0', 32);
	%bT2CU - thumb2 long conditional branch (eg: Bcc.W)		imm32 = SignExtend(S:J2:J1:imm6:imm11:'0', 32);
	%f - arm-style CPSR fields: "_cxsf"
	%m16 - arm-style reg set (for LDM/STM)
	%m8 - thumb-style reg set of 8 for LDM/STM
	%m9L - thumb-style reg set of 9 for LDM/STM with LR
	%m9P - thumb-style reg set of 9 for LDM/STM with PC
	%it - always 3 chars wide, extension to "IT" in Thumb
	%uASWI - 24-bit swi number like ARM
	%uT1SHIFT - 5-bit LSL-style immediate (1..32)
	%uT1i36 - 3-bit imm at position 6 used for thumb1 add/sub with 2 regs
	%uT1i7LSL2 - 7-bit imm at 0 for thumb1 mul 4
	%uT1i8 - 8-bit imm at 0 for thumb1
	%uT1i8LSL2 - 8-bit imm at 0 mul 4 for LDR(lit)
	%uT1i56 - 5-bit imm at 6
	%uT1i56LSL1 - 5-bit imm at 6, mul 2
	%uT1i56LSL2 - 5-bit imm at 6, mul 4
	%uT2udf16 - imm16 from T2's UDF
	%uT2imm12mem - imm12 fomr T2 LDR/STR ops, including comma and "#"
	%uT2imm12memEx - same as %uT2imm12mem, but without the comma and space, always produce a number (not blank)
	%uT2imm8memLSL2 - imm8 fom T2 LDRD/STRD ops
	%uT2imm8mem - imm8 fom T2 LDR/STR ops
	%uT2dpImm12 - imm12 for T2's ADDW/SUBW
	%uT2dpImm16 - imm16 for T2's MOVW/MOVT
	%srW - MSR spec for T2
	%srR - MRS spec for T2
	%T2memLsl - "", "LSL #1", etc up to LSL 3 for t2 mem ops
	%T2dpShift - proper decoding of data processing shift for T2, including comma
	%T2modImm - T2's modified immediate value
	%T2bfx - sbfx/ubfx decoded, both constants
	%T2bfci - bfi/bfc decoded, both constants
	%T2extRot - rotate for {U,S}XT{H,B}
*/

struct Disasm {
	uint32_t mask;
	uint32_t match;
	const char* format;	//NULL means undefined
};

//first match wins so go from most specific to least

//expects first instr as high 16 bits, second as low 16 bits
static const struct Disasm mThumb32instrs[] = {	//always begin with 11101, 11110, 11111
	//v5t
	{	0xf800d000, 0xf000d000, "BL       %bT2U",									},
	{	0xf800d000, 0xf000c000, "BLX      %bT2U",									},
	//v6m
	{	0xfff0f000, 0xf7f0a000, "UDF.W    #%uT2udf16",								},
	{	0xfff0f300, 0xf3808000, "MSR      %srW, %r16",								},
	{	0xfffff000, 0xf3ef8000, "MRS      %srR, %r8",								},
	{	0xfff0d0ff, 0xf3b08040, "SSBB"	,											},
	{	0xfff0d0ff, 0xf3b08044, "PSSBB",											},
	{	0xfff0d0f0, 0xf3b08040, "DSB",												},
	{	0xfff0d0f0, 0xf3b08050, "DMB",												},
	{	0xfff0d0f0, 0xf3b08060, "ISB",												},
	//v7m
	//Load Multiple and Store Multiple
	{	0xffd00000, 0xe8800000, "STMIA.W  %r16%_!21, {%m16}",						},
	{	0xffd00000, 0xe8900000, "LDMIA.W  %r16%_!21, {%m16}",						},
	{	0xffd00000, 0xe9000000, "STMDB.W  %r16%_!21, {%m16}",						},
	{	0xffd00000, 0xe9100000, "LDMDB.W  %r16%_!21, {%m16}",						},
	//Store single data item
	{	0xfff00c00, 0xf8000e00, "STRBT    %r12, [%r16, #%uT2imm8mem]",				},
	{	0xfff00000, 0xf8800000, "STRB     %r12, [%r16%uT2imm12mem]",				},
	{	0xfff00c00, 0xf8000c00, "STRB     %r12, [%r16, %!_-9#%uT2imm8mem]%_!8",		},
	{	0xfff00d00, 0xf8000900, "STRB     %r12, [%r16], %!_-9#%uT2imm8mem",			},
	{	0xfff00fc0, 0xf8000000, "STRB     %r12, [%r16, %r0%T2memLsl]",				},
	{	0xfff00c00, 0xf8200e00, "STRHT    %r12, [%r16, #%uT2imm8mem]",				},
	{	0xfff00000, 0xf8a00000, "STRH     %r12, [%r16%uT2imm12mem]",				},
	{	0xfff00c00, 0xf8200c00, "STRH     %r12, [%r16, %!_-9#%uT2imm8mem]%_!8",		},
	{	0xfff00d00, 0xf8200900, "STRH     %r12, [%r16], %!_-9#%uT2imm8mem",			},
	{	0xfff00fc0, 0xf8200000, "STRH     %r12, [%r16, %r0%T2memLsl]",				},
	{	0xfff00f00, 0xf8400e00, "STRT     %r12, [%r16, #%uT2imm8mem]",				},
	{	0xfff00000, 0xf8c00000, "STR      %r12, [%r16%uT2imm12mem]",				},
	{	0xfff00c00, 0xf8400c00, "STR      %r12, [%r16, %!_-9#%uT2imm8mem]%_!8",		},
	{	0xfff00d00, 0xf8400900, "STR      %r12, [%r16], %!_-9#%uT2imm8mem",			},
	{	0xfff00fc0, 0xf8400000, "STR      %r12, [%r16, %r0%T2memLsl]",				},
	//Load word
	{	0xff7f0000, 0xf8df0000, "LDR      %r12, [PC&~3, %!_-23#%uT2imm12memEx]",	},
	{	0xfff00f00, 0xf8500e00, "LDRT     %r12, [%r16, %uT2imm8mem]",				},
	{	0xfff00000, 0xf8d00000, "LDR      %r12, [%r16%uT2imm12mem]",				},
	{	0xfff00c00, 0xf8500c00, "LDR      %r12, [%r16, %!_-9#%uT2imm8mem]%_!8",		},
	{	0xfff00d00, 0xf8500900, "LDR      %r12, [%r16], %!_-9#%uT2imm8mem",			},
	{	0xfff00fc0, 0xf8500000, "LDR      %r12, [%r16, %r0%T2memLsl]",				},
	//Load halfword, memory hints
	{	0xfe70f000, 0xf830f000, NULL,												},	//unallocated instrs and hints
	{	0xfff00000, 0xf8b00000, "LDRH     %r12, [%r16%uT2imm12mem]",				},
	{	0xfff00f00, 0xf8300e00, "LDRHT    %r12, [%r16, %uT2imm8mem]",				},
	{	0xfff00c00, 0xf8300c00, "LDRH     %r12, [%r16, %!_-9#%uT2imm8mem]%_!8",		},
	{	0xfff00d00, 0xf8300900, "LDRH     %r12, [%r16], %!_-9#%uT2imm8mem",			},
	{	0xfff00fc0, 0xf8300000, "LDRH     %r12, [%r16, %r0%T2memLsl]",				},
	{	0xfff00000, 0xf9b00000, "LDRSH    %r12, [%r16%uT2imm12mem]",				},
	{	0xfff00f00, 0xf9300e00, "LDRSHT   %r12, [%r16, %uT2imm8mem]",				},
	{	0xfff00c00, 0xf9300c00, "LDRSH    %r12, [%r16, %!_-9#%uT2imm8mem]%_!8",		},
	{	0xfff00d00, 0xf9300900, "LDRSH    %r12, [%r16], %!_-9#%uT2imm8mem",			},
	{	0xfff00fc0, 0xf9300000, "LDRSH    %r12, [%r16, %r0%T2memLsl]",				},
	//Load byte, memory hints
	{	0xfe70f000, 0xf810f000, NULL,												},	//preloads, unallocated instrs, and hints - we do not disassemble those
	{	0xfff00000, 0xf8900000, "LDRB     %r12, [%r16%uT2imm12mem]",				},
	{	0xfff00f00, 0xf8100e00, "LDRBT    %r12, [%r16, %uT2imm8mem]",				},
	{	0xfff00c00, 0xf8100c00, "LDRB     %r12, [%r16, %!_-9#%uT2imm8mem]%_!8",		},
	{	0xfff00d00, 0xf8100900, "LDRB     %r12, [%r16], %!_-9#%uT2imm8mem",			},
	{	0xfff00fc0, 0xf8100000, "LDRB     %r12, [%r16, %r0%T2memLsl]",				},
	{	0xfff00000, 0xf9900000, "LDRSB    %r12, [%r16%uT2imm12mem]",				},
	{	0xfff00f00, 0xf9100e00, "LDRSBT   %r12, [%r16, %uT2imm8mem]",				},
	{	0xfff00c00, 0xf9100c00, "LDRSB    %r12, [%r16, %!_-9#%uT2imm8mem]%_!8",		},
	{	0xfff00d00, 0xf9100900, "LDRSB    %r12, [%r16], %!_-9#%uT2imm8mem",			},
	{	0xfff00fc0, 0xf9100000, "LDRSB    %r12, [%r16, %r0%T2memLsl]",				},
	//Data processing (shifted register)
	{	0xfff00f00, 0xea100f00, "TST.W    %r16, %r0%T2dpShift",						},
	{	0xfff00f00, 0xea000f00, NULL,												},
	{	0xffe00000, 0xea000000, "AND%~S 20.W   %r8, %r16, %r0%T2dpShift",			},
	{	0xffe00000, 0xea200000, "BIC%~S 20.W   %r8, %r16, %r0%T2dpShift",			},
	{	0xffef0000, 0xea4f0000, "MOV%~S 20.W   %r8, %r0%T2dpShift",					},
	{	0xffe00000, 0xea400000, "ORR%~S 20.W   %r8, %r16, %r0%T2dpShift",			},
	{	0xffef0000, 0xea6f0000, "MVN%~S 20.W   %r8, %r0%T2dpShift",					},
	{	0xffef0000, 0xfa4f0000, "MOV%~S 20.W   %r8, %r0%T2dpShift",					},
	{	0xffe00000, 0xea600000, "ORN%~S 20.W   %r8, %r16, %r0%T2dpShift",			},
	{	0xfff00f00, 0xea900f00, "TEQ.W    %r16, %r0%T2dpShift",						},
	{	0xfff00f00, 0xea800f00, NULL,												},
	{	0xffe00000, 0xea800000, "EOR%~S 20.W   %r8, %r16, %r0%T2dpShift",			},
	{	0xfff00f00, 0xeb100f00, "CMN.W    %r16, %r0%T2dpShift",						},
	{	0xfff00f00, 0xeb000f00, NULL,												},
	{	0xffe00000, 0xeb000000, "ADD%~S 20.W   %r8, %r16, %r0%T2dpShift",			},
	{	0xffe00000, 0xeb400000, "ADC%~S 20.W   %r8, %r16, %r0%T2dpShift",			},
	{	0xffe00000, 0xeb600000, "SBC%~S 20.W   %r8, %r16, %r0%T2dpShift",			},
	{	0xfff00f00, 0xebb00f00, "CMP.W    %r16, %r0%T2dpShift",						},
	{	0xfff00f00, 0xeba00f00, NULL,												},
	{	0xffe00000, 0xeba00000, "SUB%~S 20.W   %r8, %r16, %r0%T2dpShift",			},
	{	0xffe00000, 0xebc00000, "RSB%~S 20.W   %r8, %r16, %r0%T2dpShift",			},
	//Data processing (register)
	{	0xffe0f0f0, 0xfa00f000, "LSL%~S 20.W   %r8, %r16, %r0",						},
	{	0xffe0f0f0, 0xfa20f000, "LSR%~S 20.W   %r8, %r16, %r0",						},
	{	0xffe0f0f0, 0xfa40f000, "ASR%~S 20.W   %r8, %r16, %r0",						},
	{	0xffe0f0f0, 0xfa60f000, "ROR%~S 20.W   %r8, %r16, %r0",						},
	{	0xfffff080, 0xfa0ff080, "SXTH.W   %r8, %r16, %r0%T2extRot",					},
	{	0xfffff080, 0xfa1ff080, "UXTH.W   %r8, %r16, %r0%T2extRot",					},
	{	0xfffff080, 0xfa4ff080, "SXTB.W   %r8, %r16, %r0%T2extRot",					},
	{	0xfffff080, 0xfa5ff080, "UXTB.W   %r8, %r16, %r0%T2extRot",					},
	//Data processing (modified immediate)
	{	0xfbf08f00, 0xf0100f00, "TST.W    %r16, #%T2modImm",						},
	{	0xfbe08000, 0xf0000000, "AND%~S 20.W   %r8, %r16, #%T2modImm",				},
	{	0xfbe08000, 0xf0200000, "BIC%~S 20.W   %r8, %r16, #%T2modImm",				},
	{	0xfbef8000, 0xf04f0000, "MOV%~S 20.W   %r8, #%T2modImm",					},
	{	0xfbe08000, 0xf0400000, "ORR%~S 20.W   %r8, %r16, #%T2modImm",				},
	{	0xfbef8000, 0xf06f0000, "MVN%~S 20.W   %r8, #%T2modImm",					},
	{	0xfbe08000, 0xf0600000, "ORN%~S 20.W   %r8, %r16, #%T2modImm",				},
	{	0xfbf08f00, 0xf0900f00, "TEQ.W    %r16, #%T2modImm",						},
	{	0xfbe08000, 0xf0800000, "EOR%~S 20.W   %r8, %r16, #%T2modImm",				},
	{	0xfbf08f00, 0xf1100f00, "CMN.W    %r16, #%T2modImm",						},
	{	0xfbe08000, 0xf1000000, "ADD%~S 20.W   %r8, %r16, #%T2modImm",				},
	{	0xfbe08000, 0xf1400000, "ADC%~S 20.W   %r8, %r16, #%T2modImm",				},
	{	0xfbe08000, 0xf1600000, "SBC%~S 20.W   %r8, %r16, #%T2modImm",				},
	{	0xfbf08f00, 0xf1b00f00, "CMP.W    %r16, #%T2modImm",						},
	{	0xfbe08000, 0xf1a00000, "SUB%~S 20.W   %r8, %r16, #%T2modImm",				},
	{	0xfbe08000, 0xf1c00000, "RSB%~S 20.W   %r8, %r16, #%T2modImm",				},
	//Data processing (plain binary immediate)
	{	0xfbff8000, 0xf20f0000, "ADDW     %r8, PC&~3, #%uT2dpImm12",				},
	{	0xfbf08000, 0xf2000000, "ADDW     %r8, %r16, #%uT2dpImm12",					},
	{	0xfbf08000, 0xf2400000, "MOVW     %r8, #%uT2dpImm16",						},
	{	0xfbff8000, 0xf2af0000, "SUBW     %r8, PC&~3, #%uT2dpImm12",				},
	{	0xfbf08000, 0xf2a00000, "SUBW     %r8, %r16, #%uT2dpImm12",					},
	{	0xfbf08000, 0xf2c00000, "MOVT     %r8, #%uT2dpImm16",						},
		//SSAT,SSAT16,USAT,USAT16 should be here, but we do not expect to care
	{	0xfbf08000, 0xf3400000, "SBFX     %r8, %r16, %T2bfx",						},
	{	0xfbf08000, 0xf3c00000, "UBFX     %r8, %r16, %T2bfx",						},
	{	0xfbff8000, 0xf36f0000, "BFC      %r8, %T2bfci",							},
	{	0xfbf08000, 0xf3600000, "BFI      %r8, %r16, %T2bfci",						},
	//Multiply, multiply accumulate, and absolute difference
	{	0xfff0f0f0, 0xfb00f000, "MUL      %r8, %r16, %r0",							},
	{	0xfff000f0, 0xfb000000, "MLA      %r8, %r16, %r0, %r12",					},
	{	0xfff000f0, 0xfb000010, "MLS      %r8, %r16, %r0, %r12",					},
	//Long multiply, long multiply accumulate, and divide
	{	0xfff000f0, 0xfb800000, "SMULL    %r12, %r8, %r16, %r0",					},
	{	0xfff000f0, 0xfb9000f0, "SDIV     %r8, %r16, %r0",							},
	{	0xfff000f0, 0xfba00000, "UMULL    %r12, %r8, %r16, %r0",					},
	{	0xfff000f0, 0xfbb000f0, "UDIV     %r8, %r16, %r0",							},
	{	0xfff000f0, 0xfbc00000, "SMLAL    %r12, %r8, %r16, %r0",					},
	{	0xfff000f0, 0xfbe00000, "UMLAL    %r12, %r8, %r16, %r0",					},
	//Branches and miscellaneous control
	{	0xfff0d0f0, 0xf3b08020, "CLREX",											},
	{	0xf800d000, 0xf0009000, "B.W      %bT2U",									},
	{	0xf800d000, 0xf0008000, "B%C22.W    %bT2CU",								},
	//Load/store dual or exclusive, table branch
	{	0xfff00000, 0xe8400000, "STREX    %r8, %r12, [%r16, #%uT2imm8mem]",			},
	{	0xfff00000, 0xe8500000, "LDREX    %r12, [%r16, #%uT2imm8mem]",				},
	{	0xfff000f0, 0xe8c00040, "STREXB   %r0, %r12, [%r16]",						},
	{	0xfff000f0, 0xe8c00050, "STREXH   %r0, %r12, [%r16]",						},
	{	0xfff000f0, 0xe8d00040, "LDREXB   %r12, [%r16]",							},
	{	0xfff000f0, 0xe8d00050, "LDREXH   %r12, [%r16]",							},
	{	0xfff0fff0, 0xe8d0f000, "TBB      [%r16, %r0]",								},
	{	0xfff0fff0, 0xe8d0f010, "TBH      [%r16, %r0, LSL #1]",						},
	{	0xffe00000, 0xe8c00000, NULL,												},
	{	0xff500000, 0xe9400000, "STRD     %r12, %r8, [%r16, %!_-23#%uT2imm8memLSL2]%_!21",	},
	{	0xff500000, 0xe8400000, "STRD     %r12, %r8, [%r16], %!_-23#%uT2imm8memLSL2",		},
	{	0xff500000, 0xe9500000, "LDRD     %r12, %r8, [%r16, %!_-23#%uT2imm8memLSL2]%_!21",	},
	{	0xff500000, 0xe8500000, "LDRD     %r12, %r8, [%r16], %!_-23#%uT2imm8memLSL2",		},
	
	//trailer
	{	0x00000000, 0x00000000, NULL,												},
};





static const struct Disasm mThumb16instrs[] = {
	{	0xffc0, 0x0000, "MOVS.N   %l0, %l3",							},
	{	0xf800, 0x0000, "LSL.N    %l0, %l3, #%uT1SHIFT",				},
	{	0xf800, 0x0800, "LSR.N    %l0, %l3, #%uT1SHIFT",				},
	{	0xf800, 0x1000, "ASR.N    %l0, %l3, #%uT1SHIFT",				},
	{	0xfe00, 0x1800, "ADD.N    %l0, %l3, %l6",						},
	{	0xfe00, 0x1a00, "SUB.N    %l0, %l3, %l6",						},
	{	0xfe00, 0x1c00, "ADD.N    %l0, %l3, #%uT1i36",					},
	{	0xfe00, 0x1e00, "SUB.N    %l0, %l3, #%uT1i36",					},
	{	0xf800, 0x2000, "MOV.N    %l8, #%uT1i8",						},
	{	0xf800, 0x2800, "CMP.N    %l8, #%uT1i8",						},
	{	0xf800, 0x3000, "ADD.N    %l8, #%uT1i8",						},
	{	0xf800, 0x3800, "SUB.N    %l8, #%uT1i8",						},
	{	0xffc0, 0x4000, "AND.N    %l0, %l3",							},
	{	0xffc0, 0x4040, "EOR.N    %l0, %l3",							},
	{	0xffc0, 0x4080, "LSL.N    %l0, %l3",							},
	{	0xffc0, 0x40c0, "LSR.N    %l0, %l3",							},
	{	0xffc0, 0x4100, "ASR.N    %l0, %l3",							},
	{	0xffc0, 0x4140, "ADC.N    %l0, %l3",							},
	{	0xffc0, 0x4180, "SBC.N    %l0, %l3",							},
	{	0xffc0, 0x41c0, "ROR.N    %l0, %l3",							},
	{	0xffc0, 0x4200, "TST.N    %l0, %l3",							},
	{	0xffc0, 0x4240, "NEG.N    %l0, %l3",							},
	{	0xffc0, 0x4280, "CMP.N    %l0, %l3",							},
	{	0xffc0, 0x42c0, "CMN.N    %l0, %l3",							},
	{	0xffc0, 0x4300, "ORR.N    %l0, %l3",							},
	{	0xffc0, 0x4340, "MUL.N    %l0, %l3",							},
	{	0xffc0, 0x4380, "BIC.N    %l0, %l3",							},
	{	0xffc0, 0x43c0, "MVN.N    %l0, %l3",							},
	{	0xff00, 0x4400, "ADD.N    %L0, %r3 //no flags set",				},
	{	0xff00, 0x4500, "CMP.N    %L0, %r3",							},
	{	0xff00, 0x4600, "MOVhi.N  %L0, %r3",							},
	{	0xff83, 0x4700, "BX       %r3",									},
	{	0xff83, 0x4780, "BLX      %r3",									},
	{	0xf800, 0x4800, "LDR      %l8, [PC&~3, #%uT1i8LSL2]",			},
	{	0xfe00, 0x5000, "STR      %l0, [%l3, %l6]",						},
	{	0xfe00, 0x5200, "STRH     %l0, [%l3, %l6]",						},
	{	0xfe00, 0x5400, "STRB     %l0, [%l3, %l6]",						},
	{	0xfe00, 0x5600, "LDRSB    %l0, [%l3, %l6]",						},
	{	0xfe00, 0x5800, "LDR      %l0, [%l3, %l6]",						},
	{	0xfe00, 0x5a00, "LDRH     %l0, [%l3, %l6]",						},
	{	0xfe00, 0x5c00, "LDRB     %l0, [%l3, %l6]",						},
	{	0xfe00, 0x5e00, "LDRSH    %l0, [%l3, %l6]",						},
	{	0xf800, 0x6000, "STR      %l0, [%l3, #%uT1i56LSL2]",			},
	{	0xf800, 0x6800, "LDR      %l0, [%l3, #%uT1i56LSL2]",			},
	{	0xf800, 0x7000, "STRB     %l0, [%l3, #%uT1i56]",				},
	{	0xf800, 0x7800, "LDRB     %l0, [%l3, #%uT1i56]",				},
	{	0xf800, 0x8000, "STRH     %l0, [%l3, #%uT1i56LSL1]",			},
	{	0xf800, 0x8800, "LDRH     %l0, [%l3, #%uT1i56LSL1]",			},
	{	0xf800, 0x9000, "STR      %l8, [SP, #%uT1i8LSL2]",				},
	{	0xf800, 0x9800, "LDR      %l8, [SP, #%uT1i8LSL2]",				},
	{	0xf800, 0xa000, "ADD      %l8, PC&~3, #%uT1i8LSL2",				},
	{	0xf800, 0xa800, "ADD      %l8, SP, #%uT1i8LSL2",				},
	//misc instrs (v5t)
	{	0xff80, 0xb000, "ADD      SP, SP, #%uT1i7LSL2",					},
	{	0xff80, 0xb080, "SUB      SP, SP, #%uT1i7LSL2",					},
	{	0xfe00, 0xb400, "PUSH.N   {%m9L}",								},
	{	0xfe00, 0xbc00, "POP.N    {%m9P}",								},
	{	0xff00, 0xbe00, "BKPT     #%uT1i8",								},
	//misc instrs (v6m)
	{	0xffc0, 0xb200, "SXTH     %l0, %l3",							},
	{	0xffc0, 0xb240, "SXTB     %l0, %l3",							},
	{	0xffc0, 0xb280, "UXTH     %l0, %l3",							},
	{	0xffc0, 0xb2c0, "UXTB     %l0, %l3",							},
	{	0xffff, 0xb662, "CPSIE    i",									},
	{	0xffff, 0xb672, "CPSID    i",									},
	{	0xffc0, 0xba00, "REV      %l0, %l3",							},
	{	0xffc0, 0xba40, "REV16    %l0, %l3",							},
	{	0xffc0, 0xbac0, "REVSH    %l0, %l3",							},
	{	0xffff, 0xbf00, "NOP",											},
	{	0xffff, 0xbf10, "YIELD",										},
	{	0xffff, 0xbf20, "WFE",											},
	{	0xffff, 0xbf30, "WFI",											},
	{	0xffff, 0xbf40, "SEV",											},
	//misc instrs (v7m)
	{	0xffff, 0xb661, "CPSIE    f",									},
	{	0xffff, 0xb671, "CPSID    f",									},
	{	0xffff, 0xb663, "CPSIE    if",									},
	{	0xffff, 0xb673, "CPSID    if",									},
	{	0xfd00, 0xb100, "CBZ      %l0, %bT1CB",							},
	{	0xfd00, 0xb900, "CBNZ     %l0, %bT1CB",							},
	{	0xff00, 0xbf00, "IT%it    %c4",									},
	
	{	0xf0ff, 0xc000, NULL, 											}, //catch empty set as invalid
	{	0xf800, 0xc000, "STMIA.N  %l8!, {%m8}",							},
	{	0xf800, 0xc800, "LDMIA.N  %l8!, {%m8}",							},
	{	0xff00, 0xde00, "UDF.N    #%uT1i8",								},
	{	0xff00, 0xdf00, "SVC      #%uT1i8",								},
	{	0xf000, 0xd000, "B%C8      %bT1C",								},
	{	0xf800, 0xe000, "B        %bT1U",								},
	{	0x0000, 0x0000, NULL,											},
};

static const struct Disasm mArmInstrs[] = {
	//CC = 0xF
	{	0xfe000000, 0xfa000000, "BLX      %bX",							},
	{	0xf0000000, 0xf0000000, NULL									},
	//DSP instrs
	{	0x0ff00ff0, 0x01000050, "QADD     %r12, %r0, %r16",				},
	{	0x0ff00ff0, 0x01200050, "QSUB     %r12, %r0, %r16",				},
	{	0x0ff00ff0, 0x01400050, "QDADD    %r12, %r0, %r16",				},
	{	0x0ff00ff0, 0x01600050, "QDSUB    %r12, %r0, %r16",				},
	//TABLE 3.2
	{	0x0fe0f0f0, 0x00000090, "MUL%~S 20%C28   %r16, %r0, %r8",		},
	{	0x0fe000f0, 0x00200090, "MLA%~S 20%C28   %r16, %r0, %r8, %r12",	},
	{	0x0fa000f0, 0x00800090, "%~SU22MULL%~S 20%C28 %r12, %r16, %r0, %r8",	},
	{	0x0fa000f0, 0x00a00090, "%~SU22MLAL%~S 20%C28 %r12, %r16, %r0, %r8",	},
	{	0x0ff00ff0, 0x01000090, "SWP%C28    %r12, %r0, [%r16]",			},
	{	0x0ff00ff0, 0x01400090, "SWPB%C28   %r12, %r0, [%r16]",			},
	{	0x0e1000f0, 0x001000b0, "LDRH%C28   %r12, %aA3",				},
	{	0x0e1000f0, 0x000000b0, "STRH%C28   %r12, %aA3",				},
	{	0x0e1010f0, 0x000000f0, "STRD%C28   %r12, %aA3",				},
	{	0x0e1010f0, 0x000000d0, "LDRD%C28   %r12, %aA3",				},
	{	0x0e1000f0, 0x001000f0, "LDRSH%C28  %r12, %aA3",				},
	{	0x0e1000f0, 0x001000d0, "LDRSB%C28  %r12, %aA3",				},
	{	0x0e000090, 0x00000090, NULL,									},
	//TABLE 3.3 (parts we care about only)
	{	0x0fff0fff, 0x010f0000, "MRS%C28    %r12, CPSR",				},
	{	0x0fff0fff, 0x014f0000, "MRS%C28    %r12, SPSR",				},
	{	0x0fff0fff, 0x012f0000, "MSR%C28    CPSR%f, %r12",				},
	{	0x0fff0fff, 0x016f0000, "MSR%C28    SPSR%f, $r12",				},
	{	0x0ffffff0, 0x012fff10, "BX%C28     %r0",						},
	{	0x0ffffff0, 0x012fff20, "BLX%C28    %r0",						},
	{	0x0fff0ff0,	0x016f0f10, "CLZ%C28    %r12, %r0",					},
	{	0x0f900000, 0x01000000, NULL,									},
	//imm-mode test instrs with S cleared
	{	0x0ff0f000, 0x0320f000, "MSR%C28    CPSR%f, %aA1",				},	//imm mode is same as Adr mode 1, thus this
	{	0x0ff0f000, 0x0360f000, "MSR%C28    SPSR%f, %aA1",				},	//imm mode is same as Adr mode 1, thus this
	{	0x0f900000, 0x03000000, NULL,									},
	//MAIN map: DATA PROCESSING
	{	0x0de00000, 0x00000000, "AND%~S 20%C28   %r12, %r16, %aA1",		},
	{	0x0de00000, 0x00200000, "EOR%~S 20%C28   %r12, %r16, %aA1",		},
	{	0x0de00000, 0x00400000, "SUB%~S 20%C28   %r12, %r16, %aA1",		},
	{	0x0de00000, 0x00600000, "RSB%~S 20%C28   %r12, %r16, %aA1",		},
	{	0x0de00000, 0x00800000, "ADD%~S 20%C28   %r12, %r16, %aA1",		},
	{	0x0de00000, 0x00a00000, "ADC%~S 20%C28   %r12, %r16, %aA1",		},
	{	0x0de00000, 0x00c00000, "SBC%~S 20%C28   %r12, %r16, %aA1",		},
	{	0x0de00000, 0x00e00000, "RSC%~S 20%C28   %r12, %r16, %aA1",		},
	{	0x0de00000, 0x01000000, "TST%C28    %r16, %aA1",				},
	{	0x0de00000, 0x01200000, "TEQ%C28    %r16, %aA1",				},
	{	0x0de00000, 0x01400000, "CMP%C28    %r16, %aA1",				},
	{	0x0de00000, 0x01600000, "CMN%C28    %r16, %aA1",				},
	{	0x0de00000, 0x01800000, "ORR%~S 20%C28   %r12, %r16, %aA1",		},
	{	0x0def0000, 0x01a00000, "MOV%~S 20%C28   %r12, %aA1",			},
	{	0x0de00000, 0x01c00000, "BIC%~S 20%C28   %r12, %r16, %aA1",		},
	{	0x0def0000, 0x01e00000, "MVN%~S 20%C28   %r12, %aA1",			},
	//MAIN map: loads/stores
	{	0x0d700000, 0x04200000, "STRT%C28   %r12, %aA2",				},
	{	0x0d700000, 0x04300000, "LDRT%C28   %r12, %aA2",				},
	{	0x0d700000, 0x04600000, "STRBT%C28  %r12, %aA2",				},
	{	0x0d700000, 0x04700000, "LDRBT%C28  %r12, %aA2",				},
	{	0x0c500000, 0x04000000, "STR%C28    %r12, %aA2",				},
	{	0x0c500000, 0x04100000, "LDR%C28    %r12, %aA2",				},
	{	0x0c500000, 0x04400000, "STRB%C28   %r12, %aA2",				},
	{	0x0c500000, 0x04500000, "LDRB%C28   %r12, %aA2",				},
	//MAIN MAP: LDM/STM
	{	0x0e00ffff, 0x08000000, NULL,									},	//empty set is marked as invalid thus
	{	0x0e100000, 0x08000000, "STM%~ID23%~BA24%C28  %r16%_!21, {%m16}%_^22",		},
	{	0x0e100000, 0x08100000, "LDM%~ID23%~BA24%C28  %r16%_!21, {%m16}%_^22",		},
	//MAIN MAP: B/BL
	{	0x0f000000, 0x0a000000, "B%C28      %bB",						},
	{	0x0f000000, 0x0b000000, "BL%C28     %bB",						},
	//MAIN: SWI
	{	0x0f000000, 0x0f000000, "SWI%C28    %uASWI",					},
	
	
	{	0x00000000, 0x00000000, NULL,									},
};

static int32_t disasmReadInt(const char** srcP)	//return negative on error
{
	bool anyDone = false;
	uint32_t val = 0;
	char ch;
	
	while(1) {
		
		ch = **srcP;
		
		if (ch < '0' || ch > '9') {
			if (!anyDone)
				return -1;
			break;
		}
		
		(*srcP)++;
		anyDone = true;
		val = val * 10 + ch - '0';
	}
	
	return val;
}

static bool disasmFormatMatchAdjust(const char** srcP, const char* match)
{
	uint32_t len = strlen(match);
	
	if (strncmp(*srcP, match, len))
		return false;
	(*srcP) += len;
	return true;
}

static const char* disasmGetRegName(uint32_t reg)
{
	static const char* regs[] = {"R0", "R1", "R2", "R3", "R4", "R5", "R6", "R7", "R8", "R9", "R10", "R11", "R12", "SP", "LR", "PC"};
	
	if (reg < sizeof(regs) / sizeof(*regs))
		return regs[reg];
	
	return "UNKNOWN REG";
}

static char* disasmFormatBranchTarget(char *dst, uint32_t jumptTo, uint32_t instrAddr)
{
	return dst + spr(dst, "0x%08x /* $ %c 0x0%x */", jumptTo, jumptTo >= instrAddr ? '+' : '-', jumptTo >= instrAddr ? jumptTo - instrAddr: instrAddr - jumptTo);
}

static char* disasmFormatRegList(char* dst, uint32_t list)
{
	int32_t i, runStart = -1;
	bool first = true;
	
	for (i = 0; i < 17; i++) {	// "17" is VERY important
		
		bool have = !!(list & (1 << i));
		
		if (have && runStart >= 0)
			continue;
		if (!have && runStart < 0)
			continue;
		if (have) {
			runStart = i;
			continue;
		}
		//we'll output a reg, see if we need a coma
		if (!first) {
			
			*dst++ = ',';
			*dst++ = ' ';
		}
		if (runStart + 1 == i)		//single reg
			dst += spr(dst, "%s", disasmGetRegName(runStart));
		else if (runStart + 2 == i)	//two regs
			dst += spr(dst, "%s, %s", disasmGetRegName(runStart), disasmGetRegName(runStart + 1));
		else						//more than two regs
			dst += spr(dst, "%s-%s", disasmGetRegName(runStart), disasmGetRegName(i - 1));
		
		first = false;
		runStart = -1;
	}
	
	return dst;
}

static bool disasmFormatStr(char* dst, const char* src, uint32_t instr, uint32_t instrAddr, uint32_t pcVal)	//assumes enough space. returns success.
{
	static const char* psrsR[] = {"xPSR", "xPSR", "xPSR", "xPSR", NULL, NULL, NULL, NULL, "SP_main", "SP_process", NULL, NULL, NULL, NULL, NULL, NULL, "PRIMASK", "BASEPRI", "BASEPRI", "FAULTMASK", "CONTROL"};
	static const char* psrsW[] = {"APSR", "APSR|IPSR", "APSR|EPSR", "APSR|EPSR|IPSR", NULL, "IPSR", "EPSR", "EPSR|IPSR", "SP_main", "SP_process", NULL, NULL, NULL, NULL, NULL, NULL, "PRIMASK", "BASEPRI", "BASEPRI_MAX", "FAULTMASK", "CONTROL"};
	static const char* ccs[] = {"EQ", "NE", "CS", "CC", "MI", "PL", "VS", "VC", "HI", "LS", "GE", "LT", "GT", "LE", "AL", "NV", "  "/* for convenience*/};
	static const char* its[] = {"!!!", "TTT", "TT ", "TTE", "T  ", "TET", "TE ", "TEE", "   ", "ETT", "ET ", "ETE", "E  ", "EET", "EE ", "EEE", "!!!"};
	static const char* psrWmasks[] = {"_????", "_g", "_nzcvq", "_nzcvqg"};
	static const char* shifts[] = {"LSL", "LSR", "ASR", "ROR", "RRX"};
	const char *tstr;
	char tbuf[16];
	char ch, ch2;
	int32_t pos;
	uint32_t t;
	
	while (1) {
	
		ch = *src++;
		
		if (ch != '%') {
			*dst++ = ch;
			if (!ch)
				return true;
			continue;
		}
			
		if (disasmFormatMatchAdjust(&src, "_")) {
			
			ch = *src++;
			pos = disasmReadInt(&src);
			
			if (pos < 0)
				return false;
			
			if (instr & (1 << pos))
				*dst++ = ch;
		}
		else if (disasmFormatMatchAdjust(&src, "!_")) {
			
			ch = *src++;
			pos = disasmReadInt(&src);
			
			if (pos < 0)
				return false;
			
			if (!(instr & (1 << pos)))
				*dst++ = ch;
		}
		else if (disasmFormatMatchAdjust(&src, "~")) {
			
			ch = *src++;
			ch2 = *src++;
			pos = disasmReadInt(&src);
			
			if (pos < 0)
				return false;
			
			*dst++ = (instr & (1 << pos)) ? ch : ch2;
		}
		else if (disasmFormatMatchAdjust(&src, "r")) {
			
			pos = disasmReadInt(&src);
			
			if (pos < 0)
				return false;
			
			pos = (instr >> pos) & 0x0F;
			dst += spr(dst, "%s", disasmGetRegName(pos));
		}
		else if (disasmFormatMatchAdjust(&src, "l")) {
			
			pos = disasmReadInt(&src);
			
			if (pos < 0)
				return false;
			
			pos = (instr >> pos) & 0x07;
			dst += spr(dst, "%s", disasmGetRegName(pos));
		}
		else if (disasmFormatMatchAdjust(&src, "L")) {
			
			pos = disasmReadInt(&src);
			
			if (pos < 0)
				return false;
			
			pos = ((instr >> pos) & 0x07) + ((instr >> (pos + 4)) & 0x08);
			dst += spr(dst, "%s", disasmGetRegName(pos));
		}
		else if (disasmFormatMatchAdjust(&src, "C")) {
			
			pos = disasmReadInt(&src);
			
			if (pos < 0)
				return false;
			
			pos = (instr >> pos) & 0x0F;
			if (pos == 14)	//AL
				pos = 16;	//two spaces
			
			dst += spr(dst, "%s", ccs[pos]);
		}
		else if (disasmFormatMatchAdjust(&src, "c")) {
			
			pos = disasmReadInt(&src);
			
			if (pos < 0)
				return false;
			
			pos = (instr >> pos) & 0x0F;
			dst += spr(dst, "%s", ccs[pos]);
		}
		else if (disasmFormatMatchAdjust(&src, "it")) {
			
			pos = instr & 0xF;
			if (instr & 0x10)		//i promise you this works
				pos = 0x10 - pos;
			dst += spr(dst, "%s", its[pos]);
		}
		else if (disasmFormatMatchAdjust(&src, "bX")) {
			
			dst = disasmFormatBranchTarget(dst, pcVal + ((((int32_t)instr) << 8) >> 6) + ((instr & 0x01000000) ? 2 : 0), instrAddr);
		}
		else if (disasmFormatMatchAdjust(&src, "bB")) {
			
			dst = disasmFormatBranchTarget(dst, pcVal + ((((int32_t)instr) << 8) >> 6), instrAddr);
		}
		else if (disasmFormatMatchAdjust(&src, "bT1CB")) {
			
			dst = disasmFormatBranchTarget(dst, pcVal + ((instr & 0x00f8) >> 2) + ((instr & 0x0200) >> 3), instrAddr);
		}
		else if (disasmFormatMatchAdjust(&src, "bT1C")) {
			
			dst = disasmFormatBranchTarget(dst, pcVal + ((int32_t)(int8_t)instr) * 2, instrAddr);
		}
		else if (disasmFormatMatchAdjust(&src, "bT1U")) {
			
			dst = disasmFormatBranchTarget(dst, pcVal + (int32_t)(((int16_t)(instr << 5)) >> 4), instrAddr);
		}
		else if (disasmFormatMatchAdjust(&src, "bT2U")) {
			
			uint32_t imm = 0;
			
			t = instr;
			if (t & 0x04000000)
				imm = 0xff000000;
			else
				t ^= 0x2800;
			
			imm += (t & 0x000007ff) <<  1;
			imm += (t & 0x03ff0000) >>  4;
			imm += (t & 0x00000800) << 11;
			imm += (t & 0x00002000) << 10;
			
			dst = disasmFormatBranchTarget(dst, pcVal + imm, instrAddr);
		}
		else if (disasmFormatMatchAdjust(&src, "bT2CU")) {
			
			uint32_t imm = 0;
			
			t = instr;
			
			imm += (t & 0x000007ff) << 1;
			imm += (t & 0x003f0000) >> 4;
			imm += (t & 0x00002000) << 5;
			imm += (t & 0x00000800) << 8;
			if (t & 0x04000000)
				imm += 0xfff00000;
			
			dst = disasmFormatBranchTarget(dst, pcVal + imm, instrAddr);
		}
		else if (disasmFormatMatchAdjust(&src, "f")) {
			
			if (!(instr & 0x000f0000))
				dst += spr(dst, "_????");
			else {
				*dst++ = '_';
				if (instr & 0x00010000)
					*dst++ = 'c';
				if (instr & 0x00020000)
					*dst++ = 'x';
				if (instr & 0x00040000)
					*dst++ = 's';
				if (instr & 0x00080000)
					*dst++ = 'f';
			}
		}
		else if (disasmFormatMatchAdjust(&src, "srR")) {
			
			t = instr & 0xff;
			tstr = t < (sizeof(psrsR) / sizeof(*psrsR)) ? psrsR[t] : NULL;
			
			if (tstr)
				dst += spr(dst, "%s", tstr);
			else
				dst += spr(dst, "UNKNOWN_PSR_0x%02x", t);
		}
		else if (disasmFormatMatchAdjust(&src, "srW")) {
			
			t = instr & 0xff;
			tstr = t < (sizeof(psrsW) / sizeof(*psrsW)) ? psrsW[t] : NULL;
			
			if (tstr) {
				dst += spr(dst, "%s", tstr);
				
				if (!(t >> 2))
					dst += spr(dst, "%s", psrWmasks[(instr >> 10) & 3]);
			}
			else
				dst += spr(dst, "UNKNOWN_PSR_0x%02x", t);
		}
		else if (disasmFormatMatchAdjust(&src, "uT1SHIFT")) {
			
			pos = (instr >> 6) & 0x1f;
			if (!pos)
				pos = 32;
			
			dst += spr(dst, "%u", pos);
		}
		else if (disasmFormatMatchAdjust(&src, "uT1i36")) {
			
			dst += spr(dst, "%u", (instr >> 6) & 7);
		}
		else if (disasmFormatMatchAdjust(&src, "uT1i7LSL2")) {
			
			dst += spr(dst, "0x%03x", (instr & 0x7F) << 2);
		}
		else if (disasmFormatMatchAdjust(&src, "uT1i8LSL2")) {
			
			dst += spr(dst, "0x%03x", (instr & 0xFF) << 2);
		}
		else if (disasmFormatMatchAdjust(&src, "uT1i8")) {
			
			dst += spr(dst, "0x%02x", instr & 0xFF);
		}
		else if (disasmFormatMatchAdjust(&src, "uT1i56LSL2")) {
			
			dst += spr(dst, "0x%02x", ((instr >> 6) & 0x1F) << 2);
		}
		else if (disasmFormatMatchAdjust(&src, "uT1i56LSL1")) {
			
			dst += spr(dst, "0x%02x", ((instr >> 6) & 0x1F) << 1);
		}
		else if (disasmFormatMatchAdjust(&src, "uT1i56")) {
			
			dst += spr(dst, "0x%02x", (instr >> 6) & 0x1F);
		}
		else if (disasmFormatMatchAdjust(&src, "m8")) {
			
			dst = disasmFormatRegList(dst, instr & 0xff);
		}
		else if (disasmFormatMatchAdjust(&src, "m9L")) {
			
			dst = disasmFormatRegList(dst, (instr & 0xff) + ((instr & 0x0100) ? 0x4000 : 0x0000));
		}
		else if (disasmFormatMatchAdjust(&src, "m9P")) {
			
			dst = disasmFormatRegList(dst, (instr & 0xff) + ((instr & 0x0100) ? 0x8000 : 0x0000));
		}
		else if (disasmFormatMatchAdjust(&src, "m16")) {
			
			dst = disasmFormatRegList(dst, instr & 0xffff);
		}
		else if (disasmFormatMatchAdjust(&src, "aA1")) {
		
			if (instr & 0x02000000) {
				
				pos = (instr >> 7) & 0x1e;
				t = instr & 0xFF;
				if (pos)
					t = (t >> pos) | (t << (32 - pos));
				dst += spr(dst, "#0x%x", t);
			}
			else if (!(instr & 0x10)) {
				
				pos = (instr >> 7) & 0x1f;
				t = (instr >> 5) & 3;
				
				if (!t && !pos)
					dst += spr(dst, "%s", disasmGetRegName(instr & 0x0F));	//"LSL 0" is used to just use the Rm
				else if (t == 3 && !pos)
					dst += spr(dst, "%s, RRX", disasmGetRegName(instr & 0x0F));
				else
					dst += spr(dst, "%s, %s #%u", disasmGetRegName(instr & 0x0F), shifts[t], pos ? pos : 32);
			}
			else if (!(instr & 0x80)) {
			
				dst += spr(dst, "%s, %s %s", disasmGetRegName(instr & 0x0F), shifts[(instr >> 5) & 3], disasmGetRegName((instr >> 8) & 0x0F));
			}
			else {
				dst += spr(dst, "????");
			}
		}
		else if (disasmFormatMatchAdjust(&src, "aA2")) {
		
			//first get addend
			if (!(instr & 0x02000000)) {
				
				t = instr & 0xfff;
				
				if (t)
					spr(tbuf, ", %s#0x%03x", (instr & 0x00800000) ? "" : "-", t);
				else
					tbuf[0] = 0;
			}
			else {
				
				pos = (instr >> 7) & 0x1f;
				t = (instr >> 5) & 3;
				
				if (!t && !pos)
					spr(tbuf, ", %s%s", (instr & 0x00800000) ? "" : "-", disasmGetRegName(instr & 0x0F));
				else if (t == 3 && !pos)
					spr(tbuf, ", %s%s, RRX", (instr & 0x00800000) ? "" : "-", disasmGetRegName(instr & 0x0F));
				else
					spr(tbuf, ", %s%s, %s #%u", (instr & 0x00800000) ? "" : "-", disasmGetRegName(instr & 0x0F), shifts[t], pos ? pos : 32);
			}
			
			//now apply it
			if (instr & 0x01000000)
				dst += spr(dst, "[%s%s]%s", disasmGetRegName((instr >> 16) & 0x0F), tbuf, (instr & 0x00200000) ? "!" : "");
			else
				dst += spr(dst, "[%s]%s", disasmGetRegName((instr >> 16) & 0x0F), tbuf);	//is W == 1, this is the 'T' case
		}
		else if (disasmFormatMatchAdjust(&src, "aA3")) {
		
			if (!(instr & 0x00400000))
				spr(tbuf, ", %s%s", (instr & 0x00800000) ? "" : "-", disasmGetRegName(instr & 0x0F));
			else if (instr & 0xf0f)
				spr(tbuf, ", %s#0x%02x", (instr & 0x00800000) ? "" : "-", (instr & 0x0F) + ((instr & 0xf00) >> 4));
			else
				tbuf[0] = 0;
			
			if (instr & 0x01000000)
				dst += spr(dst, "[%s%s], %s", disasmGetRegName((instr >> 16) & 0x0F), tbuf, (instr & 0x00200000) ? "!" : "");
			else if (instr & 0x00200000)
				dst += spr(dst, "????");
			else
				dst += spr(dst, "[%s]%s", disasmGetRegName((instr >> 16) & 0x0F), tbuf);
		}
		else if (disasmFormatMatchAdjust(&src, "uT2udf16")) {
			
			dst += spr(dst, "0x%04x", ((instr & 0x000f0000) >> 4) + (instr & 0x0fff));
		}
		else if (disasmFormatMatchAdjust(&src, "uT2imm12memEx")) {
			
			dst += spr(dst, "#0x%03x", instr & 0xfff);
		}
		else if (disasmFormatMatchAdjust(&src, "uT2imm12mem")) {
			
			t = instr & 0xfff;
			
			if (t)
				dst += spr(dst, ", #0x%03x", t);
		}
		else if (disasmFormatMatchAdjust(&src, "uT2imm8memLSL2")) {
			
			dst += spr(dst, "0x%03x", (instr & 0xff) << 2);
		}
		else if (disasmFormatMatchAdjust(&src, "uT2imm8mem")) {
			
			dst += spr(dst, "0x%02x", instr & 0xff);
		}
		else if (disasmFormatMatchAdjust(&src, "uT2dpImm12")) {
			
			dst += spr(dst, "0x%03x", (instr & 0xff) + ((instr & 0x7000) >> 4) + ((instr & 0x04000000) >> 15));
		}
		else if (disasmFormatMatchAdjust(&src, "uT2dpImm16")) {
			
			dst += spr(dst, "0x%03x", (instr & 0xff) + ((instr & 0x7000) >> 4) + ((instr & 0x04000000) >> 15) + ((instr & 0x000f0000) >> 4));
		}
		else if (disasmFormatMatchAdjust(&src, "T2memLsl")) {
			
			t = (instr >> 4) & 3;
			
			if (t)
				dst += spr(dst, ", LSL #%u", t);
		}
		else if (disasmFormatMatchAdjust(&src, "T2dpShift")) {
			
			t = (instr >> 4) & 3;
			pos = ((instr >> 6) & 3) + ((instr >> 10) & 0x1c);
			
			if (t || pos) {
				
				if (t == 3 && !pos)
					dst += spr(dst, ", RRX");
				else
					dst += spr(dst, ", %s #%u", shifts[t], pos ? pos : 32);
			}
		}
		else if (disasmFormatMatchAdjust(&src, "T2modImm")) {
			
			t = instr & 0xff;
			pos = ((instr & 0x04000000) >> 22) + ((instr & 0x7000) >> 11) + ((instr & 0x80) >> 7);
			switch (pos) {
				
				case 0:
				case 1:
					break;
				case 2:
				case 3:
					t |= t << 16;
					break;
				case 4:
				case 5:
					t = (t << 8) | (t << 24);
					break;
				case 6:
				case 7:
					t |= t << 8;
					t |= t << 16;
					break;
				default:
					t |= 0x80;
					t = (t >> pos) | (t << (32 - pos));
					break;
			}
			dst += spr(dst, "0x%x", t);
		}
		else if (disasmFormatMatchAdjust(&src, "T2bfx")) {
			
			dst += spr(dst, "#%u, #%u", ((instr & 0x7000) >> 10) + ((instr & 0xc0) >> 6), 1 + (instr & 0x1f));
		}
		else if (disasmFormatMatchAdjust(&src, "T2bfci")) {
			
			t = ((instr & 0x7000) >> 10) + ((instr & 0xc0) >> 6);
			
			dst += spr(dst, "#%u, #%d", t, (instr & 0x1f) + 1 - t);
		}
		else if (disasmFormatMatchAdjust(&src, "T2extRot")) {
			
			t = (instr >> 4) & 3;
			if (t)
				dst += spr(dst, ", ROR #%u", t * 8);
		}
		else
			*dst++ = '%';	//copy the format string over directly if no match
	}
}

static void disasmByTable(char* dst, const struct Disasm* tbl, uint32_t instr, uint32_t instrLen, uint32_t instrAddr, uint32_t instrPcVal)
{
	while ((tbl->mask & instr) != tbl->match)	//thanks to final trailer, match is guaranteed
		tbl++;
	
	if (!tbl->format || !disasmFormatStr(dst, tbl->format, instr, instrAddr, instrPcVal))
		strcpy(dst, "UNDEFINED INSTR");
}

uint32_t disasm(char* dst, const void* instrP, bool thumb, bool specialTreatment)	//returns num bytes consumed
{
	uintptr_t addr = (uintptr_t)instrP;
	uint32_t instr;
	
	dst += spr(dst, "0x%08x: ", addr);
	
	if (thumb) {
		
		if (addr & 1)
			return 0;
		
		instr = *(const uint16_t*)instrP;
		
		if (instr < 0xe800) {
			dst += spr(dst, "%04x      ", instr);
			disasmByTable(dst, mThumb16instrs, instr, 2, addr, addr + 4);
			
			return 2;
		}
		
		instr <<= 16;
		instr += ((const uint16_t*)instrP)[1];
		
		dst += spr(dst, "%04x %04x ", instr >> 16, instr & 0xffff);
		disasmByTable(dst, mThumb32instrs, instr, 4, addr, addr + 4);
		
		if (specialTreatment) {
			if ((instr & 0xfff0f000) == 0xf7f0a000) {
				spr(dst + strlen(dst), "; .hword $0x%04x", ((const uint16_t*)instrP)[2]);
				return 6;
			}
		}
		
		return 4;
	}
	else {
		if (addr & 3)
			return 0;
		
		instr = *(const uint32_t*)instrP;
		dst += spr(dst, "%08x  ", instr);
		
		disasmByTable(dst, mArmInstrs, instr, 4, addr, addr + 8);
		return 4;
	}
}





