#include "printf.h"
#include "emit.h"


#define VERIFY_SPACE(num)		do { if ((uint16_t*)dest->bufEnd - (uint16_t*)dest->buf < (int32_t)(num)) return EmitErrNoSpace; }while(0)
#define VALIDATE_SHIFT()		do { if (!emitPrvValidateShift(shiftType, shiftAmt)) return EmitErrNotEncodeable; }while(0)
#define EMIT_HALFWORD(val)		do { *((uint16_t*)dest->buf) = val; dest->buf = ((char*)dest->buf) + sizeof(uint16_t); } while(0)

#define INSTR_IT(cond)			(0xbf08 + (cond << 4))
#define INSTR_ITT(cond)			(0xbf04 + (cond << 4) + (((cond & 1) ^ 0) << 3))
#define INSTR_ITE(cond)			(0xbf04 + (cond << 4) + (((cond & 1) ^ 1) << 3))
#define INSTR_ITTT(cond)		(0xbf02 + (cond << 4) + (((cond & 1) ^ 0) << 3) + (((cond & 1) ^ 0) << 2))
#define INSTR_ITET(cond)		(0xbf02 + (cond << 4) + (((cond & 1) ^ 1) << 3) + (((cond & 1) ^ 0) << 2))
#define INSTR_ITTE(cond)		(0xbf02 + (cond << 4) + (((cond & 1) ^ 0) << 3) + (((cond & 1) ^ 1) << 2))
#define INSTR_ITEE(cond)		(0xbf02 + (cond << 4) + (((cond & 1) ^ 1) << 3) + (((cond & 1) ^ 1) << 2))
#define INSTR_ITTTT(cond)		(0xbf01 + (cond << 4) + (((cond & 1) ^ 0) << 3) + (((cond & 1) ^ 0) << 2) + (((cond & 1) ^ 0) << 1))
#define INSTR_ITETT(cond)		(0xbf01 + (cond << 4) + (((cond & 1) ^ 1) << 3) + (((cond & 1) ^ 0) << 2) + (((cond & 1) ^ 0) << 1))
#define INSTR_ITTET(cond)		(0xbf01 + (cond << 4) + (((cond & 1) ^ 0) << 3) + (((cond & 1) ^ 1) << 2) + (((cond & 1) ^ 0) << 1))
#define INSTR_ITEET(cond)		(0xbf01 + (cond << 4) + (((cond & 1) ^ 1) << 3) + (((cond & 1) ^ 1) << 2) + (((cond & 1) ^ 0) << 1))
#define INSTR_ITTTE(cond)		(0xbf01 + (cond << 4) + (((cond & 1) ^ 0) << 3) + (((cond & 1) ^ 0) << 2) + (((cond & 1) ^ 1) << 1))
#define INSTR_ITETE(cond)		(0xbf01 + (cond << 4) + (((cond & 1) ^ 1) << 3) + (((cond & 1) ^ 0) << 2) + (((cond & 1) ^ 1) << 1))
#define INSTR_ITTEE(cond)		(0xbf01 + (cond << 4) + (((cond & 1) ^ 0) << 3) + (((cond & 1) ^ 1) << 2) + (((cond & 1) ^ 1) << 1))
#define INSTR_ITEEE(cond)		(0xbf01 + (cond << 4) + (((cond & 1) ^ 1) << 3) + (((cond & 1) ^ 1) << 2) + (((cond & 1) ^ 1) << 1))

#define INSTR_REV				0xba00
#define INSTR_REV16				0xba40
#define INSTR_REVSH				0xba60

#define INSTR32_RBIT			0xf0a0fa90
#define INSTR32_REV				0xf080fa90
#define INSTR32_REV16			0xf090fa90
#define INSTR32_REVSH			0xf0b0fa90
#define INSTR32_AND_IMM			0x0000f000
#define INSTR32_TST_IMM			0x0000f000	//same as AND, uses Rd = EMIT_REG_NO_PC, must have S set
#define INSTR32_BIC_IMM			0x0000f020
#define INSTR32_ORR_IMM			0x0000f040
#define INSTR32_ORN_IMM			0x0000f060
#define INSTR32_MOV_IMM			0x0000f040	//same as ORR, uses Rn = EMIT_REG_NO_PC
#define INSTR32_MVN_IMM			0x0000f060	//same as ORN, uses Rn = EMIT_REG_NO_PC
#define INSTR32_EOR_IMM			0x0000f080
#define INSTR32_TEQ_IMM			0x0000f080	//same as EOR, uses Rd = EMIT_REG_NO_PC, must have S set
#define INSTR32_ADD_IMM			0x0000f100
#define INSTR32_CMN_IMM			0x0000f100	//same as ADD, uses Rd = EMIT_REG_NO_PC, must have S set
#define INSTR32_ADC_IMM			0x0000f140
#define INSTR32_SBC_IMM			0x0000f160
#define INSTR32_SUB_IMM			0x0000f1a0
#define INSTR32_CMP_IMM			0x0000f1a0	//same as SUB, uses Rd = EMIT_REG_NO_PC, must have S set
#define INSTR32_RSB_IMM			0x0000f1c0
#define INSTR32_AND_REG			0x0000ea00
#define INSTR32_TST_REG			0x0000ea00	//same as AND, uses Rd = EMIT_REG_NO_PC, must have S set
#define INSTR32_BIC_REG			0x0000ea20
#define INSTR32_ORR_REG			0x0000ea40
#define INSTR32_MOV_REG			0x0000ea40	//same as ORR, uses Rn = EMIT_REG_NO_PC
#define INSTR32_ORN_REG			0x0000ea60
#define INSTR32_MVN_REG			0x0000ea60	//same as ORN, uses Rn = EMIT_REG_NO_PC
#define INSTR32_EOR_REG			0x0000ea80
#define INSTR32_TEQ_REG			0x0000ea80	//same as EOR, uses Rd = EMIT_REG_NO_PC, must have S set
#define INSTR32_ADD_REG			0x0000eb00
#define INSTR32_CMN_REG			0x0000eb00	//same as ADD, uses Rd = EMIT_REG_NO_PC, must have S set
#define INSTR32_ADC_REG			0x0000eb40
#define INSTR32_SBC_REG			0x0000eb60
#define INSTR32_SUB_REG			0x0000eba0
#define INSTR32_CMP_REG			0x0000eba0	//same as SUB, uses Rd = EMIT_REG_NO_PC, must have S set
#define INSTR32_RSB_REG			0x0000ebc0



static uint32_t emitPrvRor(uint32_t val, uint32_t rorBy)
{
	if (rorBy)
		val = (val >> rorBy) | (val << (32 - rorBy));
	
	return val;
}

static bool emitPrvIsDestWordAligned(const struct EmitBuf *dest)
{
	return !(((uintptr_t)(dest->buf)) & 3);
}

static bool emitPrvIsNullShift(enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	return shiftType == EmitShiftLsl && shiftAmt == 0;
}

static bool emitPrvValidateShift(enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	switch (shiftType) {
		case EmitShiftLsl:
		case EmitShiftLsr:
		case EmitShiftAsr:
		case EmitShiftRor:
			return shiftAmt < 32;
		
		default:
			__builtin_unreachable();
			return false;
	}
}
	
static bool emitPrvFlagsCanUseShortInstr(enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	switch (flagPrefs) {
		case EmitLeaveFlags:
			return isInIt;
		
		case EmitSetFlags:
			return !isInIt;
		
		case EmitFlagsDoNotCare:
			return true;
		
		default:
			__builtin_unreachable();
			return false;
	}
}

static bool emitPrvFlagsMustSetFlags(enum EmitFlagSettingPrefs flagPrefs)
{
	return flagPrefs == EmitSetFlags;
}

static bool emitPrvFlagsCanSetFlags(enum EmitFlagSettingPrefs flagPrefs)
{
	return flagPrefs == EmitSetFlags || flagPrefs == EmitFlagsDoNotCare;
}

static bool emitPrvFlagsMustLeaveFlags(enum EmitFlagSettingPrefs flagPrefs)
{
	return flagPrefs == EmitLeaveFlags;
}

static bool emitPrvFlagsCanLeaveFlags(enum EmitFlagSettingPrefs flagPrefs)
{
	return flagPrefs == EmitLeaveFlags || flagPrefs == EmitFlagsDoNotCare;
}

void emitBufferInit(struct EmitBuf *dest, void* buf, uint32_t len)
{
	dest->buf = dest->bufStart = buf;
	dest->bufEnd = (char*)dest->bufStart + (len &~ (sizeof(uint16_t) - 1));
}

enum EmitStatus emitSaveSpace(struct EmitBuf *dest, struct EmitBuf *spaceOutP, uint32_t numHalfwords)
{
	VERIFY_SPACE(numHalfwords);
	
	emitBufferInit(spaceOutP, dest->buf, numHalfwords * sizeof(uint16_t));
	dest->buf = ((char*)dest->buf) + sizeof(uint16_t) * numHalfwords;
	
	return EmitErrNone; 
}

uintptr_t emitGetPtrToJumpHere(const struct EmitBuf *dest)
{
	return (uintptr_t)dest->buf;
}

void* emitBufferBackup(const struct EmitBuf *dest)
{
	return (void*)dest->buf;
}

void emitBufferRestore(struct EmitBuf *dest, void* position)
{
	dest->buf = (uint16_t*)position;
}

//return negative on error (since no valid ThumbImm encoding has top bit set)
static int32_t emitPrvEncodeThumbImmWithProperCarryOut(uint32_t val, uint32_t rotBy)
{
	uint32_t i;
	
	if (val >> 8)	//only 8-bit values can be rotate-encoded
		return -1;
	
	if (!rotBy)		//non-rotated values produce no carry out
		return val << 16;
	
	if (!val)		//we cannot produce a rotated zero
		return -1;
	
	//we need the top bit set
	i = __builtin_clz(val) - 24;
	val <<= i;
	rotBy += i;
	val &= 0x7f;
	rotBy &= 31;
	
	//val is now "bcdefgh", rotBy is "i:imm3", but we have a limit on our rotation
	if (rotBy < 8)
		return -1;
	
	return ((rotBy & 0x10) << 6) + ((rotBy & 0x0e) << 27) + ((rotBy & 0x01) << 23) + (val << 16);
}

static int32_t emitPrvEncodeThumbImmWithoutProperCarryOut(uint32_t val)
{
	uint32_t i;
	
	//these forms do not set carry out
	
	if (!(val >> 8))
		return val << 16;
	
	if ((val & 0xFFFF) == (val >> 16)) {	//all special forms are of the form where top 16 bits are the same as bottom
		if (!(val & 0xFF00FF00))
			return (0x1000 + (val & 0xFF)) << 16;
		
		if (!(val & 0x00FF00FF))
			return (0x2000 + ((val >> 8) & 0xFF)) << 16;

		if (((val >> 8) & 0xFF) == (val & 0xFF))
			return (0x3000 + (val & 0xFF)) << 16;
	}
	
	i = __builtin_clz(val);
	if ((val << i) << 8)	//more than 8 bits in a row - >cannot do it
		return -1;
	
	val <<= i + 1;
	val >>= 25;
	
	i += 8;
	
	return ((i & 0x10) << 6) + ((i & 0x0e) << 27) + ((i & 0x01) << 23) + (val << 16);
}

enum EmitStatus emitLLrawHalfword(struct EmitBuf *dest, uint32_t halfword)
{
	VERIFY_SPACE(1);
	EMIT_HALFWORD(halfword);
	return EmitErrNone;
}

enum EmitStatus emitLLnop(struct EmitBuf *dest, bool w)
{
	if (w) {
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(0xf3af);
		EMIT_HALFWORD(0x8000);
		return EmitErrNone;
	}
	else {
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(0xbf00);
		return EmitErrNone;
	}
}

static enum EmitStatus emitPrvBxBlx(struct EmitBuf *dest, uint32_t regNo, bool withLink)
{
	union {
		struct {
			uint16_t instr3	: 3;
			uint16_t rmNo	: 4;
			uint16_t link	: 1;
			uint16_t inst8	: 8;
		};
		uint16_t word;
	} instr = {.word = 0x4700, };
	
	instr.rmNo = regNo;
	instr.link = withLink;

	VERIFY_SPACE(1);
	EMIT_HALFWORD(instr.word);
	return EmitErrNone;
}

enum EmitStatus emitLLbx(struct EmitBuf *dest, uint32_t regNo)
{
	return emitPrvBxBlx(dest, regNo, false);
}

enum EmitStatus emitLLblx(struct EmitBuf *dest, uint32_t regNo)
{
	return emitPrvBxBlx(dest, regNo, true);
}

enum EmitStatus emitLLcps(struct EmitBuf *dest, bool i, bool f, bool enable)
{
	union {
		struct {
			uint16_t f		: 1;
			uint16_t i		: 1;
			uint16_t inst2	: 2;
			uint16_t mask	: 1;
			uint16_t inst8	: 11;
		};
		uint16_t word;
	} instr = {.word = 0xb660, };
	
	if (!i && !f)
		return EmitErrNotEncodeable;
	
	instr.f = f;
	instr.i = i;
	instr.mask = !enable;

	VERIFY_SPACE(1);
	EMIT_HALFWORD(instr.word);
	return EmitErrNone;
}

enum EmitStatus emitLLit(struct EmitBuf *dest, enum EmitCc cc)
{
	if (cc == EmitCcNv)
		return EmitErrNotEncodeable;
	
	VERIFY_SPACE(1);
	EMIT_HALFWORD(INSTR_IT(cc));
	return EmitErrNone;
}

enum EmitStatus emitLLitt(struct EmitBuf *dest, enum EmitCc cc)
{
	if (cc == EmitCcNv)
		return EmitErrNotEncodeable;
	
	VERIFY_SPACE(1);
	EMIT_HALFWORD(INSTR_ITT(cc));
	return EmitErrNone;
}

enum EmitStatus emitLLittt(struct EmitBuf *dest, enum EmitCc cc)
{
	if (cc == EmitCcNv)
		return EmitErrNotEncodeable;
	
	VERIFY_SPACE(1);
	EMIT_HALFWORD(INSTR_ITTT(cc));
	return EmitErrNone;
}

enum EmitStatus emitLLitttt(struct EmitBuf *dest, enum EmitCc cc)
{
	if (cc == EmitCcNv)
		return EmitErrNotEncodeable;
	
	VERIFY_SPACE(1);
	EMIT_HALFWORD(INSTR_ITTTT(cc));
	return EmitErrNone;
}

enum EmitStatus emitLLite(struct EmitBuf *dest, enum EmitCc cc)
{
	if (cc == EmitCcNv)
		return EmitErrNotEncodeable;
	
	VERIFY_SPACE(1);
	EMIT_HALFWORD(INSTR_ITE(cc));
	return EmitErrNone;
}

enum EmitStatus emitLLudf(struct EmitBuf *dest, uint32_t imm, bool w)
{
	if (w) {
		
		union {
			struct {
				uint16_t imm4		: 4;
				uint16_t instr12	: 12;
			};
			uint16_t word;
		
		} instr1 = {.word = 0xf7f0};
		union {
			struct {
				uint16_t imm12		: 12;
				uint16_t instr4		: 4;
			};
			uint16_t word;
		} instr2 = {.word = 0xa000, };
		
		instr2.imm12 = imm;
		instr1.imm4 = imm >> 12;
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
	else {
		
		union {
			struct {
				uint16_t imm8	: 8;
				uint16_t instr8	: 8;
			};
			uint16_t word;
		} instr = {.word = 0xde00, };
		
		instr.imm8 = imm;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
}

static enum EmitStatus emitPrvBN(struct EmitBuf *dest, int32_t offset)
{
	union {
		struct {
			uint16_t imm11	: 11;
			uint16_t instr5	: 5;
		};
		uint16_t word;
	} instr = {.word = 0xe000, };
	
	if (offset >= 0x400 || offset < -0x400)
		return EmitErrNotEncodeable;
	
	instr.imm11 = offset;
	
	VERIFY_SPACE(1);
	EMIT_HALFWORD(instr.word);
	return EmitErrNone;
}

static enum EmitStatus emitPrvBccN(struct EmitBuf *dest, int32_t offset, enum EmitCc cc)
{
	union {
		struct {
			uint16_t imm8	: 8;
			uint16_t cc		: 4;
			uint16_t instr4	: 4;
		};
		uint16_t word;
	} instr = {.word = 0xd000, };
	
	if (cc == EmitCcAl || cc == EmitCcNv)
		return EmitErrNotEncodeable;
	
	if (offset >= 0x80 || offset < -0x80)
		return EmitErrNotEncodeable;
	
	instr.imm8 = offset;
	instr.cc = cc;
	
	VERIFY_SPACE(1);
	EMIT_HALFWORD(instr.word);
	return EmitErrNone;
}

static enum EmitStatus emitPrvBW(struct EmitBuf *dest, int32_t offset)
{
	union {
		struct {
			uint16_t imm10	: 10;
			uint16_t s		: 1;
			uint16_t instr	: 5;
		};
		uint16_t word;
	} instr1 = {.word = 0xf000};
	
	union {
		struct {
			uint16_t imm11	: 11;
			uint16_t j2		: 1;
			uint16_t instr1	: 1;
			uint16_t j1		: 1;
			uint16_t instr2	: 2;
		};
		uint16_t word;
	} instr2 = {.word = 0x9000, };
	
	if (offset >= 0x800000 || offset < -0x800000)
		return EmitErrNotEncodeable;
	
	if (offset >= 0)
		offset ^= 0x600000;
	
	instr2.imm11 = offset;
	instr1.imm10 = offset >> 11;
	instr2.j2 = offset >> 21;
	instr2.j1 = offset >> 22;
	instr1.s = offset >> 23;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

static int32_t emitPrvCalcBranchOffset(struct EmitBuf *from, uintptr_t toAddr)
{
	uint32_t fromAddr = ((uintptr_t)from->buf) + 4;
	int32_t ofst = toAddr - fromAddr;
	
	return ofst >> 1;
}

enum EmitStatus emitLLbl(struct EmitBuf *dest, uintptr_t dstAddr)
{
	int32_t offset = emitPrvCalcBranchOffset(dest, dstAddr);
	
	union {
		struct {
			uint16_t imm10	: 10;
			uint16_t s		: 1;
			uint16_t instr	: 5;
		};
		uint16_t word;
	} instr1 = {.word = 0xf000};
	
	union {
		struct {
			uint16_t imm11	: 11;
			uint16_t j2		: 1;
			uint16_t instr1	: 1;
			uint16_t j1		: 1;
			uint16_t instr2	: 2;
		};
		uint16_t word;
	} instr2 = {.word = 0xd000, };
	
	if (offset >= 0x800000 || offset < -0x800000)
		return EmitErrNotEncodeable;
	
	if (offset >= 0)
		offset ^= 0x600000;
	
	instr2.imm11 = offset;
	instr1.imm10 = offset >> 11;
	instr2.j2 = offset >> 21;
	instr2.j1 = offset >> 22;
	instr1.s = offset >> 23;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

static enum EmitStatus emitPrvBccW(struct EmitBuf *dest, int32_t offset, enum EmitCc cc)
{
	union {
		struct {
			uint16_t imm6	: 6;
			uint16_t cc		: 4;
			uint16_t s		: 1;
			uint16_t instr	: 5;
		};
		uint16_t word;
	} instr1 = {.word = 0xf000};
	
	union {
		struct {
			uint16_t imm11	: 11;
			uint16_t j2		: 1;
			uint16_t instr1	: 1;
			uint16_t j1		: 1;
			uint16_t instr2	: 2;
		};
		uint16_t word;
	} instr2 = {.word = 0x8000, };
	
	if (cc == EmitCcAl || cc == EmitCcNv)
		return EmitErrNotEncodeable;
	
	if (offset >= 0x80000 || offset < -0x80000)
		return EmitErrNotEncodeable;
	
	instr2.imm11 = offset;
	instr1.imm6 = offset >> 11;
	instr1.cc = cc;
	instr2.j1 = offset >> 17;
	instr2.j2 = offset >> 18;
	instr1.s = offset >> 19;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

enum EmitStatus emitLLabsJump(struct EmitBuf *dest, uintptr_t dstAddr)
{
	if (!emitPrvIsDestWordAligned(dest)) {	//non word aligned instr addr - need 5 codewords with a space in between
		
		//LDR PC, [PC, #4]
		EMIT(LLloadImm, EMIT_REG_NO_PC, EMIT_REG_NO_PC, 4, EmitSzWord, false, EmitAdrModeIndex);

		//NOP.N		//for alignment
		EMIT(LLnop, false);

		//.long dstAddr
		EMIT(LLrawHalfword, dstAddr);
		EMIT(LLrawHalfword, dstAddr >> 16);
	}
	else {						//word aligned instr addr - need 4 codwwords and no NOP
		VERIFY_SPACE(4);
		
		//LDR PC, [PC, #0]
		EMIT(LLloadImm, EMIT_REG_NO_PC, EMIT_REG_NO_PC, 0, EmitSzWord, false, EmitAdrModeIndex);

		//.long dstAddr
		EMIT(LLrawHalfword, dstAddr);
		EMIT(LLrawHalfword, dstAddr >> 16);
	}

	return EmitErrNone;
}

enum EmitStatus emitLLloadSp(struct EmitBuf *dest, uint32_t val)
{
	struct EmitBuf jumpOver;
	
	//LDR SP, [PC, #4]
	EMIT(LLloadImm, EMIT_REG_NO_SP, EMIT_REG_NO_PC, 4, EmitSzWord, false, EmitAdrModeIndex);

	//save space for a branch over
	EMIT(SaveSpace, &jumpOver, 1);

	if (!emitPrvIsDestWordAligned(dest)) {	//non word aligned instr addr - need 5 codewords with a space in between
		
		//NOP.N		//for alignment
		EMIT(LLnop, false);
	}
	
	//.long val
	EMIT(LLrawHalfword, val);
	EMIT(LLrawHalfword, val >> 16);

	//insert that branch
	EMIT_TO(LLbranch, &jumpOver, emitGetPtrToJumpHere(dest), EmitCcAl);

	return EmitErrNone;
}

enum EmitStatus emitLLbranch(struct EmitBuf *dest, uintptr_t dstAddr, enum EmitCc cc)
{
	int32_t offset = emitPrvCalcBranchOffset(dest, dstAddr);
	void *positionBackup;
	enum EmitStatus now;
	
	//sanity check
	if (cc == EmitCcNv)
		return EmitErrNotEncodeable;
	
	//try simple things first
	if (cc == EmitCcAl) {
		
		//try a short one
		now = emitPrvBN(dest, offset);
		if (now != EmitErrNotEncodeable)
			return now;
		
		//try a long one
		now = emitPrvBW(dest, offset);
		if (now != EmitErrNotEncodeable)
			return now;
	}
	else {
		
		//try a short one
		now = emitPrvBccN(dest, offset, cc);
		if (now != EmitErrNotEncodeable)
			return now;
		
		//try a long one
		now = emitPrvBccW(dest, offset, cc);
		if (now != EmitErrNotEncodeable)
			return now;
		
		//try an "IT" and a long branch (backup position before in case we need to roll back due to lack of range)
		positionBackup = emitBufferBackup(dest);
		
		now = emitLLit(dest, cc);
		if (now != EmitErrNone)
			return now;
		
		now = emitPrvBW(dest, offset);
		if (now != EmitErrNotEncodeable)
			return now;
		
		//if it fails, roll back the position we were at
		emitBufferRestore(dest, positionBackup);
	}
	
	return EmitErrNotEncodeable;
}

static enum EmitStatus emitPrvCbzCbnz(struct EmitBuf *dest, uint32_t rnNo, uintptr_t dstAddr, bool isCbnz)
{
	int32_t offset = emitPrvCalcBranchOffset(dest, dstAddr);
	
	union {
		struct {
			uint16_t rnNo		: 3;
			uint16_t imm5		: 5;
			uint16_t instr1a	: 1;
			uint16_t i			: 1;
			uint16_t instr1b	: 1;
			uint16_t isCbnz		: 1;
			uint16_t instr4		: 4;
		};
		uint16_t word;
	} instr = {.word = 0xb100, };
	
	if (offset < 0 || offset >= 128)
		return EmitErrNotEncodeable;
	
	instr.rnNo = rnNo;
	instr.imm5 = offset;
	instr.i = offset >> 5;
	instr.isCbnz = isCbnz;
	
	VERIFY_SPACE(1);
	EMIT_HALFWORD(instr.word);
	return EmitErrNone;
}

enum EmitStatus emitLLcbz(struct EmitBuf *dest, uint32_t rnNo, uintptr_t dstAddr)
{
	return emitPrvCbzCbnz(dest, rnNo, dstAddr, false);
}

enum EmitStatus emitLLcbnz(struct EmitBuf *dest, uint32_t rnNo, uintptr_t dstAddr)
{
	return emitPrvCbzCbnz(dest, rnNo, dstAddr, true);
}

enum EmitStatus emitHLjump(struct EmitBuf *dest, uintptr_t dstAddr)
{
	void *positionBackup;
	enum EmitStatus now;
	
	positionBackup = emitBufferBackup(dest);
	now = emitLLbranch(dest, dstAddr, EmitCcAl);
	if (now != EmitErrNotEncodeable)
		return now;
	emitBufferRestore(dest, positionBackup);

	now = emitLLabsJump(dest, dstAddr | 1);
	if (now != EmitErrNone)
		return now;
	
	return EmitErrNone;
}

static enum EmitStatus emitPrvMovwOrMovt(struct EmitBuf *dest, uint32_t regNo, uint32_t val, bool movt)
{
	union {
		struct {
			uint16_t imm8	: 8;
			uint16_t imm3	: 3;
			uint16_t imm1	: 1;
			uint16_t imm4	: 4;
		};
		uint16_t word;
	} from = {.word = (uint16_t)val,};
	
	union {
		struct {
			uint16_t imm4	: 4;
			uint16_t in3a	: 3;
			uint16_t movt	: 1;
			uint16_t ins2	: 2;
			uint16_t imm1	: 1;
			uint16_t ins5	: 5;
		};
		uint16_t word;
	} instr1 = {.word = 0xf240};
	
	union {
		struct {
			uint16_t imm8	: 8;
			uint16_t regN	: 4;
			uint16_t imm3	: 3;
			uint16_t ins1	: 1;
		};
		uint16_t word;
	} instr2 = {.imm8 = from.imm8, .regN = regNo, .imm3 = from.imm3, };
	
	
	instr1.imm1 = from.imm1;
	instr1.imm4 = from.imm4;
	instr1.movt = movt;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

//assumes a entirely valid request and processed shift values
static enum EmitStatus emitPrvThreeRegDataProcessing32(struct EmitBuf *dest, uint32_t instrVal, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool s, enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	union {
		struct {
			uint16_t rnNo		: 4;
			uint16_t s			: 1;
			uint16_t instr11	: 11;
		};
		uint16_t word;
	} instr1 = {.word = instrVal, };
	
	union {
		struct {
			uint16_t rmNo		: 4;
			uint16_t type		: 2;
			uint16_t imm2		: 2;
			uint16_t rdNo		: 4;
			uint16_t imm3		: 3;
			uint16_t instr1		: 1;
		};
		uint16_t word;
	} instr2 = {.word = instrVal >> 16, };

	if (rmNo == EMIT_REG_NO_SP || rmNo == EMIT_REG_NO_PC)	//forbidden except non-shifted "mov", but that sholuld use short instr so we do not allow it either
		return EmitErrNotEncodeable;
	
	instr1.rnNo = rnNo;
	instr1.s = s;

	instr2.rmNo = rmNo;
	instr2.type = shiftType;
	instr2.imm2 = shiftAmt;
	instr2.rdNo = rdNo;
	instr2.imm3 = shiftAmt >> 2;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

static enum EmitStatus emitPrvInstrWithExpandedImm(struct EmitBuf *dest, uint32_t instrVal, uint32_t rdNo, uint32_t rnNo, bool s, uint32_t encodedImm)
{
	union {
		struct {
			uint16_t rnNo	: 4;
			uint16_t s		: 1;
			uint16_t i5a	: 5;
			uint16_t i		: 1;
			uint16_t i5b	: 5;
		};
		uint16_t word;
	} instr1 = {.word = instrVal + encodedImm, };
	union {
		struct {
			uint16_t imm8	: 8;
			uint16_t rdNo	: 4;
			uint16_t imm3	: 3;
			uint16_t i1		: 1;
		};
		uint16_t word;
	} instr2 = {.word = (instrVal + encodedImm) >> 16, };
	
	instr1.rnNo = rnNo;
	instr1.s = s;
	instr2.rdNo = rdNo;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

enum EmitStatus emitLLextend(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo, uint32_t rotateBy, bool byte, bool unsign)
{
	if (EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo) && !rotateBy) {
		
		union {
			struct {
				uint16_t rdNo	: 3;
				uint16_t rmNo	: 3;
				uint16_t isByte	: 1;
				uint16_t unsign	: 1;
				uint16_t i8		: 8;
			};
			uint16_t word;	
		} instr = {.word = 0xb200, };
		
		instr.rdNo = rdNo;
		instr.rmNo = rmNo;
		instr.isByte = byte;
		instr.unsign = unsign;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	else {
		
		union {
			struct {
				uint16_t i4		: 4;
				uint16_t unsign	: 1;
				uint16_t i1		: 1;
				uint16_t byte	: 1;
				uint16_t i9		: 9;
			};
			uint16_t word;
		} instr1 = {.word = 0xfa0f, };
		union {
			struct {
				uint16_t rmNo	: 4;
				uint16_t rotate	: 2;
				uint16_t i2		: 2;
				uint16_t rdNo	: 4;
				uint16_t i4		: 4;
			};
			uint16_t word;
		} instr2 = {.word = 0xf080, };
		
		switch (rotateBy) {
			case 0:		rotateBy = 0;	break;
			case 8:		rotateBy = 1;	break;
			case 16:	rotateBy = 2;	break;
			case 24: 	rotateBy = 3;	break;
			default:
				return EmitErrNotEncodeable;
		};
		
		instr1.unsign = unsign;
		instr1.byte = byte;
		instr2.rmNo = rmNo;
		instr2.rotate = rotateBy;
		instr2.rdNo = rdNo;
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
}

static enum EmitStatus emitPrvReverse(struct EmitBuf *dest, int32_t instr16, uint32_t instr32, uint32_t rdNo, uint32_t rmNo)
{
	union {
		struct {
			uint16_t rmNo		: 4;
			uint16_t instr12	: 12;
		};
		uint16_t word;
	} instr1 = {.word = instr32, };
	union {
		struct {
			uint16_t rmNo		: 4;
			uint16_t instr4a	: 4;
			uint16_t rdNo		: 4;
			uint16_t instr4b	: 4;
		};
		uint16_t word;
	} instr2 = {.word = instr32 >> 16, };
	
	if (instr16 >= 0 && EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo)) {
		
		union {
			struct {
				uint16_t rdNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = instr16, };
		
		instr.rdNo = rdNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	instr1.rmNo = rmNo;
	instr2.rmNo = rmNo;
	instr2.rdNo = rdNo;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

enum EmitStatus emitLLrbit(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo)
{
	return emitPrvReverse(dest, -1, INSTR32_RBIT, rdNo, rmNo);
}

enum EmitStatus emitLLrev(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo)
{
	return emitPrvReverse(dest, INSTR_REV, INSTR32_REV, rdNo, rmNo);
}

enum EmitStatus emitLLrev16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo)
{
	return emitPrvReverse(dest, INSTR_REV16, INSTR32_REV16, rdNo, rmNo);
}

enum EmitStatus emitLLrevsh(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo)
{
	return emitPrvReverse(dest, INSTR_REVSH, INSTR32_REVSH, rdNo, rmNo);
}

enum EmitStatus emitLLbfi(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t lsb, uint32_t width)
{
	union {
		struct {
			uint16_t rnNo		: 4;
			uint16_t instr12	: 12;
		};
		uint16_t word;
	} instr1 = {.rnNo = rnNo, .instr12 = 0xf36, };
	union {
		struct {
			uint16_t msb		: 5;
			uint16_t val0a		: 1;
			uint16_t imm2		: 2;
			uint16_t rdNo		: 4;
			uint16_t imm3		: 3;
			uint16_t val0b		: 1;
		};
		uint16_t word;
	} instr2 = {.msb = lsb + width - 1, .val0a = 0, .imm2 = lsb, .rdNo = rdNo, .imm3 = lsb >> 2, .val0b = 0, };
	
	if (lsb + width > 32)
		return EmitErrNotEncodeable;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

enum EmitStatus emitLLbfx(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t lsb, uint32_t width, bool unsign, bool mayCorruptFlags)
{
	union {
		struct {
			uint16_t rnNo		: 4;
			uint16_t instr3		: 3;
			uint16_t unsign		: 1;
			uint16_t instr8		: 8;
		};
		uint16_t word;
	} instr1 = {.word = 0xf340, };
	
	union {
		struct {
			uint16_t widthm1	: 5;
			uint16_t instr1a	: 1;
			uint16_t imm2		: 2;
			uint16_t rdNo		: 4;
			uint16_t imm3		: 3;
			uint16_t instr1b	: 1;
		};
		uint16_t word;
	} instr2 = {.widthm1 = width - 1, .imm2 = lsb, .rdNo = rdNo, .imm3 = lsb >> 2, };
	
	if (lsb + width > 32)
		return EmitErrNotEncodeable;
	
	instr1.rnNo = rnNo;
	instr1.unsign = unsign;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

enum EmitStatus emitLLshiftByReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	uint32_t shiftType16;

	//no PC or SP in here
	if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	switch (shiftType) {
		case EmitShiftLsl:	shiftType16 = 0b010;	break;
		case EmitShiftLsr:	shiftType16 = 0b011;	break;
		case EmitShiftAsr:	shiftType16 = 0b100;	break;
		case EmitShiftRor:	shiftType16 = 0b111;	break;
		default: __builtin_unreachable();
	}
	
	if (EMIT_IS_LOREG(rdNo) && rdNo == rnNo && EMIT_IS_LOREG(rmNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
		
		union {
			struct {
				uint16_t rdnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t shiftType	: 3;
				uint16_t val_0x20	: 7;
			};
			uint16_t word;
		} instr = {.rdnNo = rdNo, .rmNo = rmNo, .shiftType = shiftType16, .val_0x20 = 0x20, };
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	else {
		
		union {
			struct {
				uint16_t rnNo		: 4;
				uint16_t s			: 1;
				uint16_t shiftType	: 2;
				uint16_t val_0x1F4	: 9;
			};
			uint16_t word;
		} instr1 = {.rnNo = rnNo, .s = emitPrvFlagsMustSetFlags(flagPrefs), .shiftType = shiftType, .val_0x1F4 = 0x1F4, };
		union {
			struct {
				uint16_t rmNo		: 4;
				uint16_t instr4a	: 4;
				uint16_t rdNo		: 4;
				uint16_t instr4b	: 4;
			};
			uint16_t word;
		} instr2 = {.word = 0xf000, };
		
		instr2.rmNo = rmNo;
		instr2.rdNo = rdNo;
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
}

enum EmitStatus emitLLmov(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	//MOV.N
	if (!emitPrvFlagsMustSetFlags(flagPrefs) && emitPrvIsNullShift(shiftType, shiftAmt)) {
		union {
			struct {
				uint16_t rdNo_lo	: 3;
				uint16_t rmNo		: 4;
				uint16_t rdNo_hi	: 1;
				uint16_t instr8		: 8;
			};
			uint16_t word;
		} instr = {.word = 0x4600, };
		
		instr.rdNo_lo = rdNo;
		instr.rmNo = rmNo;
		instr.rdNo_hi = rdNo >> 3;
		
		if (rdNo != rmNo) {				//do not emit useless code
			VERIFY_SPACE(1);
			EMIT_HALFWORD(instr.word);
		}

		return EmitErrNone;
	}
	
	//LSL.N (reg) LSR.N (reg) ASR.N(reg)
	if (EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt) && shiftType != EmitShiftRor) {
		
		static const uint16_t instrs[] = {[EmitShiftLsl] = 0x0000, [EmitShiftLsr] = 0x0800, [EmitShiftAsr] = 0x1000, };
		
		union {
			struct {
				uint16_t rdNo	: 3;
				uint16_t rmNo	: 3;
				uint16_t imm5	: 5;
				uint16_t instr5	: 5;
			};
			uint16_t word;
		} instr = {.word = instrs[shiftType], };
		
		instr.rdNo = rdNo;
		instr.rmNo = rmNo;
		instr.imm5 = shiftAmt;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//unshifted move to SP is allowed, but then we'd have used a short instr
	if (rdNo == EMIT_REG_NO_SP)
		return EmitErrNotEncodeable;
	
	return emitPrvThreeRegDataProcessing32(dest, INSTR32_MOV_REG, rdNo, EMIT_REG_NO_PC, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLmvnReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	//MVN.N
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
		
		union {
			struct {
				uint16_t rdNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x43c0, };
		
		instr.rdNo = rdNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//Rd, Rm are all not allowed to be SP or PC. trust me this is the fastest way to check
	if (((1 << rdNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	return emitPrvThreeRegDataProcessing32(dest, INSTR32_MVN_REG, rdNo, EMIT_REG_NO_PC, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLcmpReg(struct EmitBuf *dest, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	VALIDATE_SHIFT();
	
	//for short instrs, only PC is forbidden
	if (((1 << rnNo) | (1 << rmNo)) & (1 << EMIT_REG_NO_PC))
		return EmitErrNotEncodeable;
	
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rmNo)) {
		
		union {
			struct {
				uint16_t rnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4280, };
		
		instr.rnNo = rnNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	else if (emitPrvIsNullShift(shiftType, shiftAmt)) {
		union {
			struct {
				uint16_t rnNo_lo	: 3;
				uint16_t rmNo		: 4;
				uint16_t rnNo_hi	: 1;
				uint16_t instr8		: 8;
			};
			uint16_t word;
		} instr = {.word = 0x4500, };
		
		instr.rnNo_lo = rnNo;
		instr.rmNo = rmNo;
		instr.rnNo_hi = rnNo >> 3;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//for long instr, Rn, Rm are all not allowed to be PC. Rm is not allowed to be SP
	if (((1 << rnNo) | (1 << rmNo)) & (1 << EMIT_REG_NO_PC))
		return EmitErrNotEncodeable;
	if (rmNo == EMIT_REG_NO_SP)
		return EmitErrNotEncodeable;
	
	return emitPrvThreeRegDataProcessing32(dest, INSTR32_CMP_REG, EMIT_REG_NO_PC, rnNo, rmNo, true, shiftType, shiftAmt);
}

enum EmitStatus emitLLcmnReg(struct EmitBuf *dest, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	VALIDATE_SHIFT();
	
	if (EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rmNo) && emitPrvIsNullShift(shiftType, shiftAmt)) {
		
		union {
			struct {
				uint16_t rnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x42c0, };
		
		instr.rnNo = rnNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//for long instr, Rn, Rm are all not allowed to be PC. Rm is not allowed to be SP
	if (((1 << rnNo) | (1 << rmNo)) & (1 << EMIT_REG_NO_PC))
		return EmitErrNotEncodeable;
	if (rmNo == EMIT_REG_NO_SP)
		return EmitErrNotEncodeable;
	
	return emitPrvThreeRegDataProcessing32(dest, INSTR32_CMN_REG, EMIT_REG_NO_PC, rnNo, rmNo, true, shiftType, shiftAmt);
}

enum EmitStatus emitLLtstReg(struct EmitBuf *dest, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	VALIDATE_SHIFT();
	
	if (EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rmNo) && emitPrvIsNullShift(shiftType, shiftAmt)) {
		
		union {
			struct {
				uint16_t rnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4200, };
		
		instr.rnNo = rnNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//Rn, Rm are all not allowed to be SP or PC. trust me this is the fastest way to check
	if (((1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	return emitPrvThreeRegDataProcessing32(dest, INSTR32_TST_REG, EMIT_REG_NO_PC, rnNo, rmNo, true, shiftType, shiftAmt);
}

enum EmitStatus emitLLteqReg(struct EmitBuf *dest, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	VALIDATE_SHIFT();
	
	//Rn, Rm are all not allowed to be SP or PC. trust me this is the fastest way to check
	if (((1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	return emitPrvThreeRegDataProcessing32(dest, INSTR32_TEQ_REG, EMIT_REG_NO_PC, rnNo, rmNo, true, shiftType, shiftAmt);
}

enum EmitStatus emitLLaddReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	if (emitPrvIsNullShift(shiftType, shiftAmt)) {
		//short 3-op add?
		if (EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo) && EMIT_IS_LOREG(rnNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
		
			union {
				struct {
					uint16_t rdNo	: 3;
					uint16_t rnNo	: 3;
					uint16_t rmNo	: 3;
					uint16_t instr7	: 7;
				};
				uint16_t word;
			} instr = {.word = 0x1800, };
			
			instr.rdNo = rdNo;
			instr.rnNo = rnNo;
			instr.rmNo = rmNo;
			
			VERIFY_SPACE(1);
			EMIT_HALFWORD(instr.word);
			return EmitErrNone;
		}
		
		//short 2-op add?
		if ((rdNo == rmNo || rdNo == rnNo) && !emitPrvFlagsMustSetFlags(flagPrefs)) {
			
			union {
				struct {
					uint16_t rdnNo_lo	: 3;
					uint16_t rmNo		: 4;
					uint16_t rdnNo_hi	: 1;
					uint16_t instr8		: 8;
				};
				uint16_t word;
			} instr = {.word = 0x4400, };
			
			instr.rdnNo_lo = rdNo;
			instr.rmNo = rmNo + rnNo - rdNo;	//calculates the value of whichever is NOT same as Rd
			instr.rdnNo_hi = rdNo >> 3;
			
			VERIFY_SPACE(1);
			EMIT_HALFWORD(instr.word);
			return EmitErrNone;
		}
		
		//if Rm is SP and En is not, swap them
		if (rmNo == EMIT_REG_NO_SP && rnNo != EMIT_REG_NO_SP) {
			
			rmNo = rnNo;
			rnNo = EMIT_REG_NO_SP;
		}
	}
	
	//adding to SP is only allowed from SP, and only with limited shift abilities
	if (rdNo == EMIT_REG_NO_SP && (rnNo != EMIT_REG_NO_SP || shiftType != EmitShiftLsl || shiftAmt > 3))
		return EmitErrNotEncodeable;
	
	return emitPrvThreeRegDataProcessing32(dest, INSTR32_ADD_REG, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLsubReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	//short 3-op add?
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo) && EMIT_IS_LOREG(rnNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
	
		union {
			struct {
				uint16_t rdNo	: 3;
				uint16_t rnNo	: 3;
				uint16_t rmNo	: 3;
				uint16_t instr7	: 7;
			};
			uint16_t word;
		} instr = {.word = 0x1a00, };
		
		instr.rdNo = rdNo;
		instr.rnNo = rnNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//subtracting to SP is only allowed from SP, and onyl with limited shift abilities
	if (rdNo == EMIT_REG_NO_SP && (rnNo != EMIT_REG_NO_SP || shiftType != EmitShiftLsl || shiftAmt > 3))
		return EmitErrNotEncodeable;
	
	return emitPrvThreeRegDataProcessing32(dest, INSTR32_SUB_REG, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLsbcReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo) && rnNo == rdNo && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
	
		union {
			struct {
				uint16_t rdnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4180, };
		
		instr.rdnNo = rdNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//Rd, Rn, Rm are all not allowed to be SP or PC. trust me this is the fastest way to check
	if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	return emitPrvThreeRegDataProcessing32(dest, INSTR32_SBC_REG, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLorrReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rmNo) && (rdNo == rnNo || rdNo == rmNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
	
		union {
			struct {
				uint16_t rdnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4300, };
		
		instr.rdnNo = rdNo;
		instr.rmNo = rnNo + rmNo - rdNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//Rd, Rm are not allowed to be SP or PC. Rn is not allowed to be SP, trust me this is the fastest way to check
	if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	if (rnNo == EMIT_REG_NO_SP)
		return EmitErrNotEncodeable;
	
	return emitPrvThreeRegDataProcessing32(dest, INSTR32_ORR_REG, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLbicReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo) && rnNo == rdNo && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
	
		union {
			struct {
				uint16_t rdnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4380, };
		
		instr.rdnNo = rdNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//Rd, Rn, Rm are all not allowed to be SP or PC. trust me this is the fastest way to check
	if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	return emitPrvThreeRegDataProcessing32(dest, INSTR32_BIC_REG, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

//sets no flags ever. op = 0 is for mul/mla, 1 is mls
static enum EmitStatus emitPrvMulOrMla(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t raNo, uint32_t op)	//set Ra to PC to get MUL
{
	union {
		struct {
			uint16_t rnNo		: 4;
			uint16_t instr12	: 12;
		};
		uint16_t word;
	} instr1 = {.word = 0xfb00, };
	union {
		struct {
			uint16_t rmNo		: 4;
			uint16_t op			: 4;
			uint16_t rdNo		: 4;
			uint16_t raNo		: 4;
		};
		uint16_t word;
	} instr2;
	
	instr1.rnNo = rnNo;
	instr2.rmNo = rmNo;
	instr2.op = op;
	instr2.rdNo = rdNo;
	instr2.raNo = raNo;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
	
}

enum EmitStatus emitLLmulReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	//no SP no PC
	if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	if (EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rmNo) && (rdNo == rmNo || rdNo == rnNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
	
		union {
			struct {
				uint16_t rdmNo		: 3;
				uint16_t rnNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4340, };
		
		instr.rdmNo = rdNo;
		instr.rnNo = rnNo + rmNo - rdNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	else if (!emitPrvFlagsMustSetFlags(flagPrefs))		//MUL.W sets no flags
		return emitPrvMulOrMla(dest, rdNo, rnNo, rmNo, EMIT_REG_NO_PC, 0);
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLmlaReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t raNo)
{
	//no SP no PC in Rn, Rm, Rd, Ra
	if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo) | (1 << raNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	return emitPrvMulOrMla(dest, rdNo, rnNo, rmNo, raNo, 0);
}

enum EmitStatus emitLLmlsReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t raNo)
{
	//no SP no PC in Rn, Rm, Rd, Ra
	if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo) | (1 << raNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	return emitPrvMulOrMla(dest, rdNo, rnNo, rmNo, raNo, 1);
}

enum EmitStatus emitLLandReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rmNo) && (rdNo == rnNo || rdNo == rmNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
	
		union {
			struct {
				uint16_t rdnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4000, };
		
		instr.rdnNo = rdNo;
		instr.rmNo = rnNo + rmNo - rdNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//Rd, Rn, Rm are all not allowed to be SP or PC. trust me this is the fastest way to check
	if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	return emitPrvThreeRegDataProcessing32(dest, INSTR32_AND_REG, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLeorReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rmNo) && (rdNo == rnNo || rdNo == rmNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
	
		union {
			struct {
				uint16_t rdnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4040, };
		
		instr.rdnNo = rdNo;
		instr.rmNo = rmNo + rnNo - rdNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//Rd, Rn, Rm are all not allowed to be SP or PC. trust me this is the fastest way to check
	if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	return emitPrvThreeRegDataProcessing32(dest, INSTR32_EOR_REG, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLadcReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	if (emitPrvIsNullShift(shiftType, shiftAmt) && EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rmNo) && rnNo == rdNo && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
	
		union {
			struct {
				uint16_t rdnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.word = 0x4140, };
		
		instr.rdnNo = rdNo;
		instr.rmNo = rmNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//Rd, Rn, Rm are all not allowed to be SP or PC. trust me this is the fastest way to check
	if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	return emitPrvThreeRegDataProcessing32(dest, INSTR32_ADC_REG, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

enum EmitStatus emitLLrsbReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	VALIDATE_SHIFT();
	
	//Rd, Rn, Rm are all not allowed to be SP or PC. trust me this is the fastest way to check
	if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	return emitPrvThreeRegDataProcessing32(dest, INSTR32_RSB_REG, rdNo, rnNo, rmNo, emitPrvFlagsMustSetFlags(flagPrefs), shiftType, shiftAmt);
}

static enum EmitStatus emitPrvAddwSubw(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, bool isSub)
{
	union {
		struct {
			uint16_t rnNo	: 4;
			uint16_t instr6	: 6;
			uint16_t i		: 1;
			uint16_t instr5	: 5;
		};
		uint16_t word;
	} instr1 = {.rnNo = rnNo, .instr6 = isSub ? 0b101010 : 0b100000, .i = imm >> 11, .instr5 = 0b11110, };
	union {
		struct {
			uint16_t imm8	: 8;
			uint16_t rdNo	: 4;
			uint16_t imm3	: 3;
			uint16_t val0	: 1;
		};
		uint16_t word;
	} instr2 = {.imm8 = imm, .rdNo = rdNo, .imm3 = imm >> 8, .val0 = 0, };
	
	if (imm >= 0x1000)
		return EmitErrNotEncodeable;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

static enum EmitStatus emitPrvAddSubImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt, bool isSub)
{
	enum EmitStatus now;
	int32_t encodedImm;

	//if imm is zero and flags aren't needed, use  amov (more encoding options)
	if (imm == 0 && flagPrefs != EmitSetFlags) {
		
		return emitLLmov(dest, rdNo, rnNo, EmitShiftLsl, 0, flagPrefs, isInIt);
	}

	//ADD.N/SUB.N (two reg variant)
	if (EMIT_IS_LOREG(rdNo) && EMIT_IS_LOREG(rnNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt) && imm < 0x08) {
		
		union {
			struct {
				uint16_t rdNo	: 3;
				uint16_t rnNo	: 3;
				uint16_t imm3	: 3;
				uint16_t isSub	: 1;
				uint16_t instr6	: 6;
			};
			uint16_t word;
		} instr = {.rdNo = rdNo, .rnNo = rnNo, .imm3 = imm, .isSub = isSub, .instr6 = 0b000111};
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//ADD.N/SUB.N (one reg variant)
	if (rdNo == rnNo && EMIT_IS_LOREG(rnNo) && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt) && imm < 0x100) {
		
		union {
			struct {
				uint16_t imm8	: 8;
				uint16_t rdnNo	: 3;
				uint16_t isSub	: 1;
				uint16_t instr4	: 4;
			};
			uint16_t word;
		} instr = {.imm8 = imm, .rdnNo = rdNo, .isSub = isSub, .instr4 = 0b0011, };
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//ADD.N/SUB.N SP, SP, #imm
	if (rdNo == rnNo && rnNo == EMIT_REG_NO_SP && !emitPrvFlagsMustSetFlags(flagPrefs) && !(imm & 3) && imm < 0x200) {
		
		union {
			struct {
				uint16_t imm7	: 7;
				uint16_t isSub	: 1;
				uint16_t instr8	: 8;
			};
			uint16_t word;
		} instr = {.imm7 = imm >> 2, .isSub = isSub, .instr8 = 0b10110000, };
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//ADD.N Rx, SP, #imm
	if (!isSub && EMIT_IS_LOREG(rdNo) && rnNo == EMIT_REG_NO_SP && !emitPrvFlagsMustSetFlags(flagPrefs) && !(imm & 3) && imm < 0x400) {
		
		union {
			struct {
				uint16_t imm8	: 8;
				uint16_t rdNo	: 3;
				uint16_t instr5	: 5;
			};
			uint16_t word;
		} instr = {.imm8 = imm >> 2, .rdNo = rdNo, .instr5 = 0b10101, };
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//ADDW/SUBW & ADD.W/SUB.W share some common restrictions
	if (rdNo == EMIT_REG_NO_PC || (rdNo == EMIT_REG_NO_SP && rnNo != EMIT_REG_NO_SP))
		return EmitErrNotEncodeable;
	
	//ADDW/SUBW
	if (!emitPrvFlagsMustSetFlags(flagPrefs) && imm < 0x1000) {
		now = emitPrvAddwSubw(dest, rdNo, rnNo, imm, isSub);
		if (now != EmitErrNotEncodeable)
			return now;
	}
	
	//ADD.W/SUB.W
	encodedImm = emitPrvEncodeThumbImmWithoutProperCarryOut(imm);	//ADD/SUB will replace C flag so we do not care how imm is produced
	if (encodedImm < 0)
		return EmitErrNotEncodeable;
	
	return emitPrvInstrWithExpandedImm(dest, isSub ? INSTR32_SUB_IMM : INSTR32_ADD_IMM, rdNo, rnNo, emitPrvFlagsMustSetFlags(flagPrefs), encodedImm);
}

enum EmitStatus emitLLaddImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	return emitPrvAddSubImm(dest, rdNo, rnNo, imm, flagPrefs, isInIt, false);
}

enum EmitStatus emitLLsubImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	return emitPrvAddSubImm(dest, rdNo, rnNo, imm, flagPrefs, isInIt, true);
}

//use if we do not care about shifter carry out
static enum EmitStatus emitPrvMathematicalOpImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t val, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs, uint32_t instr32)
{
	int32_t encodedImm = emitPrvEncodeThumbImmWithoutProperCarryOut(emitPrvRor(val, rorBy));
	
	if (encodedImm < 0)
		return EmitErrNotEncodeable;
	
	return emitPrvInstrWithExpandedImm(dest, instr32, rdNo, rnNo, emitPrvFlagsMustSetFlags(flagPrefs), encodedImm);
}

//use if we DO care about shifter carry out
static enum EmitStatus emitPrvLogicalOpImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t val, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs, uint32_t instr32)
{
	int32_t encodedImm;
	
	if (emitPrvFlagsMustSetFlags(flagPrefs))
		encodedImm = emitPrvEncodeThumbImmWithProperCarryOut(val, rorBy);	//RSB will replace C flag so we do not care how imm is produced
	else
		encodedImm = emitPrvEncodeThumbImmWithoutProperCarryOut(emitPrvRor(val, rorBy));
	
	if (encodedImm < 0)
		return EmitErrNotEncodeable;
	
	return emitPrvInstrWithExpandedImm(dest, instr32, rdNo, rnNo, emitPrvFlagsMustSetFlags(flagPrefs), encodedImm);
}

enum EmitStatus emitLLeorImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t val, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs)
{
	return emitPrvLogicalOpImm(dest, rdNo, rnNo, val, rorBy, flagPrefs, INSTR32_EOR_IMM);
}

enum EmitStatus emitLLandImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t val, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs)
{
	return emitPrvLogicalOpImm(dest, rdNo, rnNo, val, rorBy, flagPrefs, INSTR32_AND_IMM);
}

enum EmitStatus emitLLbicImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t val, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs)
{
	return emitPrvLogicalOpImm(dest, rdNo, rnNo, val, rorBy, flagPrefs, INSTR32_BIC_IMM);
}

enum EmitStatus emitLLorrImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t val, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs)
{
	return emitPrvLogicalOpImm(dest, rdNo, rnNo, val, rorBy, flagPrefs, INSTR32_ORR_IMM);
}

enum EmitStatus emitLLtstImm(struct EmitBuf *dest, uint32_t rnNo, uint32_t val, uint32_t rorBy)
{
	return emitPrvLogicalOpImm(dest, EMIT_REG_NO_PC, rnNo, val, rorBy, EmitSetFlags, INSTR32_TST_IMM);
}

enum EmitStatus emitLLteqImm(struct EmitBuf *dest, uint32_t rnNo, uint32_t val, uint32_t rorBy)
{
	return emitPrvLogicalOpImm(dest, EMIT_REG_NO_PC, rnNo, val, rorBy, EmitSetFlags, INSTR32_TEQ_IMM);
}

enum EmitStatus emitLLadcImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	return emitPrvMathematicalOpImm(dest, rdNo, rnNo, imm, 0, flagPrefs, INSTR32_ADC_IMM);
}

enum EmitStatus emitLLsbcImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	return emitPrvMathematicalOpImm(dest, rdNo, rnNo, imm, 0, flagPrefs, INSTR32_SBC_IMM);
}

enum EmitStatus emitLLrsbImm(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t imm, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	//NEG.N
	if (EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rdNo) && !imm && emitPrvFlagsCanUseShortInstr(flagPrefs, isInIt)) {
		
		union {
			struct {
				uint16_t rdNo		: 3;
				uint16_t rnNo		: 3;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr = {.rdNo = rdNo, .rnNo = rnNo, .instr10 =0b0100001001, };
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//RSB.W
	return emitPrvMathematicalOpImm(dest, rdNo, rnNo, imm, 0, flagPrefs, INSTR32_RSB_IMM);
}

enum EmitStatus emitLLcmpImm(struct EmitBuf *dest, uint32_t rnNo, uint32_t imm)
{
	int32_t encodedImm;
	
	//CMP.N
	if (EMIT_IS_LOREG(rnNo) && imm < 0x100) {
		
		union {
			struct {
				uint16_t imm8	: 8;
				uint16_t rnNo	: 3;
				uint16_t instr5	: 5;
			};
			uint16_t word;
		} instr = {.imm8 = imm, .rnNo = rnNo, .instr5 = 0b00101, };
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//CMP.W
	return emitPrvMathematicalOpImm(dest, EMIT_REG_NO_PC, rnNo, imm, 0, EmitSetFlags, INSTR32_CMP_IMM);
}

enum EmitStatus emitLLcmnImm(struct EmitBuf *dest, uint32_t rnNo, uint32_t imm)
{
	return emitPrvMathematicalOpImm(dest, EMIT_REG_NO_PC, rnNo, imm, 0, EmitSetFlags, INSTR32_CMN_IMM);
}

static enum EmitStatus emitPrvMovImm_16bit(struct EmitBuf *dest, uint32_t regNo, uint32_t val)	//keeping track of flag allowances is your business
{
	if (EMIT_IS_LOREG(regNo) && !(val >> 24)) {
		
		union {
			struct {
				uint16_t imm8	: 8;
				uint16_t reg	: 3;
				uint16_t in5	: 5;
			};
			uint16_t word;
		} instr = {.word = 0x2000};
		
		instr.imm8 = val;
		instr.reg = regNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLmovImm(struct EmitBuf *dest, uint32_t regNo, uint32_t valIn, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	uint32_t val = emitPrvRor(valIn, rorBy);
	enum EmitStatus now;
	
	if (regNo == EMIT_REG_NO_PC)
		return EmitErrNotEncodeable;
	
	//maybe a MOV.N will do? we cannot use emitPrvFlagsCanUseShortInstr due to how flags are set with rotation
	if (EMIT_IS_LOREG(regNo) && !(val >> 8) && ((flagPrefs == EmitSetFlags && !rorBy && !isInIt) || flagPrefs == EmitFlagsDoNotCare || (flagPrefs == EmitLeaveFlags && isInIt)))
		return emitPrvMovImm_16bit(dest, regNo, val);
	
	//how about a MOVW ?
	if (!emitPrvFlagsMustSetFlags(flagPrefs) && !(val >> 16))
		return emitPrvMovwOrMovt(dest, regNo, val, false);
	
	//maybe a MOV.W ?
	return emitPrvLogicalOpImm(dest, regNo, EMIT_REG_NO_PC, valIn, rorBy, flagPrefs, INSTR32_MOV_IMM);
}

enum EmitStatus emitLLmvnImm(struct EmitBuf *dest, uint32_t regNo, uint32_t valIn, uint32_t rorBy, enum EmitFlagSettingPrefs flagPrefs, bool isInIt)
{
	return emitPrvLogicalOpImm(dest, regNo, EMIT_REG_NO_PC, valIn, rorBy, flagPrefs, INSTR32_MVN_IMM);
}

enum EmitStatus emitHLloadImmToReg(struct EmitBuf *dest, uint32_t regNo, uint32_t val, bool canCorruptNZ, bool canCorruptC, bool isInIt)
{
	//this instr is guaranteed to never corrupt the V flag, and some code relies on that!

	enum EmitStatus now;
	int32_t encodedImm;
	
	if (regNo == EMIT_REG_NO_PC || regNo == EMIT_REG_NO_SP)
		return EmitErrNotEncodeable;
	
	//mov.n does not corrupt C flag, but does corrupt NV outside of IT
	if ((isInIt || canCorruptNZ) && !(val >> 8) && EMIT_IS_LOREG(regNo))
		return emitPrvMovImm_16bit(dest, regNo, val);
	
	//try a basic mov.imm?
	now = emitLLmovImm(dest, regNo, val, 0, (canCorruptNZ || canCorruptC)  ? EmitFlagsDoNotCare : EmitLeaveFlags, isInIt);
	if (now != EmitErrNotEncodeable)
		return now;
	
	//try a mvn.imm
	encodedImm = emitPrvEncodeThumbImmWithoutProperCarryOut(~val);
	if (encodedImm >= 0)
		return emitPrvLogicalOpImm(dest, regNo, EMIT_REG_NO_PC, ~val, 0, (canCorruptNZ || canCorruptC) ? EmitFlagsDoNotCare : EmitLeaveFlags, INSTR32_MVN_IMM);
	
	//if the top 17 bits are all ones, and it is a loreg, and we do not need to set flags, a MOVW and a SXTH are shorter
	// note that this is irrelevant if all top bits are zeroes since a normal MOVW would have worked above
	if (EMIT_IS_LOREG(regNo) && (val & 0xffff8000) == 0xffff8000) {
		
		now = emitPrvMovwOrMovt(dest, regNo, val, false);
		if (now != EmitErrNone)
			return now;
		
		now = emitLLextend(dest, regNo, regNo, 0, false, false);
		if (now != EmitErrNone)
			return now;
		
		return EmitErrNone;
	}
	
	//if all else fails, a MOVW and MOVT pair will work as long as this is not SP (which we are not allowed to set piece-wise)
	if (regNo != EMIT_REG_NO_SP) {
		
		now = emitPrvMovwOrMovt(dest, regNo, val, false);
		if (now != EmitErrNone)
			return now;
		
		now = emitPrvMovwOrMovt(dest, regNo, val >> 16, true);
		if (now != EmitErrNone)
			return now;
		
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

static enum EmitStatus emitPrvLdmStmW(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak, bool load, bool ia)
{
	uint32_t pcMask = 1 << EMIT_REG_NO_PC, lrMask = 1 << EMIT_REG_NO_LR, spMask = 1 << EMIT_REG_NO_SP, loRegsMask = 0x00ff;
	bool isSingleReg = !(regsMask & (regsMask - 1));
	
	if (rnNo == EMIT_REG_NO_PC || isSingleReg || (regsMask & spMask))
		return EmitErrNotEncodeable;
	else {
		
		union {
			struct {
				uint16_t rnNo		: 4;
				uint16_t load		: 1;
				uint16_t w			: 1;
				uint16_t instr10	: 10;
			};
			uint16_t word;
		} instr1 = {.word = ia ? 0xe880 : 0xe900, };
		
		instr1.rnNo = rnNo;
		instr1.load = load;
		instr1.w = wbak;
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(regsMask);
		return EmitErrNone;
	}
}

enum EmitStatus emitLLldmia(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	uint32_t pcMask = 1 << EMIT_REG_NO_PC, lrMask = 1 << EMIT_REG_NO_LR, loRegsMask = 0x00ff, rnMask = (1 << rnNo);
	
	if (!regsMask || (regsMask >> 16) || (wbak && (regsMask & rnMask)))
		return EmitErrNotEncodeable;
	
	//POP.N?
	if (rnNo == EMIT_REG_NO_SP && wbak && !(regsMask & ~(pcMask | loRegsMask))) {
		
		union {
			struct {
				uint16_t regsListLo	: 8;
				uint16_t pcToo		: 1;
				uint16_t instr7		: 7;
			};
			uint16_t word;
		} instr = {.word = 0xbc00, };
		
		instr.regsListLo = regsMask;
		instr.pcToo = regsMask >> EMIT_REG_NO_PC;
	
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//LDMIA.N?
	if (EMIT_IS_LOREG(rnNo) && !(regsMask & ~loRegsMask) && !wbak != !(regsMask & rnMask)) {
		
		union {
			struct {
				uint16_t regsList	: 8;
				uint16_t rnNo		: 3;
				uint16_t instr7		: 5;
			};
			uint16_t word;
		} instr = {.word = 0xc800, };
		
		instr.regsList = regsMask;
		instr.rnNo = rnNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//can never load PC & LR both
	if ((regsMask & (pcMask | lrMask)) == (pcMask | lrMask))
		return EmitErrNotEncodeable;
	
	//LDMIA.W?
	return emitPrvLdmStmW(dest, rnNo, regsMask, wbak, true, true);
}

enum EmitStatus emitLLldmdb(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	uint32_t pcMask = 1 << EMIT_REG_NO_PC, lrMask = 1 << EMIT_REG_NO_LR, rnMask = (1 << rnNo);
	
	if (!regsMask || (regsMask >> 16) || (wbak && (regsMask & (1 << rnNo))))
		return EmitErrNotEncodeable;
	
	if (rnNo == EMIT_REG_NO_SP)	//loading below SP is never a good idea and i refuse to help you!
		return EmitErrNotEncodeable;
	
	//can never load PC & LR both
	if ((regsMask & (pcMask | lrMask)) == (pcMask | lrMask))
		return EmitErrNotEncodeable;
	
	//the only option is an LDMDB.W
	return emitPrvLdmStmW(dest, rnNo, regsMask, wbak, true, false);
}

enum EmitStatus emitLLstmia(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	uint32_t pcMask = 1 << EMIT_REG_NO_PC, loRegsMask = 0x00ff, rnMask = (1 << rnNo);
	
	if (!regsMask || (regsMask >> 16))
		return EmitErrNotEncodeable;
	
	//STMIA.N?
	if (EMIT_IS_LOREG(rnNo) && wbak && !(regsMask & ~loRegsMask)) {
		
		union {
			struct {
				uint16_t regsList	: 8;
				uint16_t rnNo		: 3;
				uint16_t instr7		: 5;
			};
			uint16_t word;
		} instr = {.word = 0xc000, };
		
		//if Rn is in the list, it must be the lowest numbered reg
		if ((regsMask & rnMask) && (regsMask & (rnMask - 1)))
			return EmitErrNotEncodeable;
		
		instr.regsList = regsMask;
		instr.rnNo = rnNo;
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//STMIA.W
	return emitPrvLdmStmW(dest, rnNo, regsMask, wbak, false, true);
}

enum EmitStatus emitLLstmdb(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	uint32_t pcMask = 1 << EMIT_REG_NO_PC, lrMask = 1 << EMIT_REG_NO_LR, spMask = 1 << EMIT_REG_NO_SP, loRegsMask = 0x00ff, rnMask = (1 << rnNo);
	
	if (!regsMask || (regsMask >> 16) || (regsMask & (pcMask | spMask)) || (wbak && (regsMask & rnMask)))
		return EmitErrNotEncodeable;
	
	//PUSH.N?
	if (rnNo == EMIT_REG_NO_SP && wbak && !(regsMask & ~(lrMask | loRegsMask))) {
		
		union {
			struct {
				uint16_t regsListLo	: 8;
				uint16_t lrToo		: 1;
				uint16_t instr7		: 7;
			};
			uint16_t word;
		} instr = {.word = 0xb400, };
		
		instr.regsListLo = regsMask;
		instr.lrToo = regsMask >> EMIT_REG_NO_LR;
	
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	
	//STMDB.W
	return emitPrvLdmStmW(dest, rnNo, regsMask, wbak, false ,false);
}

#ifdef HAVE_FPU
	static enum EmitStatus emitPrvVldmStmSP(struct EmitBuf *dest, uint32_t rnNo, uint32_t regFirst, uint32_t regNum, bool wbak, bool ia, bool load)
	{
		union {
			struct {
				uint16_t rnNo		: 4;
				uint16_t L			: 1;
				uint16_t W			: 1;
				uint16_t D			: 1;
				uint16_t U			: 1;
				uint16_t P			: 1;
				uint16_t instr7		: 7;
			};
			uint16_t word;
		} instr1 = {.rnNo = rnNo, .L = load, .W = wbak, .D = regFirst , .U = ia ? 1 : 0, .P = ia ? 0 : 1, .instr7 = 0b1110110, };
		union {
			struct {
				uint16_t numRegs	: 8;
				uint16_t val_0x0a	: 4;
				uint16_t Vd			: 4;
			};
			uint16_t word;
		} instr2 = {.numRegs = regNum, .val_0x0a = 10, .Vd = regFirst >> 1, };
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
	
	enum EmitStatus emitLLvldmiaSP(struct EmitBuf *dest, uint32_t rnNo, uint32_t regFirst, uint32_t regNum, bool wbak)
	{
		return emitPrvVldmStmSP(dest, rnNo, regFirst, regNum, wbak, true, true);
	}
	
	enum EmitStatus emitLLvstmiaSP(struct EmitBuf *dest, uint32_t rnNo, uint32_t regFirst, uint32_t regNum, bool wbak)
	{
		return emitPrvVldmStmSP(dest, rnNo, regFirst, regNum, wbak, true, false);
	}
	
	enum EmitStatus emitLLvldmdbSP(struct EmitBuf *dest, uint32_t rnNo, uint32_t regFirst, uint32_t regNum, bool wbak)
	{
		return emitPrvVldmStmSP(dest, rnNo, regFirst, regNum, wbak, false, true);
	}
	
	enum EmitStatus emitLLvstmdbSP(struct EmitBuf *dest, uint32_t rnNo, uint32_t regFirst, uint32_t regNum, bool wbak)
	{
		return emitPrvVldmStmSP(dest, rnNo, regFirst, regNum, wbak, false, false);
	}
	
	static enum EmitStatus emitPrvVmovBetweenVfpAndArmSP(struct EmitBuf *dest, uint32_t armReg, uint32_t vfpReg, bool toArm)
	{
		union {
			struct {
				uint16_t Vn			: 4;
				uint16_t toArm		: 1;
				uint16_t instr11	: 11;
			};
			uint16_t word;
		} instr1 = {.Vn = vfpReg >> 1, .toArm = toArm, .instr11 = 0b11101110000, };
		union {
			struct {
				uint16_t instr6		: 7;
				uint16_t N			: 1;
				uint16_t instr4		: 4;
				uint16_t rtNo		: 4;
			};
			uint16_t word;
		} instr2 = {.instr6 = 0b0010000, .N = vfpReg, .instr4 = 0b1010, .rtNo = armReg, };
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
	
	enum EmitStatus emitLLvmovArmToVfpSP(struct EmitBuf *dest, uint32_t dstReg, uint32_t srcReg)
	{
		return emitPrvVmovBetweenVfpAndArmSP(dest, srcReg, dstReg, false);
	}
	
	enum EmitStatus emitLLvmovVfpToArmSP(struct EmitBuf *dest, uint32_t dstReg, uint32_t srcReg)
	{
		return emitPrvVmovBetweenVfpAndArmSP(dest, dstReg, srcReg, true);
	}
	
	static enum EmitStatus emitPrvVmovBetweenVfpAndArm2xSP(struct EmitBuf *dest, uint32_t armReg1, uint32_t armReg2, uint32_t vfpReg, bool toArm)
	{
		union {
			struct {
				uint16_t rtNo2		: 4;
				uint16_t toArm		: 1;
				uint16_t instr11	: 11;
			};
			uint16_t word;
		} instr1 = {.rtNo2 = armReg2, .toArm = toArm, .instr11 = 0b11101100010, };
		union {
			struct {
				uint16_t Vm			: 4;
				uint16_t val1		: 1;
				uint16_t M			: 1;
				uint16_t instr6		: 6;
				uint16_t rtNo		: 4;
			};
			uint16_t word;
		} instr2 = {.Vm = vfpReg >> 1, .val1 = 1, .M = vfpReg, .instr6 = 0b101000, .rtNo = armReg1, };
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
	
	enum EmitStatus emitLLvmovArmToVfp2xSP(struct EmitBuf *dest, uint32_t dstReg, uint32_t srcReg1, uint32_t srcReg2)
	{
		return emitPrvVmovBetweenVfpAndArm2xSP(dest, srcReg1, srcReg2, dstReg, false);
	}
	
	enum EmitStatus emitLLvmovVfpToArm2xSP(struct EmitBuf *dest, uint32_t dstReg1, uint32_t dstReg2, uint32_t srcReg)
	{
		return emitPrvVmovBetweenVfpAndArm2xSP(dest, dstReg1, dstReg2, srcReg, true);
	}
	
	static enum EmitStatus emitPrvVcvtFixedPtSP(struct EmitBuf *dest, uint32_t reg, uint32_t size, uint32_t fracBits, bool unsign, bool toFixed)
	{
		union {
			struct {
				uint16_t unsign		: 1;
				uint16_t instr1		: 1;
				uint16_t toFixed	: 1;
				uint16_t instr3		: 3;
				uint16_t D			: 1;
				uint16_t instr9		: 9;
			};
			uint16_t word;
		} instr1 = {.word = 0xeeba, };
		union {
			struct {
				uint16_t imm4		: 4;
				uint16_t instr1a	: 1;
				uint16_t i			: 1;
				uint16_t instr1b	: 1;
				uint16_t szIs32		: 1;
				uint16_t instr4		: 4;
				uint16_t Vd			: 4;
			};
			uint16_t word;
		} instr2 = {.word = 0x0a40, };
		
		if ((size != 16 && size != 32) || size < fracBits)
			return EmitErrNotEncodeable;
		
		fracBits = size - fracBits;
		
		instr1.unsign = unsign;
		instr1.toFixed = toFixed;
		instr1.D = reg;
		instr2.imm4 = fracBits >> 1;
		instr2.i = fracBits;
		instr2.szIs32 = size == 32;
		instr2.Vd = reg >> 1;
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
	
	enum EmitStatus emitLLvcvtToFixedPtSP(struct EmitBuf *dest, uint32_t reg, uint32_t size, uint32_t fracBits, bool unsign)
	{
		return emitPrvVcvtFixedPtSP(dest, reg, size, fracBits, unsign, true);
	}
	
	enum EmitStatus emitLLvcvtFromFixedPtSP(struct EmitBuf *dest, uint32_t reg, uint32_t size, uint32_t fracBits, bool unsign)
	{
		return emitPrvVcvtFixedPtSP(dest, reg, size, fracBits, unsign, false);
	}

#endif

enum EmitStatus emitPrvLoadStoreImm(struct EmitBuf *dest, bool load, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz, bool sext, enum EmitAddrMode adrMode)
{
	static const uint16_t instrs[] = {[EmitSzByte] = 0xf800, [EmitSzHalfword] = 0xf820, [EmitSzWord] = 0xf840, };
	bool emitted = false, add = (imm >= 0);
	uint32_t immAbs = add ? imm : -imm;
	
	//first half of instr32
	union {
		struct {
			uint16_t rnNo		: 4;
			uint16_t load		: 1;
			uint16_t instr2		: 2;
			uint16_t imm12form	: 1;
			uint16_t sext		: 1;
			uint16_t instr7		: 7;
		};
		uint16_t word;
	} instr32;
		
	//writeback with Rn == Rt is not allowed in v7M
	if (adrMode != EmitAdrModeIndex && rtNo == rnNo)
		return EmitErrNotEncodeable;
	
	//loading SP has special restrictions
	if (load && rtNo == EMIT_REG_NO_SP) {
		
		//Erratum 752419: loading with writeback can cause issues in v7M if an exception happens at the same time - avoid emitting that
		if (adrMode != EmitAdrModeIndex) {
			logw("refusing to load SP with writeback!\n");
			return EmitErrNotEncodeable;
		}
		
		//non-word loads to SP are forbidden
		if (opSz != EmitSzWord)
			return EmitErrNotEncodeable;
	}
	
	//try short ones
	if (EMIT_IS_LOREG(rtNo) && adrMode == EmitAdrModeIndex) {
		
		//LDR/STR Rx, [SP, #...]
		if (rnNo == EMIT_REG_NO_SP && opSz == EmitSzWord && add && !(imm & 3) && imm < 0x400 && !sext) {
			
			union {
				struct {
					uint16_t imm8	: 8;
					uint16_t rtNo	: 3;
					uint16_t load	: 1;
					uint16_t instr4	: 4;
				};
				uint16_t word;
			} instr = {.word = 0x9000, };
			
			instr.imm8 = imm >> 2;
			instr.rtNo = rtNo;
			instr.load = load;
			
			VERIFY_SPACE(1);
			EMIT_HALFWORD(instr.word);
			return EmitErrNone;
		}
		
		if (EMIT_IS_LOREG(rnNo) && add && !sext) {
			
			union {
				struct {
					uint16_t rtNo	: 3;
					uint16_t rnNo	: 3;
					uint16_t imm5	: 5;
					uint16_t load	: 1;
					uint16_t instr4	: 4;
					
				};
				uint16_t word;
			} instr = {.rtNo = rtNo, .rnNo = rnNo, .load = load, };
			
			switch (opSz) {
				case EmitSzByte:
					if (imm < 0x20) {
						instr.imm5 = imm;
						instr.instr4 = 0x07;
						emitted = true;
					}
					break;
				case EmitSzHalfword:
					if (!(imm & 1) && imm < 0x40) {
						instr.imm5 = imm >> 1;
						instr.instr4 = 0x08;
						emitted = true;
					}
					break;
				case EmitSzWord:
					if (!(imm & 1) && imm < 0x80) {
						instr.imm5 = imm >> 2;
						instr.instr4 = 0x06;
						emitted = true;
					}
					break;
				default:
					__builtin_unreachable();
					break;
			}
			if (emitted) {
				VERIFY_SPACE(1);
				EMIT_HALFWORD(instr.word);
				return EmitErrNone;
			}
		}
		
		//LDR Rx, [PC, #...]
		if (rnNo == EMIT_REG_NO_PC && opSz == EmitSzWord && add && !(imm & 3) && imm < 0x400 && !sext && load) {
			
			union {
				struct {
					uint16_t imm8	: 8;
					uint16_t rtNo	: 3;
					uint16_t instr5	: 5;
				};
				uint16_t word;
			} instr = {.word = 0x4800, };
			
			instr.imm8 = imm >> 2;
			instr.rtNo = rtNo;
			
			VERIFY_SPACE(1);
			EMIT_HALFWORD(instr.word);
			return EmitErrNone;
		}
	}
	
	//preload first half of instr32 with values
	instr32.word = instrs[opSz];
	instr32.rnNo = rnNo;
	instr32.load = load;
	instr32.sext = sext;
	
	//try the imm12 variants
	if (adrMode == EmitAdrModeIndex && add && imm < 0x1000) {
		
		union {
			struct {
				uint16_t imm12		: 12;
				uint16_t rtNo		: 4;
			};
			uint16_t word;
		} instr2 = {.imm12 = imm, .rtNo = rtNo, };
		
		instr32.imm12form = 1;
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr32.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
	
	//imm8 variants (all cases that imm12 could handle have been handled for sure)
	if (immAbs < 0x100) {
		
		union {
			struct {
				uint16_t rnNo		: 4;
				uint16_t instr12	: 12;
			};
			uint16_t word;
		} instr1 = {.rnNo = rnNo, .instr12 = instrs[opSz], };
		
		union {
			struct {
				uint16_t imm8		: 8;
				uint16_t w			: 1;
				uint16_t u			: 1;
				uint16_t p			: 1;
				uint16_t instr1		: 1;
				uint16_t rtNo		: 4;
			};
			uint16_t word;
		} instr2 = {.imm8 = immAbs, .rtNo = rtNo, .u = add, .instr1 = 1, };
		
		switch (adrMode) {
			case EmitAdrModeIndexWbak:
				instr2.w = 1;
				//fallthrough
			case EmitAdrModeIndex:
				instr2.p = 1;
				break;
			case EmitAdrModePostindex:
				instr2.w = 1;
				break;
			default:
				__builtin_unreachable();
				break;
		}
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr32.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLstoreImm(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz, enum EmitAddrMode adrMode)
{
	//all the special things we can do are for words
	if (opSz == EmitSzWord) {
		
		//STMIA.N ?
		if (imm == 4 && adrMode == EmitAdrModePostindex && EMIT_IS_LOREG(rtNo) && EMIT_IS_LOREG(rnNo))
			return emitLLstmia(dest, rnNo, 1 << rtNo, true);
		
		//PUSH.N ?
		if (imm == -4 && adrMode == EmitAdrModeIndexWbak && EMIT_IS_LOREG(rtNo)&& rnNo == EMIT_REG_NO_SP)
			return emitLLstmdb(dest, rnNo, 1 << rtNo, true);
	}
	
	return emitPrvLoadStoreImm(dest, false, rtNo, rnNo, imm, opSz, false, adrMode);
}

enum EmitStatus emitLLloadImm(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz, bool sext, enum EmitAddrMode adrMode)
{
	//cannot sign extend a loaded word
	if (sext && opSz == EmitSzWord)
		return EmitErrNotEncodeable;
	
	//only word sized loads allowed into PC or SP
	if ((rtNo == EMIT_REG_NO_PC || rtNo == EMIT_REG_NO_SP) && opSz != EmitSzWord)
		return EmitErrNotEncodeable;
	
	//all the special things we can do are for words in postindex mode
	if (opSz == EmitSzWord && imm == 4 && adrMode == EmitAdrModePostindex) {
		
		//POP.N or LDMIA.N?
		if ((rnNo == EMIT_REG_NO_SP && (EMIT_IS_LOREG(rtNo) || rtNo == EMIT_REG_NO_PC)) || (EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rtNo)))
			return emitLLldmia(dest, rnNo, 1 << rtNo, true);
	}
	
	return emitPrvLoadStoreImm(dest, true, rtNo, rnNo, imm, opSz, sext, adrMode);
}

static enum EmitStatus emitPrvLoadStoreRegReg(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, uint32_t rmNo, uint32_t lslAmt, enum EmitMemOpSz opSz, uint32_t instr16, uint32_t instr32)
{
	if (opSz != EmitSzWord && (rtNo == EMIT_REG_NO_PC || rtNo == EMIT_REG_NO_SP))
		return EmitErrNotEncodeable;
	
	if (rmNo == EMIT_REG_NO_PC || rmNo == EMIT_REG_NO_SP)
		return EmitErrNotEncodeable;
	
	if (lslAmt > 3)
		return EmitErrNotEncodeable;
	
	if (EMIT_IS_LOREG(rtNo) && EMIT_IS_LOREG(rnNo) && EMIT_IS_LOREG(rmNo) && !lslAmt) {
		
		union {
			struct {
				uint16_t rtNo		: 3;
				uint16_t rnNo		: 3;
				uint16_t rmNo		: 3;
				uint16_t instr7		: 7;
			};
			uint16_t word;
		} instr = {.rtNo = rtNo, .rnNo = rnNo, .rmNo = rmNo, .instr7 = instr16, };
		
		VERIFY_SPACE(1);
		EMIT_HALFWORD(instr.word);
		return EmitErrNone;
	}
	else if (lslAmt < 4) {
		
		union {
			struct {
				uint16_t rnNo		: 4;
				uint16_t instr12	: 12;
			};
			uint16_t word;
		} instr1 = {.rnNo = rnNo, .instr12 = instr32, };
		union {
			struct {
				uint16_t rmNo		: 4;
				uint16_t imm2		: 2;
				uint16_t val_0		: 6;
				uint16_t rtNo		: 4;
			};
			uint16_t word;
		} instr2 = {.rmNo = rmNo, .imm2 = lslAmt, .val_0 = 0, .rtNo = rtNo, };
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitLLstoreRegReg(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, uint32_t rmNo, uint32_t lslAmt, enum EmitMemOpSz opSz)
{
	uint32_t instr16, instr32;
	
	switch (opSz) {
		case EmitSzByte:		instr16 = 0b0101010;	instr32 = 0b111110000000;	break;
		case EmitSzHalfword:	instr16 = 0b0101001;	instr32 = 0b111110000010;	break;
		case EmitSzWord:		instr16 = 0b0101000;	instr32 = 0b111110000100;	break;
		default:	__builtin_unreachable();
	}
	
	if (rnNo == EMIT_REG_NO_PC || rtNo == EMIT_REG_NO_PC)
		return EmitErrNotEncodeable;
	
	return emitPrvLoadStoreRegReg(dest, rtNo, rnNo, rmNo, lslAmt, opSz, instr16, instr32);
}

enum EmitStatus emitLLloadRegReg(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, uint32_t rmNo, uint32_t lslAmt, enum EmitMemOpSz opSz, bool sext)
{
	uint32_t instr16, instr32;
	
	switch (opSz) {
		case EmitSzByte:
			instr16 = sext ? 0b0101011 : 0b0101110;
			instr32 = sext ? 0b111110010001 : 0b111110000001;
			break;
		
		case EmitSzHalfword:
			instr16 = sext ? 0b0101111 : 0b0101101;
			instr32 = sext ? 0b111110010011 : 0b111110000011;
			break;
		
		case EmitSzWord:
			if (sext)
				return EmitErrNotEncodeable;
			instr16 = 0b0101100;
			instr32 = 0b111110000101;
			break;
		
		default:
			__builtin_unreachable();
	}
	
	return emitPrvLoadStoreRegReg(dest, rtNo, rnNo, rmNo, lslAmt, opSz, instr16, instr32);
}

static enum EmitStatus emitPrvLdrexStrex(struct EmitBuf *dest, uint32_t rdNo, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz, bool load)
{
	if (opSz == EmitSzWord) {
		
		union {
			struct {
				uint16_t rnNo		: 4;
				uint16_t load		: 1;
				uint16_t instr11	: 11;
			};
			uint16_t word;
		} instr1 = {.rnNo = rnNo, .load = load, .instr11 = 0b11101000010, };
		
		union {
			struct {
				uint16_t imm8		: 8;
				uint16_t rdNo		: 4;	//0x0f for LDREX
				uint16_t rtNo		: 4;
			};
			uint16_t word;
		} instr2 = {.imm8 = imm >> 2, .rdNo = rdNo, .rtNo = rtNo, };
		
		if ((imm & 3) || (imm > 0x400))
			return EmitErrNotEncodeable;
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
	else {
		
		union {
			struct {
				uint16_t rnNo		: 4;
				uint16_t load		: 1;
				uint16_t instr11	: 11;
			};
			uint16_t word;
		} instr1 = {.rnNo = rnNo, .load = load, .instr11 = 0b11101000110, };
		
		union {
			struct {
				uint16_t rdNo		: 4;	//0x0f for LDREXx
				uint16_t halfword	: 1;
				uint16_t instr7		: 7;
				uint16_t rtNo		: 4;
			};
			uint16_t word;
		} instr2 = {.rdNo = rdNo, .halfword = opSz == EmitSzHalfword, .instr7 = 0b1111010, .rtNo = rtNo, };
		
		if (imm)
			return EmitErrNotEncodeable;
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
}

enum EmitStatus emitLLstrex(struct EmitBuf *dest, uint32_t rdNo, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz)
{
	return emitPrvLdrexStrex(dest, rdNo, rtNo, rnNo, imm, opSz, false);
}

enum EmitStatus emitLLldrex(struct EmitBuf *dest, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitMemOpSz opSz)
{
	return emitPrvLdrexStrex(dest, EMIT_REG_NO_PC, rtNo, rnNo, imm, opSz, true);
}

enum EmitStatus emitHLldmia(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	bool isSingleReg = regsMask && !(regsMask & (regsMask - 1));
	uint32_t whichReg = __builtin_ctz(regsMask);
	enum EmitStatus now;
	
	//no regs - we're done
	if (!regsMask)
		return EmitErrNone;
	
	//try the usual
	now = emitLLldmia(dest, rnNo, regsMask, wbak);
	if (now != EmitErrNotEncodeable)
		return now;
	
	//the only extra case we can handle is single-reg op
	if (isSingleReg)
		return emitLLloadImm(dest, whichReg, rnNo, wbak ? 4 : 0, EmitSzWord, false, wbak ? EmitAdrModePostindex : EmitAdrModeIndex);
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitHLstmia(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	bool isSingleReg = regsMask && !(regsMask & (regsMask - 1));
	uint32_t whichReg = __builtin_ctz(regsMask);
	enum EmitStatus now;
	
	//no regs - we're done
	if (!regsMask)
		return EmitErrNone;
	
	//try the usual
	now = emitLLstmia(dest, rnNo, regsMask, wbak);
	if (now != EmitErrNotEncodeable)
		return now;
	
	//the only extra case we can handle is single-reg op
	if (isSingleReg)
		return emitLLstoreImm(dest, whichReg, rnNo, wbak ? 4 : 0, EmitSzWord, wbak ? EmitAdrModePostindex : EmitAdrModeIndex);
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitHLldmdb(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	bool isSingleReg = regsMask && !(regsMask & (regsMask - 1));
	uint32_t whichReg = __builtin_ctz(regsMask);
	enum EmitStatus now;
	
	//no regs - we're done
	if (!regsMask)
		return EmitErrNone;
	
	//try the usual
	now = emitLLldmdb(dest, rnNo, regsMask, wbak);
	if (now != EmitErrNotEncodeable)
		return now;
	
	//the only extra case we can handle is single-reg op
	if (isSingleReg)
		return emitLLloadImm(dest, whichReg, rnNo, -4, EmitSzWord, false, wbak ? EmitAdrModeIndexWbak : EmitAdrModeIndex);
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitHLstmdb(struct EmitBuf *dest, uint32_t rnNo, uint32_t regsMask, bool wbak)
{
	bool isSingleReg = regsMask && !(regsMask & (regsMask - 1));
	uint32_t whichReg = __builtin_ctz(regsMask);
	enum EmitStatus now;
	
	//no regs - we're done
	if (!regsMask)
		return EmitErrNone;
	
	//try the usual
	now = emitLLstmdb(dest, rnNo, regsMask, wbak);
	if (now != EmitErrNotEncodeable)
		return now;
	
	//the only extra case we can handle is single-reg op
	if (isSingleReg)
		return emitLLstoreImm(dest, whichReg, rnNo, -4, EmitSzWord, wbak ? EmitAdrModeIndexWbak : EmitAdrModeIndex);
	
	return EmitErrNotEncodeable;
}

enum EmitStatus emitHLpush(struct EmitBuf *dest, uint32_t regsMask)
{
	return emitHLstmdb(dest, EMIT_REG_NO_SP, regsMask, true);
}

enum EmitStatus emitHLpop(struct EmitBuf *dest, uint32_t regsMask)
{
	return emitHLldmia(dest, EMIT_REG_NO_SP, regsMask, true);
}

static enum EmitStatus emitPrvMsrMrs(struct EmitBuf *dest, uint32_t rdNo, uint32_t sysm, uint32_t mask, uint32_t rnNo, bool isMrs)
{
	union {
		struct {
			uint16_t rnNo		: 4;
			uint16_t instr12	: 12;
		};
		uint16_t word;
	} instr1 = {.word = isMrs ? 0xf3ef : 0xf380, };
	union {
		struct {	//msr version
			uint16_t sysmA		: 8;
			uint16_t instr2		: 2;
			uint16_t mask		: 2;
			uint16_t instr4a	: 4;
		};
		struct {	//mrs version
			uint16_t sysmB		: 8;
			uint16_t rdNo		: 4;
			uint16_t instr4b	: 4;
		};
		uint16_t word;
	} instr2 = {.word = 0x8000, };
	
	if (isMrs) {
		instr2.sysmB = sysm;
		instr2.rdNo = rdNo;
	}
	else {
		instr2.sysmA = sysm;
		instr1.rnNo = rnNo;
		instr2.mask = mask;
	}
	
	if(rdNo == EMIT_REG_NO_SP || rnNo == EMIT_REG_NO_SP || rdNo == EMIT_REG_NO_PC || rnNo == EMIT_REG_NO_PC)
		return EmitErrNotEncodeable;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

enum EmitStatus emitLLmrs(struct EmitBuf *dest, uint32_t rdNo, uint32_t sysm)
{
	return emitPrvMsrMrs(dest, rdNo, sysm, 0, 0, true);
}

enum EmitStatus emitLLmsr(struct EmitBuf *dest, uint32_t sysm, uint32_t mask, uint32_t rnNo)
{
	return emitPrvMsrMrs(dest, 0, sysm, mask, rnNo, false);
}

enum EmitStatus emitLLsvc(struct EmitBuf *dest, uint32_t imm)
{
	if (imm >> 8)
		return EmitErrNotEncodeable;
	
	VERIFY_SPACE(1);
	EMIT_HALFWORD(0xdf00 + imm);
	return EmitErrNone;
}

enum EmitStatus emitLLclz(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo)
{
	union {
		struct {
			uint16_t rmNo		: 4;
			uint16_t instr12	: 12;
		};
		uint16_t word;
	} instr1 = {.word = 0xfab0, };
	union {
		struct {
			uint16_t rmNo		: 4;
			uint16_t instr4a	: 4;
			uint16_t rdNo		: 4;
			uint16_t instr4b	: 4;
		};
		uint16_t word;
	} instr2 = {.word = 0xf080, };
	
	//PC and SP not allowed
	if(((1 << rdNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	instr1.rmNo = rmNo;
	instr2.rmNo = rmNo;	//yes, encoded twice
	instr2.rdNo = rdNo;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

static enum EmitStatus emitPrvLdrdStrdImm(struct EmitBuf *dest, bool load, uint32_t rtNo, uint32_t rtNo2, uint32_t rnNo, int32_t imm, enum EmitAddrMode adrMode)
{
	bool add = imm >= 0;
	uint32_t immAbs = add ? imm : -imm;
	union {
		struct {
			uint16_t rnNo	: 4;
			uint16_t load	: 1;
			uint16_t w		: 1;
			uint16_t instr1	: 1;
			uint16_t u		: 1;
			uint16_t p		: 1;
			uint16_t instr7	: 7;
		};
		uint16_t word;
	} instr1 = {.word = 0xe840, };
	union {
		struct {
			uint16_t imm8	: 8;
			uint16_t rtNo2	: 4;
			uint16_t rtNo	: 4;
		};
		uint16_t word;
	} instr2 = {.imm8 = immAbs >> 2, .rtNo2 = rtNo2, .rtNo = rtNo, };
	
	//no wbak if one of the Rt regs is also base
	if (adrMode != EmitAdrModeIndex && (rtNo == rnNo || rtNo2 == rnNo))
		return EmitErrNotEncodeable;
	
	//neither of them is allowed to be SP or PC. this is the fastest way to check for that
	if (((1 << rtNo) | (1 << rtNo2)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	//imm must be in range
	if ((immAbs & 3) || (immAbs >= 0x400))
		return EmitErrNotEncodeable;
	
	instr1.rnNo = rnNo;
	instr1.load = load;
	instr1.u = add;
	
	switch (adrMode) {
		case EmitAdrModeIndexWbak:
			instr1.w = 1;
			//fallthrough
		case EmitAdrModeIndex:
			instr1.p = 1;
			break;
		case EmitAdrModePostindex:
			instr1.w = 1;
			break;
		default:
			__builtin_unreachable();
			break;
	}

	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

enum EmitStatus emitLLldrdImm(struct EmitBuf *dest, uint32_t rtNo, uint32_t rtNo2, uint32_t rnNo, int32_t imm, enum EmitAddrMode adrMode)
{
	//erratum 602117
	if (rtNo == rnNo)
		return EmitErrNotEncodeable;
	
	//regs must differ
	if (rtNo == rtNo2)
		return EmitErrNotEncodeable;
	
	return emitPrvLdrdStrdImm(dest, true, rtNo, rtNo2, rnNo, imm, adrMode);
}

enum EmitStatus emitLLstrdImm(struct EmitBuf *dest, uint32_t rtNo, uint32_t rtNo2, uint32_t rnNo, int32_t imm, enum EmitAddrMode adrMode)
{
	if (rnNo == EMIT_REG_NO_PC)
		return EmitErrNotEncodeable;
	
	return emitPrvLdrdStrdImm(dest, false, rtNo, rtNo2, rnNo, imm, adrMode);
}

enum EmitStatus emitLLlongMul(struct EmitBuf *dest, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo, bool isUnsigned, bool accumulate)
{
	union {
		struct {
			uint16_t rnNo		: 4;
			uint16_t val0		: 1;
			uint16_t isUnsigned	: 1;
			uint16_t accumulate	: 1;
			uint16_t instr9		: 9;
		};
		uint16_t word;
	} instr1 = {.rnNo = rnNo, .val0 = 0, .isUnsigned = isUnsigned, .accumulate = accumulate, .instr9 = 0b111110111, };
	union {
		struct {
			uint16_t rmNo		: 4;
			uint16_t val0		: 4;
			uint16_t rdHiNo		: 4;
			uint16_t rdLoNo		: 4;
		};
		uint16_t word;
	} instr2 = {.rmNo = rmNo, .val0 = 0, .rdHiNo = rdHiNo, .rdLoNo = rdLoNo, };
	
	if (rdHiNo == rdLoNo)
		return EmitErrNotEncodeable;
	
	//none of the regs are allowed to be SP or PC. this is the easiest way to check
	if (((1 << rdLoNo) | (1 << rdHiNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_PC) | (1 << EMIT_REG_NO_SP)))
		return EmitErrNotEncodeable;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

enum EmitStatus emitLLumull(struct EmitBuf *dest, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo)
{
	return emitLLlongMul(dest, rdLoNo, rdHiNo, rnNo, rmNo, true, false);
}

enum EmitStatus emitLLumlal(struct EmitBuf *dest, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo)
{
	return emitLLlongMul(dest, rdLoNo, rdHiNo, rnNo, rmNo, true, true);
}

enum EmitStatus emitLLsmull(struct EmitBuf *dest, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo)
{
	return emitLLlongMul(dest, rdLoNo, rdHiNo, rnNo, rmNo, false, false);
}

enum EmitStatus emitLLsmlal(struct EmitBuf *dest, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo)
{
	return emitLLlongMul(dest, rdLoNo, rdHiNo, rnNo, rmNo, false, true);
}

static enum EmitStatus emitPrvPkh(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, bool tb)
{
	union {
		struct {
			uint16_t rnNo		: 4;
			uint16_t instr12	: 12;
		};
		uint16_t word;
	} instr1 = {.word = 0xeac0, };
	union {
		struct {
			uint16_t rmNo		: 4;
			uint16_t instr1a	: 1;
			uint16_t tb			: 1;
			uint16_t imm2		: 2;
			uint16_t rdNo		: 4;
			uint16_t imm3		: 3;
			uint16_t instr1b	: 1;
		};
		uint16_t word;
	} instr2 = {.word = 0, };
	
	//if there is a shift, only some types are allowed
	if (shiftAmt && ((tb && shiftType != EmitShiftAsr) || (!tb && shiftType != EmitShiftLsl)))
		return EmitErrNotEncodeable;
	
	instr1.rnNo = rnNo;
	instr2.rmNo = rmNo;
	instr2.tb = tb;
	instr2.imm2 = shiftAmt;
	instr2.rdNo = rdNo;
	instr2.imm3 = shiftAmt >> 2;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

enum EmitStatus emitLLpkhbt(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	return emitPrvPkh(dest, rdNo, rnNo, rmNo, shiftType, shiftAmt, false);
}

enum EmitStatus emitLLpkhtb(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	return emitPrvPkh(dest, rdNo, rnNo, rmNo, shiftType, shiftAmt, true);
}

static enum EmitStatus emitPrvDiv(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool unsign)
{
	//No SP or PC allowed
	if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;
	
	union {
		struct {
			uint16_t rnNo		: 4;
			uint16_t instr1		: 1;
			uint16_t unsign		: 1;
			uint16_t instr10	: 10;
		};
		uint16_t word;
	} instr1 = {.word = 0xfb90, };
	union {
		struct {
			uint16_t rmNo		: 4;
			uint16_t instr4a	: 4;
			uint16_t rdNo		: 4;
			uint16_t instr4b	: 4;
		};
		uint16_t word;
	} instr2 = {.word = 0xf0f0, };
	
	instr1.rnNo = rnNo;
	instr1.unsign = unsign ? 1 : 0;
	instr2.rmNo = rmNo;
	instr2.rdNo = rdNo;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

enum EmitStatus emitLLsdiv(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
{
	return emitPrvDiv(dest, rdNo, rnNo, rmNo, false);
}

enum EmitStatus emitLLudiv(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
{
	return emitPrvDiv(dest, rdNo, rnNo, rmNo, true);
}

static enum EmitStatus emitPrvSat(struct EmitBuf *dest, uint32_t rdNo, uint32_t immTo, uint32_t rnNo, enum EmitShiftType shiftType, uint32_t shiftAmt, bool unsign)
{
	//No SP or PC allowed
	if (((1 << rdNo) | (1 << rnNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
		return EmitErrNotEncodeable;

	//saturation has its limits
	if (!immTo || immTo > 32)
		return EmitErrNotEncodeable;

	union {
		struct {
			uint16_t rnNo		: 4;
			uint16_t instr1a	: 1;
			uint16_t isAsr		: 1;
			uint16_t instr1b	: 1;
			uint16_t unsign		: 1;
			uint16_t instr10	: 8;
		};
		uint16_t word;
	} instr1 = {.word = 0xf300, };
	union {
		struct {
			uint16_t sat_imm	: 5;
			uint16_t instr1a	: 1;
			uint16_t imm2		: 2;
			uint16_t rdNo		: 4;
			uint16_t imm3		: 3;
			uint16_t instr1b	: 1;
		};
		uint16_t word;
	} instr2 = {.word = 0x0000, };
	
	switch (shiftType) {
		case EmitShiftLsl:
			break;
		
		case EmitShiftAsr:
			instr1.isAsr = 1;
			break;
		
		default:
			return EmitErrNotEncodeable;
	}
	
	instr1.rnNo = rnNo;
	instr1.unsign = unsign ? 1 : 0;
	instr2.sat_imm = immTo - 1;
	instr2.imm2 = shiftAmt;
	instr2.rdNo = rdNo;
	instr2.imm3 = shiftAmt >> 2;
	
	VERIFY_SPACE(2);
	EMIT_HALFWORD(instr1.word);
	EMIT_HALFWORD(instr2.word);
	return EmitErrNone;
}

enum EmitStatus emitLLssat(struct EmitBuf *dest, uint32_t rdNo, uint32_t immTo, uint32_t rnNo, enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	return emitPrvSat(dest, rdNo, immTo, rnNo, shiftType, shiftAmt, false);
}

enum EmitStatus emitLLusat(struct EmitBuf *dest, uint32_t rdNo, uint32_t immTo, uint32_t rnNo, enum EmitShiftType shiftType, uint32_t shiftAmt)
{
	return emitPrvSat(dest, rdNo, immTo, rnNo, shiftType, shiftAmt, true);
}

enum EmitStatus emitLLtableBranch(struct EmitBuf *dest, struct EmitTableBranchState *state, uint32_t rmNo, bool halfwordSized, uint32_t numCases)
{
	uint32_t i, spaceNeeded;
	
	union {
		struct {
			uint16_t rmNo		: 4;
			uint16_t h			: 1;
			uint16_t instr11	: 11;
		};
		uint16_t word;
	} instr2 = {.rmNo = rmNo, .h = halfwordSized, .instr11 = 0b11110000000, };
	
	if (halfwordSized)
		spaceNeeded = numCases;
	else
		spaceNeeded = (numCases + 1) / sizeof(uint16_t);
	
	VERIFY_SPACE(2 + spaceNeeded);
	EMIT_HALFWORD(0xe8df);
	EMIT_HALFWORD(instr2.word);
	
	state->srcPtr = dest->buf;
	state->numCases = numCases;
	state->isHalfword = halfwordSized;
	
	for (i = 0; i < spaceNeeded; i++)
		EMIT_HALFWORD(0);
	
	return EmitErrNone;
}

enum EmitStatus emitLLtableBranchSetCase(struct EmitTableBranchState *state, uint32_t caseIdx, uintptr_t dstAddr)
{
	uint32_t ofst = (dstAddr - state->srcAddr) / sizeof(uint16_t);
	
	if (caseIdx >= state->numCases)
		return EmitErrNotEncodeable;
	
	if (state->isHalfword) {
		
		if (ofst >> 16)
			return EmitErrNotEncodeable;
		
		state->ptrH[caseIdx] = ofst;
	}
	else {
		
		if (ofst >> 24)
			return EmitErrNotEncodeable;
		
		state->ptrB[caseIdx] = ofst;
	}
	
	return EmitErrNone;
}

#ifdef HAVE_v7E_SUPPORT
	static enum EmitStatus emitPrvSatAddSub(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool doubling, bool subtract)
	{
		union {
			struct {
				uint16_t rnNo		: 4;
				uint16_t instr12	: 12;
			};
			uint16_t word;
		} instr1 = {.word = 0xfa80, };
		union {
			struct {
				uint16_t rmNo		: 4;
				uint16_t doubling	: 1;
				uint16_t subtract	: 1;
				uint16_t instr2		: 2;
				uint16_t rdNo		: 4;
				uint16_t instr4		: 4;
			};
			uint16_t word;
		} instr2 = {.word = 0xf080, };
		
		//no SP no PC
		if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_PC) | (1 << EMIT_REG_NO_SP)))
			return EmitErrNotEncodeable;
		
		instr1.rnNo = rnNo;
		instr2.rmNo = rmNo;
		instr2.doubling = doubling;
		instr2.subtract = subtract;
		instr2.rdNo = rdNo;
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
	
	enum EmitStatus emitLLqadd(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvSatAddSub(dest, rdNo, rnNo, rmNo, false, false);
	}
	
	enum EmitStatus emitLLqsub(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvSatAddSub(dest, rdNo, rnNo, rmNo, false, true);
	}
	
	enum EmitStatus emitLLqdadd(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvSatAddSub(dest, rdNo, rnNo, rmNo, true, false);
	}
	
	enum EmitStatus emitLLqdsub(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvSatAddSub(dest, rdNo, rnNo, rmNo, true, true);
	}
	
	static enum EmitStatus emitPrvDspMulsWXY(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t raNo, bool nTop, bool mTop, bool wide)
	{
		union {
			struct {
				uint16_t rnNo		: 4;
				uint16_t instr1		: 1;
				uint16_t wide		: 1;
				uint16_t instr12	: 10;
			};
			uint16_t word;
		} instr1 = {.word = 0xfb10, };
		union {
			struct {
				uint16_t rmNo		: 4;
				uint16_t mTop		: 1;
				uint16_t nTop		: 1;
				uint16_t instr2		: 2;
				uint16_t rdNo		: 4;
				uint16_t raNo		: 4;
			};
			uint16_t word;
		} instr2 = {.word = 0x0000, };
		
		instr1.rnNo = rnNo;
		instr1.wide = wide ? 1 : 0;
		instr2.rmNo = rmNo;
		instr2.mTop = mTop;
		instr2.nTop = nTop;
		instr2.rdNo = rdNo;
		instr2.raNo = raNo;
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
	
	enum EmitStatus emitLLsmulxy(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool nTop, bool mTop)
	{
		//no SP no PC
		if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_PC) | (1 << EMIT_REG_NO_SP)))
			return EmitErrNotEncodeable;
		
		return emitPrvDspMulsWXY(dest, rdNo, rnNo, rmNo, EMIT_REG_NO_PC, nTop, mTop, false);
	}
	
	enum EmitStatus emitLLsmlaxy(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t raNo, bool nTop, bool mTop)
	{
		//no SP no PC
		if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo) | (1 << raNo)) & ((1 << EMIT_REG_NO_PC) | (1 << EMIT_REG_NO_SP)))
			return EmitErrNotEncodeable;
		
		return emitPrvDspMulsWXY(dest, rdNo, rnNo, rmNo, raNo, nTop, mTop, false);
	}
	
	enum EmitStatus emitLLsmulwy(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool mTop)
	{
		//no SP no PC
		if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_PC) | (1 << EMIT_REG_NO_SP)))
			return EmitErrNotEncodeable;
		
		return emitPrvDspMulsWXY(dest, rdNo, rnNo, rmNo, EMIT_REG_NO_PC, false, mTop, true);
	}
	
	enum EmitStatus emitLLsmlawy(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t raNo, bool mTop)
	{
		//no SP no PC
		if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo) | (1 << raNo)) & ((1 << EMIT_REG_NO_PC) | (1 << EMIT_REG_NO_SP)))
			return EmitErrNotEncodeable;
		
		return emitPrvDspMulsWXY(dest, rdNo, rnNo, rmNo, raNo, false, mTop, true);
	}
	
	enum EmitStatus emitLLsmlalxy(struct EmitBuf *dest, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo, bool nTop, bool mTop)
	{
		union {
			struct {
				uint16_t rnNo		: 4;
				uint16_t instr12	: 12;
			};
			uint16_t word;
		} instr1 = {.word = 0xfbc0, };
		union {
			struct {
				uint16_t rmNo		: 4;
				uint16_t mTop		: 1;
				uint16_t nTop		: 1;
				uint16_t instr2		: 2;
				uint16_t rdHiNo		: 4;
				uint16_t rdLoNo		: 4;
			};
			uint16_t word;
		} instr2 = {.word = 0x0080, };
		
		//no SP no PC
		if (((1 << rdLoNo) | (1 << rdHiNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_PC) | (1 << EMIT_REG_NO_SP)))
			return EmitErrNotEncodeable;
		
		instr1.rnNo = rnNo;
		instr2.rmNo = rmNo;
		instr2.mTop = mTop;
		instr2.nTop = nTop;
		instr2.rdHiNo = rdHiNo;
		instr2.rdLoNo = rdLoNo;
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
	
	static enum EmitStatus emitPrvDspSimdAddSub(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool unsign, bool halving, bool saturating, bool sub, bool halfwords)
	{
		//No SP or PC allowed
		if (((1 << rdNo) | (1 << rnNo) | (1 << rmNo)) & ((1 << EMIT_REG_NO_SP) | (1 << EMIT_REG_NO_PC)))
			return EmitErrNotEncodeable;
		
		union {
			struct {
				uint16_t rnNo		: 4;
				uint16_t sz16		: 1;
				uint16_t instr1		: 1;
				uint16_t subtract	: 1;
				uint16_t instr12	: 9;
			};
			uint16_t word;
		} instr1 = {.word = 0xfa80, };
		union {
			struct {
				uint16_t rmNo		: 4;
				uint16_t q			: 1;
				uint16_t halve		: 1;
				uint16_t unsign		: 1;
				uint16_t instr1		: 1;
				uint16_t rdNo		: 4;
				uint16_t inst4		: 4;
			};
			uint16_t word;
		} instr2 = {.word = 0xf000, };
		
		instr1.rnNo = rnNo;
		instr1.sz16 = halfwords ? 1 : 0;
		instr1.subtract = sub ? 1 : 0;
		instr2.rmNo = rmNo;
		instr2.q = saturating ? 1 : 0;
		instr2.halve = halving ? 1 : 0;
		instr2.unsign = unsign ? 1 : 0;
		instr2.rdNo = rdNo;
		
		VERIFY_SPACE(2);
		EMIT_HALFWORD(instr1.word);
		EMIT_HALFWORD(instr2.word);
		return EmitErrNone;
	}
	
	enum EmitStatus emitLLuhadd16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, true, true, false, false, true);
	}
	
	enum EmitStatus emitLLuhadd8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, true, true, false, false, false);
	}
	
	enum EmitStatus emitLLuhsub16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, true, true, false, true, true);
	}
	
	enum EmitStatus emitLLuhsub8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, true, true, false, true, false);
	}
	
	enum EmitStatus emitLLshadd16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, false, true, false, false, true);
	}
	
	enum EmitStatus emitLLshadd8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, false, true, false, false, false);
	}
	
	enum EmitStatus emitLLshsub16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, false, true, false, true, true);
	}
	
	enum EmitStatus emitLLshsub8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, false, true, false, true, false);
	}
	
	enum EmitStatus emitLLuqadd16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, true, false, true, false, true);
	}
	
	enum EmitStatus emitLLuqadd8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, true, false, true, false, false);
	}
	
	enum EmitStatus emitLLuqsub16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, true, false, true, true, true);
	}
	
	enum EmitStatus emitLLuqsub8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, true, false, true, true, false);
	}
	
	enum EmitStatus emitLLqadd16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, false, false, true, false, true);
	}
	
	enum EmitStatus emitLLqadd8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, false, false, true, false, false);
	}
	
	enum EmitStatus emitLLqsub16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, false, false, true, true, true);
	}
	
	enum EmitStatus emitLLqsub8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, false, false, true, true, false);
	}
	
	enum EmitStatus emitLLuadd16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, true, false, false, false, true);
	}
	
	enum EmitStatus emitLLuadd8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, true, false, false, false, false);
	}
	
	enum EmitStatus emitLLusub16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, true, false, false, true, true);
	}
	
	enum EmitStatus emitLLusub8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, true, false, false, true, false);
	}
	
	enum EmitStatus emitLLsadd16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, false, false, false, false, true);
	}
	
	enum EmitStatus emitLLsadd8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, false, false, false, false, false);
	}
	
	enum EmitStatus emitLLssub16(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, false, false, false, true, true);
	}
	
	enum EmitStatus emitLLssub8(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo)
	{
		return emitPrvDspSimdAddSub(dest, rdNo, rnNo, rmNo, false, false, false, true, false);
	}
#endif

void* emitBufPtrToFuncPtr(void *buf)
{
	return (void*)(((uintptr_t)buf) + 1);
}