#include <PalmOS.h>
#include <SonyCLIE.h>
#include "PalmOSextra.h"
#include "SonyMSIO.h"
#include "deviceID.h"
#include "globals.h"
#include "asm.h"


#pragma GCC optimize ("Os")			//no need to waste space

#define LED_VIB_ATTR_TYPE				0		//value is readonly, u32, LED_VIB_TYPE_*
#define LED_VIB_ATTR_ACTIVE				1		//value is u8 (treated as boolean) as to whether this is on or off. can be turned on in middle of doing what was programmed. else will self turn on when number of cycles is completed as requested
#define LED_VIB_ATTR_RATE				2		//value is u16 specifying length of one cycle in units of 10ms (one cycle is 32 subcycles, each of which is described by a bit in "pattern" below
#define LED_VIB_ATTR_PATTERN			3		//value is u32, each bit is whether vib/led are on during that period. used big to little bits. eg: 0xFFFFFFFF=stay on, no vibrating,  0x0F0F0000=vibrate twice, then pause, 0x0F0F0F0F=vibrate four times in equal intervals
#define LED_VIB_ATTR_DELAY				4		//value is u16 - delay between cycles in units of 10ms
#define LED_VIB_ATTR_REPEAT_COUNT		5		//value is u16 - how many cycles to do when turned on
#define LED_VIB_ATTR_INTERRUPT_OFF		8		//value is ignored, writeonly, disables immediately

struct DriverRequirement {
	UInt32 dev;
	UInt16 minVer;
	const char dbName[16];	//can be up to dmDBNameLength, but we know ours is less, so we save space
};

struct DevInfoEntry {
	uint32_t devID;
	uint8_t screenRes		: 2;	//StdRes, HiRes, HiRes+
	uint8_t colorDepths		: 2;	//4, 8, 16
	uint8_t hasLed			: 1;
	uint8_t hasVib			: 1;
	uint8_t hasSound		: 1;
	uint8_t brokenMsio		: 1;
};


static const struct DriverRequirement mDriverReqs[]  = {
	{
		.dev = sonyHwrOEMDeviceIDYosemite2,
		.dbName = "MsSlot",
		.minVer = 200,
	},
	{
		.dev = sonyHwrOEMDeviceIDYosemite2,
		.dbName = "1859HandlerExtn",
		.minVer = 100,
	},
	{
		.dev = sonyHwrOEMDeviceIDYellowstone,
		.dbName = "MsSlot MB",
		.minVer = 200
	},
	{
		.dev = sonyHwrOEMDeviceIDVenice,
		.dbName = "MsSlot MB",
		.minVer = 300,
	},
	{
		.dev = sonyHwrOEMDeviceIDModena,
		.dbName = "MsSlot MB",
		.minVer = 300,
	},
};

static const struct DevInfoEntry mDevs[] = {
	
	#define STANDARD_RES		0
	#define HI_RES				1
	#define HI_RES_PLUS			2
	
	#define DEPTH_16_GRAYS		0
	#define DEPTH_256_COLORS	1
	#define DEPTH_64K			2
	
	#define NO_LED				0
	#define HAS_LED				1
	#define NO_VIBRATOR			0
	#define HAS_VIBRATOR		1
	#define NO_SOUND			0
	#define HAS_SOUND			1
	#define WORKING_MSIO		0	//raw tpcs broken?
	#define BROKEN_MSIO			1
	
	#define DEF_DEVICE_EX(_id, _res, _depth, _led, _vib, _snd, _brokenMsio)	{.devID = _id, .screenRes = _res, .colorDepths = _depth, .hasLed = _led, .hasVib = _vib, .hasSound = _snd, .brokenMsio = _brokenMsio, }
	#define DEF_DEVICE(_id, _res, _depth, _led, _vib, _snd)	DEF_DEVICE_EX(_id, _res, _depth, _led, _vib, _snd, WORKING_MSIO)

//thoise maerked with /**/ are untested
	
	//devices with MSIO support
	DEF_DEVICE(sonyHwrOEMDeviceIDModena,		HI_RES,			DEPTH_64K, 			HAS_LED,	HAS_VIBRATOR,	HAS_SOUND),		//PEG-T600C/15C/25C (OS 4.1)
/**/	DEF_DEVICE(sonyHwrOEMDeviceIDNaples,		HI_RES,			DEPTH_64K, 			HAS_LED,	HAS_VIBRATOR,	HAS_SOUND),		//PEG-T650C/65C/75C (OS 4.1)
	DEF_DEVICE(sonyHwrOEMDeviceIDFortaleza,		HI_RES,			DEPTH_16_GRAYS,		NO_LED,		NO_VIBRATOR,	NO_SOUND),		//PEG-SL10 (OS 4.1)
/**/	DEF_DEVICE(sonyHwrOEMDeviceIDCocos,			HI_RES,			DEPTH_16_GRAYS,		HAS_LED,	NO_VIBRATOR,	NO_SOUND),		//PEG-SJ20 (OS 4.1)
	DEF_DEVICE(sonyHwrOEMDeviceIDGalapagos,		HI_RES,			DEPTH_64K,			HAS_LED,	NO_VIBRATOR,	NO_SOUND),		//PEG-SJ22 (OS 4.1)
	DEF_DEVICE(sonyHwrOEMDeviceIDHawaii,		HI_RES,			DEPTH_64K,			HAS_LED,	NO_VIBRATOR,	NO_SOUND),		//PEG-SJ30 (OS 4.1)
/**/	DEF_DEVICE(sonyHwrOEMDeviceIDMcdonald,		HI_RES,			DEPTH_64K,			HAS_LED,	NO_VIBRATOR,	HAS_SOUND),		//PEG-SJ33 (OS 4.1)
	DEF_DEVICE(sonyHwrOEMDeviceIDRedwood,		HI_RES_PLUS,	DEPTH_64K,			HAS_LED,	NO_VIBRATOR,	HAS_SOUND),		//PEG-NR70/V (OS 4.1)
	
	//ACER devices. same msio api except no checking for "SoNy" creator ID. LOLz...
	DEF_DEVICE_EX(acerHwrOEMDeviceIDMP500,			HI_RES,			DEPTH_64K,			NO_LED,		NO_VIBRATOR,	NO_SOUND, BROKEN_MSIO),	//Acer s61/65 (OS 4.1)
	
	//devices shipped without MSIO support, but upgradeable using sony-provided drivers
	DEF_DEVICE(sonyHwrOEMDeviceIDYellowstone,	HI_RES,			DEPTH_64K, 			HAS_LED,	NO_VIBRATOR,	NO_SOUND),		//PEG-N600C/10C (OS 4.0)
	DEF_DEVICE(sonyHwrOEMDeviceIDVenice,		HI_RES,			DEPTH_16_GRAYS, 	HAS_LED,	HAS_VIBRATOR,	HAS_SOUND),		//PEG-T400/15/25 (OS 4.1)
	DEF_DEVICE(sonyHwrOEMDeviceIDYosemite2,		HI_RES,			DEPTH_64K, 			HAS_LED,	NO_VIBRATOR,	HAS_SOUND),		//PEG-N750C/60C/70C (OS 4.1)

	//devices without MSIO support due to software. but whose MSIO support i created
	DEF_DEVICE(sonyHwrOEMDeviceIDYosemite,		HI_RES,			DEPTH_256_COLORS, 	HAS_LED,	NO_VIBRATOR,	HAS_SOUND),		//PEG-N700C/10C (OS 3.5)		ysm2 driver might work... if upgraded to OS4 has 64K display too
	
	//devices with no ROM msio support and need driverless support
	DEF_DEVICE(acerHwrOEMDeviceIDMP300,			STANDARD_RES,	DEPTH_16_GRAYS,		NO_LED,		NO_VIBRATOR,	NO_SOUND),		//Acer s11/12/15 (OS 4.1)		s65 driver should work unmodified here
	DEF_DEVICE(sonyHwrOEMDeviceIDPda1Mono,		STANDARD_RES,	DEPTH_16_GRAYS,		NO_LED,		NO_VIBRATOR,	NO_SOUND),		//PEG-S300 (OS 3.5)
	DEF_DEVICE(sonyHwrOEMDeviceIDPda1Color,		STANDARD_RES,	DEPTH_256_COLORS, 	NO_LED,		NO_VIBRATOR,	NO_SOUND),		//PEG-S500C (OS 3.5)
	DEF_DEVICE(sonyHwrOEMDeviceIDNasca,			STANDARD_RES,	DEPTH_16_GRAYS,		NO_LED,		NO_VIBRATOR,	NO_SOUND),		//PEG-S320 (OS 4.0)				another sony driver might work?
	DEF_DEVICE(sonyHwrOEMDeviceIDNasca2,		STANDARD_RES,	DEPTH_16_GRAYS,		NO_LED,		NO_VIBRATOR,	NO_SOUND),		//PEG-S360 (OS 4.0)				another sony driver might work?
	
	#undef DEF_DEVICE
};

//		/**/ = tested at least once

//N610 MSIO driver might be optimized for faster reads
//N700/10/50/60 have a LED but used for charging only. no AttnMgr support
//N7xx have VERY slow MSIO due to old DSP firmware. 176Kbit/s is the speed. this is not likely to work well...
//s65 could likely go faster in driverless mode

enum DeviceIdResult deviceIdentify(struct Globals* g, struct MsioPktBootParams *bp, UInt32 *screenRxBufSzP)
{
	UInt8 i, nDevs = sizeof(mDevs) / sizeof(*mDevs), maxDepthShift;
	struct MsioScreenData *sd = NULL;
	UInt32 t, devID;
	Err e;
	
	g->hrLibRef = sysErrLibNotFound;
	g->silkLibRef = sysErrLibNotFound;
	g->deviceMsioFlags = 0;
	
	bp->hdr.pktTyp = MSIO_PKT_SET_BOOT_PARAMS;
	
	e = FtrGet(sysFtrCreator, sysFtrNumOEMDeviceID, &devID);
	if (e != errNone)
		return DeviceIdentificationError;
	
	//check for requirements
	for (i = 0; i < sizeof(mDriverReqs) / sizeof(*mDriverReqs); i++) {
		
		if (mDriverReqs[i].dev == devID) {
			
			LocalID LID = DmFindDatabase(0, mDriverReqs[i].dbName);
			UInt16 ver;
			
			if (!LID)
				return DeviceNeedsMissingDriver;
			
			e = DmDatabaseInfo(0, LID, NULL, NULL, &ver, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
			if (e != errNone)
				return DeviceIdentificationError;
			
			if (ver < mDriverReqs[i].minVer)
				return DeviceNeedsMissingDriver;
		}
	}
	
	//find device
	for (i = 0; i < nDevs && mDevs[i].devID != devID; i++);
	if (i == nDevs)
		return DeviceUnknown;
	
	if (mDevs[i].brokenMsio)
		g->deviceMsioFlags |= DEVICE_MSIO_FLAG_RAW_TPCS_BROKEN;
	
	bp->miscBits = 0;
	
	if (HwrBacklight(false, false))
		bp->miscBits |= MISC_BIT_BACKLIGHT_ON;
	
	switch (mDevs[i].screenRes) {
		case 0:
			bp->dispW = 160;
			bp->dispH = 160;
			break;
		
		case 1:
			bp->dispW = 320;
			bp->dispH = 320;
			bp->miscBits |= MISC_BIT_HIGH_DENSITY;
			break;
		
		case 2:
			bp->dispW = 320;
			bp->dispH = 480;
			bp->miscBits |= MISC_BIT_HIGH_DENSITY;
			break;
		
		default:
			return DeviceIdentificationError;
	}
	
	if (bp->miscBits & MISC_BIT_HIGH_DENSITY) {
		
		e = FtrGet(sysFtrCreator, sysFtrNumWinVersion, &t);
		if (e != errNone || t < 4) {	//no proper PalmOS high density API is vailable (it is in Acer s65)
		
			UInt32 val320 = 320;
			UInt16 hrLibRef;
					
			e = SysLibFind(sonySysLibNameHR, &hrLibRef);
			if (e == sysErrLibNotFound)
				e = SysLibLoad(sysFileTLibrary, sonySysFileCHRLib, &hrLibRef);
			
			if (e == errNone)
				e = HROpen(hrLibRef);
			
			if (e == errNone) {
				
				e = HRWinScreenMode(hrLibRef, winScreenModeSet, &val320, &val320, NULL, NULL);
				if (e != errNone) {
					
					if (errNone == HRClose(hrLibRef))
						SysLibRemove(hrLibRef);
				}
			}
			
			if (e != errNone)
				return DeviceUnknown;
			
			g->hrLibRef = hrLibRef;
		}
	}
	if (bp->dispW != bp->dispH) {
		UInt16 silkLibRef;
		
		e = SysLibFind(sonySysLibNameSilk, &silkLibRef);
		if (e == sysErrLibNotFound)
			e = SysLibLoad(sysFileTLibrary, sonySysFileCSilkLib, &silkLibRef);
		
		if (e == errNone)
			e = SilkLibOpen(silkLibRef);
		
		if (e == errNone) {
			
			e = SilkLibEnableResize(silkLibRef);
			if (e == errNone)
				e = SilkLibResizeDispWin(silkLibRef, silkResizeMax);
			
			if (e != errNone) {
				
				if (errNone == SilkLibClose(silkLibRef))
					SysLibRemove(silkLibRef);
			}
		}
	
		if (e != errNone)
			return DeviceUnknown;
		
		g->silkLibRef = silkLibRef;
	}
		
	switch (mDevs[i].colorDepths) {
		case 0:
			bp->supportedDepths = ((1UL << 1) | (1UL << 2) | (1UL << 4)) >> 1;
			maxDepthShift = 2;
			break;
		
		case 1:
			bp->supportedDepths = ((1UL << 1) | (1UL << 2) | (1UL << 4) | (1UL << 8)) >> 1;
			maxDepthShift = 3;
			break;
		
		case 2:
			bp->supportedDepths = ((1UL << 1) | (1UL << 2) | (1UL << 4) | (1UL << 8) | (1UL << 16)) >> 1;
			maxDepthShift = 4;
			break;
		
		default:
			return DeviceIdentificationError;
	}
	
	if (mDevs[i].hasLed)
		bp->miscBits += MISC_BIT_LED_SUPPORTED;
	if (mDevs[i].hasVib)
		bp->miscBits += MISC_BIT_VIB_SUPPORTED;
	if (mDevs[i].hasSound)
		bp->miscBits += MISC_BIT_AUDIO_SUPPORTED;
	
	e = WinScreenMode(winScreenModeGet, NULL, NULL, &t, NULL);
	if (e != errNone)
		return e;

	bp->curDepth = t;
	bp->hwrMiscFlags = (errNone == FtrGet(sysFtrCreator, sysFtrNumHwrMiscFlags, &t)) ? t : 0;
	bp->hwrMiscExtFlags = (errNone == FtrGet(sysFtrCreator, sysFtrNumHwrMiscFlagsExt, &t)) ? t : 0;
	
	t = (UInt32)(UInt16)bp->dispW * (UInt32)(UInt16)bp->dispH;				//num pixels
	t = (t << maxDepthShift) >> 3;											//num bytes in a full screen image
	t = (t * 257 + 255) / 256 + 1;											//num bytes in largest possible compressed screen image
	*screenRxBufSzP = t;
		
	//driverless support?
	switch (driverlessCheck(g, devID)) {
		case DriverlessDeviceUnknown:
		case DriverlessDeviceKnownAndSupported:
			return DeviceOK;
		
		case DriverlessDeviceKnownAndError:
		default:
			return DeviceNeedsMissingDriver;
	}
}
