/*
	(c) 2021 Dmitry Grinberg   https://dmitry.gr
	Non-commercial use only OR licensing@dmitry.gr
*/


#pragma GCC optimize ("Os")


#include <string.h>
#include <stdarg.h>
#include "printf.h"


#ifdef _CPU_HAS_32_BIT_DIV_

	static uint_fast8_t prvDiv10(uint64_t *valP)
	{
		uint64_t val = *valP;
		uint32_t retHi, retMid, retLo, retChar;
		
		retHi = val >> 32;
		retMid = retHi % 10;
		retHi /= 10;
		
		retMid = (retMid << 16) + (uint16_t)(val >> 16);
		retLo = retMid % 10;
		retMid /= 10;
		
		retLo = (retLo << 16) + (uint16_t)val;
		retChar = retLo % 10;
		retLo /= 10;
		
		val = retHi;
		val <<= 16;
		val += retMid;
		val <<= 16;
		val += retLo;
		
		*valP = val;
		
		return retChar;
	}
	
	static void hexToDec(uint8_t *dst, uint64_t val)	//dst must be 10 bytes
	{
		uint_fast8_t i, prev = 0;
		
		for (i = 0; i < 20; i++) {
			
			uint_fast8_t now = prvDiv10(&val);
			
			if (i & 1) {
				now = now * 16 + prev;
				dst[i / 2] = now;
			}
			else
				prev = now;
			
		}
	}
#else

	static void hexToDec(uint8_t *dst, uint64_t val)	//double-dabble algorithm, dst must be 10 bytes
	{
		uint_fast8_t i, j;
		
		for (i = 0; i < 10; i++)
			dst[i] = 0;
		
		for (i = 0; i < 64; i++) {
			
			uint_fast8_t shiftIn;
			
			for (j = 0; j < 10; j++) {
				
				if (dst[j] >= 0x50)
					dst[j] += 0x30;
				
				if ((dst[j] & 0x0f) >= 5)
					dst[j] += 3;
			}
			shiftIn = val >> 63;
			val <<= 1;
			
			for (j = 0; j < 10; j++) {
				
				shiftIn += 2 * dst[j];
				dst[j] = shiftIn;
				shiftIn >>= 8;
			}
		}
	}
#endif

static void copyOver(uint8_t *dst, uint64_t val)	//dst must be 10bytes
{
	uint_fast8_t i;
	
	for (i = 0; i < 10; i++, val >>= 8)
		dst[i] = val & 0xff;
}

static uint8_t getBcdDigit(const uint8_t *buf, uint_fast8_t digit)
{
	uint_fast8_t t = buf[digit / 2];
	
	if (digit & 1)
		return t >> 4;
	else
		return t & 0x0f;
}

static void StrPrvPrintfEx_number(uint64_t number, bool baseTen, bool zeroExtend, bool isSigned, uint32_t padToLength)
{
	uint_fast8_t numLen, printedLen, padChars, i;
	bool wasNegative = false;
	uint8_t buf[10];
	
	if (isSigned && ((int64_t)number) < 0) {
		
		wasNegative = true;
		number = -number;
	}
	
	if (baseTen)
		hexToDec(buf, number);
	else
		copyOver(buf, number);
	
	//find the length of the number (1 minimum)
	for (numLen = sizeof(buf) * 2; numLen >= 2 && !getBcdDigit(buf, numLen - 1); numLen--);
	
	printedLen = numLen + (wasNegative ? 1 : 0);
	
	if (printedLen > padToLength)
		padChars = 0;
	else
		padChars = padToLength - printedLen;
	
	//padding works differently for zeroes and for spaced w.r.t. the minus sign
	
	if (zeroExtend && wasNegative)
		prPutchar('-');
	while (padChars--)
		prPutchar(zeroExtend ? '0' : ' ');
	if (!zeroExtend && wasNegative)
		prPutchar('-');
	for (i = 0; i < numLen; i++) {
		char ch = getBcdDigit(buf, numLen - 1 - i);
		
		ch = (ch < 10) ? ch + '0' : ch + 'A' - 10;
		prPutchar(ch);
	}
}

static uint32_t StrVPrintf_StrLen_withMax(const char* s, uint32_t max)
{
	uint32_t len = 0;
	
	while ((*s++) && (len < max))
		len++;
	
	return len;
}

void prvRaw(const char* fmtStr, va_list vl)
{	
	uint64_t val64;
	char c, t;
	
	while((c = *fmtStr++) != 0){
		
		if (c == '%') {
			
			bool zeroExtend = false, useLong = false, useVeryLong = false, isSigned = false, baseTen = false;
			uint32_t padToLength = 0, len, i;
			const char* str;
			
more_fmt:
			switch (c = *fmtStr++) {
				
				case '%':
				default:	
					prPutchar(c);
					break;
				
				case 'c':
					
					t = va_arg(vl,unsigned int);
					prPutchar(t);
					break;
				
				case 's':
					
					str = va_arg(vl,char*);
					if (!str)
						str = "(null)";
					if (padToLength)
						len = StrVPrintf_StrLen_withMax(str, padToLength);
					else
						padToLength = len = strlen(str);
					
					for (i = len; i < padToLength; i++)
						prPutchar(' ');
					for (i = 0; i < len; i++)
						prPutchar(*str++);
					break;
				
				case '0':
					
					if (!zeroExtend && !padToLength) {
						
						zeroExtend = true;
						goto more_fmt;
					}
					//fallthrough
				
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
					
					padToLength = (padToLength * 10) + c - '0';
					goto more_fmt;
				
				case 'd':
					isSigned = true;
					//fallthrough
					
				case 'u':
					baseTen = true;
					//fallthrough
					
				case 'x':
				case 'X':
					val64 = useVeryLong ? va_arg(vl,uint64_t) : va_arg(vl,uint32_t);
					if (isSigned && !useVeryLong)
						val64 = (int64_t)(int32_t)val64;
					StrPrvPrintfEx_number(val64, baseTen, zeroExtend, isSigned, padToLength);
					break;
					
				case 'l':
					if(useLong)
						useVeryLong = true;
					useLong = true;
					goto more_fmt;

			}
		}
		else
			prPutchar(c);
	}
}

void prRaw(const char *format, ...)
{
	va_list vl;
	
	va_start(vl, format);
	prv(format, vl);
	va_end(vl);
}
