#include <PalmOS.h>
#include <SerialMgrOld.h>
#include <HsExt.h>
#include <stdint.h>
#include <stdarg.h>
#include "visorCommsProto.h"


//missing defines (as per emulator srcs)
Boolean	HwrBacklight(Boolean set, Boolean newState)	SYS_TRAP(sysTrapHwrBacklightV33);


#define USE_SCREEN_LOCKING				0

#define MAX_CARD_NUMBER_TO_GUESS		16

#define RX_BUF_SZ						16384						//we have a lot of RAM. let's go wild!

enum LedState {
	LedStateOff,
	LedStateInPattern,
	LedStateInterPattern,
};


struct CommsInfo {
	UInt16 cardNo;
	UInt16 *dataPtr;
	UInt16 *signalPtr;
	
	volatile UInt16 *dataOut;
	UInt16 dataOutWords;

	volatile UInt16 *dataIn;
	UInt16 dataInWords;
	
	void *outMsg;
	void *inMsg;
	
	UInt32 prevKeyState;
	Int16 prevPenX, prevPenY;
	
	UInt16 serLibRef;
	void *serLibBuffer;
	
	UInt8 *scrCompressedData;
	UInt32 scrCompressedSz;
	
	//LED
	volatile UInt8* ledAddr;
	UInt8 ledBit;
	enum LedState ledState;
	UInt8 pattBit;
	UInt32 ledPattern;
	UInt16 ledCsecPerPiece;
	UInt16 ledCsecBetween;
	UInt16 ledNumTimes;
	UInt32 ledDue;
	
};

#define logStart(...)		//logStart_(__VA_ARGS__)
#define log(...)			//log_(__VA_ARGS__)
#define logBytes(...)		//logBytes_(__VA_ARGS__)


void scrCopy(void* dst, void* src, UInt16 nWords);
uint8_t* screenDecompress(uint8_t *dst, const uint8_t *src, uint32_t srcLen);	


static void msgbox(const char *fmt, ...)
{
	char x[128];
	va_list vl;
	
	va_start(vl, fmt);
	StrVPrintF(x, fmt, vl);
	va_end(vl);
	
	FrmCustomAlert(10024, x, NULL, NULL);
}

static void logStart_(void)
{
	UInt16 len, ofst = 0;
	DmOpenRef db;
	MemHandle mh;
	

	db = DmOpenDatabase(0, DmFindDatabase(0, "MemoDB"), dmModeReadWrite);
	if (db) {
		

		UInt16 at = dmMaxRecordIndex;
			
		while (DmNumRecords(db))
			DmRemoveRecord(db, 0);
			
		(void)DmNewRecord(db, &at, 1);
			
		DmReleaseRecord(db, 0, true);
		DmCloseDatabase(db);
	}
}

static void log_(const char *fmt, ...)
{
	UInt16 len, ofst = 0;
	DmOpenRef db;
	MemHandle mh;
	char x[128];
	va_list vl;
	char *mp;
	
	va_start(vl, fmt);
	len = StrVPrintF(x, fmt, vl);
	va_end(vl);
	
	db = DmOpenDatabase(0, DmFindDatabase(0, "MemoDB"), dmModeReadWrite);
	if (db) {
		
		mh = DmQueryRecord(db, 0);
		ofst = MemHandleSize(mh);
		//skip trailing zeroes
		mp = MemHandleLock(mh);
		while (ofst && mp[ofst - 1] == 0)
			ofst--;
		MemHandleUnlock(mh);
		mh = DmResizeRecord(db, 0, ofst + len);
		
		DmWrite(MemHandleLock(mh), ofst, x, len);
		MemHandleUnlock(mh);
			
		DmReleaseRecord(db, 0, true);
		DmCloseDatabase(db);
	}
}

static char hexch(UInt8 v)
{
	if (v < 10)
		return '0' + v;
	else if (v < 16)
		return 'A' - 10 + v;
	else
		return '?';
}

static void logBytes_(const char *name, const void *bytes, UInt16 len)
{
	const UInt8 *b = (const UInt8*)bytes;
	UInt16 i, j;
	
	log("%s(%u bytes):\n", name, len);
	for (i = 0; i < len; i += 16) {
		log("%24x: ", i);
		for (j = 0; j < 16 && i+ j < len; j++)
			log(" %c%c", hexch(b[j + i] >> 4), hexch(b[j + i] & 15));
		log("\n");
	};
}

static UInt16 mailboxMsg(struct CommsInfo *ci, UInt16 message)
{
	volatile UInt16 *dataPtr = ci->dataPtr;
	volatile UInt16 *signalPtr = ci->signalPtr;
	UInt16 t;

	for (t = 0; t < 64; t++)
		dataPtr[1] = message;
	for (t = 0; t < 64; t++)
		dataPtr[3] = MSG_MRK_REQUEST;
	
	while (1) {
		
		do {
			t = dataPtr[3];
		} while (dataPtr[3] != t);
		
		if (t == MSG_MRK_REQUEST)
			(void)signalPtr[0];
		else
			break;
	}
	
	if (t != MSG_MRK_REPLY)
		return MSG_S_NOT_UNDERSTOOD;
	
	do {
		t = dataPtr[1];
	} while (dataPtr[1] != t);
	
	if (t == 0)
		msgbox("got %04x, now %04x", t, dataPtr[1]);
	
	return t;
}

static UInt16 mailboxMsgBasic(struct CommsInfo *ci, UInt16 message)
{
	return mailboxMsg(ci, message);
}

static UInt16 mailboxMsgComplex(struct CommsInfo *ci, UInt16 message, UInt16 bytesToCopyOut)
{
	UInt16 wordToCopyOut = (bytesToCopyOut + 1) / 2;
	const UInt16 *src = (const UInt16*)ci->outMsg;
	volatile UInt16 *dst = ci->dataOut;

	while (wordToCopyOut--)
		*dst++ = *src++;

	return mailboxMsg(ci, message);
}

static UInt32 isOsAtLeast(UInt32 desiredVer)
{
	UInt32 ver;
	
	return (errNone == FtrGet(sysFtrCreator, sysFtrNumROMVersion, &ver) && ver >= desiredVer);
}

static Int32 ledGetEvtGettingTimeout(const struct CommsInfo *ci)
{
	UInt32 due = ci->ledDue, now = TimGetTicks();
	
	if (!due)
		return -1;
	
	if (due <= now)
		return 0;
	
	return due - now;
}

static void ledOn(const struct CommsInfo *ci, Boolean on)
{
	if (on)
		*ci->ledAddr |= ci->ledBit;
	else
		*ci->ledAddr &=~ ci->ledBit;
}

static void ledProcess(struct CommsInfo *ci)
{
	UInt32 due, now;
	
	while ((due = ci->ledDue) && (now = TimGetTicks()) >= due) {
	
		switch (ci->ledState) {
			case LedStateOff:
				ci->ledDue = 0;
				ledOn(ci, false);
				break;
			
			case LedStateInterPattern:
				ci->ledState = LedStateInPattern;
				ci->pattBit = 31;
				ci->ledDue += ci->ledCsecPerPiece;
				ledOn(ci, (ci->ledPattern >> ci->pattBit) & 1);
				break;
			
			case LedStateInPattern:
				if (ci->pattBit) {
					ci->pattBit--;
					ci->ledDue += ci->ledCsecPerPiece;
					ledOn(ci, (ci->ledPattern >> ci->pattBit) & 1);
				}
				else if (ci->ledNumTimes > 1) {
					ci->ledState = LedStateInterPattern;
					ci->ledDue += ci->ledCsecBetween;
					ledOn(ci, 0);
					ci->ledNumTimes--;
				}
				else {
					ci->ledState = LedStateOff;
					ci->ledNumTimes = 0;
					ci->ledDue = 0;
				}
				break;
		}		
	}
}

static void __attribute__((noinline)) handleFwUpdate(struct CommsInfo *ci)
{
	RectangleType rect, rectFull;
	UInt16 progress;
	char x[8];
	FormPtr f;
	
	f = FrmNewForm(1000, "Updating...", 10, 50, 140, 60, true, 0, 0, 0);
	if (!f)
		goto fail;
	
	if (!FrmNewLabel(&f, 1001, "Do not power off...", 1, 15, largeBoldFont))
		goto fail;
	
	if (!FrmNewLabel(&f, 1002, "Device will reset after the update", 1, 28, stdFont))
		goto fail;
	
	FrmSetActiveForm(f);
	FrmDrawForm(f);
	
	RctSetRectangle(&rectFull, 4, 43, 132, 14);
	WinDrawRectangleFrame(simpleFrame, &rectFull);
	WinEraseRectangle(&rectFull, 1);
	RctInsetRectangle(&rectFull, 1);
	rect = rectFull;
	FntSetFont(boldFont);
	
	do {
		UInt16 len;
		
		progress = ci->dataPtr[0];
		len = StrPrintF(x, "%lu%%", progress);
		rect.extent.x = (UInt16)((UInt16)13 * progress + (UInt16)5) / (UInt16)10;
		WinEraseRectangle(&rectFull, 0);
		WinDrawChars(x, len, rectFull.topLeft.x + (UInt16)(rectFull.extent.x - FntCharsWidth(x, len)) / 2, rectFull.topLeft.y);
		WinInvertRectangle(&rect, 0);
		SysTaskDelay(50);
		
	} while(progress != 100);
	
	SysReset();
	
	fail:
		SysFatalAlert("Failed to show update ui\n");
}


//ErrAlertCustom() is OS3.2+, and we need to operate on 3.1
static void MyErrAlertCustom(Err errCode, char *errMsgP, char *preMsgP, char *postMsgP)
{
	#define TEMP_DB_NAME "__tmp__"
	
	UInt32 ver;
	
	if (isOsAtLeast(sysMakeROMVersion(3,2,0,sysROMStageRelease,0))) {
		
		(void)ErrAlertCustom(errCode, errMsgP, preMsgP, postMsgP);
	}
	else {	//this gets hard fast...
		
		UInt16 len = 0;
		char *str;
		
		if (errMsgP)
			len += StrLen(errMsgP);
		if (preMsgP)
			len += StrLen(preMsgP);
		if (postMsgP)
			len += StrLen(postMsgP);
		
		str = MemPtrNew(len + 32);
		if (str) {
			
			DmOpenRef db;
			LocalID LID;
			UInt16 attr;
			MemHandle mh;
			char* dst;
			
			str[0] = 0;
			
			if (errCode)
				StrPrintF(str + StrLen(str), "(Err 0x%24x) ", errCode);
			
			if (preMsgP)
				StrCat(str, preMsgP);
			if (errMsgP)
				StrCat(str, errMsgP);
			if (postMsgP)
				StrCat(str, postMsgP);
				
			if (errNone == DmCreateDatabase(0,TEMP_DB_NAME,'psys','TEMP',true)) {
				
				LID = DmFindDatabase(0, TEMP_DB_NAME);
				
				if (LID) {
					
					db = DmOpenDatabase(0, LID, dmModeReadWrite);
					
					if (db) {
						
						if (errNone == DmDatabaseInfo(0,LID,NULL,&attr,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL)) {
							
							attr |= dmHdrAttrRecyclable;
							
							if (errNone == DmSetDatabaseInfo(0,LID,NULL,&attr,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL,NULL)) {
								
								mh = DmNewResource(db, 'tSTR', 1000, StrLen(str) + 1);
								
								if (mh) {
									
									dst = MemHandleLock(mh);
									
									if (dst){
										
										if (errNone == DmStrCopy(dst, 0, str))
											FrmHelp(1000);
										
										MemHandleUnlock(mh);
									}
									DmReleaseResource(mh);
								}
							}
						}
						DmCloseDatabase(db);
					}
				}
				
				LID = DmFindDatabase(0,TEMP_DB_NAME);
				
				if (LID)		//still not deleted? why?
					DmDeleteDatabase(0,LID);
			}
						
			MemPtrFree(str);
		}
	}
}

//WinScreenLock() is OS3.5+, and we need to operate on 3.1
static UInt8* MyWinScreenLock(WinLockInitType initMode)
{
	if (!isOsAtLeast(sysMakeROMVersion(3,5,0,sysROMStageRelease,0)))
		return NULL;
	
	return WinScreenLock(initMode);
}

static Boolean prepareBoot(struct CommsInfo *ci)
{
	struct MsgContinueBoot *msg = (struct MsgContinueBoot*)ci->outMsg;
	UInt32 maxDepth, depth = 0, w, h, t;
	UInt16 i, ret;

	//get supported depths, w, h
	if (errNone != WinScreenMode(winScreenModeGetSupportedDepths, NULL, NULL, &depth, NULL))
		depth = 1;	//1bpp always supported
	msg->supportedDepths = depth;

	//find max supported depth
	for (maxDepth = 16; maxDepth; maxDepth >>= 1) {
		if (depth & (1 << (maxDepth - 1)))
			break;
	}
	
	//go to max supported depth (but no more than 8bpp)
	depth = maxDepth > 8 ? 8 : maxDepth;
	
	if (errNone != WinScreenMode(winScreenModeSet, NULL, NULL, &depth, NULL))
		return false;

	if (errNone != WinScreenMode(winScreenModeGet, &w, &h, &depth, NULL))
		return false;
	msg->curDepth = depth;
	msg->dispW = w;
	msg->dispH = h;

	//allocate compressed screen buffer
	ci->scrCompressedData = MemPtrNew((257UL * (w * h * maxDepth / 8) + 255) / 256 + 1 + 4/* overage */);
	if (!ci->scrCompressedData) {
	
		MyErrAlertCustom(0,  "compressed LCD buffer", "Out Of Memory allocating ", NULL);
		return false;
	}
	
	//get misc flag
	msg->hwrMiscFlags = (errNone == FtrGet(sysFtrCreator, sysFtrNumHwrMiscFlags, &t)) ? t : 0;
	msg->hwrMiscExtFlags = (errNone == FtrGet(sysFtrCreator, sysFtrNumHwrMiscFlagsExt, &t)) ? t : 0;
	
	//other bits
	msg->miscBits = 0;
	
	if (errNone == FtrGet(sysFtrCreator, sysFtrNumOEMDeviceID, &t)){
		
		switch (t) {
			case 0x09:	//edge
				ci->ledAddr = (volatile UInt8*)0xFFFFF441;
				ci->ledBit = 0x10;
				msg->miscBits |= MISC_BIT_LED_SUPPORTED;
				break;
		
			case 0x0A:	//prism
				ci->ledAddr = (volatile UInt8*)0xFFFFF411;
				ci->ledBit = 0x80;
				msg->miscBits |= MISC_BIT_LED_SUPPORTED;
				break;
			
			case 0x8B:	//pro
				ci->ledAddr = (volatile UInt8*)0xFFFFF441;
				ci->ledBit = 0x08;
				msg->miscBits |= MISC_BIT_LED_SUPPORTED;
				break;
			
			default:
				break;
		}
	}
	
	if (HwrBacklight(false, false))
		msg->miscBits |= MISC_BIT_BACKLIGHT_ON;
	
	return MSG_C_ACK == mailboxMsgComplex(ci, MSG_C_CONTINUE_BOOT, sizeof(*msg));
}

static void irqSetState(struct CommsInfo *ci, UInt8 on)
{
	(void)HsCardAttrSet(ci->cardNo, hsCardAttrIntEnable, &on);
}

static void irqHandler(UInt32 cardParam, Boolean* sysAwakeP)
{
	struct CommsInfo *ci = (struct CommsInfo*)cardParam;
	
	(void)EvtWakeup();
	irqSetState(ci, false);		//irqs off till we handle it
}

static void prepareBattInfoMsg(struct CommsInfo *ci)
{
	struct MsgBatteryInfo *bi = (struct MsgBatteryInfo*)ci->outMsg;
	SysBatteryKind kind;
	Boolean pluggedIn;
	UInt8 percent;
	
	bi->centivolts = SysBatteryInfo(false,NULL, NULL, NULL, &kind, &pluggedIn, &percent);
	bi->flags = ((((UInt16)percent) << BATT_INFO_PERCENT_SHIFT) & BATT_INFO_PERCENT_MASK) | 
				 (pluggedIn ? BATT_INFO_PLUGGED_IN_MASK : 0) |
				 ((((UInt16)kind) << BATT_INFO_KIND_SHIFT) & BATT_INFO_KIND_MASK);
}

static void* getDisplayWriteAddress(void)	//where to write the data?
{
	if (isOsAtLeast(sysMakeROMVersion(3,5,0,sysROMStageRelease,0)))
		return BmpGetBits(WinGetBitmap(WinGetDisplayWindow()));
	else
		return *(void**)0xFFFFFA00;			//i shit you not (all <3,1 devices are EZs using built-in controller. Their blitter also hides the sxcreen base address well. no API gets it
}

static void mySerWakeupHandler(UInt32 refCon)
{
	(void)EvtWakeup();
}

static void serialOp(struct CommsInfo *ci)
{
	const struct MsgSerOp *msgIn = (const struct MsgSerOp*)ci->dataIn;
	struct MsgSerOp *msgOut = (struct MsgSerOp*)ci->outMsg;
	struct SerSettingsType settings;
	Boolean ctsOn, dsrOn;
	UInt16 val16, len;
	UInt32 val32;
	void* data;
	Err e;
	
	msgOut->op = SerOpFailure;					//by default
	
	switch (msgIn->op) {
		case SerOpOpen:
			if (ci->serLibRef)
				break;
			e = SysLibFind("Serial Library", &ci->serLibRef);
			if (e || !ci->serLibRef)
				break;
			e = SerOpen(ci->serLibRef, msgIn->data.open.port, (((UInt32)msgIn->data.open.baudHi) << 16) | msgIn->data.open.baudLo);
			if (e)
				break;
			
			ci->serLibBuffer = MemChunkNew(0, RX_BUF_SZ + 32, 0x1200);
			SerSetReceiveBuffer(ci->serLibRef, ci->serLibBuffer, RX_BUF_SZ); 
			
			e = SerSetWakeupHandler(ci->serLibRef, &mySerWakeupHandler, (UInt32)ci);
			if (e)
				break;
			
			e = SerPrimeWakeupHandler(ci->serLibRef, 1);
			if (e)
				break;
			
			msgOut->op = SerOpSuccess;
			break;
		
		case SerOpClose:
			if (!ci->serLibRef)
				break;
			
			SerSetReceiveBuffer(ci->serLibRef, NULL, 0); 
			MemChunkFree(ci->serLibBuffer);
			
			e = SerClose(ci->serLibRef);
			if (e)
				break;
			ci->serLibRef = 0;
			msgOut->op = SerOpSuccess;
			break;
		
		case SerOpSetSettings:
			if (!ci->serLibRef)
				break;
			settings.baudRate = (((UInt32)msgIn->data.settings.baudHi) << 16) | msgIn->data.settings.baudLo;
			settings.flags = msgIn->data.settings.flags &~ (SER_FLAG_IR_RX_ON | SER_FLAG_IR_MODE);
			settings.ctsTimeout = (((UInt32)msgIn->data.settings.ctsTimeoutHi) << 16) | msgIn->data.settings.ctsTimeoutLo;
			e = SerSetSettings(ci->serLibRef, &settings);
			if (e)
				break;
			
			if (msgIn->data.settings.flags & SER_FLAG_IR_MODE) {
				
				e = SerControl(ci->serLibRef, serCtlIrDAEnable, NULL, NULL);
				if (e == errNone)
					e = SerControl(ci->serLibRef, (msgIn->data.settings.flags & SER_FLAG_IR_RX_ON) ? serCtlRxEnable : serCtlRxDisable, NULL, NULL);
			}
			else
				e = SerControl(ci->serLibRef, serCtlIrDADisable, NULL, NULL);
			
			if (e)
				break;
			
			msgOut->op = SerOpSuccess;
			break;
		
		case SerOpGetSettings:
			if (!ci->serLibRef)
				break;
			e = SerGetSettings(ci->serLibRef, &settings);
			if (e)
				break;
			msgOut->data.settings.baudHi = settings.baudRate >> 16;
			msgOut->data.settings.baudLo = settings.baudRate;
			msgOut->data.settings.flags = settings.flags;
			msgOut->data.settings.ctsTimeoutHi = settings.ctsTimeout >> 16;
			msgOut->data.settings.ctsTimeoutLo = settings.ctsTimeout;
			msgOut->op = SerOpCurSettings;
			break;
		
		case SerOpGetState:
			if (!ci->serLibRef)
				break;
			msgOut->data.state.errors = SerGetStatus(ci->serLibRef, &ctsOn, &dsrOn);
			msgOut->data.state.ctsOn = ctsOn;
			msgOut->data.state.dsrOn = dsrOn;
			msgOut->op = SerOpCurState;
			break;
		
		case SerOpControl:
			if (!ci->serLibRef)
				break;
			switch (msgIn->data.control.selector) {
				case serCtlStartBreak:				//no params in or out
				case serCtlStopBreak:
				case serCtlStartLocalLoopback:
				case serCtlStopLocalLoopback:
				case serCtlIrDAEnable:
				case serCtlIrDADisable:
				case serCtlIrScanningOn:
				case serCtlIrScanningOff:
				case serCtlRxEnable:
				case serCtlRxDisable:
					data = NULL;
					len = 0;
					break;
				
				case serCtlBreakStatus:				//UInt16 out
					data = &val16;
					len = sizeof(val16);
					break;
				
				case serCtlMaxBaud:					//UInt32 out
				case serCtlHandshakeThreshold:
					data = &val32;
					len = sizeof(val32);
					break;
				
				default:
					data = NULL;
					len = 0xffff;
					break;
			}
			if (len == 0xffff)
				break;
			e = SerControl(ci->serLibRef, msgIn->data.control.selector, data, &len);
			log("SerControl(libref, %d, ...) => %24x", msgIn->data.control.selector, e);
			msgOut->data.control.selector = msgIn->data.control.selector;
			msgOut->data.control.err = e;
			switch (msgOut->data.control.selector) {
				case serCtlBreakStatus:				//UInt16 out
					msgOut->data.control.valHi = 0;
					msgOut->data.control.valLo = val16;
					break;
				
				case serCtlMaxBaud:					//UInt32 out
				case serCtlHandshakeThreshold:
					msgOut->data.control.valHi = val32 >> 16;
					msgOut->data.control.valLo = val32;
					break;
			}
			msgOut->op = SerOpControlReply;
			break;
		
		case SerOpSendWait:
			if (!ci->serLibRef)
				break;
			msgOut->op = SerSendWait(ci->serLibRef, -1) == errNone ? SerOpSuccess : SerOpFailure;
			break;
		
		default:
			break;
	}
}

static void serialTx(struct CommsInfo *ci)
{
	struct MsgSerialDataTxReply *msgOut = (struct MsgSerialDataTxReply*)ci->outMsg;
	const struct MsgSerialData *msgIn = (const struct MsgSerialData*)ci->dataIn;
	UInt16 wantToSend = (msgIn->bytesAndFlags & MSD_BYTES_AND_FLAGS_BYTES_MASK) >> MSD_BYTES_AND_FLAGS_BYTES_SHIFT;
	Boolean autoTx = !!(msgIn->bytesAndFlags & MSD_FLAG_AUTOTX);	//no verifiction is done. assumes caller is sane and onyl asks this of an IR port in RX mode
	UInt32 sent = 0;
	Err e;
	
	if (ci->serLibRef) {
		
		const UInt8* dataP = (const UInt8*)msgIn->data;	// 68K is big endian so this works as per spec - high byte first
		UInt32 done = 0;
		
		logBytes("TX", dataP, wantToSend);
		
		if (autoTx)
			SerControl(ci->serLibRef, serCtlRxDisable, NULL, NULL);
		
		//save the round trips - send it all now
		while (wantToSend) {
		
			sent = SerSend(ci->serLibRef, dataP + done, wantToSend - done, &e);
			if (e != errNone)
				break;
			
			done += sent;
			wantToSend -= sent;
		}
		
		e = SerSendWait(ci->serLibRef, -1);	//wait for tx to end before turning on RX
		
		if (autoTx)
			SerControl(ci->serLibRef, serCtlRxEnable, NULL, NULL);
		
		msgOut->sentBytesHi = done >> 16;
		msgOut->sentBytesLo = done;
		msgOut->errVal = e;
	}
	else {
	
		msgOut->errVal = 0xffff;
		msgOut->sentBytesHi = 0;
		msgOut->sentBytesLo = 0;
	}
}

static void serialMaybeSendMessage(struct CommsInfo *ci, UInt16 *replyMsgP, UInt16 *bytesToCopyOutP)
{
	UInt32 bytesAvail, bytesWeCanSendToSpringboard = ci->dataOutWords * sizeof(UInt16) - sizeof(struct MsgSerialData);
	struct MsgSerialData *msgOut = (struct MsgSerialData*)ci->outMsg;
	UInt8 *dst = (UInt8*)msgOut->data, *t = dst; //68K is big endian so this works as per spec - high byte first
	Boolean doSend = false;
	UInt16 sta = 0;
	Err e;
	
	if (!ci->serLibRef)
		return;
	
	if (bytesWeCanSendToSpringboard > (MSD_BYTES_AND_FLAGS_BYTES_MASK >> MSD_BYTES_AND_FLAGS_BYTES_SHIFT))
		bytesWeCanSendToSpringboard = MSD_BYTES_AND_FLAGS_BYTES_MASK >> MSD_BYTES_AND_FLAGS_BYTES_SHIFT;
	
	msgOut->bytesAndFlags = 0;
	*bytesToCopyOutP = sizeof(struct MsgSerialData);

	SerPrimeWakeupHandler(ci->serLibRef, 1);	//re-prime

	do {
		Boolean ctsOn, dsrOn;
		UInt8 *data;
		UInt16 now;
		
		e = SerReceiveWindowOpen(ci->serLibRef, &data, &bytesAvail);
		if (e == serErrLineErr) {
			sta |= SerGetStatus(ci->serLibRef, &ctsOn, &dsrOn);
			(void)SerClearErr(ci->serLibRef);
			bytesWeCanSendToSpringboard = 0;	//break out of the loop
			doSend = true;
		}
		else if (e == errNone) {
			
			if (bytesAvail > bytesWeCanSendToSpringboard)
				bytesAvail = bytesWeCanSendToSpringboard;
			
			bytesWeCanSendToSpringboard -= bytesAvail;
			now = bytesAvail;
			
			if (now) {
				while (now--)
					*t++ = *data++; 
			
				doSend = true;
			}
		}
		(void)SerReceiveWindowClose(ci->serLibRef, bytesAvail);
		
	} while (bytesAvail && bytesWeCanSendToSpringboard);
	
	(*bytesToCopyOutP) += t - dst;
	msgOut->bytesAndFlags = (msgOut->bytesAndFlags &~ MSD_BYTES_AND_FLAGS_BYTES_MASK) | (((t - dst) << MSD_BYTES_AND_FLAGS_BYTES_SHIFT) & MSD_BYTES_AND_FLAGS_BYTES_MASK);
			
	if (doSend) {
	
		logBytes("RX", dst, t - dst);
		*replyMsgP = MSG_C_SER_DATA_RX;
	}
}

#ifdef __MWERKS__
	asm void scrCopy(void* dst, void* src, UInt16 nWords)
	{
		#define DONT_INCLUDE_SCR_DECOMPRESS
		#include "asm.S"
		#undef DONT_INCLUDE_SCR_DECOMPRESS
	}


	asm uint8_t* screenDecompress(uint8_t *dst, const uint8_t *src, uint32_t srcLen)
	{
		#define DONT_INCLUDE_SCR_COPY
		#include "asm.S"
		#undef DONT_INCLUDE_SCR_COPY
	}

#else
	
	
#endif

static void sendMsgAndProcessReply(struct CommsInfo *ci, UInt16 message, UInt16 bytesToCopyOut)
{
	struct MsgSingleWord *msgSingle = (struct MsgSingleWord*)ci->dataIn;
	struct MsgScreenData *msgScreen = (struct MsgScreenData*)ci->dataIn;
	struct MsgLedControl *msgLedCtl = (struct MsgLedControl*)ci->dataIn;
	struct MsgClut *msgClut = (struct MsgClut*)ci->dataIn;
	UInt16 L, reply, i, *scrData, *fromData;
	UInt32 tmp32;

send_another_msg:
	if (message == MSG_C_NOP) {		//do not waste an opportunity to send something else
	
		UInt32 keys;
		Int16 x, y;
		Boolean dn;
		
		
		EvtGetPen(&x, &y, &dn);
		if (!dn)
			x = y = -1;
		
		if (x != ci->prevPenX || y != ci->prevPenY) {	//send pen
			
			struct MsgPenEvt *msgPen = (struct MsgPenEvt*)ci->outMsg;
			
			msgPen->x = x;
			msgPen->y = y;
			ci->prevPenX = x;
			ci->prevPenY = y;
			message = MSG_C_PEN_INFO;
			bytesToCopyOut = sizeof(*msgPen);
		}
	}
	
	if (message == MSG_C_NOP) {		//do not waste an opportunity to send something else
	
		UInt32 keys = KeyCurrentState();
		
		if (keys != ci->prevKeyState) {	//send button
			
			struct MsgButtonEvt *msgBtn = (struct MsgButtonEvt*)ci->outMsg;
			
			msgBtn->keysState = keys;
			ci->prevKeyState = keys;
			message = MSG_C_BUTTON_INFO;
			bytesToCopyOut = sizeof(*msgBtn);
		}
	}
	
	if (message == MSG_C_NOP) {		//do not waste an opportunity to send something else
		
		serialMaybeSendMessage(ci, &message, &bytesToCopyOut);
	}
	
	reply = mailboxMsgComplex(ci, message, bytesToCopyOut);
	
reprocess_reply:
	
	switch (reply) {
	
		case MSG_C_ACK:
			//nothing
			break;
		
		case MSG_C_GET_BATT_INFO:
			prepareBattInfoMsg(ci);
			reply = mailboxMsgComplex(ci, MSG_C_BATTERY_INFO, sizeof(struct MsgBatteryInfo));
			goto reprocess_reply;
		
		case MSG_C_SET_BRIGHTNESS:
			//brightness support is in 3.5+, luckily we do not need brightness in devices that do not have suport for it
			if (isOsAtLeast(sysMakeROMVersion(3,5,0,sysROMStageRelease,0))) {
			
				SysLCDBrightness(true, msgSingle->value);
			}
			break;
		
		case MSG_C_SET_CONTRAST:
			SysLCDContrast(true, msgSingle->value);
			break;
		
		case MSG_C_SET_SCREEN_DEPTH:
			tmp32 = msgSingle->value;
			WinScreenMode(winScreenModeSet, NULL, NULL, &tmp32, NULL);
			break;
		
		case MSG_C_SET_CLUT:
			//palete support is in 3.5+, luckily we do not need palettes if we have no color. color is also in 3.5+
			if (isOsAtLeast(sysMakeROMVersion(3,5,0,sysROMStageRelease,0))) {
			
				WinPalette(winPaletteSet, 0, msgClut->numEntries, (struct RGBColorType*)msgClut->colors);
			}
			break;
		
		case MSG_C_SCREEN_IMAGE:
			
			L = msgScreen->sz;
			
			//most often an entire screen fits in a single message. in that case there is no point copying the data out
			
			if (!ci->scrCompressedSz && (L & 0x8000) && !msgScreen->ofst) {
			
#if USE_SCREEN_LOCKING
				Boolean screenLocked  = !!MyWinScreenLock(winLockDontCare);
#endif
				(void)screenDecompress(getDisplayWriteAddress(), (void*)msgScreen->data, L & 0x7fff);
				
#if USE_SCREEN_LOCKING
				if (screenLocked)
					WinScreenUnlock();
#endif
			}
			else {
				
				if (msgScreen->ofst != ci->scrCompressedSz)	{	//we missed something
					ci->scrCompressedSz = 0;
					message = MSG_C_NOP;
					goto send_another_msg;
				}
				
				scrCopy(ci->scrCompressedData + ci->scrCompressedSz, msgScreen->data, ((L & 0x7fff) + 1) / 2);	//over-copy is ok by a byte
				ci->scrCompressedSz += L & 0x7fff;
				
				if (L & 0x8000) {
				
#if USE_SCREEN_LOCKING
					Boolean screenLocked  = !!MyWinScreenLock(winLockDontCare);
#endif

					(void)screenDecompress(getDisplayWriteAddress(), ci->scrCompressedData, ci->scrCompressedSz);
					ci->scrCompressedSz = 0;
					
#if USE_SCREEN_LOCKING
					if (screenLocked)
						WinScreenUnlock();
#endif
				}
				else {				//we are in screen refresh so rx more right away
				
					message = MSG_C_NOP;
					goto send_another_msg;
				}
			}
			
			break;
		
		case MSG_C_UPDATE:
			irqSetState(ci, false);
			handleFwUpdate(ci);
			//not reached
			break;
		
		case MSG_C_SER_OP:
			serialOp(ci);
			reply = mailboxMsgComplex(ci, MSG_C_SER_OP_REPLY, sizeof(struct MsgSerOp));
			goto reprocess_reply;
		
		case MSG_C_SER_DATA_TX:
			serialTx(ci);
			reply = mailboxMsgComplex(ci, MSG_C_SER_DATA_TX_REPLY, sizeof(struct MsgSerialDataTxReply));
			goto reprocess_reply;
		
		case MSG_C_LED_CONTROL:
			if (!msgLedCtl->numTimes) {
				ledOn(ci, 0);
				ci->ledState = LedStateOff;
				ci->ledDue = 0;
			}
			else {
				ci->ledState = LedStateInterPattern;
				ci->ledPattern = (((UInt32)msgLedCtl->patternHi) << 16) + msgLedCtl->patternLo;
				ci->ledCsecPerPiece = msgLedCtl->csecPerPiece;
				ci->ledCsecBetween = msgLedCtl->csecBetween;
				ci->ledNumTimes = msgLedCtl->numTimes;
				ci->ledDue = TimGetTicks();
			}
			break;
		
		case MSG_C_SET_BACKLIGHT:
			(void)HwrBacklight(true, !!msgSingle->value);
			break;
			
		
		default: {
		
			char x[32];
			
			StrPrintF(x, "WTF 0x%04x", reply);
			
			MyErrAlertCustom(0, x, NULL, NULL);
		
			break;
		}
	}	
}

void
#ifdef __GNUC__
__attribute__((section(".vectors")))
#endif
__Startup__(void)
{
	struct EventType evt;
	struct CommsInfo *ci;
	UInt32 tmp;
	Err e;
	
	
	ci = MemChunkNew(0, sizeof(*ci), 0x1200);
	MemSet(ci, sizeof(*ci), 0);
	
	logStart();
	
	//find the card
	for (ci->cardNo = 0; ci->cardNo < MAX_CARD_NUMBER_TO_GUESS; ci->cardNo++) {
		UInt8 isRemovable;
		
		if (errNone == HsCardAttrGet(ci->cardNo, hsCardAttrRemovable, &isRemovable) && isRemovable)
			break;
	}
	if (ci->cardNo == MAX_CARD_NUMBER_TO_GUESS)
		goto out;

	//access time is 25 nanoseconds
	tmp = 25;
	e = HsCardAttrSet(ci->cardNo, hsCardAttrAccessTime, &tmp);
	if (e != errNone) {
		MyErrAlertCustom(e, "setting access time", NULL, NULL);
		goto out;
	}
	
	//get addresses of chip selects
	e = HsCardAttrGet(ci->cardNo, hsCardAttrCsBase, (void*)&ci->dataPtr);
	if (e != errNone) {
		MyErrAlertCustom(e, "getting cs base", NULL, NULL);
		goto out;
	}
	e = HsCardAttrGet(ci->cardNo, hsCardAttrCsSize, &tmp);
	if (e != errNone) {
		MyErrAlertCustom(e, "getting cs size", NULL, NULL);
		goto out;
	}
	ci->signalPtr = (UInt16*)(((char*)ci->dataPtr) + tmp);
	
	if (mailboxMsgBasic(ci, MSG_S_SHABBAT) != MSG_S_SHALOM) {
	
		MyErrAlertCustom(0, "Communications failed to be established", NULL, NULL);
		goto out;
	}
	
	//set up irq handler data
	e = HsCardAttrSet(ci->cardNo, hsCardAttrCardParam, (UInt32*)&ci);
	if (e != errNone) {
		MyErrAlertCustom(e, "setting irq handler data", NULL, NULL);
		goto out;
	}
	
	//set up irq handler
	tmp = (UInt32)&irqHandler;
	e = HsCardAttrSet(ci->cardNo, hsCardAttrIntHandler, &tmp);
	if (e != errNone) {
		MyErrAlertCustom(e, "setting irq handler func", NULL, NULL);
		goto out;
	}
	
	//enable irqs
	irqSetState(ci, true);
	
	ci->dataOut = ci->dataPtr + mailboxMsgBasic(ci, MSG_S_MAINBOX_INFO_0);
	ci->dataOutWords = mailboxMsgBasic(ci, MSG_S_MAINBOX_INFO_1);
	ci->dataIn = ci->dataPtr + mailboxMsgBasic(ci, MSG_S_MAINBOX_INFO_2);
	ci->dataInWords = mailboxMsgBasic(ci, MSG_S_MAINBOX_INFO_3);
	
	ci->outMsg = MemPtrNew(sizeof(UInt16) * ci->dataOutWords);
	if (!ci->outMsg) {
	
		MyErrAlertCustom(0,  "comms buffer ", "Out Of Memory allocating ", "1");
		goto out;
	}
	
	ci->inMsg = MemPtrNew(sizeof(UInt16) * ci->dataInWords);
	if (!ci->inMsg) {
	
		MyErrAlertCustom(0,  "comms buffer ", "Out Of Memory allocating ", "1");
		goto out;
	}
	
	if (!prepareBoot(ci)) {
		
		MyErrAlertCustom(0, "Failed to boot the module", NULL, NULL);
		goto out;
	}
	
	while (1) {
		
		struct MsgButtonEvt *msgBtn = (struct MsgButtonEvt*)ci->outMsg;
		struct MsgPenEvt *msgPen = (struct MsgPenEvt*)ci->outMsg;
		
		EvtGetEvent(&evt, KeyCurrentState() ? 1 : ledGetEvtGettingTimeout(ci));
		
		switch (evt.eType) {
			
			case nilEvent:
				ledProcess(ci);
				//fallthrough
				
			case keyDownEvent:
			case keyUpEvent:
			case penDownEvent:
			case penMoveEvent:
			case penUpEvent:
				sendMsgAndProcessReply(ci, MSG_C_NOP, 0);
				ledProcess(ci);
				irqSetState(ci, true);
				break;
			
			default:
				//nothing to do here
				break;
		}
	}

out:

	if (ci->inMsg)
		MemPtrFree(ci->inMsg);

	if (ci->outMsg)
		MemPtrFree(ci->outMsg);
	
	MemChunkFree(ci);
}