#include <string.h>
#include "halSound.h"
#include "printf.h"
#include "audio.h"
#include "heap.h"
#include "kal.h"
#include "dal.h"


#define SIMPLE_SOUND_MIN_FREQ		20
#define SIMPLE_SOUND_MIN_SAMPLES	64



struct OutputStream {
	struct OutputStream *next, *prev;
	uint32_t handle;
	struct AudioOutputStrmHandle *strm;
	void* fakeBufs[2];				//for faking HAL_SOUND_IOCTL_GET_BUFFS support
	uint16_t vol;
	int16_t pan;
};

struct InputStream {
	enum AudioChannelConfig chCfg;
	enum AudioSampleType sampTyp;
	uint32_t volL, volR;
	uint32_t handle;
	uint16_t vol;
	int16_t pan;
	bool inited;
};

static const uint16_t mSampleRateToSampleFreq[] = {
	[AudioRate8000] = 8000,
	[AudioRate11025] = 11025,
	[AudioRate16000] = 16000, 
	[AudioRate22050] = 22050,
	[AudioRate24000] = 24000,
	[AudioRate32000] = 32000,
	[AudioRate44100] = 44100,
	[AudioRate48000] = 48000,
};

static uint32_t mSoundMutex, mSimpleSoundTimer = 0, mSimpleSoundTimerCookie, mSampledSoundNextHandle = 0;
static bool mHaveAudioOut = false, mHaveAudioIn = false, mHaveOnlySimpleAudioOut = false /* only relevant if no sampled audio out */;
static struct AudioOutputStrmHandle *mSimpleSoundStream = NULL;
static enum AudioSampleRate mNativeOutputRate;
static struct OutputStream *mOutputStreams;
static struct InputStream *mInputStream;

Err DALEXPORT impl_HALSoundInitialize(void)
{
	logi("%s\n", __func__);
	
	if (audioOutInit(&mNativeOutputRate, NULL))
		mHaveAudioOut = true;
	else {
		logw("Failed to init sampled audio subsystem and mixer\n");
		if (audioOnlySimpleOutInit())
			mHaveOnlySimpleAudioOut = true;
		else
			logw("Failed to init simple audio\n");
	}
	
	if (audioInInit())
		mHaveAudioIn = true;
	else
		logw("Failed to init mic subsystem\n");
	
	if (errNone != KALMutexCreate(&mSoundMutex, CREATE_4CC('_','s','s','L')))
		fatal("Failed to init sound lock\n");
	
	return errNone;
}

static struct OutputStream* streamLookupOutputLocked(SndStreamRef ref)
{
	struct OutputStream *s;
	
	for (s = mOutputStreams; s && s->handle != ref; s = s->next);
	
	return s;
}

static struct InputStream* streamLookupInputLocked(SndStreamRef ref)
{
	if (mInputStream && mInputStream->handle == ref)
		return mInputStream;
	
	return NULL;
}

static uint32_t streamPrvHandleAllocate(void)	//call only while mutex is locked
{
	uint32_t ret;
	
	do {
		if ((++mSampledSoundNextHandle) & 0x80000000)
			mSampledSoundNextHandle = 1;
		
		ret = mSampledSoundNextHandle;
	} while (streamLookupOutputLocked(ret) || streamLookupInputLocked(ret));
	
	return ret;
}

static void soundPrvCalcVolumes(uint32_t *volLP, uint32_t *volRP, uint32_t vol, int32_t pan)
{
	uint32_t volL = vol, volR = vol;
	
	if (pan < 0)
		volR = volR * (1024 + pan) / 1024;
	else if (pan > 0)
		volL = volL * (1024 - pan) / 1024;
	
	if (volLP)
		*volLP = volL;
	if (volRP)
		*volRP = volR;
}

SndStreamRef DALEXPORT impl_HALSoundOpen(const char *deviceP, UInt32 flags, Err *errP)		//HAL_SOUND_DEVICE_NAME_*
{
	SndStreamRef ret = INVALID_SND_STREAM;
	Err e = sndErrNotImpl;
	
	logi("%s(%s, ...)\n", __func__, deviceP);
	(void)flags;
	
	if (mHaveAudioOut && !strcmp(HAL_SOUND_DEVICE_NAME_OUTPUT, deviceP)) {
		
		struct OutputStream *s = (struct OutputStream*)kheapAlloc(sizeof(struct OutputStream));
		if (s) {
			s->strm = NULL;
			s->vol = HAL_SOUND_UNITY_VOLUME;
			s->pan = 0;
			s->prev = NULL;
			s->fakeBufs[0] = NULL;
			s->fakeBufs[1] = NULL;
			
			KALMutexReserve(mSoundMutex, -1);
			s->handle = streamPrvHandleAllocate();
			s->next = mOutputStreams;
			if (mOutputStreams)
				mOutputStreams->prev = s;
			mOutputStreams = s;
			KALMutexRelease(mSoundMutex);
			
			ret = s->handle;
			e = errNone;
		}
	}
	else if (mHaveAudioIn && !strcmp(HAL_SOUND_DEVICE_NAME_INPUT, deviceP)) {
			
		struct InputStream *s = (struct InputStream*)kheapAlloc(sizeof(struct InputStream));
		
		s->inited = 0;
		s->vol = HAL_SOUND_UNITY_VOLUME;
		s->pan = 0;
		soundPrvCalcVolumes(&s->volL, &s->volR, s->vol, s->pan);
		
		KALMutexReserve(mSoundMutex, -1);
		if (mInputStream)
			e = sndErrMemory;
		else {
			s->handle = streamPrvHandleAllocate();
			
			ret = s->handle;
			mInputStream = s;
			s = NULL;
			e = errNone;
		}
		KALMutexRelease(mSoundMutex);
		if (s)
			kheapFree(s);
	}
	else {
		
		logi("Audio stream type '%s' not supported\n", deviceP);
	}
	
	logi("%s(%s, ...) -> %u, err=0x%04x\n", __func__, deviceP, ret, e);
	
	if (errP)
		*errP = e;
	return ret;
}

Err DALEXPORT impl_HALSoundClose(SndStreamRef streamRef)
{
	struct OutputStream *os = NULL;
	Err ret = sndErrInvalidStream;
	struct InputStream *is = NULL;
	uint32_t i;
	
	logi("%s(%u)\n", __func__, streamRef);
	
	KALMutexReserve(mSoundMutex, -1);
	os = streamLookupOutputLocked(streamRef);
	if (os) {
		
		if (os->next)
			os->next->prev = os->prev;
		if (os->prev)
			os->prev->next = os->next;
		else
			mOutputStreams = os->next;
	}
	else {
		
		is = streamLookupInputLocked(streamRef);
		mInputStream = NULL;
	}
	KALMutexRelease(mSoundMutex);
	
	if (os) {
		
		if (os->strm)
			audiostreamDestroyOutput(os->strm);
		
		for (i = 0; i < 2; i++) {
			if (os->fakeBufs[i])
				kheapFree(os->fakeBufs[i]);
			os->fakeBufs[i] = NULL;
		}
		kheapFree(os);
		
		ret = errNone;
	}
	
	if (is) {
		
		audioMicStop();
		kheapFree(is);
		
		ret = errNone;
	}
	
	return ret;
}

//should block until all samples have been read (or stream is terminated)
uint32_t DALEXPORT impl_HALSoundRead(SndStreamRef streamRef, void *bufP, uint32_t size, Err *errP)
{
	struct InputStream *is;
	bool found = false;
	uint32_t ret = 0;
	Err e = errNone;

	logst("%s\n", __func__);
	KALMutexReserve(mSoundMutex, -1);
	is = streamLookupInputLocked(streamRef);
	if (is)
		found = true;
	else
		e = sndErrInvalidStream;
	KALMutexRelease(mSoundMutex);
	
	if (found)
		ret = audioMicGetSamples(bufP, size, is->volL, is->volR);

	if (errP)
		*errP = e;

	return ret;
}

//should block until all samples can be written (or stream is terminated)
uint32_t DALEXPORT impl_HALSoundWrite(SndStreamRef streamRef, void *bufP, uint32_t size, Err *errP)
{
	struct AudioOutputStrmHandle *strm = NULL;
	struct OutputStream *os;
	uint32_t ret = 0;
	Err e = errNone;
	
	logst("%s\n", __func__);
	KALMutexReserve(mSoundMutex, -1);
	os = streamLookupOutputLocked(streamRef);
	if (!os)
		e = sndErrInvalidStream;
	else
		strm = os->strm;
	KALMutexRelease(mSoundMutex);
	
	if (strm)
		ret = audiostreamAddSamplesOutput(strm, bufP, size);
	
	if (errP)
		*errP = e;
	
	return ret;
}

static void soundSetVolumes(struct OutputStream *s)
{
	uint32_t volL, volR;
	
	if (!s->strm)
		return;
	
	soundPrvCalcVolumes(&volL, &volR, s->vol, s->pan);

	audiostreamSetVolumesOutput(s->strm, volL, volR);
}

static bool soundStreamValidateParams(const void *si, enum AudioChannelConfig *chansP, enum AudioSampleType *sampTypP, enum AudioSampleRate *rateP)
{
	uint32_t req_format, req_rate, req_sampTyp, req_streamW;
	enum AudioChannelConfig chans;
	enum AudioSampleType sampTyp;
	enum AudioSampleRate rate;
	
	if (is50()) {
		
		const struct HalSoundInitInfo50 *si50 = (const struct HalSoundInitInfo50*)si;
		
		req_rate = si50->sampleRate;
		req_format = sndFormatPCM;
		req_sampTyp = si50->sampleType;
		req_streamW = si50->streamWidth;
	}
	else {
		
		const struct HalSoundInitInfo52 *si52 = (const struct HalSoundInitInfo52*)si;
		
		req_rate = si52->sampleRate;
		req_format = si52->format;
		req_sampTyp = si52->sampleType;
		req_streamW = si52->streamWidth;
	}
	
	//XXX: is we ever want to support formats other than PCM, keep in mind that OS 5.2.1 PACE truncates "format" to 16 bits so compare carefullt
	
	if (req_format != sndFormatPCM) {
		
		logw("Audio format 0x%08x not suported\n", req_format);
		return false;
	}
	
	switch (req_rate) {
		case 8000:				rate = AudioRate8000;			break;
		case 11025:				rate = AudioRate11025;			break;
		case 16000:				rate = AudioRate16000;			break;
		case 22050:				rate = AudioRate22050;			break;
		case 24000:				rate = AudioRate24000;			break;
		case 32000:				rate = AudioRate32000;			break;
		case 44100:				rate = AudioRate44100;			break;
		case 48000:				rate = AudioRate48000;			break;
		default:
			logw("requested sample rate %u is unsupported\n", req_rate);
			return false;
	}
	
	switch (req_streamW) {
		case sndMono:			chans = AudioMono;				break;
		case sndStereo:			chans = AudioStereo;			break;
		default:
			logw("requested stream width %u is unsupported\n", req_streamW);
			return false;
	}
	
	switch (req_sampTyp) {
		case sndInt8:			sampTyp = AudioSampleS8;		break;
		case sndUInt8:			sampTyp = AudioSampleU8;		break;
		case sndInt16Big:		sampTyp = AudioSampleS16BE;		break;
		case sndInt16Little:	sampTyp = AudioSampleS16LE;		break;
		case sndInt32Big:		sampTyp = AudioSampleS32BE;		break;
		case sndInt32Little:	sampTyp = AudioSampleS32LE;		break;
		case sndFloatBig:		sampTyp = AudioSampleFloatBE;	break;
		default:
			logw("requested sample type %u is unsupported\n", req_sampTyp);
			return false;
	}
	
	if (chansP)
		*chansP = chans;
	
	if (sampTypP)
		*sampTypP = sampTyp;
	
	if (rateP)
		*rateP = rate;
	
	return true;
}

static Err soundStreamOutputInit(struct OutputStream *s, const void *si)
{
	enum AudioChannelConfig chans;
	enum AudioSampleType sampTyp;
	enum AudioSampleRate rate;
	
	if (!soundStreamValidateParams(si, &chans, &sampTyp, &rate))
		return sndErrFormat;
	
	s->strm = audiostreamCreateOutput(rate, sampTyp, chans, 1024, false);
	
	if (!s->strm) {
		logw("Failed to create sound stream\n");
		return sndErrMemory;
	}
	
	soundSetVolumes(s);
	return errNone;
}

static Err soundStreamInputInit(struct InputStream *s, const void *si)
{
	enum AudioSampleRate rate;
	
	if (!soundStreamValidateParams(si, &s->chCfg, &s->sampTyp, &rate))
		return sndErrFormat;
	
	if (!audioMicSetup(rate, 1024, s->sampTyp, s->chCfg)) {
		logw("Failed to create sound in stream\n");
		return sndErrMemory;
	}
	
	s->inited = true;
	
	return errNone;
}

static Err soundPrvIoctlOutput(struct OutputStream *s, uint32_t cmd, void *paramP)
{
	struct HalSonySoundStreamStats *sss = (struct HalSonySoundStreamStats*)paramP;
	struct HalSoundBuffersInfo *bi = (struct HalSoundBuffersInfo*)paramP;
	uint32_t t, vol;
	int32_t pan;
	Err e;
	
	switch (cmd) {
		case HAL_SOUND_IOCTL_SETUP:
			if (s->strm) {
				logw("Duplicate audio setup disallowed\n");
				e = sndErrInvalidStream;
				break;
			}
			e = soundStreamOutputInit(s, paramP);
			if (e)
				logw("Sound stream creating produced error 0x%04x\n", e);
			break;
		
		case HAL_SOUND_IOCTL_SET_VOL:
			vol = *(const uint32_t*)paramP;
			if (vol > HAL_SOUND_MAX_VOLUME)
				vol = HAL_SOUND_UNITY_VOLUME;
			s->vol = vol;
			soundSetVolumes(s);
			e = errNone;
			break;
		
		case HAL_SOUND_IOCTL_GET_VOL:
			*(uint32_t*)paramP = s->vol;
			e = errNone;
			break;
		
		case HAL_SOUND_IOCTL_SET_PAN:
			pan = (int32_t)(int16_t)*(const int32_t*)paramP;		//yes this is correct
			if (pan > HAL_SOUND_PAN_FULL_RIGHT)
				pan = HAL_SOUND_PAN_FULL_RIGHT;
			else if (pan < HAL_SOUND_PAN_FELL_LEFT)
				pan = HAL_SOUND_PAN_FELL_LEFT;
			s->pan = pan;
			soundSetVolumes(s);
			e = errNone;
			break;
		
		case HAL_SOUND_IOCTL_GET_PAN:
			*(int32_t*)paramP = (int32_t)s->pan;
			e = errNone;
			break;
		
		case HAL_SOUND_IOCTL_GET_BUFFS:
			//our audio layer does its own buffering and usually to a depth over 2
			//this is incompatible with the assumptions of this ioctl so we'll
			//just allocate two buffers, and let them pretend they go to hardware
			//sadly lots of code in the OS assumes these are exactly 4K in size
			t = 4096;
			if (!s->fakeBufs[0]) {
				
				s->fakeBufs[0] = kheapAlloc(t);
				if (!s->fakeBufs[0]) {
					e = sndErrMemory;
					break;
				}
				
				s->fakeBufs[1] = kheapAlloc(t);
				if (!s->fakeBufs[1]) {
					e = sndErrMemory;
					kheapFree(s->fakeBufs[0]);
					s->fakeBufs[0] = NULL;
					break;
				}
			}
			bi->sz = t;
			bi->bufs[0] = s->fakeBufs[0];
			bi->bufs[1] = s->fakeBufs[1];
			e = errNone;
			break;
		
		case HAL_SOUND_IOCTL_STREAM_STOP:
			e = errNone;
			break;
		
	#ifdef SONY_SUPPORT_ENABLED
	
		case HAL_SOUND_IOCTL_GET_STATS:
			if (paramP && audiostreamOutputGetSonyStats(s->strm, &sss->numOutSamplesProduced, &sss->lastMixedTicks))
				e = errNone;
			else
				e = sysErrParamErr;
			break;
			
	#endif
	
		default:
			e = sndErrInvalidStream;
			break;
	}
	
	return e;
}

static Err soundPrvIoctlInput(struct InputStream *s, uint32_t cmd, void *paramP)
{
	struct HalSoundBuffersInfo *bi = (struct HalSoundBuffersInfo*)paramP;
	uint32_t t, vol;
	int32_t pan;
	Err e;
	
	switch (cmd) {
		case HAL_SOUND_IOCTL_SETUP:
			if (s->inited) {
				logw("Duplicate audio in setup disallowed\n");
				e = sndErrInvalidStream;
				break;
			}
			e = soundStreamInputInit(s, paramP);
			if (e)
				logw("Sound in stream creating produced error 0x%04x\n", e);
			break;
		
		case HAL_SOUND_IOCTL_SET_VOL:
			vol = *(const uint32_t*)paramP;
			if (vol > HAL_SOUND_MAX_VOLUME)
				vol = HAL_SOUND_UNITY_VOLUME;
			s->vol = vol;
			
			soundPrvCalcVolumes(&s->volL, &s->volR, s->vol, s->pan);
			e = errNone;
			break;
		
		case HAL_SOUND_IOCTL_GET_VOL:
			*(uint32_t*)paramP = s->vol;
			e = errNone;
			break;
		
		case HAL_SOUND_IOCTL_SET_PAN:
			pan = (int32_t)(int16_t)*(const int32_t*)paramP;		//yes this is correct
			if (pan > HAL_SOUND_PAN_FULL_RIGHT)
				pan = HAL_SOUND_PAN_FULL_RIGHT;
			else if (pan < HAL_SOUND_PAN_FELL_LEFT)
				pan = HAL_SOUND_PAN_FELL_LEFT;
			s->pan = pan;
			soundPrvCalcVolumes(&s->volL, &s->volR, s->vol, s->pan);
			e = errNone;
			break;
		
		case HAL_SOUND_IOCTL_GET_PAN:
			*(int32_t*)paramP = (int32_t)s->pan;
			e = errNone;
			break;
		
		case HAL_SOUND_IOCTL_STREAM_STOP:
			e = errNone;
			break;
	
		default:
			e = sndErrInvalidStream;
			break;
	}
	
	return e;
}

Err DALEXPORT impl_HALSoundIoctl(SndStreamRef streamRef, uint32_t cmd, void *paramP)
{
	struct OutputStream *os;
	struct InputStream *is;
	Err e;
	
	logi("%s(%u, %u, ...)\n", __func__, streamRef, cmd);
	
	KALMutexReserve(mSoundMutex, -1);
	
	#ifdef SONY_SUPPORT_ENABLED
	
		if (streamRef == SONY_STREAM_REF_SPECIAL) switch (cmd) {
			case HAL_SOUND_IOCTL_NUM_ACTIVE_STREAMS:
				*(uint32_t*)paramP = audioGetNumOutStreams();
				break;
		}
	
	#endif
	
	if ((os = streamLookupOutputLocked(streamRef)) != NULL)
		e = soundPrvIoctlOutput(os, cmd, paramP);
	else if ((is = streamLookupInputLocked(streamRef)) != NULL)
		e = soundPrvIoctlInput(is, cmd, paramP);
	else
		e = sndErrInvalidStream;
	KALMutexRelease(mSoundMutex);
	
	return e;
}

//simple sound api:



static int16_t soundSinWaveVal(uint16_t pos)			//full wave is 0..64k
{
	static const uint16_t sintab[] = {					//256 samples for a quarter of a wave, so 1024 for full wave
		0x0000, 0x00c9, 0x0192, 0x025b, 0x0324, 0x03ed, 0x04b6, 0x057f, 0x0648, 0x0711, 0x07d9, 0x08a2, 0x096a, 0x0a33, 0x0afb, 0x0bc4,
		0x0c8c, 0x0d54, 0x0e1c, 0x0ee3, 0x0fab, 0x1072, 0x113a, 0x1201, 0x12c8, 0x138f, 0x1455, 0x151c, 0x15e2, 0x16a8, 0x176e, 0x1833,
		0x18f9, 0x19be, 0x1a82, 0x1b47, 0x1c0b, 0x1ccf, 0x1d93, 0x1e57, 0x1f1a, 0x1fdd, 0x209f, 0x2161, 0x2223, 0x22e5, 0x23a6, 0x2467,
		0x2528, 0x25e8, 0x26a8, 0x2767, 0x2826, 0x28e5, 0x29a3, 0x2a61, 0x2b1f, 0x2bdc, 0x2c99, 0x2d55, 0x2e11, 0x2ecc, 0x2f87, 0x3041,
		0x30fb, 0x31b5, 0x326e, 0x3326, 0x33df, 0x3496, 0x354d, 0x3604, 0x36ba, 0x376f, 0x3824, 0x38d9, 0x398c, 0x3a40, 0x3af2, 0x3ba5,
		0x3c56, 0x3d07, 0x3db8, 0x3e68, 0x3f17, 0x3fc5, 0x4073, 0x4121, 0x41ce, 0x427a, 0x4325, 0x43d0, 0x447a, 0x4524, 0x45cd, 0x4675,
		0x471c, 0x47c3, 0x4869, 0x490f, 0x49b4, 0x4a58, 0x4afb, 0x4b9d, 0x4c3f, 0x4ce0, 0x4d81, 0x4e20, 0x4ebf, 0x4f5d, 0x4ffb, 0x5097,
		0x5133, 0x51ce, 0x5268, 0x5302, 0x539b, 0x5432, 0x54c9, 0x5560, 0x55f5, 0x568a, 0x571d, 0x57b0, 0x5842, 0x58d3, 0x5964, 0x59f3,
		0x5a82, 0x5b0f, 0x5b9c, 0x5c28, 0x5cb3, 0x5d3e, 0x5dc7, 0x5e4f, 0x5ed7, 0x5f5d, 0x5fe3, 0x6068, 0x60eb, 0x616e, 0x61f0, 0x6271,
		0x62f1, 0x6370, 0x63ee, 0x646c, 0x64e8, 0x6563, 0x65dd, 0x6656, 0x66cf, 0x6746, 0x67bc, 0x6832, 0x68a6, 0x6919, 0x698b, 0x69fd,
		0x6a6d, 0x6adc, 0x6b4a, 0x6bb7, 0x6c23, 0x6c8e, 0x6cf8, 0x6d61, 0x6dc9, 0x6e30, 0x6e96, 0x6efb, 0x6f5e, 0x6fc1, 0x7022, 0x7083,
		0x70e2, 0x7140, 0x719d, 0x71f9, 0x7254, 0x72ae, 0x7307, 0x735e, 0x73b5, 0x740a, 0x745f, 0x74b2, 0x7504, 0x7555, 0x75a5, 0x75f3,
		0x7641, 0x768d, 0x76d8, 0x7722, 0x776b, 0x77b3, 0x77fa, 0x783f, 0x7884, 0x78c7, 0x7909, 0x794a, 0x7989, 0x79c8, 0x7a05, 0x7a41,
		0x7a7c, 0x7ab6, 0x7aee, 0x7b26, 0x7b5c, 0x7b91, 0x7bc5, 0x7bf8, 0x7c29, 0x7c59, 0x7c88, 0x7cb6, 0x7ce3, 0x7d0e, 0x7d39, 0x7d62,
		0x7d89, 0x7db0, 0x7dd5, 0x7dfa, 0x7e1d, 0x7e3e, 0x7e5f, 0x7e7e, 0x7e9c, 0x7eb9, 0x7ed5, 0x7eef, 0x7f09, 0x7f21, 0x7f37, 0x7f4d,
		0x7f61, 0x7f74, 0x7f86, 0x7f97, 0x7fa6, 0x7fb4, 0x7fc1, 0x7fcd, 0x7fd8, 0x7fe1, 0x7fe9, 0x7ff0, 0x7ff5, 0x7ff9, 0x7ffd, 0x7ffe,
		0x7fff,	0 /* we might read this last value but we'll never really use it */
	};
	
	uint32_t ret, iF, iS, vF, vS, dF, dS;	//for first and second: index, distance, value
	bool inv = false;
	
	//second half of sin is an inverse of first
	if (pos >= 0x8000) {
		
		pos -= 0x8000;
		inv = true;
	}
	
	//second quarter of sin is the reverse of first
	if (pos >= 0x4000)
		pos = 0x8000 - pos;
	
	
	//now calc the first quarter of sin using linear interpolation
	
	//calc indices we'll use
	iF = pos / 64;
	iS = iF + 1;
	
	//calc distances from them to our point
	dF = pos - iF * 64;
	dS = iS * 64 - pos;
	
	//grab the values
	vF = sintab[iF];
	vS = sintab[iS];
	
	//interpolate
	ret = (vF * dS + vS * dF) / 64;
	
	if (inv)
		ret = -ret;
	
	return ret;
}

static void soundCreateWave(int16_t* dst, uint32_t freq)
{
	uint32_t i, nsampl = (mSampleRateToSampleFreq[mNativeOutputRate] + freq / 2) / freq;

	for (i = 0; i < nsampl; i++)
		*dst++ = soundSinWaveVal(((i * 65536) + nsampl / 2) / nsampl);
}

static void soundStopSimpleLocked(void)
{
	mSimpleSoundTimerCookie++;
	if (mHaveOnlySimpleAudioOut)
		audioOnlySimpleTone(0, 0);
	else if (mSimpleSoundStream) {
		audiostreamDestroyOutput(mSimpleSoundStream);
		mSimpleSoundStream = NULL;
	}
	
	if (mSimpleSoundTimer) {
		KALTimerDelete(mSimpleSoundTimer);
		mSimpleSoundTimer = 0;
	}
}

Err DALEXPORT impl_HALSoundOff(void)	//stop the simple sound that we have been playing (or do nothing if we had none)
{
	//this func can be called before HALSoundInitialize. If so, do nothing as we do not even have a mutex
	
	logt("%s\n", __func__);
	
	if (mSoundMutex) {
	
		KALMutexReserve(mSoundMutex, -1);
		soundStopSimpleLocked();
		KALMutexRelease(mSoundMutex);
	}
	
	return errNone;
}

static void soundSimpleSoundTimerCbk(void *userData)
{
	uint32_t expectedCookie = (uintptr_t)userData;
	
	KALMutexReserve(mSoundMutex, -1);
	if (expectedCookie == mSimpleSoundTimerCookie)
		soundStopSimpleLocked();
	KALMutexRelease(mSoundMutex);
}

Err DALEXPORT impl_HALSoundPlay(uint32_t freq, uint32_t amplitude, int32_t timeDuration)	// play a simple sound
{
	Err e = errNone;
	uint32_t len;
	
	logt("%s(freq=%u ampl=%u dur=%u)\n", __func__, freq, amplitude, timeDuration);
	
	if (!timeDuration)
		return sndErrBadParam;
	
	if (mHaveOnlySimpleAudioOut) {
		
		KALMutexReserve(mSoundMutex, -1);
		audioOnlySimpleTone(freq, amplitude);
		KALMutexRelease(mSoundMutex);
	}
	else if (mHaveAudioOut) {
		
		uint32_t i, nIter, sampsInBuf, sampsPerWave = (mSampleRateToSampleFreq[mNativeOutputRate] + freq / 2) / freq;
		
		if (freq < SIMPLE_SOUND_MIN_FREQ)
			freq = SIMPLE_SOUND_MIN_FREQ;
		
		for (sampsInBuf = 0, nIter = 0; sampsInBuf < SIMPLE_SOUND_MIN_SAMPLES; sampsInBuf += sampsPerWave, nIter++);
		
		KALMutexReserve(mSoundMutex, -1);
		soundStopSimpleLocked();
		mSimpleSoundStream = audiostreamCreateOutput(mNativeOutputRate, AudioSampleS16LE, AudioMono, sampsInBuf, true);
		if (!mSimpleSoundStream)
			e = sndErrMemory;
		else {
			uint32_t sampledSoundVolume = amplitude * AUDIO_VOLUME_UNITY / sndMaxAmp;
			int16_t *dst = (int16_t*)audioOutputLoopStreamGetBuffer(mSimpleSoundStream);
			
			audiostreamSetVolumesOutput(mSimpleSoundStream, sampledSoundVolume, sampledSoundVolume);
			
			for (i = 0; i < nIter; i++, dst += sampsPerWave)
				soundCreateWave(dst, freq);
			
			audioOutputLoopStreamEnable(mSimpleSoundStream);
		}
		KALMutexRelease(mSoundMutex);
	}
	else {
		
		e = sndErrNotImpl;
	}
	
	if (e == errNone) {
		
		if (timeDuration > 0) {
			e = KALTimerCreate(&mSimpleSoundTimer, CREATE_4CC('_','s','s','L'), soundSimpleSoundTimerCbk, (void*)mSimpleSoundTimerCookie);
			if (e == errNone)
				e = KALTimerSet(mSimpleSoundTimer, timeDuration);
		}
	}
	
	return e;
}

Err DALEXPORT impl_HALPlaySmf(void *channel, SndSmfCmdEnum command, UInt8 *midiData, SndSmfOptionsType *options, SndSmfChanRangeType *channelRange, SndSmfCallbacksType *callbacks, Boolean noWait)
{
	logt("%s: refusing to do this - let Boot.prc sort it out\n", __func__);
	
	//if we return an error, boot.prc will do the
	// sequencing and use HALSoundPlay() and
	// HALSoundOff() to play. Let it do that. Why
	// should we?

	return 0xffff;
}


