#include <string.h>
#include "printf.h"
#include "kernel.h"
#include "heap.h"
#include "boot.h"
#include "kal.h"
#include "dal.h"




#define TAGFMT	"(%c%c%c%c)"
#define TAGPARAM(x)	(((x) >> 24) & 0xff),(((x) >> 16) & 0xff),(((x) >> 8) & 0xff),(((x) >> 0) & 0xff)


Err DALEXPORT impl_KALTaskCreate(uint32_t *tidP, const struct KalTaskCreateInfo *info)
{
	uint32_t flags = MEM_USABLE_AS_STACK | MEM_FAST, stackSz = (info->stackSz + 7) &~ 7;	//round to a multiple of 8
	kstatus_t sta;
	void* stackMem;
	
	logt("%s() tag " TAGFMT " from 0x%08x\n", __func__, TAGPARAM(info->tag), __builtin_return_address(0));
	#ifdef KAL_USE_STATIC_MEM_FOR_STACKS
		flags |= MEM_NO_OS_HEAP;
	#endif
		
	stackMem = kheapAllocEx(stackSz, flags);

	if (!stackMem)
		sta = KERN_STATUS_MEM_ERR;
	else {
		sta = KTaskCreate(info->tag, (void*)info->codeToRun, stackMem, stackSz, info->exinf, 0x200 - (uint32_t)info->prio, false, tidP);
		if (sta != KERN_STATUS_OK)
			kheapFree(stackMem);
	}
	logd("Task created with tid %u exinf 0x%08x and tag 0x%08x(%c%c%c%c) (sta %u)\n", *tidP, info->exinf, info->tag, (info->tag >> 24) & 0xff, (info->tag >> 16) & 0xff, (info->tag >> 8) & 0xff, (info->tag >> 0) & 0xff, sta);
	
	return kernelStatusToPalmOsErr(sta);
}

Err DALEXPORT impl_KALTaskStart(uint32_t tid, void *taskParam)
{
	logt("%s\n", __func__);
	return kernelStatusToPalmOsErr(KTaskStart(tid, taskParam));
}

Err DALEXPORT impl_KALTaskWait(int32_t timeoutMSec)
{
	tid_t tid;
	
	if (KERN_STATUS_OK != KTaskGetTid(&tid))
		tid = -1;
	
	logst("%s(%d) my tid is %u\n", __func__, timeoutMSec, tid);
	
	return kernelStatusToPalmOsErr(KTaskWait(timeoutMSec));
}

Err DALEXPORT impl_KALTaskWaitClr(void)
{
	tid_t tid;
	
	if (KERN_STATUS_OK != KTaskGetTid(&tid))
		tid = -1;
	
	logt("%s() my tid is %u\n", __func__, tid);

	return kernelStatusToPalmOsErr(KTaskWaitClr());
}

Err DALEXPORT impl_KALTaskWake(uint32_t tid)
{
	logst("%s(%u)\n", __func__, tid);
	
	return kernelStatusToPalmOsErr(KTaskWake(tid));
}

Err DALEXPORT impl_KALTaskGetCurrentID(UInt32 *tidP)
{
	logvst("%s\n", __func__);
	
	return kernelStatusToPalmOsErr(KTaskGetTid(tidP));
}

Err DALEXPORT impl_KALTaskSuspend(uint32_t tid)
{
	logi("%s(%u)\n", __func__, tid);
	
	return kernelStatusToPalmOsErr(KTaskSuspend(tid));
}

Err DALEXPORT impl_KALTaskResume(uint32_t tid)
{
	logi("%s(%u)\n", __func__, tid);
	
	return kernelStatusToPalmOsErr(KTaskResume(tid));
}


Err DALEXPORT impl_KALTaskGetInfo(uint32_t tid, struct KalTaskInfo* info)
{
	struct KernTaskInfo ki;
	kstatus_t sta;
	
	//PalmOS audio code likes to request status on NULL task in low-memory situations. It'll hang until it gets an error
	if (!tid)
		return ERR_KAL_ITEM_NOT_FOUND;
	
	logvst("%s\n", __func__);
	sta = KTaskGetInfo(tid, &ki);
	if (sta == KERN_STATUS_OK) {
		info->startFunc = (void*)ki.pc;	//close enough
		info->exinf = ki.exinf;
		info->tag = ki.tag;
		info->curStack = (void*)ki.sp;
		info->stackSz = (char*)ki.stackLimit - (char*)ki.sp;//we claim stack size is whatever stack we have FREE. who cares?
		info->prio = ki.prio;
	}
	return kernelStatusToPalmOsErr(sta);
}

Err DALEXPORT impl_KALTaskDelete(uint32_t tid)
{
	logt("%s(%d)\n", __func__, tid);
	
	return kernelStatusToPalmOsErr(KTaskDestroy(tid));
}

Err DALEXPORT impl_KALTaskDelay(uint32_t msec)
{
	logst("%s(%u)\n", __func__, msec);
	
	return kernelStatusToPalmOsErr(KTaskDelay(msec));
}

Err DALEXPORT impl_KALMutexCreate(uint32_t *mutHandleP, UInt32 tag)
{
	Err ret;
	
	logt("%s() tag " TAGFMT " from 0x%08x\n", __func__, TAGPARAM(tag), __builtin_return_address(0));
	ret = kernelStatusToPalmOsErr(KMutexCreate(tag, true, mutHandleP));
	logt("%s -> 0x%08x\n", __func__, *mutHandleP);
	
	return ret;
}

Err DALEXPORT impl_KALMutexDelete(uint32_t mutHandle)
{
	Err ret;
	
	logt("%s(0x%08x) from 0x%08x\n", __func__, mutHandle, __builtin_return_address(0));
	if (!mutHandle)
		fatal("NULL mutex @ %s from 0x%08x\n", __func__, __builtin_return_address(0));
	
	ret = kernelStatusToPalmOsErr(KMutexDestroy(mutHandle));
	logt("%s(0x%08x) from 0x%08x -> 0x%04x\n", __func__, mutHandle, __builtin_return_address(0), ret);
	
	return ret;
}

Err DALEXPORT impl_KALMutexRelease(uint32_t mutHandle)
{
	Err ret;
	
	logvst("%s(0x%08x) from 0x%08x\n", __func__, mutHandle, __builtin_return_address(0));
	if (!mutHandle) {		//nvfs devices have weird startup, including reserving a null semaphore
		loge("NULL mutex @ %s from 0x%08x\n", __func__, __builtin_return_address(0));
		return -1;
	}
	ret = kernelStatusToPalmOsErr(KMutexRelease(mutHandle));
	logvst("%s(0x%08x) from 0x%08x -> 0x%04x\n", __func__, mutHandle, __builtin_return_address(0), ret);
	
	return ret;
}

Err DALEXPORT impl_KALMutexReserve(uint32_t mutHandle, int32_t timeoutMsec)
{
	Err ret;
	
	logvst("%s(0x%08x, %ld) from 0x%08x\n", __func__, mutHandle, timeoutMsec, __builtin_return_address(0));
	if (!mutHandle) {		//nvfs devices have weird startup, including reserving a null semaphore
		loge("NULL mutex @ %s from 0x%08x\n", __func__, __builtin_return_address(0));
		return -1;
	}
	
	ret = kernelStatusToPalmOsErr(KMutexReserve(mutHandle, timeoutMsec));
	logvst("%s(0x%08x, %ld) from 0x%08x -> 0x%04x\n", __func__, mutHandle, timeoutMsec, __builtin_return_address(0), ret);
	
	return ret;
}

Err DALEXPORT impl_KALTaskSwitching(bool enable)
{
	logvst("%s(%s)\n", __func__, enable ? "true" : "false");
	
	return kernelStatusToPalmOsErr(KTaskSwitching(enable));
}

Err DALEXPORT impl_KALEventGroupCreate(uint32_t *evtGrpHandleP, uint32_t tag, uint32_t startingState)
{
	Err ret;
	
	logt("%s(0x%08lx, '" TAGFMT "', 0x%08lx) from 0x%08x\n", __func__, evtGrpHandleP, TAGPARAM(tag), startingState, __builtin_return_address(0));
	ret = kernelStatusToPalmOsErr(KEventGroupCreate(tag, startingState, evtGrpHandleP));
	logt("%s -> 0x%04x,0x%08x\n", __func__, ret, *evtGrpHandleP);
	
	return ret;
}

Err DALEXPORT impl_KALEventGroupDelete(uint32_t evtGrpHandle)
{
	Err ret;
	
	logt("%s(0x%08x) from 0x%08x\n", __func__, evtGrpHandle, __builtin_return_address(0));
	ret = kernelStatusToPalmOsErr(KEventGroupDestroy(evtGrpHandle));
	logt("%s(0x%08x) from 0x%08x -> 0x%04x\n", __func__, evtGrpHandle, __builtin_return_address(0), ret);
	
	return ret;
}

Err DALEXPORT impl_KALEventGroupRead(uint32_t evtGrpHandle, uint32_t *evtsP)
{
	Err ret;
	
	logt("%s(0x%08x) from 0x%08x\n", __func__, evtGrpHandle, __builtin_return_address(0));
	ret = kernelStatusToPalmOsErr(KEventGroupRead(evtGrpHandle, evtsP));
	logt("%s(0x%08x) from 0x%08x -> (0x%08x), 0x%04x\n", __func__, evtGrpHandle, __builtin_return_address(0), *evtsP, ret);
	
	return ret;
}

Err DALEXPORT impl_KALEventGroupSignal(uint32_t evtGrpHandle, uint32_t evts)
{
	Err ret;
	
	logst("%s(0x%08x) from 0x%08x\n", __func__, evtGrpHandle, __builtin_return_address(0));
	ret = kernelStatusToPalmOsErr(KEventGroupSignal(evtGrpHandle, evts));
	logst("%s(0x%08x) from 0x%08x -> 0x%04x\n", __func__, evtGrpHandle, __builtin_return_address(0), ret);
	
	return ret;
}

Err DALEXPORT impl_KALEventGroupClear(uint32_t evtGrpHandle, uint32_t evts)
{
	Err ret;
	
	logst("%s(0x%08x) from 0x%08x\n", __func__, evtGrpHandle, __builtin_return_address(0));
	ret = kernelStatusToPalmOsErr(KEventGroupClear(evtGrpHandle, evts));
	logst("%s(0x%08x) from 0x%08x -> 0x%04x\n", __func__, evtGrpHandle, __builtin_return_address(0), ret);
	
	return ret;
}

Err DALEXPORT impl_KALEventGroupWait(uint32_t evtGrpHandle, const struct KalEvtGrpWaitReq *req, UInt32 *evtsP)
{
	Err ret;
	
	logst("%s(0x%08x) from 0x%08x\n", __func__, evtGrpHandle, __builtin_return_address(0));
	
	if (req->waitType != KAL_EVT_GRP_WAIT_TYPE_AND && req->waitType != KAL_EVT_GRP_WAIT_TYPE_OR)
		fatal("unknown wait type %u\n", req->waitType);
	
	ret = kernelStatusToPalmOsErr(KEventGroupWait(evtGrpHandle, req->wantedBits, evtsP, req->timeoutMsec, req->waitType == KAL_EVT_GRP_WAIT_TYPE_AND));
	logst("%s(0x%08x) from 0x%08x -> (0x%08x), 0x%04x\n", __func__, evtGrpHandle, __builtin_return_address(0), *evtsP, ret);
	
	return ret;
}

Err DALEXPORT impl_KALSemaphoreCreate(uint32_t *semHandleP, uint32_t tag, uint32_t initialVal)
{
	Err ret;
	
	logt("%s() tag " TAGFMT " from 0x%08x\n", __func__, TAGPARAM(tag), __builtin_return_address(0));
	ret = kernelStatusToPalmOsErr(KSemaphoreCreate(tag, initialVal, semHandleP));
	logt("%s -> 0x%08x\n", __func__, *semHandleP);
	
	return ret;
}

Err DALEXPORT impl_KALSemaphoreDelete(uint32_t semHandle)
{
	Err ret;
	
	logt("%s(0x%08x) from 0x%08x\n", __func__, semHandle, __builtin_return_address(0));
	ret = kernelStatusToPalmOsErr(KSemaphoreDestroy(semHandle));
	logt("%s(0x%08x) from 0x%08x -> 0x%04x\n", __func__, semHandle, __builtin_return_address(0), ret);
	
	return ret;
}

Err DALEXPORT impl_KALSemaphoreSignal(uint32_t semHandle)
{
	Err ret;
	
	logst("%s(0x%08x) from 0x%08x\n", __func__, semHandle, __builtin_return_address(0));
	ret = kernelStatusToPalmOsErr(KSemaphorePost(semHandle));
	logst("%s(0x%08x) from 0x%08x -> 0x%04x\n", __func__, semHandle, __builtin_return_address(0), ret);
	
	return ret;
}

Err DALEXPORT impl_KALSemaphoreWait(uint32_t semHandle, int32_t timeout)
{
	Err ret;
	
	logst("%s(0x%08x) from 0x%08x\n", __func__, semHandle, __builtin_return_address(0));
	ret = kernelStatusToPalmOsErr(KSemaphoreWait(semHandle, timeout));
	logst("%s(0x%08x) from 0x%08x -> 0x%04x\n", __func__, semHandle, __builtin_return_address(0), ret);
	
	return ret;
}

Err DALEXPORT impl_KALTimerCreate(uint32_t *timerIDP, uint32_t tag, KALTimerProcFn cbk, void *userData)
{
	Err ret;
	
	logst("%s() tag " TAGFMT " from 0x%08x\n", __func__, TAGPARAM(tag), __builtin_return_address(0));
	ret = kernelStatusToPalmOsErr(KTimerCreate(tag, (KernTimerCbk)(uintptr_t)cbk, userData, timerIDP));	//cllback cast only safe bc arm calling convention makes sure the one param is properly passed
	logst("%s(0x%08x) from 0x%08x -> 0x%04x\n", __func__, *timerIDP, __builtin_return_address(0), ret);
	
	return ret;
}

Err DALEXPORT impl_KALTimerDelete(uint32_t timerID)
{
	Err ret;
	
	logt("%s(0x%08x) from 0x%08x\n", __func__, timerID, __builtin_return_address(0));
	ret = kernelStatusToPalmOsErr(KTimerDestroy(timerID));
	logt("%s(0x%08x) from 0x%08x -> 0x%04x\n", __func__, timerID, __builtin_return_address(0), ret);
	
	return ret;
}

Err DALEXPORT impl_KALTimerSet(uint32_t timerID, uint32_t milliseconds)
{
	Err ret;
	
	logt("%s(0x%08x) from 0x%08x\n", __func__, timerID, __builtin_return_address(0));
	ret = kernelStatusToPalmOsErr(KTimerSet(timerID, milliseconds));
	logt("%s(0x%08x) from 0x%08x -> 0x%04x\n", __func__, timerID, __builtin_return_address(0), ret);
	
	return ret;
}


Err DALEXPORT impl_KALMailboxCreate(uint32_t *mailboxIDP, uint32_t tag, uint32_t depth)
{
	void* storage = kheapAlloc(KERN_MAILBOX_STORAGE_NEEDS(depth));
	kstatus_t sta;
	Err ret;
	
	logst("%s() tag " TAGFMT " from 0x%08x\n", __func__, TAGPARAM(tag), __builtin_return_address(0));
	
	if (!storage)
		sta = KERN_STATUS_MEM_ERR;
	else {
		sta = KMailboxCreate(tag, depth, storage, mailboxIDP);
		if (sta != KERN_STATUS_OK)
			kheapFree(storage);
	}
	ret = kernelStatusToPalmOsErr(sta);
	logst("%s(0x%08x) from 0x%08x -> 0x%04x\n", __func__, *mailboxIDP, __builtin_return_address(0), ret);
	
	return ret;
}

Err DALEXPORT impl_KALMailboxDelete(uint32_t mailboxID)
{
	void* storage;
	Err ret;
	
	logt("%s(0x%08x) from 0x%08x\n", __func__, mailboxID, __builtin_return_address(0));
	ret = kernelStatusToPalmOsErr(KMailboxDestroy(mailboxID, &storage));
	logt("freeing mbx mem\n");
	kheapFree(storage);
	logt("%s(0x%08x) from 0x%08x -> 0x%04x\n", __func__, mailboxID, __builtin_return_address(0), ret);
	
	return ret;
}

Err DALEXPORT impl_KALMailboxSend(uint32_t mailboxID, uint32_t message)
{
	Err ret;
	
	logst("%s(0x%08x) from 0x%08x\n", __func__, mailboxID, __builtin_return_address(0));
	ret = kernelStatusToPalmOsErr(KMailboxSend(mailboxID, message));
	logst("%s(0x%08x) from 0x%08x -> 0x%04x\n", __func__, mailboxID, __builtin_return_address(0), ret);
	
	return ret;
}

Err DALEXPORT impl_KALMailboxWait(uint32_t mailboxID, uint32_t *messageP, int32_t timeout)
{
	Err ret;
	
	logst("%s(0x%08x) from 0x%08x\n", __func__, mailboxID, __builtin_return_address(0));
	ret = kernelStatusToPalmOsErr(KMailboxWait(mailboxID, messageP, timeout));
	logst("%s(0x%08x) from 0x%08x -> 0x%04x\n", __func__, mailboxID, __builtin_return_address(0), ret);
	
	return ret;
}
