#include "input.h"
#include "joystick.h"
#include "hw/CortexEmu/inputIface.h"
#include "CortexEmuCpu.h"
#include "printf.h"


static DrvInputKeyCbk mKeyCbk = NULL;
static DrvInputBtnCbk mBtnCbk = NULL;
static DrvInputPenCbk mPenCbk = NULL;
static JoystickReportF mJoyCbk = NULL;
static uint32_t mJoyInterval = 0;

void __attribute__((used)) Input_IRQHandler(void)
{
	static uint32_t mPrevKeyState = 0, mPrevPenState = 0xFFFFFFFF;
	uint32_t t;
	
	t = *(volatile uint32_t*)(INPUT_UNIT_BASE + INPUT_OFST_BTN_STATE);
	if (t != mPrevKeyState) {
		
		if (mBtnCbk) {
			uint32_t diff = mPrevKeyState ^ t;
			uint32_t newlyPressed = t & diff;
			uint32_t newlyReleased = mPrevKeyState & diff;
			
			while (newlyPressed) {
				uint32_t bit = newlyPressed - (newlyPressed & (newlyPressed - 1));
				newlyPressed &=~ bit;
				mBtnCbk(bit, true);
			}
			while (newlyReleased) {
				uint32_t bit = newlyReleased - (newlyReleased & (newlyReleased - 1));
				newlyReleased &=~ bit;
				mBtnCbk(bit, false);
			}
		}
		mPrevKeyState = t;
	}
	
	t = *(volatile uint32_t*)(INPUT_UNIT_BASE + INPUT_OFST_PEN_STATE);
	if (t != mPrevPenState) {
		
		if (mPenCbk)
			mPenCbk((t & 0x80000000) ? (int32_t)(t >> 12) & 0xFFF : -1, (t & 0x80000000) ? (int32_t)(t >> 0) & 0xFFF : -1);
		mPrevPenState = t;
	}
	
	t = *(volatile uint32_t*)(INPUT_UNIT_BASE + INPUT_OFST_KBD_STATE);
	if ((t & 0x80000000) && mKeyCbk)
		mKeyCbk((t >> 16) & 0x0F, t & 0xFFFF);
}

kstatus_t drvInputInit(DrvInputKeyCbk keyCbk, DrvInputBtnCbk btnCbk, DrvInputPenCbk penCbk)
{
	mKeyCbk = keyCbk;
	mBtnCbk = btnCbk;
	mPenCbk = penCbk;
	
	NVIC_EnableIRQ(Input_IRQn);
	
	return KERN_STATUS_OK;
}

kstatus_t drvInputPreSleep(void)
{
	NVIC_DisableIRQ(Input_IRQn);
	
	return KERN_STATUS_OK;
}

kstatus_t drvInputPostWake(void)
{
	NVIC_EnableIRQ(Input_IRQn);
	
	return KERN_STATUS_OK;
}

void __attribute__((used)) Joystick_IRQHandler(void)
{
	int16_t x, y;
	
	joystickRead(&x, &y);
	if (mJoyCbk)
		mJoyCbk(x, y);
	
	NVIC_ClearPendingIRQ(Joystick_IRQn);
}

void joystickInit(JoystickReportF reportF)
{
	mJoyCbk = reportF;
	joystickSetRate(0);
	NVIC_EnableIRQ(Joystick_IRQn);
}

void joystickSetRate(uint32_t ticksBetween)
{
	*(volatile uint32_t*)(INPUT_UNIT_BASE + INPUT_OFST_JOY_INTERVAL) = ticksBetween;
}

void joystickRead(int16_t *x, int16_t *y)
{
	*x = *(volatile uint32_t*)(INPUT_UNIT_BASE + INPUT_OFST_JOYSTICK_X);
	*y = *(volatile uint32_t*)(INPUT_UNIT_BASE + INPUT_OFST_JOYSTICK_Y);
}

uint32_t drvInputReadKeysEarly(void)
{
	return *(volatile uint32_t*)(INPUT_UNIT_BASE + INPUT_OFST_BTN_STATE);
}