#include <stdint.h>
#include "irqsCortex.h"
#include "timers.h"
#include "printf.h"
#include "irqs.h"
#include "rtc.h"
#include "cpu.h"

static const uint16_t mCumulativeDays[] = {0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334};		//index [n] is how many days in the year BEFORE the 1st of month N+1
static DrvRtcCbk mAlmCbk = NULL, mHzCbk = NULL;


void __attribute__((used)) RTC_IRQHandler(void)
{
	NVIC_ClearPendingIRQ(RTC_IRQn);
	rtc_hw->inte = 0;
	
	logt("*ALARM*\n");
	if (mAlmCbk)
		mAlmCbk();
}

void __attribute__((used)) TIMER1_IRQHandler(void)	//our Hz tick
{
	uint32_t prevV = timer_hw->alarm[1];
	
	timer_hw->intr = 1 << 1;
	timer_hw->alarm[1] = timer_hw->timerawl + CPU_CLOCK_RATE / SYSTICK_CLOCKS_PER_TIMER_TICK;
	
	if (mHzCbk)
		mHzCbk();
}

void __attribute__((used)) TIMER0_IRQHandler(void)
{
	timer_hw->intr = 1 << 0;
	timerExternalIrq();
}

void hwTimerInit(void)
{
	watchdog_hw->tick = WATCHDOG_TICK_ENABLE_BITS + 12000000ull * SYSTICK_CLOCKS_PER_TIMER_TICK / CPU_CLOCK_RATE;
	TIMER1_IRQHandler();
	timer_hw->inte |= (1 << 0) | (1 << 1);
	NVIC_EnableIRQ(TIMER1_IRQn);
}

uint32_t hwTimerIntOff(void)
{
	uint32_t ret = NVIC_IsEnabledIRQ(TIMER0_IRQn);		//allows re-entrancy
	NVIC_DisableIRQ(TIMER0_IRQn);
	
	return ret;
}

uint32_t hwTimerIntOn(void)
{
	uint32_t ret = NVIC_IsEnabledIRQ(TIMER0_IRQn);		//allows re-entrancy
	NVIC_EnableIRQ(TIMER0_IRQn);
	
	return ret;
}

void hwTimerIntRestore(uint32_t state)
{
	if (state)
		NVIC_EnableIRQ(TIMER0_IRQn);
	else
		NVIC_DisableIRQ(TIMER0_IRQn);
}

void hwTimerSet(uint32_t ticksFromNow)
{
	if (ticksFromNow)
		timer_hw->alarm[0] = timer_hw->timerawl + ticksFromNow;
	else
		timer_hw->armed = 1 << 0;
}

static uint32_t rtcPrvRawRead(void)
{
	uint32_t hi, lo, t, days = 0, seconds;
	
	do {
		lo = rtc_hw->rtc_0;
		hi = rtc_hw->rtc_1;
	} while (hi != rtc_hw->rtc_1 || lo != rtc_hw->rtc_0);
	
	days += 365 * ((hi & RTC_RTC_1_YEAR_BITS) >> RTC_RTC_1_YEAR_LSB);		//we force the RTC to never think of leap years so as to use it as a counter
	days += ((hi & RTC_RTC_1_DAY_BITS) >> RTC_RTC_1_DAY_LSB) - 1;
	days += mCumulativeDays[((hi & RTC_RTC_1_MONTH_BITS) >> RTC_RTC_1_MONTH_LSB) - 1];
	
	seconds = days * 24 * 60 * 60;
	seconds += 60 * 60 * (lo & RTC_RTC_0_HOUR_BITS) >> RTC_RTC_0_HOUR_LSB;
	seconds += 60 * (lo & RTC_RTC_0_MIN_BITS) >> RTC_RTC_0_MIN_LSB;
	seconds += (lo & RTC_RTC_0_SEC_BITS) >> RTC_RTC_0_SEC_LSB;
	
	return seconds;
}

static void rtcPrvTimestampToPieces(uint32_t *secP, uint32_t *minP, uint32_t *hrP, uint32_t *dayP, uint32_t *monthP, uint32_t *yrP, uint32_t time)
{
	uint32_t month;
	
	*secP = time % 60;
	time /= 60;
	*minP = time % 60;
	time /= 60;
	*hrP = time % 24;
	time /= 24;
	
	//time is now in days - calculate year
	*yrP = time / 365;
	time %= 365;
	
	//now we need to allocate days and months
	for (month = 1; month < 12 && mCumulativeDays[month] < time; month++);
	*dayP = time - mCumulativeDays[month] + 1;
	*monthP = month;
}

static void rtcPrvRawWrite(uint32_t time)
{
	uint32_t sec, min, hr, day, month, yr;
	
	rtcPrvTimestampToPieces(&sec, &min, &hr, &day, &month, &yr, time);
	
	//produce the register values
	rtc_hw->setup_0 = (yr << RTC_SETUP_0_YEAR_LSB) + (month << RTC_SETUP_0_MONTH_LSB) + (day << RTC_SETUP_0_DAY_LSB);
	rtc_hw->setup_1 = (hr << RTC_SETUP_1_HOUR_LSB) + (min << RTC_SETUP_1_MIN_LSB) + (sec << RTC_SETUP_1_SEC_LSB);
	
	rtc_hw->ctrl = RTC_CTRL_LOAD_BITS;
	while (rtc_hw->ctrl & RTC_CTRL_LOAD_BITS);
}

static void rtcPrvSetAlarm(uint32_t time)
{
	rtc_hw->inte = 1;
	rtc_hw->irq_setup_0 = 0;
	
	if (time) {
		
		uint32_t sec, min, hr, day, month, yr;
		
		rtcPrvTimestampToPieces(&sec, &min, &hr, &day, &month, &yr, time);
		while (rtc_hw->irq_setup_0 & RTC_IRQ_SETUP_0_MATCH_ACTIVE_BITS);
		rtc_hw->irq_setup_1 = (RTC_IRQ_SETUP_1_HOUR_ENA_BITS | RTC_IRQ_SETUP_1_MIN_ENA_BITS | RTC_IRQ_SETUP_1_SEC_ENA_BITS) + (hr << RTC_IRQ_SETUP_1_HOUR_LSB) + (min << RTC_IRQ_SETUP_1_MIN_LSB) + (sec << RTC_IRQ_SETUP_1_SEC_LSB);
		rtc_hw->irq_setup_0 = (RTC_IRQ_SETUP_0_MATCH_ENA_BITS | RTC_IRQ_SETUP_0_YEAR_ENA_BITS | RTC_IRQ_SETUP_0_MONTH_ENA_BITS | RTC_IRQ_SETUP_0_DAY_ENA_BITS) + (yr << RTC_IRQ_SETUP_0_YEAR_LSB) + (month << RTC_IRQ_SETUP_0_MONTH_LSB) + (day << RTC_IRQ_SETUP_0_DAY_LSB);
		while (!(rtc_hw->irq_setup_0 & RTC_IRQ_SETUP_0_MATCH_ACTIVE_BITS));
		NVIC_ClearPendingIRQ(RTC_IRQn);
		rtc_hw->inte = 0;
	}
}

kstatus_t drvRtcInit(DrvRtcCbk hzCbk, DrvRtcCbk almCbk)
{
	uint32_t time;
	kstatus_t sta;
	
	mAlmCbk = almCbk;
	mHzCbk = hzCbk;
		
	//clock it up to XOSC speed / 256
	clocks_hw->clk[clk_rtc].ctrl = (clocks_hw->clk[clk_rtc].ctrl &~ (CLOCKS_CLK_RTC_CTRL_AUXSRC_BITS | CLOCKS_CLK_RTC_CTRL_ENABLE_BITS | CLOCKS_CLK_RTC_CTRL_KILL_BITS)) | CLOCKS_CLK_RTC_CTRL_ENABLE_BITS | CLOCKS_CLK_RTC_CTRL_AUXSRC_VALUE_XOSC_CLKSRC;
	clocks_hw->clk[clk_rtc].div = (256 << CLOCKS_CLK_RTC_DIV_INT_LSB);
	
	rtc_hw->ctrl = 0;
	while (rtc_hw->ctrl & RTC_CTRL_RTC_ACTIVE_BITS);
	rtc_hw->ctrl = RTC_CTRL_FORCE_NOTLEAPYEAR_BITS;
	
	rtc_hw->clkdiv_m1 = 46874;
	rtcPrvRawWrite(0xE0498B2B);	//the time when this code was built
	
	rtc_hw->ctrl |= RTC_CTRL_RTC_ENABLE_BITS;
	
	drvRtcSetAlarm(0);
	NVIC_EnableIRQ(RTC_IRQn);
	
	return KERN_STATUS_OK;
}

kstatus_t drvRtcPreSleep(void)
{
	timer_hw->inte &=~ (1 << 1);
	
	return KERN_STATUS_OK;
}

kstatus_t drvRtcPostWake(void)
{
	timer_hw->inte |= (1 << 1);
	
	return KERN_STATUS_OK;
}

kstatus_t drvRtcGet(uint32_t *valP)
{
	*valP = rtcPrvRawRead();
	
	return KERN_STATUS_OK;
}

kstatus_t drvRtcSet(uint32_t val)
{
	rtcPrvRawWrite(val);
	
	return KERN_STATUS_OK;
}

kstatus_t drvRtcSetAlarm(uint32_t at)
{
	//*(volatile uint32_t*)(RTC_UNIT_BASE + RTC_OFST_ALARM) = at;

	return KERN_STATUS_OK;
}
