#include "../dma_driver/DmaDriver.h"
#include "audioCommon.h"
#include "ac97codec.h"
#include <stdint.h>
#include <string.h>
#include <stdio.h>
#include "audiohw.h"
#include "printf.h"
#include "cpu.h"


#define AUDIO_BUFFER_ENTRIES	1024			//L+R is one sample: U32 (we only read left though)

static AudioInHwSamplesReadyF mAcceptSamplesF;
static DmaStream mAudioInDmaStream;
static uint16_t *mAudioInBuf;			//must be 32-bit but we use 16, teehee


static void audioInHwPrvSendData(bool secondHalf)
{
	mAcceptSamplesF(mAudioInBuf + (secondHalf ? (AUDIO_BUFFER_ENTRIES / 2) : 0), 0);
}

static void audioInPrvDmaIrqHandler(void* userData, uint32_t strmSta)
{
	if (strmSta & DMA_STRM_IRQ_HALF)
		audioInHwPrvSendData(false);
	else if (strmSta & DMA_STRM_IRQ_DONE)
		audioInHwPrvSendData(true);
}

bool audioInHwInit(AudioInHwSamplesReadyF acceptSamplesF, uint32_t *numSamplesPerBufP)
{
	static const struct DmaStreamUserCfg audioInDmaCfg = {
		.magic = CFG_STRUCT_MAGIX,
		.chan = XSCALE_DMA_CH_AC97_AUD_RX,
		.circBuf = 1,
		.perSz = __builtin_ctz(sizeof(uint16_t)),
		.memSz = __builtin_ctz(sizeof(uint16_t)),
		.perControlsFlow = 1,
		.memIncr = true,
		.perBurstSz = 16,
		.memBurstSz = 16,
		.toMem = true,
		.numItems = AUDIO_BUFFER_ENTRIES,
	};
	uintptr_t audioBufPa;
	uint16_t val;
	
	aximAudioCtlInit();
	
	if (!platDmaBufAlloc(AUDIO_BUFFER_ENTRIES * sizeof(uint16_t), &audioBufPa, (void**)&mAudioInBuf)) {
		loge("Cannot alloc audio in DMA buffer\n");
		return false;
	}
	
	//recording is from "mic" source
	ac97codecRegWrite(0x1a, 0x0000);
	
	//enable 20dB mic boost but mute the mic path into audio mixer
	ac97codecRegRead(0x0e, &val);
	ac97codecRegWrite(0x0e, val | 0x0040);
	
	//set vol
	ac97codecRegWrite(0x1c, 0x0505);

	//variable sampling rate enabled
	ac97codecRegWrite(0x2a, 1);
	
	memset(mAudioInBuf, 0, AUDIO_BUFFER_ENTRIES * sizeof(uint16_t));
	
	mAudioInDmaStream = DmaLibStreamReserve(0, 13);
	if (!mAudioInDmaStream)
		fatal("Cannot grab input DMA stream\n");
		
	if (!DmaLibStreamConfigure(mAudioInDmaStream, &audioInDmaCfg))
		fatal("Cannot configure audio out DMA stream\n");
	
	if (!DmaLibStreamSetIrqHandler(mAudioInDmaStream, audioInPrvDmaIrqHandler, NULL))
		fatal("Cannot configure audio out DMA irq handler\n");
	
	//addr should be PA, hence this weird construct
	if (!DmaLibStreamSetPeriphAddr(mAudioInDmaStream, (uintptr_t)&((struct PxaAC97*)PXA_BASE_AC97)->PCDR))
		fatal("Cannot configure audio out DMA PADDR\n");

	if (!DmaLibStreamSetMemAddr(mAudioInDmaStream, 0, audioBufPa))
		fatal("Cannot configure audio out DMA MADDR\n");
	
	if (!DmaLibStreamSetIrqState(mAudioInDmaStream, DMA_STRM_IRQ_HALF | DMA_STRM_IRQ_DONE))
		fatal("Cannot enable audio out DMA irqs\n");
	
	
	mAcceptSamplesF = acceptSamplesF;
	*numSamplesPerBufP = AUDIO_BUFFER_ENTRIES / 2;
	
	return true;
}

bool audioInHwSetState(bool on, enum AudioSampleRate rate)
{
	if (on) {
		
		uint32_t rateVal;
		uint16_t read;
		
		switch (rate) {
			case AudioRate8000:
				rateVal = 8000;
				break;
			
			case AudioRate11025:
				rateVal = 11025;
				break;
			
			case AudioRate16000:
				rateVal = 16000;
				break;
			
			case AudioRate22050:
				rateVal = 22050;
				break;
			
			case AudioRate32000:
				rateVal = 32000;
				break;
			
			case AudioRate44100:
				rateVal = 44100;
				break;
			
			case AudioRate48000:
				rateVal = 48000;
				break;
			
			default:
				//this includes 24K sampling rate since this chip does not support it
				loge("Sampling rate requested is unsupported on this hardware\n");
				return false;
		}
		aximAudioCtlAdc(true);
		aximAudioCtlAdc(false);
		aximAudioCtlAdc(true);
		do {
			logi("Setting rate to %u\n", rateVal);
			ac97codecRegWrite(0x32, rateVal);
			ac97codecRegRead(0x32, &read);
		} while (read != rateVal);

		if (!DmaLibStreamSetEnabled(mAudioInDmaStream, true))
			fatal("Cannot set state for audio in DMA\n");
	}
	else {
		if (!DmaLibStreamSetEnabled(mAudioInDmaStream, false))
			fatal("Cannot set state for audio in DMA\n");
		
		aximAudioCtlAdc(false);
	}
	
	return true;
}


