#include "../dma_driver/DmaDriver.h"
#include "audioCommon.h"
#include "ac97codec.h"
#include "aximCpld.h"
#include "platform.h"
#include "audiohw.h"
#include "printf.h"
#include "xscale.h"
#include <stdint.h>
#include <string.h>
#include <stdio.h>

#define AUDIO_BUFFER_ENTRIES	1024			//L+R is one sample: U32

static AudioOutHwReadyForMoreSamplesF mReadyForSamplesF;
static DmaStream mAudioOutDmaStream;
static uint32_t *mAudioOutBuf;

//codec state
static uint8_t mAdcRefCt = 0xff, mDacRefCt = 0xff;

#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC push_options
	#pragma GCC target ("arm")
#endif


static void __attribute__((naked)) audioHwConvertSamples(uint32_t *dst, const int32_t *src, uint32_t *dstEnd)
{
	asm volatile(
		"	ldr    r12, =0x7fff		\n\t"
		"	push   {r4-r7}			\n\t"
		"1:							\n\t"
		
		"	ldmia  r1!, {r4-r7}		\n\t"
		
		"	mov    r4, r4, asr #8	\n\t"
		"	movs   r3, r4, asr #15	\n\t"
		"	addmis r3, r3, #1		\n\t"
		"	movne  r4, r12			\n\t"
		"	negmi  r4, r4			\n\t"
		
		"	mov    r5, r5, asr #8	\n\t"
		"	movs   r3, r5, asr #15	\n\t"
		"	addmis r3, r3, #1		\n\t"
		"	movne  r5, r12			\n\t"
		"	negmi  r5, r5			\n\t"
		
		"	mov    r6, r6, asr #8	\n\t"
		"	movs   r3, r6, asr #15	\n\t"
		"	addmis r3, r3, #1		\n\t"
		"	movne  r6, r12			\n\t"
		"	negmi  r6, r6			\n\t"
		
		"	mov    r7, r7, asr #8	\n\t"
		"	movs   r3, r7, asr #15	\n\t"
		"	addmis r3, r3, #1		\n\t"
		"	movne  r7, r12			\n\t"
		"	negmi  r7, r7			\n\t"
		
		"	strh   r7, [r0, #6]		\n\t"
		"	strh   r6, [r0, #4]		\n\t"
		"	strh   r5, [r0, #2]		\n\t"
		"	strh   r4, [r0], #8		\n\t"
		
		"	cmp    r0, r2			\n\t"
		"	bne    1b				\n\t"
		
		"	pop    {r4-r7}			\n\t"
		"	bx    lr				\n\t"
		:
		:
		:"memory", "cc"
	);
}

#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC pop_options
#endif

static void audioHwPrvReqData(bool secondHalf)
{
	audioHwConvertSamples(mAudioOutBuf + (secondHalf ? AUDIO_BUFFER_ENTRIES / 2 : 0), mReadyForSamplesF(true), mAudioOutBuf + (secondHalf ? AUDIO_BUFFER_ENTRIES : AUDIO_BUFFER_ENTRIES / 2));
	mReadyForSamplesF(false);
}

static void audioOutPrvDmaIrqHandler(void* userData, uint32_t strmSta)
{
	if (strmSta & DMA_STRM_IRQ_HALF)
		audioHwPrvReqData(false);
	else if (strmSta & DMA_STRM_IRQ_DONE)
		audioHwPrvReqData(true);
}

bool audioOutHwInit(AudioOutHwReadyForMoreSamplesF readyForSamplesF, uint32_t *numSamplesPerBufP, enum AudioSampleRate* nativeRateP, bool *nativeStereoP)
{
	static const struct DmaStreamUserCfg audioOutDmaCfg = {
		.magic = CFG_STRUCT_MAGIX,
		.chan = XSCALE_DMA_CH_AC97_AUD_TX,
		.circBuf = 1,
		.perSz = __builtin_ctz(sizeof(uint32_t)),
		.memSz = __builtin_ctz(sizeof(uint32_t)),
		.perControlsFlow = 1,
		.memIncr = true,
		.perBurstSz = 32,
		.memBurstSz = 32,
		.toMem = 0,
		.numItems = AUDIO_BUFFER_ENTRIES,
	};
	uintptr_t audioBufPa;
	uint_fast8_t i;
	
	aximAudioCtlInit();
	
	if (!platDmaBufAlloc(AUDIO_BUFFER_ENTRIES * sizeof(uint32_t), &audioBufPa, (void**)&mAudioOutBuf)) {
		loge("Cannot alloc audio DMA buffer\n");
		return false;
	}
	
	//max vol
	ac97codecRegWrite(0x02, 0x0000);	//affects headphone volume only (line out)
	ac97codecRegWrite(0x06, 0x0000);	//affects internal speaker only (mono out)
	
	//clear POP bit, clear MIX so mono out is from DAC
	ac97codecRegWrite(0x20, 0x0000);
	
	//do not tristate mono out - it is our speaker output
	ac97codecRegWrite(0x5c, 0x0000);
	
	//mono amp on
	ac97codecRegWrite(0x74, 0x0001);
	
	//variable sampling rate enabled
	ac97codecRegWrite(0x2a, 1);
	
	//do not play mic stuff into speaker
	for (i = 0x0c; i <= 0x16; i += 2) {
		
		uint16_t val;
	
		ac97codecRegRead(i, &val);
		ac97codecRegWrite(i, val | 0x8000);
	}
	
	//do play our DAC
	ac97codecRegWrite(0x18, 0);
	
	memset(mAudioOutBuf, 0, AUDIO_BUFFER_ENTRIES * sizeof(uint32_t));
	
	mAudioOutDmaStream = DmaLibStreamReserve(0, 14);
	if (!mAudioOutDmaStream)
		fatal("Cannot grab output DMA stream\n");
		
	if (!DmaLibStreamConfigure(mAudioOutDmaStream, &audioOutDmaCfg))
		fatal("Cannot configure audio out DMA stream\n");
	
	if (!DmaLibStreamSetIrqHandler(mAudioOutDmaStream, audioOutPrvDmaIrqHandler, NULL))
		fatal("Cannot configure audio out DMA irq handler\n");
	
	//addr should be PA, hence this weird construct
	if (!DmaLibStreamSetPeriphAddr(mAudioOutDmaStream, (uintptr_t)&((struct PxaAC97*)PXA_BASE_AC97)->PCDR))
		fatal("Cannot configure audio out DMA PADDR\n");

	if (!DmaLibStreamSetMemAddr(mAudioOutDmaStream, 0, audioBufPa))
		fatal("Cannot configure audio out DMA MADDR\n");
	
	if (!DmaLibStreamSetIrqState(mAudioOutDmaStream, DMA_STRM_IRQ_HALF | DMA_STRM_IRQ_DONE))
		fatal("Cannot enable audio out DMA irqs\n");
	
	
	*nativeStereoP = true;
	*nativeRateP = AudioRate48000;
	*numSamplesPerBufP = AUDIO_BUFFER_ENTRIES / 2;
	mReadyForSamplesF = readyForSamplesF;
	
	return true;
}

void audioOutHwSetState(bool on)
{
	aximAudioCtlDac(on);
	if (on)		//set rate
		ac97codecRegWrite(0x2c, 48000);
	aximCpldSetBits(AXIM_CPLD_BIT_AUDIO_CODEC_AMP_POWER, on);
	if (!DmaLibStreamSetEnabled(mAudioOutDmaStream, on))
		fatal("Cannot enable audio out DMA\n");
}

bool audioOnlySimpleOutInit(void)
{
	return false;
}

void audioOnlySimpleTone(uint32_t freq, uint32_t amp)
{
	fatal("Unexpected call to %s\n", __func__);
}



//the common code to manage codec state


static void aximAudioCtlPrvCodecRegsRefresh(void)
{
	uint32_t val = 0xef00;	//aclink on so digi can work
	
	if (mAdcRefCt) {
		
		logi("ERR: fix up minimu mic cfg\n");
		
		val &=~ (1 << (8 + 0));	//PR0 - ADCs on
		val &=~ (1 << (8 + 2));	//PR2 - mixer on
		val &=~ (1 << (8 + 3));	//PR3 - mixer vref on
		val &=~ (1 << (8 + 5));	//PR5 - internal clock must be on
		
	}
	if (mDacRefCt) {
		
		val &=~ (1 << (8 + 1));	//PR1 - DACs need to be on
		val &=~ (1 << (8 + 2));	//PR2 - mixer on
		val &=~ (1 << (8 + 3));	//PR3 - mixer vref on
		val &=~ (1 << (8 + 5));	//PR5 - internal clock must be on
		val &=~ (1 << (8 + 6));	//PR6 - headphone out on
		val &=~ (1 << (8 + 7));	//EAPD
	}
	ac97codecRegWrite(0x26, val);
	ac97codecRegWrite(0x2a, 1);
}

void aximAudioCtlInit(void)
{
	if (mAdcRefCt == 0xff && mDacRefCt == 0xff) {
		
		mAdcRefCt = 0;
		mDacRefCt = 0;
		aximAudioCtlPrvCodecRegsRefresh();
	}
}

void aximAudioCtlDac(bool up)
{
	aximAudioCtlInit();
	
	if (up)
		mDacRefCt++;
	else
		mDacRefCt--;
	
	aximAudioCtlPrvCodecRegsRefresh();
}

void aximAudioCtlAdc(bool up)
{
	aximAudioCtlInit();
	
	if (up)
		mAdcRefCt++;
	else
		mAdcRefCt--;
	
	aximAudioCtlPrvCodecRegsRefresh();
}
