#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include <stdio.h>
#include "aximPwrCtl.h"
#include "halMemory.h"
#include "platform.h"
#include "aximCpld.h"
#include "xscale.h"
#include "memmap.h"
#include "printf.h"
#include "timers.h"
#include "xscale.h"
#include "wm9705.h"
#include "input.h"
#include "entry.h"
#include "misc.h"
#include "heap.h"
#include "irqs.h"
#include "ac97.h"
#include "mpu.h"

#define NON_PORTABLE
#include <HwrMiscFlags.h>
#undef NON_PORTABLE

static char mSnum[12];
static uint8_t mDevFlags;	//just the relevant bits

#define DEV_FLAGS_KNOWN		0x80
#define DEV_FLAG_HIGH_END	0x10

static void setupAximX3gpios(void)
{
	uint32_t i;
	
	//general cleanup
	for (i = 0; i < 90; i++) {
		platGpioSetEdgeDetect(i, false, false);
		platGpioClearEdgeDetected(i);
	}
	
	//AC97
	platGpioSetDir(28, false);
	platGpioSetDir(29, false);
	platGpioSetDir(30, true);
	platGpioSetDir(31, true);
	platGpioSetDir(89, true);
	platGpioSetFunc(28, 1);
	platGpioSetFunc(29, 1);
	platGpioSetFunc(30, 2);
	platGpioSetFunc(31, 2);
	platGpioSetFunc(89, 0);	//AFR0 *IS* used for GPIOS > 85
	
	//IR
	platGpioSetDir(46, false);
	platGpioSetDir(47, true);
	platGpioSetFunc(46, 2);
	platGpioSetFunc(47, 1);
	
	//sd
	platGpioSetFunc(33, 0);	//reset pin
	platGpioSetDir(33, true);
	platGpioSetFunc(12, 0);	//irq pin
	platGpioSetDir(12, false);
}

static void debugUartInit(void)
{
	struct PxaUart *uart = platPeriphP2V(PXA_BASE_FFUART);
	uint32_t i;
	
	//setup gpios
	platGpioSetDir(39, true);	//TX
	platGpioSetFunc(39, 2);
	
	platEnablePeriphClock(XSCALE_CLOCK_ID_FFUART, true);
	
	uart->IER = 0x00;	//uart off while we config it
	uart->FCR = 0x07;	//reset fifos, enable them
	uart->LCR = 0x80;	//access DLAB
	uart->DLL = 0x18;	//divisor of 24 gives us 38,400
	uart->DLH = 0x00;
	uart->LCR = 0x03;	//DLAB off, 8n1 selected
	uart->MCR = 0x00;	//no loopback or weird modem shit
	
	uart->IER = 0x40;	//unit on, no irqs
}

static void __attribute__((naked, target("thumb"))) getSnumAsm(uint32_t romPtr, void *dst, uint32_t idCmd, uint32_t resetCmd)
{
	asm volatile(
		"	str   r2, [r0]		\n\t"
		"	ldmia r0!, {r2}		\n\t"
		"	strh  r2, [r1, #0]	\n\t"
		"	ldmia r0!, {r2}		\n\t"
		"	strh  r2, [r1, #2]	\n\t"
		"	ldmia r0!, {r2}		\n\t"
		"	strh  r2, [r1, #4]	\n\t"
		"	ldmia r0!, {r2}		\n\t"
		"	strh  r2, [r1, #6]	\n\t"
		"	str   r3, [r0]		\n\t"
		"	bx    lr			\n\t"
	);
}

static void getSnum(void)
{
	static const char base41[] = "0123456789ABCDEFGHJKLMNPQRSTUVWXYZabdefgh";
	uint16_t codeInRam[11];	//as per size of getSnumAsm
	uint64_t raw = 0;
	irq_state_t sta;
	uint32_t i;
	
	//code must be in ram since we could be running from rom and then switching rom away from "read" mode will crash
	memcpy(codeInRam, (void*)(((uintptr_t)&getSnumAsm) &~ 1), sizeof(codeInRam));
	platInstrCacheClearDataCacheClean((uintptr_t)codeInRam, sizeof(codeInRam));
	
	sta = irqsAllOff();
	((void (*)(uint32_t romPtr, void *dst, uint32_t idCmd, uint32_t resetCmd))(((uintptr_t)codeInRam) | 1))(CPU_FLASH_1_BASE + 4 * 0x81, &raw, 0x00900090, 0x00ff00ff);
	irqsRestoreState(sta);
	
	for (i = 0; i < sizeof(mSnum); i++) {
		uint32_t len = sizeof(base41) - 1;
		
		mSnum[i] = base41[raw % len];
		raw /= len;
	}
}

static void aximGetDeviceFlags(void)		//needs ints on
{
	struct PxaOsTimer *tmr = platPeriphP2V(PXA_BASE_OSTIMER);
	struct PxaGpio *gpio = platPeriphP2V(PXA_BASE_GPIO);
	uint32_t t, old;
	
	old = gpio->GAFR[3];
	gpio->GAFR[3] = old &~ 0xfff00000;
	gpio->GPSR[1] = 0xfc000000;
	gpio->GPDR[1] |= 0xfc000000;
	t = tmr->OSCR;
	while (tmr->OSCR - t < 0x1CCA);
	gpio->GPDR[1] &=~ 0xfc000000;
	
	for (t = 0; t < 100; t++)
		(void)gpio->GPLR[1];
	
	t = gpio->GPLR[1];
	gpio->GAFR[3] = old;
	
	mDevFlags = ((t & 0xfc000000) >> 26) | DEV_FLAGS_KNOWN;
	
	logi("DEVICE FLAGS: 0x%02x\n", mDevFlags &~ DEV_FLAGS_KNOWN);
}

const struct HalMemMap* DALEXPORT impl_HALMemoryGetMemoryMap(void)
{
	static struct HalMemMapPiece mMapPieces[] = {
		{
			.memType = HAL_MEM_PIECE_TYPE_ROM,
			.base = CPU_ROM_BASE,
			.size = CPU_ROM_SIZE,
		},
		{
			.memType = HAL_MEM_PIECE_TYPE_DYNAMIC_RAM,
			.base = CPU_DYN_RAM_BASE,
			.size = CPU_DYN_RAM_SIZE,
		},
		{
			.memType = HAL_MEM_PIECE_TYPE_STORAGE_RAM,
			.base = CPU_STORAGE_RAM_BASE,
			.size = 0,
		},
	};
	static const struct HalMemMap mMap = {
		.numPieces = sizeof(mMapPieces) / sizeof(*mMapPieces),
		.pieces = mMapPieces,
	};
	
	if (!mMapPieces[2].size) {
		
		bool isHighEnd = !!(mDevFlags & DEV_FLAG_HIGH_END);
		
		if (!(mDevFlags & DEV_FLAGS_KNOWN))
			fatal("flags not yet known\n");
		
		logi("Configuring mem map for %s-END device\n", isHighEnd ? "HIGH" : "LOW");
		mMapPieces[2].size = CPU_STORAGE_RAM_SIZE - (isHighEnd ? 0 : (32 << 20));
	}
	
	return &mMap;
}

void __attribute__((used)) machInit(uint32_t stage, const void* data)
{
	if (stage == STAGE_INIT_EARLY) {		//no globals, no BSS
		
		platMapPeriphs();
		platInitPowerAndClocks();
		//get debug uart up as early as possible
		debugUartInit();
		
		
		//map more things
		mmuMapIoSeg((void*)0x89000000, 0x0c000000);	//sd controller
		setupAximX3gpios();
	}
	else if (stage == STAGE_INIT_SET_VTOR) {		//we have globals now so setup more things
		
		aximCpldInit();
		
		getSnum();
		
		aximGetDeviceFlags();
		
		/*
			//clear first 64K of storage heap to make sure it does not appear valid
			mpuSetStorageRamWriteable(true);
			memset((void*)CPU_STORAGE_RAM_BASE, 0, 0x10000);
			mpuSetStorageRamWriteable(false);
			logi("Storage heap cleared\n");
		*/
	}
	else if (stage == STAGE_SETUP_HEAPS) {
		
		static uint32_t heap[65536];
		
		kheapRegisterHeap((uintptr_t)heap, sizeof(heap), MEM_USABLE_AS_STACK | MEM_USABLE_FOR_DMA | MEM_USABLE_FOR_EXEC | MEM_FAST);
		kheapRegisterHeap(CPU_PAGETABLE_BASE + CPU_DMA_MEM_START, CPU_DMA_MEM_SIZE, MEM_UNCACHED);
	}
	else if (stage == STAGE_INIT_INTERRUPTS) {
		
		platInitIrqs();
	}
	else if (stage == STAGE_INIT_PRE_SCHED) {
		
		
	}
	else if (stage == STAGE_INIT_IN_THREAD) {
		
		//by now we expect to be all done modifying vectors
		mmuLockExcPage();
	}
	else if (stage == STAGE_INIT_POST_RAL) {
		
		platExportFuncs();
		
		aximCpldSetBits(AXIM_CPLD_BIT_AUDIO_CODEC_POWER, true);
		ac97init();
		
		if (!ac97LinkReset(true))
			fatal("Failed to cold-boot AC97\n");
		
		if (!wm9705init(AC97primaryAudioCodec))
			fatal("Failed to init WM9705\n");
		
		drvInputInitLate();
	}
	else if (stage == STAGE_CRASH_LOCKS_BREAK) {
		
		
	}
}

bool hwMaybeGetRomToken(uint32_t name, const void **dataP, uint16_t *szP)
{
	if (name == CREATE_4CC('s','n','u','m')) {
	
		if (dataP)
			*dataP = mSnum;
		if(szP)
			*szP = sizeof(mSnum);
		
		return true;
	}
	
	return false;
}

void hwGetMiscFlags(uint16_t *miscFlagsP, uint16_t *extMiscFlagsP)
{
	if (miscFlagsP)
		*miscFlagsP = hwrMiscFlagHasBacklight | hwrMiscFlagHasMiscFlagExt | hwrMiscFlagHasCradleDetect | hwrMiscFlagNoRTCBug | hwrMiscFlagHasMbdIrDA;
	
	if (extMiscFlagsP)
		*extMiscFlagsP = hwrMiscFlagExtHasSWBright | hwrMiscFlagExt115KIrOK | hwrMiscFlagExtHasLiIon;
}

int32_t cpuGetClockRate(enum ClockRateDevice dev)
{
	switch (dev) {
		case CpuClockRate:
			return CPU_CLOCK_RATE;
			
		case TimerClockRate:
			return TIMER_TICKS_PER_SEC;
		
		case UartUnitClockRate:
			return 14745600;
		
		default:
			return -1;
	}
}

bool hwPwrCtl(uint32_t selector, const uint32_t *newValP, uint32_t *oldValP)
{	
	switch (selector) {
		case AXIM_PWR_BIT_SD_CHIP:
			if (oldValP)
				*oldValP = !!(aximCpldRead() & AXIM_CPLD_BIT_SD_CHIP_POWER);
			if (newValP)
				aximCpldSetBits(AXIM_CPLD_BIT_SD_CHIP_POWER, !!*newValP);
			break;
		
		case AXIM_PWR_BIT_SD_CHIP_RST:
			if (oldValP)
				*oldValP = !!platGpioGetVal(33);
			if (newValP)
				platGpioSetVal(33, !!*newValP);
			break;
		
		case AXIM_PWR_BIT_USB_IFACE:
			if (oldValP)
				*oldValP = !!(aximCpldRead() & AXIM_CPLD_BIT_USB_IFACE_POWER);
			if (newValP)
				aximCpldSetBits(AXIM_CPLD_BIT_USB_IFACE_POWER, !!*newValP);
			break;
		
		case AXIM_PWR_BIT_IR_UART_CLK:
			if (oldValP)
				*oldValP = platIsPeriphClockEnabled(XSCALE_CLOCK_ID_STUART);
			if (newValP)
				platEnablePeriphClock(XSCALE_CLOCK_ID_STUART, !!*newValP);
			break;
		
		default:
			return false;
	}
	return true;
}

#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC push_options
	#pragma GCC target ("arm")
#endif

void __attribute__((noinline)) machIdle(void)	//noinline will stop this form being inlined into thumb and them failing to link
{
	uint32_t dummy, dummy2;
	
	asm volatile(
		"ldr %0, [%2]					\n\t"
		"orr %0, #1 << 20				\n\t"
		"str %0, [%2]					\n\t"
		"mcr p14, 0, %1, c7, c0, 0		\n\t"
		"bic %0, #1 << 20				\n\t"
		"str %0, [%2]					\n\t"
		:"=&r"(dummy)
		:"r"(1), "r"(&((struct PxaMemCtrl*)platPeriphP2V(PXA_BASE_MEM_CTRL))->MDREFR)
		:"memory", "cc"
	);
}

#if 1
	//	--this does what the stock bootloader does. if we ever want to boot WITHOUT IT, we'll need this. doing it also improves stability, somehow
	
void __attribute__((noinline, used)) machPreRamInit(void)
{
	asm volatile(
	
	#define FLUSHPIPELINE											\
		"	sub   pc, pc ,#4								\n\t"	\
		"88:												\n\t"
		
	
	#define CPWAIT(reg)												\
		"	mrc   p15, 0, " reg ", c2, c0, 0				\n\t"	\
		"	nop												\n\t"	\
		FLUSHPIPELINE
	
		//configure 
		"	ldr   r0, =0x2001								\n\t"	//set CPAR and wait
		"	mcr   p15, 0, r0, c15, c1, 0					\n\t"
		CPWAIT("r0")
		"	mov   r0, #0x78									\n\t"	//some system control register settings
		"	mcr   p15, 0, r0, c1, c0, 0						\n\t"
		CPWAIT("r0")
		"	mov   r0, #0									\n\t"
		"	mcr   p15, 0, r0, c8, c7, 0						\n\t"	//inval I & D TLB
		"	mcr   p15, 0, r0, c7, c7, 0						\n\t"	//inval I & D cache & BTB
		"	mcr   p15, 0, r0, c7, c10, 4					\n\t"	//drain write and fill buffer
		"	nop												\n\t"
		"	nop												\n\t"
		"	nop												\n\t"
		"	mvn   r0, #0									\n\t"	//set all bits in DACR
		"	mcr   p15, 0, r0, c3, c0, 0						\n\t"
		"	ldr   r1, =0x40f00000							\n\t"	//get & clear RCSR
		"	ldr   r2, [r1,#0x30]							\n\t"
		"	and   r2, r2, #0x0f								\n\t"
		"	str   r3, [r1,#0x30]							\n\t"
		"	ldr   r3, [r1,#0x04]							\n\t"	//get PSSR
		
		//set PGSR regs
		"	ldr   r0, =0x40F00020							\n\t"
		"	mov   r1, #0x00008000							\n\t"
		"	ldr   r2, =0x00400302							\n\t"
		"	ldr   r3, =0x0081c400							\n\t"
		"	stmia r0, {r1-r3}								\n\t"
		
		//delay for ram init
		"	ldr   r2, =0x40a00000							\n\t"
		"	mov   r1, #0									\n\t"
		"	str   r1, [r2, #0x10]							\n\t"
		"2:													\n\t"
		"	ldr   r1, [r2, #0x10]							\n\t"
		"	cmp   r1, #0x400								\n\t"
		"	blt   2b										\n\t"
		
		//ram init
		"	mov   r0, #0x48000000							\n\t"
		"	ldr   r1, =0x128212B3							\n\t"
		"	str   r1, [r0, #0x08]							\n\t"
		"	ldr   r1, [r0, #0x08]							\n\t"
		"	ldr   r1, =0x77797FF1							\n\t"
		"	str   r1, [r0, #0x0c]							\n\t"
		"	ldr   r1, [r0, #0x0c]							\n\t"
		"	ldr   r1, =0x72FC7FFC							\n\t"
		"	str   r1, [r0, #0x10]							\n\t"
		"	ldr   r1, [r0, #0x10]							\n\t"
		"	mov   r1, #0									\n\t"
		"	str   r1, [r0, #0x14]							\n\t"
		"	ldr   r1, =0x00010285							\n\t"
		"	str   r1, [r0, #0x28]							\n\t"
		"	ldr   r2, =0x00044833							\n\t"
		"	str   r2, [r0, #0x2c]							\n\t"
		"	str   r1, [r0, #0x30]							\n\t"
		"	str   r2, [r0, #0x34]							\n\t"
		"	str   r1, [r0, #0x38]							\n\t"
		"	str   r2, [r0, #0x3c]							\n\t"
		"	ldr   r1, [r0, #0x04]							\n\t"
		"	bic   r1, #0x00000f00							\n\t"
		"	bic   r1, #0x000000ff							\n\t"
		"	add   r1, #0x00000024							\n\t"
		"	str   r1, [r0, #0x04]							\n\t"
		"	bic   r1, #0x03800000							\n\t"
		"	bic   r1, #0x00020000							\n\t"
		"	orr   r1, #0x00010000							\n\t"
		"	str   r1, [r0, #0x04]							\n\t"
		"	ldr   r1, [r0, #0x04]							\n\t"
		"	bic   r1, #0x00400000							\n\t"
		"	str   r1, [r0, #0x04]							\n\t"
		"	orr   r1, #0x00008000							\n\t"
		"	str   r1, [r0, #0x04]							\n\t"
		"	ldr   r1, [r0, #0x04]							\n\t"
		"	orr   r1, #0x00007000							\n\t"
		"	str   r1, [r0, #0x04]							\n\t"
		"	ldr   r1, [r0, #0x04]							\n\t"
		
		//internal flash init
		"	ldr   r2, =0x00600060							\n\t"
		"	ldr   r3, =0x00030003							\n\t"
		"	ldr   r12, =0x0000a308							\n\t"
		"	ldr   r1, [r0, #0x1c]							\n\t"
		"	and   r1, #0x00000002							\n\t"
		"	orr   r1, #0x00006000							\n\t"
		"	orr   r1, #0x000000f5							\n\t"
		FLUSHPIPELINE
		"	str   r2, [r12]									\n\t"
		"	str   r3, [r12]									\n\t"
		"	str   r1, [r0, #0x1c]							\n\t"
		FLUSHPIPELINE
		
		//now config the external flash, if detected
		"	mov   r2, #0x04000000							\n\t"
		"	ldr   r1, =0x00900090							\n\t"
		"	str   r1, [r2]									\n\t"
		"	ldr   r1, [r2]									\n\t"
		"	sub   r1, #0x00890000							\n\t"
		"	subs  r1, #0x00000089							\n\t"
		"	ldreq r1, =0x04028308							\n\t"
		"	ldreq r12, =0x00600060							\n\t"
		"	ldreq r3, =0x00030003							\n\t"
		"   streq r12, [r1]									\n\t"
		"   streq r3, [r1]									\n\t"
		"	ldreq r1, =0x000060f5							\n\t"
		"	streq r1, [r0, #0x1c]							\n\t"
		"	ldr   r1, =0x00ff00ff							\n\t"
		"	str   r1, [r2]									\n\t"
		"	nop												\n\t"
		"	nop												\n\t"
		"	ldr   r1, =0x000009c8							\n\t"
		"	str   r1, [r0, #0x00]							\n\t"
		
		//delay a little for init
		"	ldr   r2, =0x40a00000							\n\t"
		"	mov   r1, #0									\n\t"
		"	str   r1, [r2, #0x10]							\n\t"
		"2:													\n\t"
		"	ldr   r1, [r2, #0x10]							\n\t"
		"	cmp   r1, #0x400								\n\t"
		"	blt   2b										\n\t"
		
		//more ram init
		"	mov   r1, #0xa0000000							\n\t"
		"	str   r1, [r1]									\n\t"
		"	str   r1, [r1]									\n\t"
		"	str   r1, [r1]									\n\t"
		"	str   r1, [r1]									\n\t"
		"	str   r1, [r1]									\n\t"
		"	str   r1, [r1]									\n\t"
		"	str   r1, [r1]									\n\t"
		"	str   r1, [r1]									\n\t"
		"	str   r1, [r1]									\n\t"
		"	ldr   r1, [r0, #0x00]							\n\t"
		"	orr   r1, #0x00000001							\n\t"
		"	str   r1, [r0, #0x00]							\n\t"
		"	mov   r1, #0									\n\t"
		"	str   r1, [r0, #0x40]							\n\t"
		"	nop												\n\t"
		"	nop												\n\t"
		
		//probe 32M vs 64M of RAM
		"	ldr   r1, =0xb00cca5e							\n\t"
		"	mov   r2, #0xa0000000							\n\t"
		"	ldr   r3, [r2]									\n\t"
		"	str   r1, [r2]									\n\t"
		"	ldr   r1, =0xabadf00d							\n\t"
		"	mov   r2, #0xa2000000							\n\t"
		"	ldr   r12, [r2]									\n\t"
		"	str   r1, [r2]									\n\t"
		"	mov   r2, #0xa0000000							\n\t"
		"	ldr   r1, [r2]									\n\t"
		"	str   r3, [r2]									\n\t"
		"	mov   r2, #0xa2000000							\n\t"
		"	str   r12, [r2]									\n\t"
		"	ldr   r2, =0xabadf00d							\n\t"
		"	cmp   r1, r2									\n\t"		//"EQ" if we have 32M of RAM
		"	beq   ram_amt_32M								\n\t"
		
		"ram_amt_64M:										\n\t"
		"	ldr   r1, [r0, #0x04]							\n\t"
		"	orr   r1, #0x00400000							\n\t"
		"	str   r1, [r0, #0x04]							\n\t"
		"	ldr   r1, =0x000009c8							\n\t"
		"	str   r1, [r0, #0x00]							\n\t"
		"	mov   r2, #0xa0000000							\n\t"
		"	ldr   r3, [r2]									\n\t"
		"	ldr   r3, [r2]									\n\t"
		"	ldr   r3, [r2]									\n\t"
		"	ldr   r3, [r2]									\n\t"
		"	ldr   r3, [r2]									\n\t"
		"	ldr   r3, [r2]									\n\t"
		"	ldr   r3, [r2]									\n\t"
		"	ldr   r3, [r2]									\n\t"
		"	orr   r1, #0x00000001							\n\t"
		"	str   r1, [r0, #0x00]							\n\t"
		"	ldr   r1, [r0, #0x04]							\n\t"
		"	bic   r1, #0x00400000							\n\t"
		"	str   r1, [r0, #0x04]							\n\t"
		"	b     ram_init_done								\n\t"
		
		"ram_amt_32M:										\n\t"
		"	ldr   r1, [r0, #0x04]							\n\t"
		"	bic   r1, #0x00000f00							\n\t"
		"	bic   r1, #0x000000ff							\n\t"
		"	add   r1, #0x00000034							\n\t"
		"	str   r1, [r0, #0x04]							\n\t"
		"	ldr   r1, =0x000009a8							\n\t"
		"	str   r1, [r0, #0x00]							\n\t"
		"	mov   r2, #0xa0000000							\n\t"
		"	ldr   r3, [r2]									\n\t"
		"	ldr   r3, [r2]									\n\t"
		"	ldr   r3, [r2]									\n\t"
		"	ldr   r3, [r2]									\n\t"
		"	ldr   r3, [r2]									\n\t"
		"	ldr   r3, [r2]									\n\t"
		"	ldr   r3, [r2]									\n\t"
		"	ldr   r3, [r2]									\n\t"
		"	orr   r1, #0x00000001							\n\t"
		"	str   r1, [r0, #0x00]							\n\t"
		
		"ram_init_done:										\n\t"
		
		
		"gpio_setup:										\n\t"
		"	adr   r0, gpio_init_pairs						\n\t"
		"	adr   r1, gpio_init_done						\n\t"
		"4:													\n\t"
		"	cmp   r0, r1									\n\t"
		"	bxeq  lr										\n\t"
		"	ldmia r0!, {r2, r3}								\n\t"
		"	str   r3, [r2]									\n\t"
		"	b     4b										\n\t"
		
		//init pairs {addr, data}
		"gpio_init_pairs:									\n\t"
		
		//mine
		"	.word 0x40E0000C, 0x00008000					\n\t"
		"	.word 0x40E00014, 0x0000C000					\n\t"
		
		
		//stock
		"	.word 0x40E00018, 0x0A008002					\n\t"
		"	.word 0x40E0001C, 0x00FF03AA					\n\t"
		"	.word 0x40E00020, 0x0001C000					\n\t"
		"	.word 0x40E00024, 0x00204140					\n\t"
		"	.word 0x40E00028, 0x00000051					\n\t"
		"	.word 0x40E0002C, 0x00C00000					\n\t"
		"	.word 0x40E0000C, 0x0A22C140					\n\t"
		"	.word 0x40E00010, 0x00FFAB83					\n\t"
		"	.word 0x40E00014, 0x0001C000					\n\t"
		"	.word 0x40E00054, 0x80000004					\n\t"
		"	.word 0x40E00058, 0x00000018					\n\t"
		"	.word 0x40E0005C, 0x999A9550					\n\t"
		"	.word 0x40E00060, 0x0005AAAA					\n\t"
		"	.word 0x40E00064, 0xA0000000					\n\t"
		"	.word 0x40E00068, 0x00005000					\n\t"
		"gpio_init_done:									\n\t"
		"													\n\t"
		"	.ltorg											\n\t"
	);
}

#else

void __attribute__((noinline, used)) machPreRamInit(void)
{
	
}

#endif

#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC pop_options
#endif


Err machinePaceDispatch(EmulStateRef ref, uint16_t call, Err *ret68kP)
{
	return sysErrNotAllowed;
}