#include "platform.h"
#include "printf.h"
#include "xscale.h"
#include <stdio.h>
#include "ac97.h"


static bool ac97PrvWaitCodecAccess(struct PxaAC97 *ac97)
{
	uint32_t nRetries = 8192;
	
	while(ac97->CAR & 1) {
		
		if (!--nRetries)
			return false;
	}
	return true;
}

static bool ac97PrvWaitGsrBit(struct PxaAC97 *ac97, uint32_t bit, uint32_t desiredVal, uint32_t nRetries)
{
	while((ac97->GSR & bit) != desiredVal) {
		
		if (!--nRetries)
			return false;
	}
	
	return true;
}

static volatile uint32_t* ac97GetRegsPtrForType(struct PxaAC97 *ac97, enum AC97deviceSelection dev)
{
	switch (dev) {
		case AC97primaryAudioCodec:
			return ac97->primaryAudioRegs;
		
		case AC97secondaryAudioCodec:
			return ac97->secondaryAudioRegs;
		
		case AC97primaryModem:
			return ac97->primaryModemRegs;
		
		case AC97secondaryModem:
			return ac97->secondaryModemRegs;
		
		default:
			return NULL;
	}
}

bool ac97regRead(enum AC97deviceSelection dev, uint8_t reg, uint16_t *valP)
{
	struct PxaAC97 *ac97 = platPeriphP2V(PXA_BASE_AC97);
	uint32_t nRetries = 8192;
	bool success;
	
	if (reg & 1)		//all reg addrs are even
		return false;
	reg >>= 1;
	
	if (!ac97PrvWaitCodecAccess(ac97))
		return false;
	
	ac97->GSR = (1 << 18) | (1 << 15);	//clear the "s done" and "timedout" flag
	(void)ac97GetRegsPtrForType(ac97, dev)[reg];	//a dummy read
	
	if (!ac97PrvWaitGsrBit(ac97, 1 << 18, 1 << 18, 81920)) {
		
		ac97->CAR &=~ 1;
		return false;
	}

	success = !(ac97->GSR & (1 << 15));
	ac97->GSR = 1 << 18;	//clear the "s done" flag
	*valP = ac97GetRegsPtrForType(ac97, dev)[reg];	//the real read
	
	if (!ac97PrvWaitGsrBit(ac97, 1 << 18, 1 << 18, 81920)) {
		
		ac97->CAR &=~ 1;
		return false;
	}

	return success;
}

bool ac97regWrite(enum AC97deviceSelection dev, uint8_t reg, uint_fast16_t val)
{
	struct PxaAC97 *ac97 = platPeriphP2V(PXA_BASE_AC97);
	
	if (reg & 1)		//all reg addrs are even
		return false;
	reg >>= 1;
	
	if (!ac97PrvWaitCodecAccess(ac97))
		return false;
	
	ac97->GSR = 1 << 19;	//clear the "c done" flag
	ac97GetRegsPtrForType(ac97, dev)[reg] = val;
	
	if (!ac97PrvWaitGsrBit(ac97, 1 << 19, 1 << 19, 81920)) {
		ac97->CAR &=~ 1;
		return false;
	}

	return true;
}

void ac97init(void)
{
	struct PxaAC97 *ac97 = platPeriphP2V(PXA_BASE_AC97);
	
	platEnablePeriphClock(XSCALE_CLOCK_ID_AC97, true);
}

void ac97linkPowerDown(void)
{
	struct PxaAC97 *ac97 = platPeriphP2V(PXA_BASE_AC97);
	
	ac97->GCR = (ac97->GCR &~ (1 << 2)) | (1 << 3) | (1 << 1);
}

bool ac97LinkReset(bool cold)
{
	struct PxaAC97 *ac97 = platPeriphP2V(PXA_BASE_AC97);
	
	if (cold) {
		
		ac97->GCR = (ac97->GCR &~ (1 << 2)) | (1 << 1);	//set to 1 since it may be at zero
		ac97->GCR = (ac97->GCR &~ (1 << 2));			//set to zero for cold reset
		
		platDelay(100);		//that literally asserted a line - let the codec feel it
		
		ac97->GCR = (ac97->GCR &~ (1 << 2)) | (1 << 1);	//end the cold reset
		
		platDelay(50);		//that literally deasseted a line - let the codec wake up
	}
	else {
		
		ac97->GCR |= (1 << 1) | (1 << 2);
		while (ac97->GCR & (1 << 2));
	}
	
	return true;
}

bool ac97IsCodecReady(enum AC97deviceSelection dev)
{
	struct PxaAC97 *ac97 = platPeriphP2V(PXA_BASE_AC97);
	uint32_t bit;
	
	switch (dev) {
		case AC97primaryAudioCodec:
		case AC97primaryModem:
			bit = 1 << 8;
			break;
		
		case AC97secondaryAudioCodec:
		case AC97secondaryModem:
			bit = 1 << 9;
			break;
		
		default:
			return false;
	}
	
	return !!(ac97->GSR & bit);
}
