#include "ral_export.h"
#include "platform.h"
#include <stdbool.h>
#include "hwTimer.h"
#include "printf.h"
#include "memmap.h"
#include "timers.h"
#include "xscale.h"
#include <stdio.h>
#include "heap.h"
#include "irqs.h"
#include "dal.h"
#include "ral.h"
#include "mpu.h"
#include "cpu.h"



//#define USE_POWER_BUTTON_FOR_DEBUGGING



#define XSCALE_PERIPHS_OFFSET		0x50000000UL




struct IrqHandlerInfo {
	HalIrqHandlerF handler;
	void* data;
};

static struct IrqHandlerInfo mIrqHandlers[CPU_NUM_IRQS] = {};
static uint32_t mCurTimeHi = 0;

static void timerAdvance(void);
static void setCCLKCFG(uint32_t val);


bool cpuIrqGetHandler(uint32_t irqNo, HalIrqHandlerF *curHandlerP, void** curUserDataP)
{
	irq_state_t t;
	if (irqNo >= CPU_NUM_IRQS)
		return false;
	
	t = irqsAllOff();
	if (curHandlerP)
		*curHandlerP = mIrqHandlers[irqNo].handler;
	if (curUserDataP)
		*curUserDataP = mIrqHandlers[irqNo].data;
	irqsRestoreState(t);
	
	return true;
}

bool cpuIrqSetHandler(uint32_t irqNo, HalIrqHandlerF newHandler, void* newUserData)
{
	irq_state_t t;
	if (irqNo >= CPU_NUM_IRQS)
		return false;
	
	t = irqsAllOff();
	mIrqHandlers[irqNo].handler = newHandler;
	mIrqHandlers[irqNo].data = newUserData;
	irqsRestoreState(t);
	
	return true;
}

static void* platPrvPeriphP2V(uintptr_t pa)		//so it i sinlined
{
	return (void*)(pa + XSCALE_PERIPHS_OFFSET);
}

void* platPeriphP2V(uintptr_t pa)
{
	return platPrvPeriphP2V(pa);
}

bool cpuIrqSetState(uint32_t irqNo, bool on)
{
	struct PxaGpio *gpio = platPrvPeriphP2V(PXA_BASE_GPIO);
	struct PxaIc *ic = platPrvPeriphP2V(PXA_BASE_IC);
	irq_state_t t;
	bool wasOn;
	
	if (irqNo >= CPU_NUM_IRQS)
		return false;
	
	if (irqNo >= 32)
		fatal("gpio irqs cannot be turned on or off\n");
	
	t = irqsAllOff();
	wasOn = (ic->ICMR >> irqNo) & 1;
	if (on)
		ic->ICMR |= 1 << irqNo;
	else
		ic->ICMR &=~ (1 << irqNo);
	irqsRestoreState(t);
	
	return wasOn;
}

static void xscalePrvCallIrqHandler(uint32_t irqNo)
{
	if (!mIrqHandlers[irqNo].handler)
		fatal("no irq handler for irq %u\n", irqNo);
	else
		mIrqHandlers[irqNo].handler(mIrqHandlers[irqNo].data);
}

static void xscalePrvDispatchGpioIrqs(void)
{
	static const uint32_t masks[] = {0xfffffffc, 0xffffffff, 0x03ffffff};
	struct PxaGpio *gpio = platPrvPeriphP2V(PXA_BASE_GPIO);
	uint32_t i, bits, which;
	bool anythingDone;
	
	do {
		anythingDone = false;
		
		for (i = 0; i < 3; i++) {
			
			while ((bits = gpio->GEDR[i] & masks[i]) != 0) {
				
				which = 31 - __builtin_clz(bits);
				gpio->GEDR[i] = 1 << which;
				
				xscalePrvCallIrqHandler(XSCALE_IRQ_NO_GPIO(i * 32 + which));
				anythingDone = true;
			}
		}
		
	} while (anythingDone);
}

void __attribute__((used)) kernelDispatchIrq(void)
{
	static const uint8_t prioIrqs[] = {		//timer0 is delivered via fiq and thus isn't seen here
		XSCALE_IRQ_NO_DMA,		XSCALE_IRQ_NO_TMR1,		XSCALE_IRQ_NO_AC97,		XSCALE_IRQ_NO_SSP,
		XSCALE_IRQ_NO_I2S,		XSCALE_IRQ_NO_STUART,	XSCALE_IRQ_NO_GPIO0,	XSCALE_IRQ_NO_GPIO1,
		XSCALE_IRQ_NO_GPIO_2_89,XSCALE_IRQ_NO_USB,		XSCALE_IRQ_NO_HWUART,	XSCALE_IRQ_NO_BTUART,
		XSCALE_IRQ_NO_LCD,		XSCALE_IRQ_NO_TMR2,		XSCALE_IRQ_NO_TMR3,		XSCALE_IRQ_NO_MMC,
		XSCALE_IRQ_NO_I2C,		XSCALE_IRQ_NO_PMU,		XSCALE_IRQ_NO_FFUART,	XSCALE_IRQ_NO_RTC_HZ,
		XSCALE_IRQ_NO_RTC_ALM,	XSCALE_IRQ_NO_ICP,		XSCALE_IRQ_NO_ASSP,		XSCALE_IRQ_NO_NSSP,
	};
	struct PxaIc *ic = platPrvPeriphP2V(PXA_BASE_IC);
	uint32_t bitfield;
	
	while ((bitfield = ic->ICIP) != 0) {	//these are in priority order
		
		int32_t i, irqNo = -1;
		
		for (i = 0; !(bitfield & (1 << (irqNo = prioIrqs[i]))); i++);
		
		if (irqNo == -1)
			fatal("WTF: irqs did not line up\n");
		if (irqNo == XSCALE_IRQ_NO_GPIO_2_89)
			xscalePrvDispatchGpioIrqs();
		else
			xscalePrvCallIrqHandler(irqNo);
	}
}

void platInitIrqs(void)
{
	struct PxaGpio *gpio = platPeriphP2V(PXA_BASE_GPIO);
	struct PxaIc *ic = platPrvPeriphP2V(PXA_BASE_IC);
	uint32_t i;
	
	//disanle gpio irqs before we able irqh for all gpio irqs
	for (i = 0; i < 90; i++) {
		
		platGpioSetEdgeDetect(i, false, false);
		platGpioClearEdgeDetected(i);
	}
	
	ic->ICLR = 0;	//all irqs are IRQs and not FIQs
	ic->ICMR = 1 << XSCALE_IRQ_NO_GPIO_2_89;	//all irqs masked except generic gpio change
	ic->ICCR = 1;	//only unmasked interrupts wake us
}

void platMapPeriphs(void)
{
	uint32_t i;
	
	//memory config
	mmuMapIoSeg(platPrvPeriphP2V(PXA_BASE_MEM_CTRL), PXA_BASE_MEM_CTRL);
	
	//LCD
	mmuMapIoSeg(platPrvPeriphP2V(PXA_BASE_LCD_CTRL), PXA_BASE_LCD_CTRL);
	
	//all else
	for (i = 0; i <= 0x16; i++)
		mmuMapIoSeg(platPrvPeriphP2V(0x40000000 + (i << 20)), 0x40000000 + (i << 20));
}

void platInitPowerAndClocks(void)
{
	struct PxaClockMgr *clk = platPrvPeriphP2V(PXA_BASE_CLOCK_MGR);
	struct PxaPwrMgr *pwr = platPrvPeriphP2V(PXA_BASE_PWR_MGR);
	
	//go to 400MHz
	clk->OSCC = 2;										//32K osc on
	clk->CKEN = 0;										//no clocks on
	clk->CCCR = 0b1001000001;							//mem 100, run 200, turbo 2x (400)
	setCCLKCFG(3);										//turbo on, clock change on
	
	//config pwr mgr a bit
	pwr->PMCR = 0;
	pwr->PCFR = 0;
	pwr->PWER = (1 << 0) | (1 << 10) | (1 << 11) | (1 << 12) | (1 << 31);	//wake on buttons and rtc alarm
	pwr->PRER = 0;
	pwr->PFER = (1 << 0) | (1 << 10) | (1 << 11) | (1 << 12);				//button gpios wake on falling edge
	pwr->PEDR = 0x0000ffff;													//clear statusses
	pwr->PSSR = 0x00000037;													//clear statusses
	pwr->PSPR = 0;
	pwr->PMFWR = 0;
	pwr->PGSR[0] = 0;		//todo: reconsider this later
	pwr->PGSR[1] = 0;
	pwr->PGSR[2] = 0;
	
	//we leave pwr->RCSR alone so we can read it later
}

void platEnablePeriphClock(uint32_t periph, bool clocked)
{
	struct PxaClockMgr *clk = platPrvPeriphP2V(PXA_BASE_CLOCK_MGR);
	uint32_t bit = (1 << periph);
	irq_state_t sta;
	
	sta = irqsAllOff();

	if (clocked)
		clk->CKEN |= bit;
	else
		clk->CKEN &=~ bit;
	
	irqsRestoreState(sta);
}

bool platIsPeriphClockEnabled(uint32_t periph)
{
	struct PxaClockMgr *clk = platPrvPeriphP2V(PXA_BASE_CLOCK_MGR);
	
	return (clk->CKEN >> periph) & 1;
}

void timersInit(void)
{
	struct PxaOsTimer *tmr = platPrvPeriphP2V(PXA_BASE_OSTIMER);
	struct PxaIc *ic = platPrvPeriphP2V(PXA_BASE_IC);
	void **handlers = (void**)0xffff0020;
	
	handlers[7] = timerAdvance;	//FIQ reserved for general timekeeping
	
	tmr->OSCR = 0;
	tmr->OIER = 0;		//int on for match #0
	tmr->OWER = 0;		//watchdog off
	tmr->OSMR[0] = 0;	//#0: irq on rollover
	tmr->OSSR = 0x0f;	//clear irq flags
	
	//make it a fiq
	#ifdef USE_POWER_BUTTON_FOR_DEBUGGING
		ic->ICLR |= 1 << XSCALE_IRQ_NO_GPIO0;
	#else
		ic->ICLR |= 1 << XSCALE_IRQ_NO_TMR0;
	#endif
	cpuIrqSetState(XSCALE_IRQ_NO_TMR0, true);

	//other init
	timersCommonInit();
}

uint64_t timerGetTime(void)
{
	struct PxaOsTimer *tmr = platPrvPeriphP2V(PXA_BASE_OSTIMER);
	union {		//arm-gcc sucks for v5T, we resort to this monstrocity...
		struct {
			uint32_t lo, hi;
		};
		uint64_t vl;
	} ret;
	uint32_t hi, lo;
	
	do {
		hi = mCurTimeHi;
		lo = tmr->OSCR;
		asm volatile("":::"memory");
	} while (mCurTimeHi != hi);
	
	ret.lo = lo;
	ret.hi = hi;
	
	return ret.vl;
}

static void hwTimerIrqHandler(void *data)
{
	struct PxaOsTimer *tmr = (struct PxaOsTimer*)data;
	
	tmr->OSSR = 2;
	timerExternalIrq();
}

void hwTimerInit(void)
{
	cpuIrqSetHandler(XSCALE_IRQ_NO_TMR1, hwTimerIrqHandler, platPrvPeriphP2V(PXA_BASE_OSTIMER));
}

uint32_t hwTimerIntOff(void)
{
	return cpuIrqSetState(XSCALE_IRQ_NO_TMR1, false);
}

uint32_t hwTimerIntOn(void)
{
	return cpuIrqSetState(XSCALE_IRQ_NO_TMR1, true);
}

void hwTimerIntRestore(uint32_t state)
{
	cpuIrqSetState(XSCALE_IRQ_NO_TMR1, !!state);
}

void hwTimerSet(uint32_t ticksFromNow)
{
	struct PxaOsTimer *tmr = platPrvPeriphP2V(PXA_BASE_OSTIMER);
	
	tmr->OIER &=~ 2;
	
	if (ticksFromNow) {
		
		if (ticksFromNow >= 0x80000000)
			ticksFromNow = 0x80000000;
		
		tmr->OSMR[1] = tmr->OSCR + ticksFromNow;
		tmr->OIER |= 2;
	}
}

void platGpioSetDir(uint32_t gpioNum, bool out)
{
	struct PxaGpio *gpio = platPrvPeriphP2V(PXA_BASE_GPIO);
	
	if (gpioNum > 85)
		out = !out;
	
	if (out)
		gpio->GPDR[gpioNum / 32] |= 1 << (gpioNum % 32);
	else
		gpio->GPDR[gpioNum / 32] &=~ (1 << (gpioNum % 32));
}

void platGpioSetVal(uint32_t gpioNum, bool hi)
{
	struct PxaGpio *gpio = platPrvPeriphP2V(PXA_BASE_GPIO);
	
	if (hi)
		gpio->GPSR[gpioNum / 32] = 1 << (gpioNum % 32);
	else
		gpio->GPCR[gpioNum / 32] = 1 << (gpioNum % 32);
}

bool platGpioGetVal(uint32_t gpioNum)
{
	struct PxaGpio *gpio = platPrvPeriphP2V(PXA_BASE_GPIO);
	
	return (gpio->GPLR[gpioNum / 32] >> (gpioNum % 32)) & 1;
}

void platGpioSetFunc(uint32_t gpioNum, uint32_t func)
{
	struct PxaGpio *gpio = platPrvPeriphP2V(PXA_BASE_GPIO);
	
	gpio->GAFR[gpioNum / 16] = (gpio->GAFR[gpioNum / 16] &~ (3 << ((gpioNum % 16) * 2))) | (func  << ((gpioNum % 16) * 2));
}

void platGpioSetEdgeDetect(uint32_t gpioNum, bool rising, bool falling)
{
	struct PxaGpio *gpio = platPrvPeriphP2V(PXA_BASE_GPIO);
	
	if (rising)
		gpio->GRER[gpioNum / 32] |= 1 << (gpioNum % 32);
	else
		gpio->GRER[gpioNum / 32] &=~ (1 << (gpioNum % 32));
	
	if (falling)
		gpio->GFER[gpioNum / 32] |= 1 << (gpioNum % 32);
	else
		gpio->GFER[gpioNum / 32] &=~ (1 << (gpioNum % 32));
}

bool platGpioGetEdgeDetected(uint32_t gpioNum)
{
	struct PxaGpio *gpio = platPrvPeriphP2V(PXA_BASE_GPIO);
	
	return (gpio->GEDR[gpioNum / 32] >> (gpioNum % 32)) & 1;
}

void platGpioClearEdgeDetected(uint32_t gpioNum)
{
	struct PxaGpio *gpio = platPrvPeriphP2V(PXA_BASE_GPIO);
	
	gpio->GEDR[gpioNum / 32] = 1 << (gpioNum % 32);
}

void platDelay(uint32_t usec)		//clock is 3.6864 MHz
{
	uint32_t clockCycles = (usec * 0x4571C71CULL) >> 32;
	uint64_t start = timerGetTime();
	
	while (timerGetTime() - start < clockCycles);
}

bool platDmaBufAlloc(uint32_t sz, uintptr_t *paP, void **vaP)
{
	void *va = kheapAllocEx(sz, MEM_UNCACHED);
	
	if (!va)
		return false;
	
	*vaP = va;
	*paP = ((uintptr_t)va) - CPU_PAGETABLE_BASE + CPU_PAGETABLE_PA;
	
	return true;
}

void platDmaBufFree(void *va)
{
	kheapFree(va);
}

void platExportFuncs(void)
{
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_PLAT_PERIPH_P2V, &platPeriphP2V))
		fatal("PlatP2V export fail\n");
	
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_DMA_BUF_ALLOC, &platDmaBufAlloc))
		fatal("PlatP2V export fail\n");
	
	if (!ralSetRePalmTabFunc(REPALM_FUNC_IDX_DMA_BUF_FREE, &platDmaBufFree))
		fatal("PlatP2V export fail\n");
}	


#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC push_options
	#pragma GCC target ("arm")
#endif

static void __attribute__((noinline)) setCCLKCFG(uint32_t val)	//if allowed to inline, will try to compile in thumb mode inlined
{
	asm("mcr p14, 0, %0, c6, c0, 0"::"r"(val):"memory");
}

#ifdef USE_POWER_BUTTON_FOR_DEBUGGING
	static void __attribute__((used)) report(uint32_t *regs)
	{
		uint32_t i;
		
		loge("report as requested SR=0x%08x\n", regs[16]);
		for (i = 0; i < 8; i++)
			loge("R%02u = 0x%08x  R%02u = 0x%08x\n", i, regs[i], i + 8, regs[i + 8]);
		
		for (i = 0; i < 32; i++)
			loge("[SP, 0x%03x]: [0x%08x] = 0x%08x\n", regs[13] + i * 4, *(uint32_t*)(regs[13] + i * 4));
		
		platGpioClearEdgeDetected(0);
		
		extern void schedShowTasks(void);
		schedShowTasks();
		
		extern void tmrShowTimers(void);
		tmrShowTimers();
	}

	static void __attribute__((naked)) timerAdvance(void)
	{
		asm volatile(			//we have banked regs - use them
			"	mrs  r8, spsr				\n\t"
			"	stmfd sp!, {r8}				\n\t"
			"	stmfd sp!, {lr}				\n\t"
			"	sub sp, #4 * 15				\n\t"
			"	stmia sp, {r0-r14}^			\n\t"
			"	nop							\n\t"
			"	mov   r0, sp				\n\t"
			"	bl		report				\n\t"
			"	ldmia sp, {r0-r14}^			\n\t"
			"	nop							\n\t"
			"	add   sp, #4 * 15			\n\t"
			"	ldmfd sp!, {lr}				\n\t"
			"	ldmfd sp!, {r8}				\n\t"
			"	msr   spsr, r8				\n\t"
			"	subs pc, lr, #4				\n\t"
			:
			:
			:"memory"
		);
	}

#else

	static void __attribute__((naked)) timerAdvance(void)
	{
		asm volatile(			//we have banked regs - use them
			"	ldr  r8, =%[p_mCurTimeHi]	\n\t"
			"	ldr  r9, [r8]				\n\t"
			"	add  r9, #1					\n\t"
			"	str  r9, [r8]				\n\t"
			"	ldr  r8, =%[loc_OSSR]		\n\t"
			"	mov  r9, #1					\n\t"
			"	str  r9, [r8]				\n\t"
			"	subs pc, lr, #4				\n\t"
			:
			: [p_mCurTimeHi] "i"(&mCurTimeHi), [loc_OSSR] "i"(&((struct PxaOsTimer*)(PXA_BASE_OSTIMER + XSCALE_PERIPHS_OFFSET))->OSSR)
			:"memory"
		);
	}

#endif

bool __attribute__((noinline)) platSetStackGuard(uint32_t addr)	//noinline will stop this form being inlined into thumb and them failing to link
{
	//theory of operation: data breakpoint sized 32 bytes, triggering on access
	//DBR0 = addr
	//DBR1 = mask
	//DBCON = 0x0102 (any access, DBR1 is mask)
	
	//always disable first to prevent mis-triggering
	asm volatile(
		"	mcr p15, 0, %0, c14, c4, 0		\n\t"	//DBCON
		:
		:"r"(0)
		:"memory"
	);
	
	if (addr) {	// enable
		
		asm volatile(
			"	mcr p15, 0, %0, c14, c0, 0		\n\t"	//DBR0
			"	mcr p15, 0, %1, c14, c3, 0		\n\t"	//DBR1
			"	mcr p15, 0, %2, c14, c4, 0		\n\t"	//DBCON
			:
			:"r"(addr), "r"(31), "r"(0x0102)
			:"memory"
		);
	}
	
	return true;
}

void platInstrCacheClearDataCacheClean(uintptr_t addr, int32_t sz)
{
	volatile uint32_t *dcacheCleaningAddr = (volatile uint32_t*)CPU_MINIDCACHE_CLEAN_BASE;
	uint32_t i, dummy;
	
	asm volatile("mcr p15, 0, %0, c7, c10, 4"::"r"(0):"memory");	//drain write buffer
	
	//clean the mini data cache (required in all cases)
	for (i = 0; i < 64; i++) {
		(void)*dcacheCleaningAddr;
		dcacheCleaningAddr += 8;
	}
	
	if (sz < 0) {	//no range - clean all
		
		for (i = 0; i < 32768; i += 32) {
			
			asm volatile(
				"mcr p15, 0, %0, c7, c2, 5"			//allocate line
				:
				:"r"(CPU_CACHE_CLEAN_BASE + i)
				:"memory"
			);
		}
		
		asm volatile("mcr p15, 0, %0, c7, c10, 4"::"r"(0):"memory");	//drain write buffer
	
		//inval icache
		asm volatile(
			"	mcr   p15, 0, %1, c7, c5, 0			\n\t"
			"	mrc   p15, 0, %0, c2, c0, 0			\n\t"
			"	mov   %0, %0						\n\t"
			"	sub   pc, pc, #4					\n\t"
			:"=r"(dummy)
			:"r"(0)
			:"memory"
		);
	}
	else {			//have a range - use it
		
		uint32_t end = addr + sz + 31;
		
		addr &= ~31;
		end &=~ 31;
		
		for (i = addr; i != end; i += 32) {
			
			asm volatile(
				"mcr p15, 0, %0, c7, c10, 1"		//clean line by MVA
				:
				:"r"(i)
				:"memory"
			);
		}
		
		asm volatile("mcr p15, 0, %0, c7, c10, 4"::"r"(0):"memory");	//drain write buffer
		
		for (i = addr; i != end; i += 32) {
			
			asm volatile(
				"mcr p15, 0, %0, c7, c5, 1"		//clean line by MVA
				:
				:"r"(i)
				:"memory"
			);
		}
	}
}

#ifdef BUILDING_FOR_BIG_ARM
	#pragma GCC pop_options
#endif
















