#define WEAK __attribute__ ((weak))
#define ALIAS(f) __attribute__ ((weak, alias (#f)))

#define NON_PORTABLE
#include <HwrMiscFlags.h>
#undef NON_PORTABLE

#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include <stdio.h>
#include "stm32f469xx.h"
#include "memmap.h"
#include "printf.h"
#include "timers.h"
#include "entry.h"
#include "heap.h"
#include "irqs.h"
#include "boot.h"
#include "mpu.h"

#define EARLYPRINT(...)		//pr(__VA_ARGS__)

void __attribute__((used)) IntDefaultHandler(void)
{
	cpuIrqDefaultHandler();
}

#define VEC_(nm, pfx)	void nm##pfx(void) __attribute__ ((weak, alias ("IntDefaultHandler"))) 
#define VEC(nm)		VEC_(nm, Handler)
#define VECI(nm)	VEC_(nm, IRQHandler)


VEC(NMI_);
VEC(HardFault_);
VEC(MemManage_);
VEC(BusFault_);
VEC(UsageFault_);
VEC(SVC_);
VEC(PendSV_);
VEC(SysTick_);

VECI(WWDG_);
VECI(PVD_);
VECI(TAMP_STAMP_);
VECI(RTC_WKUP_);
VECI(FLASH_);
VECI(RCC_);
VECI(EXTI0_);
VECI(EXTI1_);
VECI(EXTI2_);
VECI(EXTI3_);
VECI(EXTI4_);
VECI(DMA1_Stream0_);
VECI(DMA1_Stream1_);
VECI(DMA1_Stream2_);
VECI(DMA1_Stream3_);
VECI(DMA1_Stream4_);
VECI(DMA1_Stream5_);
VECI(DMA1_Stream6_);
VECI(ADC_);
VECI(CAN1_TX_);
VECI(CAN1_RX0_);
VECI(CAN1_RX1_);
VECI(CAN1_SCE_);
VECI(EXTI9_5_);
VECI(TIM1_BRK_TIM9_);
VECI(TIM1_UP_TIM10_);
VECI(TIM1_TRG_COM_TIM11_);
VECI(TIM1_CC_);
VECI(TIM2_);
VECI(TIM3_);
VECI(TIM4_);
VECI(I2C1_EV_);
VECI(I2C1_ER_);
VECI(I2C2_EV_);
VECI(I2C2_ER_);
VECI(SPI1_);
VECI(SPI2_);
VECI(USART1_);
VECI(USART2_);
VECI(USART3_);
VECI(EXTI15_10_);
VECI(RTC_Alarm_);
VECI(OTG_FS_WKUP_);
VECI(TIM8_BRK_TIM12_);
VECI(TIM8_UP_TIM13_);
VECI(TIM8_TRG_COM_TIM14_);
VECI(TIM8_CC_);
VECI(DMA1_Stream7_);
VECI(FMC_);
VECI(SDIO_);
VECI(TIM5_);
VECI(SPI3_);
VECI(UART4_);
VECI(UART5_);
VECI(TIM6_DAC_);
VECI(TIM7_);
VECI(DMA2_Stream0_);
VECI(DMA2_Stream1_);
VECI(DMA2_Stream2_);
VECI(DMA2_Stream3_);
VECI(DMA2_Stream4_);
VECI(ETH_);
VECI(ETH_WKUP_);
VECI(CAN2_TX_);
VECI(CAN2_RX0_);
VECI(CAN2_RX1_);
VECI(CAN2_SCE_);
VECI(OTG_FS_);
VECI(DMA2_Stream5_);
VECI(DMA2_Stream6_);
VECI(DMA2_Stream7_);
VECI(USART6_);
VECI(I2C3_EV_);
VECI(I2C3_ER_);
VECI(OTG_HS_EP1_OUT_);
VECI(OTG_HS_EP1_IN_);
VECI(OTG_HS_WKUP_);
VECI(OTG_HS_);
VECI(DCMI_);
VECI(HASH_RNG_);
VECI(FPU_);
VECI(UART7_);
VECI(UART8_);
VECI(SPI4_);
VECI(SPI5_);
VECI(SPI6_);
VECI(SAI1_);
VECI(LTDC_);
VECI(LTDC_ER_);
VECI(DMA2D_);
VECI(CRYP_);
VECI(QSPI_);
VECI(DSI_);


//aligned by linker script as needed
__attribute__ ((section(".ramvecs"))) void (*__ISR_VECTORS[]) (void) =
{
	0,		// unused: initial sp
	0,		// unused: reset handler
	NMI_Handler,
	HardFault_Handler,
	MemManage_Handler,
	BusFault_Handler,
	UsageFault_Handler,
	0,
	0,
	0,
	0,
	SVC_Handler,		// SVCall handler
	0,					// Reserved
	0,					// Reserved
	PendSV_Handler,		// The PendSV handler
	SysTick_Handler,	// The SysTick handler
	
	// Chip Level - STM32F469
	WWDG_IRQHandler,
	PVD_IRQHandler,
	TAMP_STAMP_IRQHandler,
	RTC_WKUP_IRQHandler,
	FLASH_IRQHandler,
	RCC_IRQHandler,
	EXTI0_IRQHandler,
	EXTI1_IRQHandler,
	EXTI2_IRQHandler,
	EXTI3_IRQHandler,
	EXTI4_IRQHandler,
	DMA1_Stream0_IRQHandler,
	DMA1_Stream1_IRQHandler,
	DMA1_Stream2_IRQHandler,
	DMA1_Stream3_IRQHandler,
	DMA1_Stream4_IRQHandler,
	DMA1_Stream5_IRQHandler,
	DMA1_Stream6_IRQHandler,
	ADC_IRQHandler,
	CAN1_TX_IRQHandler,
	CAN1_RX0_IRQHandler,
	CAN1_RX1_IRQHandler,
	CAN1_SCE_IRQHandler,
	EXTI9_5_IRQHandler,
	TIM1_BRK_TIM9_IRQHandler,
	TIM1_UP_TIM10_IRQHandler,
	TIM1_TRG_COM_TIM11_IRQHandler,
	TIM1_CC_IRQHandler,
	TIM2_IRQHandler,
	TIM3_IRQHandler,
	TIM4_IRQHandler,
	I2C1_EV_IRQHandler,
	I2C1_ER_IRQHandler,
	I2C2_EV_IRQHandler,
	I2C2_ER_IRQHandler,
	SPI1_IRQHandler,
	SPI2_IRQHandler,
	USART1_IRQHandler,
	USART2_IRQHandler,
	USART3_IRQHandler,
	EXTI15_10_IRQHandler,
	RTC_Alarm_IRQHandler,
	OTG_FS_WKUP_IRQHandler,
	TIM8_BRK_TIM12_IRQHandler,
	TIM8_UP_TIM13_IRQHandler,
	TIM8_TRG_COM_TIM14_IRQHandler,
	TIM8_CC_IRQHandler,
	DMA1_Stream7_IRQHandler,
	FMC_IRQHandler,
	SDIO_IRQHandler,
	TIM5_IRQHandler,
	SPI3_IRQHandler,
	UART4_IRQHandler,
	UART5_IRQHandler,
	TIM6_DAC_IRQHandler,
	TIM7_IRQHandler,
	DMA2_Stream0_IRQHandler,
	DMA2_Stream1_IRQHandler,
	DMA2_Stream2_IRQHandler,
	DMA2_Stream3_IRQHandler,
	DMA2_Stream4_IRQHandler,
	ETH_IRQHandler,
	ETH_WKUP_IRQHandler,
	CAN2_TX_IRQHandler,
	CAN2_RX0_IRQHandler,
	CAN2_RX1_IRQHandler,
	CAN2_SCE_IRQHandler,
	OTG_FS_IRQHandler,
	DMA2_Stream5_IRQHandler,
	DMA2_Stream6_IRQHandler,
	DMA2_Stream7_IRQHandler,
	USART6_IRQHandler,
	I2C3_EV_IRQHandler,
	I2C3_ER_IRQHandler,
	OTG_HS_EP1_OUT_IRQHandler,
	OTG_HS_EP1_IN_IRQHandler,
	OTG_HS_WKUP_IRQHandler,
	OTG_HS_IRQHandler,
	DCMI_IRQHandler,
	CRYP_IRQHandler,
	HASH_RNG_IRQHandler,
	FPU_IRQHandler,
	UART7_IRQHandler,
	UART8_IRQHandler,
	SPI4_IRQHandler,
	SPI5_IRQHandler,
	SPI6_IRQHandler,
	SAI1_IRQHandler,
	LTDC_IRQHandler,
	LTDC_ER_IRQHandler,
	DMA2D_IRQHandler,
	QSPI_IRQHandler,
	DSI_IRQHandler,
};

static void clockTreeInit(void)
{	
	//turn on PWR
	RCC->APB1ENR |= RCC_APB1ENR_PWREN;
	
	//setup flash wait states and optimizations, reset caches
	FLASH->ACR = FLASH_ACR_LATENCY_5WS;
	FLASH->ACR |= FLASH_ACR_DCRST | FLASH_ACR_ICRST;
	FLASH->ACR &=~ (FLASH_ACR_DCRST | FLASH_ACR_ICRST);
	FLASH->ACR |= FLASH_ACR_PRFTEN | FLASH_ACR_ICEN | FLASH_ACR_DCEN;

	//first go to safe settings: HSI
	RCC->CR = RCC_CR_HSION;													//HSI on, PLL off
	RCC->CFGR = (RCC->CFGR &~ RCC_CFGR_SW_Msk) | RCC_CFGR_SW_HSI;			//switch to HSI
	
	
	EARLYPRINT("setting voltage scaling\n");
	if ((PWR->CR & PWR_CR_VOS) != (PWR_CR_VOS_0 | PWR_CR_VOS_1)) {
		PWR->CR |= PWR_CR_VOS_0 | PWR_CR_VOS_1;
		while (!(PWR->CSR & PWR_CSR_VOSRDY));
	}
	
	EARLYPRINT("PWR->CSR=0x%08x\n", PWR->CSR);
	PWR->CR |= PWR_CR_ODEN;
	while (!(PWR->CSR & PWR_CSR_ODRDY));
	
	EARLYPRINT("setting over drive on\n");
	PWR->CR |= PWR_CR_ODSWEN;
	while (!(PWR->CSR & PWR_CSR_ODSWRDY));
	
	//turn on HSE and wait for it
	RCC->CR |= RCC_CR_HSEON;
	while (!(RCC->CR & RCC_CR_HSERDY));
	
	//HSE / 4 as source (2MHz), VCO outputs 384MHZ, P output (main clock) is 192MHz, Q output (usb&SDIO is 48MHz)
	RCC->PLLCFGR = (192 << RCC_PLLCFGR_PLLN_Pos) | (8 << RCC_PLLCFGR_PLLQ_Pos) | (4 << RCC_PLLCFGR_PLLM_Pos) | RCC_PLLCFGR_PLLSRC_HSE;
	
	//we need PLLSAI for LCD at precisely thid freq. CFG: HSE / 4 as source (2MHz), VCO outputs 384MHZ, SAI1 clock min speed, SAIDLCD clock = VCO/7 = 54.857 MHz
	RCC->PLLSAICFGR = (192 << RCC_PLLSAICFGR_PLLSAIN_Pos) | (15 << RCC_PLLSAICFGR_PLLSAIQ_Pos) | (7 << RCC_PLLSAICFGR_PLLSAIR_Pos);
	
	//set up lcd clock
	RCC->DCKCFGR = 0 << RCC_DCKCFGR_PLLSAIDIVR_Pos;	//LCD = SAILCD / 2 = 27.429 MHz
	
	//turn on the main PLL and wait for it
	RCC->CR |= RCC_CR_PLLON;
	while (!(RCC->CR & RCC_CR_PLLRDY));
	
	//turnon SAI PLL & wait for it
	RCC->CR |= RCC_CR_PLLSAION;
	while (!(RCC->CR & RCC_CR_PLLSAIRDY));
	
	//set up other divisors (RTC = HSE/8 = 1MHz, APB2 = sysclk/2 = 96MZ, APB1 = sysclk/4 = 48MZ, AHB = sysclk = 192MHz) and go to it
	RCC->CFGR = RCC_CFGR_I2SSRC | (8 << RCC_CFGR_RTCPRE_Pos) | RCC_CFGR_PPRE2_DIV2 | RCC_CFGR_PPRE1_DIV4 | RCC_CFGR_SW_PLL;
	while (((RCC->CFGR & RCC_CFGR_SW_Msk) >> RCC_CFGR_SW_Pos) != ((RCC->CFGR & RCC_CFGR_SWS_Msk) >> RCC_CFGR_SWS_Pos));
}

static void clockUpPeriphs(void)
{
	RCC->AHB1ENR = RCC_AHB1ENR_GPIOAEN | RCC_AHB1ENR_GPIOBEN | RCC_AHB1ENR_GPIOCEN | RCC_AHB1ENR_GPIODEN | RCC_AHB1ENR_GPIOEEN | RCC_AHB1ENR_GPIOFEN | RCC_AHB1ENR_GPIOGEN | RCC_AHB1ENR_GPIOHEN | RCC_AHB1ENR_GPIOIEN | RCC_AHB1ENR_GPIOJEN | RCC_AHB1ENR_GPIOKEN | RCC_AHB1ENR_DMA2DEN | RCC_AHB1ENR_CCMDATARAMEN;
	RCC->AHB3ENR = RCC_AHB3ENR_FMCEN;
	RCC->APB2ENR = RCC_APB2ENR_LTDCEN | RCC_APB2ENR_SPI5EN | RCC_APB2ENR_DSIEN | RCC_APB2ENR_SYSCFGEN;
	RCC->APB1ENR = RCC_APB1ENR_TIM2EN | RCC_APB1ENR_PWREN | RCC_APB1ENR_USART3EN;
}

static void gpiosInit(void)
{
	/*-- GPIOs Configuration -----------------------------------------------------*/
	/*
	 SDRAM:
	 		C: 0
	 		D: 0 1 8 9 10 14 15
	 		E: 0 1 7 8 9 10 11 12 13 14 15
	 		F: 0 1 2 3 4 5 11 12 13 14 15
	 		G: 0 1 4 5 8 15
	 		H: 2 3 8 9 10 11 12 13 14 15
	 		I: 0 1 2 3 4 5 6 7 9 10

	DSI:
			F: 12 13
			H: 12 13
			J: 2 12 13 
			
			
	LCD:
			int: J5
			blctrl: A3
			reset: gpio OpenDrain with no pull H7

	SWD:
			A: 13 14

	I2C1:
			B: 8(C) 9(D)
			
	touch interrpt:
			pj5
	
	AlarmLed: K3
	
	STLINK uart
			PB10 = USART3.DO (af 7)
			PB11 = USART3.DI (af 7)
	

	*/
	
	GPIOA->MODER	= 0x28000000;	//A13,A14 - SWD
	GPIOA->OTYPER	= 0x00000000;
	GPIOA->OSPEEDR	= 0x3C000000;
	GPIOA->PUPDR	= 0x24000000;
	GPIOA->AFR[0]	= 0x00000000;
	GPIOA->AFR[1]	= 0x00000000;
	
	GPIOB->MODER	= 0x00a50000;	//B8,B9 - I2C1, B10,B11-debug uart
	GPIOB->OTYPER	= 0x00000300;
	GPIOB->OSPEEDR	= 0x000a0000;
	GPIOB->PUPDR	= 0x00000000;
	GPIOB->AFR[0]	= 0x00000000;
	GPIOB->AFR[1]	= 0x00007700;

	GPIOC->MODER	= 0x00000002;	//C0 - SDRAM
	GPIOC->OTYPER	= 0x00000000;
	GPIOC->OSPEEDR	= 0x00000002;
	GPIOC->PUPDR	= 0x00000001;
	GPIOC->AFR[0]	= 0x0000000C;
	GPIOC->AFR[1]	= 0x00000000;

	GPIOD->MODER	= 0xA02A000A;	//D0,D1,D8,D9,D10,D14,D15 - SDRAM
	GPIOD->OTYPER	= 0x00000000;
	GPIOD->OSPEEDR	= 0xA02A000A;
	GPIOD->PUPDR	= 0x50150005;
	GPIOD->AFR[0]	= 0x000000CC;
	GPIOD->AFR[1]	= 0xCC000CCC;

	GPIOE->MODER	= 0xAAAA800A;	//E0,E1,E7,E8,E9,E10,E11,E12,E13,E14,E15 - SDRAM
	GPIOE->OTYPER	= 0x00000000;
	GPIOE->OSPEEDR	= 0xAAAA800A;
	GPIOE->PUPDR	= 0x55554005;
	GPIOE->AFR[0]	= 0xC00000CC;
	GPIOE->AFR[1]	= 0xCCCCCCCC;

	GPIOF->MODER	= 0xAA800AAA;	//F0,F1,F2,F3,F4,F5,F11,F12,F13,F14,F15 - SDRAM
	GPIOF->OTYPER	= 0x00000000;
	GPIOF->OSPEEDR	= 0xAA800AAA;
	GPIOF->PUPDR	= 0x55400555;
	GPIOF->AFR[0]	= 0x00CCCCCC;
	GPIOF->AFR[1]	= 0xCCCCC000;
	
	GPIOG->MODER	= 0x80020A0A;	//G0,G1,G4,G5,G8,G15 = SDRAM
	GPIOG->OTYPER	= 0x00000000;
	GPIOG->OSPEEDR	= 0x80020A0A;
	GPIOG->PUPDR	= 0x40010505;
	GPIOG->AFR[0]	= 0x00CC00CC;
	GPIOG->AFR[1]	= 0xC000000C;
	
	GPIOH->MODER	= 0xAAAA40A0;	//H2,H3,H8,H9,H10,H11,H12,H13,H14,H15 = SDRAM, H7-lcd_reset
	GPIOH->OTYPER	= 0x00000000;
	GPIOH->OSPEEDR	= 0xAAAA00A0;
	GPIOH->PUPDR	= 0x55550050;
	GPIOH->AFR[0]	= 0x0000CC00;
	GPIOH->AFR[1]	= 0xCCCCCCCC;
	
	GPIOI->MODER	= 0x0028AAAA;	//I0,I1,I2,I3,I4,I5,I6,I7,I9,I10 = SDRAM
	GPIOI->OTYPER	= 0x00000000;
	GPIOI->OSPEEDR	= 0x0028AAAA;
	GPIOI->PUPDR	= 0x00145555;
	GPIOI->AFR[0]	= 0xCCCCCCCC;
	GPIOI->AFR[1]	= 0x00000CC0;
	
	GPIOJ->MODER	= 0x00000000;	//J5 - touch interrupt
	GPIOJ->OTYPER	= 0x00000000;
	GPIOJ->OSPEEDR	= 0x00000000;
	GPIOJ->PUPDR	= 0x00000000;
	GPIOJ->AFR[0]	= 0x00000000;
	GPIOJ->AFR[1]	= 0x00000000;
	
	GPIOK->MODER	= 0x00000040;	//K3 - AlarmLed
	GPIOK->OTYPER	= 0x00000000;
	GPIOK->OSPEEDR	= 0x00000000;
	GPIOK->PUPDR	= 0x00000000;
	GPIOK->AFR[0]	= 0x00000000;
	GPIOK->AFR[1]	= 0x00000000;
	GPIOK->BSRR		= 0x00000008;	//set K3 to turn off LED
		
	//setup which GPIOs are used for external ints
	SYSCFG->EXTICR[0] = 0x00000000;
	SYSCFG->EXTICR[1] = 0x00000000;
	SYSCFG->EXTICR[2] = 0x00000000;
	SYSCFG->EXTICR[3] = 0x00000000;
	
	//uart's APB is CPU_CLOCK_RATE / 2
	#define DEBUG_UART_BAUDRATE		115200
	
	USART3->BRR = (CPU_CLOCK_RATE / 2 + DEBUG_UART_BAUDRATE / 2) / DEBUG_UART_BAUDRATE;
	USART3->GTPR = 1;	//as per spec
	USART3->CR1 = USART_CR1_UE | USART_CR1_TE;
}

static bool sdramSendCmd(uint32_t cmdModeAndTarget, uint32_t autoRefreshNmbr, uint32_t modeRegisterDef, uint32_t timeout)
{
	uint32_t ticks;
	
	FMC_Bank5_6->SDCMR = cmdModeAndTarget | ((autoRefreshNmbr - 1) << FMC_SDCMR_NRFS_Pos) | (modeRegisterDef << FMC_SDCMR_MRD_Pos);
	
	for (ticks = 0; ticks < timeout; ticks++) {
		if (FMC_Bank5_6->SDSR & FMC_SDSR_BUSY)
			continue;
		return true;
	}
	
	return false;
}

static void sdramInit(void)
{
	uint32_t cfg;
	
	#define FMC_SDRAM_CMD_CLK_ENABLE				1
	#define FMC_SDRAM_CMD_PALL						2
	#define FMC_SDRAM_CMD_AUTOREFRESH_MODE			3
	#define FMC_SDRAM_CMD_LOAD_MODE					4
	
	#define SDRAM_MODEREG_BURST_LENGTH_1			0
	#define SDRAM_MODEREG_BURST_LENGTH_2			1
	#define SDRAM_MODEREG_BURST_TYPE_SEQUENTIAL		0
	#define SDRAM_MODEREG_CAS_LATENCY_3				0x30
	#define SDRAM_MODEREG_OPERATING_MODE_STANDARD	0
	#define SDRAM_MODEREG_WRITEBURST_MODE_SINGLE	0x200
	
	
	//SDRAM config
	cfg = FMC_SDCR1_SDCLK_1 | FMC_SDCR1_RBURST | FMC_SDCR1_NB | (3 << FMC_SDCR1_CAS_Pos) | FMC_SDCR1_MWID_1 | FMC_SDCR1_NR_0;
	FMC_Bank5_6->SDCR[0] = cfg;
	FMC_Bank5_6->SDCR[1] = cfg;

	//SDRAM timing config
	cfg =	((2 /* LoadToActiveDelay*/ - 1) << FMC_SDTR1_TMRD_Pos) |
			((6 /* ExitSelfRefreshDelay */ - 1) << FMC_SDTR1_TXSR_Pos) |
			((4 /* SelfRefreshTime */ - 1) << FMC_SDTR1_TRAS_Pos) |
			((6 /* RowCycleDelay */ - 1) << FMC_SDTR1_TRC_Pos) |
			((2 /* WriteRecoveryTime */ - 1) << FMC_SDTR1_TWR_Pos) |
			((2 /* RPDelay */ - 1) << FMC_SDTR1_TRP_Pos) |
			((2 /* RCDDelay */ - 1) << FMC_SDTR1_TRCD_Pos);
	
	FMC_Bank5_6->SDTR[0] = cfg;
	FMC_Bank5_6->SDTR[1] = cfg;

	//Clock configuration enable command
	if (!sdramSendCmd(FMC_SDRAM_CMD_CLK_ENABLE | FMC_SDCMR_CTB1, 1, 0, 0x1000))
		fatal("SDRAM init failed: %s\n", "Cfg Ena");
	
	//100 ms delay or so
	asm volatile(
		"1:					\n\t"
		"	subs %0, #1		\n\t"
		"	bne 1b			\n\t"
		:"=l"(cfg)
		:"0"(6400000)
		:"cc"
	);

	//PALL (precharge all) command
	if (!sdramSendCmd(FMC_SDRAM_CMD_PALL | FMC_SDCMR_CTB1, 1, 0, 0x1000))
		fatal("SDRAM init failed: %s\n", "PALL");
	
	//Auto-Refresh command
	if (!sdramSendCmd(FMC_SDRAM_CMD_AUTOREFRESH_MODE | FMC_SDCMR_CTB1, 8, 0, 0x1000))
		fatal("SDRAM init failed: %s\n", "AutoRefresh");

	//Program external memory mode register
	if (!sdramSendCmd(FMC_SDRAM_CMD_LOAD_MODE | FMC_SDCMR_CTB1, 1, SDRAM_MODEREG_BURST_LENGTH_1 | SDRAM_MODEREG_BURST_TYPE_SEQUENTIAL | SDRAM_MODEREG_CAS_LATENCY_3 | SDRAM_MODEREG_OPERATING_MODE_STANDARD | SDRAM_MODEREG_WRITEBURST_MODE_SINGLE, 0x1000))
		fatal("SDRAM init failed: %s\n", "External Memory Mode");

	//Set the refresh rate counter => (15.62 us x Freq) - 20, Set the device refresh counter */
	FMC_Bank5_6->SDRTR |= (0x050C << FMC_SDRTR_COUNT_Pos);
	
	//map it at 0x60000000 instead of 0xc0000000
	SYSCFG->MEMRMP = (SYSCFG->MEMRMP &~ SYSCFG_MEMRMP_SWP_FMC_Msk) | SYSCFG_MEMRMP_SWP_FMC_0;
	
	//for now, clear it
	memset((void*)0x60000000, 0, 16*1024*1024);
}

void machIdle(void)
{
	//wfi fails like on STM32F429
}

static void mpuRegCfg(uint32_t idx, uint32_t addr, uint32_t cfg)
{
	MPU->RBAR = addr | 0x10 | idx;
	MPU->RASR = cfg;
}

//configures mmu for storage ram
void machSetStorageAreaWriteable(bool writeable)
{
	mpuRegCfg(2, 0x60000000, (writeable ? MPU_PERM_U_RW_S_RW : MPU_PERM_U_RO_S_RO) | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | (23 << 1) | 0x0000);
}

void __attribute__((used)) machInit(uint32_t stage, const void* data)
{
	if (stage == STAGE_INIT_EARLY) {
		
		clockTreeInit();
		clockUpPeriphs();
		gpiosInit();
		sdramInit();
	}
	else if (stage == STAGE_INIT_SET_VTOR) {
		SCB->VTOR = (uint32_t)__ISR_VECTORS;
	}
	else if (stage == STAGE_SETUP_HEAPS) {
		
		//register CCM first so it is checked first
		kheapRegisterHeap(HAL_CCM_MEM_BASE, HAL_CCM_MEM_SIZE, MEM_USABLE_AS_STACK | MEM_FAST);
		kheapRegisterHeap(HAL_STATIC_MEM_BASE, HAL_STATIC_MEM_SIZE, MEM_USABLE_AS_STACK | MEM_USABLE_FOR_DMA | MEM_USABLE_FOR_EXEC | MEM_FAST);
	}
	else if (stage == STAGE_INIT_MPU) {
		
		// * CCM 0x10000000 + 0x00010000. cannot be DMAd or executed from
		// * internal ram (0x20000000) is jit TC (160K), .data&.bss (32K) and heap (128K) (320K total)
		// * rom 0x08000000
		// * ystsem memory at 0x1fff0000
		//configured together to save regions, SRD *IS* correct
		mpuRegCfg(0, 0x00000000, MPU_PERM_U_RW_S_RW | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | (29 << 1) | 0xe100);

		//a region for NULL faulting (0x08000000 in size)
		mpuRegCfg(1, 0x00000000, MPU_PERM_U_XX_S_XX | MPU_PERM_NX | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | (26 << 1) | 0x0000);
		
		//16MB region for storage ram (dyn ram and vram overlay it)
		machSetStorageAreaWriteable(false);
		
		//4MB region for dynamic ram + vram
		mpuRegCfg(3, 0x60000000, MPU_PERM_U_RW_S_RW | MPU_MEM_TYPE_RAM | MPU_FLAG_ENABLED | (21 << 1) | 0x0000);
		
		//Periphs (0x40000000 + 0x10000000, device, nx)
		mpuRegCfg(4, 0x40000000, MPU_PERM_U_XX_S_RW | MPU_MEM_TYPE_DEVICE | MPU_PERM_NX | MPU_FLAG_ENABLED | (27 << 1) | 0x0000);
		
		//SCS/SCB/etc in 0xE0000000..0xE0100000 is always acessed using the default map
		
		mpuRegCfg(5, 0, 0);
		mpuRegCfg(6, 0, 0);
		
		//mpu on
		MPU->CTRL = MPU_CTRL_ENABLE_Msk;
	}
	else if (stage == STAGE_INIT_INTERRUPTS) {
		
		const struct MachInitDataInterrupts *info = (const struct MachInitDataInterrupts*)data;
		uint32_t mediumPrio = (info->lowestAllowablePrio + info->highestAllowablePrio) / 2;
		
		//scheduler timer interrupt is high prio too so nobody else can interrupt it (and more importantly - it will not interrupt syscalls and vise-versa)
		NVIC_SetPriority(TIM2_IRQn, info->schedulingTimerPrio);
		
		//lcd copying int is highest prio as well. safe since it doesnt touch any structs
		NVIC_SetPriority(TIM5_IRQn, info->lowestAllowablePrio);
		
		//set all HW ints to medium prio
		NVIC_SetPriority(EXTI9_5_IRQn, mediumPrio);
		NVIC_SetPriority(RTC_WKUP_IRQn, mediumPrio);
		NVIC_SetPriority(RTC_Alarm_IRQn, mediumPrio);
	}
}

static void machBusyWaitDelay(uint64_t ticks)
{
	uint64_t start = timerGetTime();
	
	while (timerGetTime() - start < ticks);
}

void machBusyWaitDelayMsec(uint32_t msec)
{
	machBusyWaitDelay((uint64_t)msec * TIMER_TICKS_PER_MSEC);
}

void machBusyWaitDelayUsec(uint32_t usec)
{
	machBusyWaitDelay((uint64_t)usec * (TIMER_TICKS_PER_MSEC / 1000));
}

bool hwMaybeGetRomToken(uint32_t name, const void **dataP, uint16_t *szP)
{
	if (name == CREATE_4CC('s','n','u','m')) {
		
		static char snumStr[12];
		
		if (!snumStr[0]) {
			static const char *base32 = "23456789ABCDEFGHJKLMNPQRSTUVWXYZ";
			volatile uint32_t *uid = (volatile uint32_t*)UID_BASE;
			uint32_t snumLo = uid[0] ^ uid[1];
			uint32_t snumHi = uid[1] ^ uid[2];
			uint64_t snum = (((uint64_t)snumHi) << 32) + snumLo;
			uint32_t i;
			
			for (i = 0; i < sizeof(snumStr); i++, snum >>= 5)
				snumStr[i] = base32[snum & 31];
		}
		if (dataP)
			*dataP = snumStr;
		if(szP)
			*szP = sizeof(snumStr);
		
		return true;
	}
	return false;
}

void hwGetMiscFlags(uint16_t *miscFlagsP, uint16_t *extMiscFlagsP)
{
	if (miscFlagsP)
		*miscFlagsP = hwrMiscFlagHasMiscFlagExt | hwrMiscFlagHasCradleDetect | hwrMiscFlagNoRTCBug | hwrMiscFlagHasMbdIrDA | hwrMiscFlagHasBacklight;
	
	if (extMiscFlagsP)
		*extMiscFlagsP = hwrMiscFlagExtHasSWBright | hwrMiscFlagExt115KIrOK | hwrMiscFlagExtHasLiIon;
}

int32_t cpuGetClockRate(enum ClockRateDevice dev)
{
	switch (dev) {
		case CpuClockRate:
		case TimerClockRate:
			return CPU_CLOCK_RATE;
		
		case SdioUnitClockRate:
			return CPU_CLOCK_RATE / 4;
		
		case UartUnitClockRate:
			return CPU_CLOCK_RATE / 2;
		
		default:
			return-1;
	}
}

bool hwPwrCtl(uint32_t selector, const uint32_t *newValP, uint32_t *oldValP)
{
	return false;
}

void machSleep(void)
{
	logi("pretending to sleep\n");
	SysTaskDelay(10000);
	logi("waking up\n");
	dalModifyWakeFlags(DAL_WAKE_FLAG_GENERAL, 0);
	//nothing yet
}

Err machinePaceDispatch(EmulStateRef ref, uint16_t call, Err *ret68kP)
{
	return sysErrNotAllowed;
}

