#include "machSpecific.h"
#include "printf.h"
#include "input.h"
#include "cpu.h"
#include "kal.h"
#include "ral.h"
#include "dal.h"


static DrvInputKeyCbk mKeyCbk = NULL;
static DrvInputBtnCbk mBtnCbk = NULL;
static DrvInputPenCbk mPenCbk = NULL;
static uint8_t mTouchChipI2cId;
static tid_t mI2cThread;

#define TS_CHIP_ADDR1			0x2a		//7-bit addr		FT6206
#define TS_CHIP_ADDR2			0x38		//7-bit addr		FT6336G

#define TS_REG_DEV_MODE			0x00
#define TS_DEV_MODE_FACTORY		0x40
#define TS_REG_DATA				0x02
#define TS_CALIB_REG			0x02	// in factory mode
#define TS_CALIB_VAL			0x04
#define TS_REG_CTRL				0x86
#define TS_CTRL_PWR_SV_VAL		0x01
#define TS_REG_G_MODE			0xA4
#define TS_G_MODE_INT_POL_VAL	0x00
#define TS_REG_CHIP_ID			0xA8
#define TS_CHIP_ID_VAL			0x11

struct TsTouch {
	uint8_t xh			: 4;
	uint8_t reserved0	: 2;
	uint8_t evt			: 2;
	uint8_t xl;
	uint8_t yh			: 4;
	uint8_t id			: 4;
	uint8_t yl;
	uint8_t weight;
	uint8_t area;
};

struct TsTouchPartial {
	uint8_t xh			: 4;
	uint8_t reserved0	: 2;
	uint8_t evt			: 2;
	uint8_t xl;
	uint8_t yh			: 4;
	uint8_t id			: 4;
	uint8_t yl;
};

struct TsDataFull {
	uint8_t numTouchPoints	: 4;
	uint8_t reserved0		: 4;
	struct TsTouch touches[2];
};

struct TsDataOneTouch {
	uint8_t numTouchPoints	: 4;
	uint8_t reserved0		: 4;
	struct TsTouch touch;
};

struct TsDataOneTouchPartial {			//for i2c speed
	uint8_t numTouchPoints	: 4;
	uint8_t reserved0		: 4;
	struct TsTouchPartial touch;
};

static void i2cDelay(void)			//FT6202 is very slow and if the clock is fast it will NOT clock stretch, just give us duplicate bytes (i guess it doesnt reload internal buffer fast enough)
{
	uint32_t tmp;
	
	asm volatile(
		"1:						\n\t"
		"	subs %0, #1			\n\t"
		"	bne 1b				\n\t"
		:"=l"(tmp)
		:"0"(500)
		:"memory", "cc"
	);
}

static void i2cSdaHi(void)
{
	GPIOB->BSRR = 1 << 9;
}

static void i2cSdaLo(void)
{
	GPIOB->BSRR = 1 << (16 + 9);
}

static bool i2cSdaRead(void)
{
	return !!(GPIOB->IDR & (1 << 9));
}

static bool i2cSclRead(void)
{
	return !!(GPIOB->IDR & (1 << 8));
}

static void i2cSclHi(void)
{
	GPIOB->BSRR = 1 << 8;
	while(!i2cSclRead());	//wait for clock stretching to end
}

static void i2cSclLo(void)
{
	GPIOB->BSRR = 1 << (16 + 8);
}

static void i2cBitW(bool hi)
{
	if (hi)
		i2cSdaHi();
	else
		i2cSdaLo();
	i2cDelay();
	i2cSclHi();
	i2cDelay();
	i2cSclLo();
}

static bool i2cBitR(void)
{
	bool ret;
	
	i2cSdaHi();
	i2cDelay();
	i2cSclHi();
	i2cDelay();
	
	ret = i2cSdaRead();
	
	i2cSclLo();
	
	return ret;
}

static void i2cStart(void)
{
	i2cSclHi();
	i2cSdaHi();
	i2cDelay();
	i2cSdaLo();
	i2cDelay();
	i2cSclLo();
}

static void i2cStop(void)
{
	i2cSdaLo();
	i2cDelay();
	i2cSclHi();
	i2cDelay();
	i2cSdaHi();
	i2cDelay();
}

static bool i2cByteW(uint32_t val)
{
	uint32_t i;
	
	for (i = 0; i < 8; i++, val <<= 1)
		i2cBitW(!!(val & 0x80));
	
	return !i2cBitR();
}

static uint32_t i2cByteR(bool ack)
{
	uint32_t i, v = 0;
	
	for (i = 0; i < 8; i++)
		v = (v << 1) | (i2cBitR() ? 1 : 0);
	
	i2cBitW(!ack);
	
	return v;
}

static bool i2cWrite(uint8_t dev, uint8_t reg, const uint8_t *bytes, uint32_t len)	//guaranteed to permanently fuck up the I2C unit internal state machine if you get a NAK. FUCK ST for this shitty I2C unit!
{
	i2cStart();
	if (!i2cByteW(dev * 2 + 0)) {
		loge("write address NAKed\n");
		return false;
	}
	if (!i2cByteW(reg)) {
		loge("write reg NAKed\n");
		return false;
	}
	
	while (len) {
		
		len--;
		if (!i2cByteW(*bytes++)) {
			
			if (len) {
				loge("early nak\n");
				i2cStop();
				return false;
			}
		}
	}
	
	i2cStop();
	return true;
}

static bool i2cRead(uint8_t dev, uint8_t reg, uint8_t *bytes, uint32_t len)	//guaranteed to permanently fuck up the I2C unit internal state machine if you get a NAK. FUCK ST for this shitty I2C unit!
{
	i2cStart();
	if (!i2cByteW(dev * 2 + 0)) {
		loge("write address NAKed\n");
		return false;
	}
	if (!i2cByteW(reg)) {
		loge("write reg NAKed\n");
		return false;
	}
	i2cStart();
	if (!i2cByteW(dev * 2 + 1)) {
		loge("read address NAKed\n");
		return false;
	}
	
	while (len) {
		
		len--;
		*bytes++ = i2cByteR(!!len);
	}
	
	i2cStop();
	return true;
}

static int32_t touchRegRead8(uint8_t regAddr)
{
	uint8_t reg = 0;
	
	if (!i2cRead(mTouchChipI2cId, regAddr, &reg, 1))
		return -1;
	
	return (uint32_t)reg;
}

static bool touchRegReadBurst(uint8_t regAddr, void* dst, uint32_t num)
{
	return i2cRead(mTouchChipI2cId, regAddr, dst, num);
}

static bool touchRegWrite8(uint32_t regAddr, uint8_t val)
{
	return i2cWrite(mTouchChipI2cId, regAddr, &val, 1);
}

static bool touchRegWriteBurst(uint32_t regAddr, const uint8_t *vals, uint32_t num)
{
	return i2cWrite(mTouchChipI2cId, regAddr, vals, num);
}

void __attribute__((used)) EXTI9_5_IRQHandler(void)
{
	uint32_t r9 = ralSetSafeR9();	//called form irq with random r9
	
	if (EXTI->PR & 0x0020) {
		EXTI->PR = 0x0020;
		KALTaskWake(mI2cThread);
	}
	
	ralRestoreR9(r9);
}

static void i2cThreadFunc(void *param)
{
	static const uint8_t calibdata[] = {0x16, 0x3C, 0xE9, 0x01, 0x01, 0xA0, 0x00/*ctrl*/, 0x0A, 0x06, 0x28};	//undocumented settings regs
	int32_t i, val;

	for (i = 0, mTouchChipI2cId = TS_CHIP_ADDR1; i < 2; i++, mTouchChipI2cId = TS_CHIP_ADDR2) {
		
		bool found = false;
		
		i2cStart();
		i2cDelay();
		found = i2cByteW(mTouchChipI2cId * 2 + 0);
		i2cDelay();
		i2cStop();
		
		if (found)
			break;
	}
	if (i == 2)
		fatal("Touch chip not found\n");
	
	
	logi("Potential touch chip found at i2c id 0x%02x\n", mTouchChipI2cId);	
	val = touchRegRead8(TS_REG_CHIP_ID);
		
	logi("TOUCH: CHIP_ID = 0x%02x\n", val);
	if (val != TS_CHIP_ID_VAL)
		fatal("Touch chip not verified due to ID 0x%x\n", val);
	
	//calibrate
	if (!touchRegWriteBurst(0x80, calibdata, sizeof(calibdata)))
		fatal("Cannot setup touch chip\n");

	if (!touchRegWrite8(TS_REG_G_MODE, TS_G_MODE_INT_POL_VAL)) {
		loge("failed to write MOD\n");
		return;
	}


	while(1) {
		
		uint32_t prevX = (uint32_t)-1, prevY = (uint32_t)-1;
		
		KALTaskWait(-1);
		
		while (!(GPIOJ->IDR & 0x0020)) {		//low until touch lifted
			
			struct TsDataOneTouchPartial ts;
			uint32_t i;
			
			if (!touchRegReadBurst(TS_REG_DATA, &ts, sizeof(ts)))
				loge("data read fail\n");
			else if (ts.numTouchPoints == 1 && ts.touch.evt == 2) {
				
				uint32_t x = ts.touch.xh, y = ts.touch.yh;
				
				x = (x << 8) + ts.touch.xl;
				y = (y << 8) + ts.touch.yl;
				
				if (x == prevX && y == prevY)	//if same as before, wait a bit
					KALTaskDelay(50);
				else {
					mPenCbk(x, y);
					prevX = x;
					prevY = y;
					KALTaskDelay(20);
				}
			}
			
			if (GPIOJ->IDR & 0x0020)		//wait 20 ms before believing that finger was lifted
				KALTaskDelay(20);
		}
		mPenCbk(-1, -1);
	}
}

kstatus_t drvInputInit(DrvInputKeyCbk keyCbk, DrvInputBtnCbk btnCbk, DrvInputPenCbk penCbk)
{
	static const struct KalTaskCreateInfo i2cThCrNfo = {
		.codeToRun = i2cThreadFunc,
		.stackSz = 1024,
		.prio = 5,	//hi prio
		.tag = CREATE_4CC('_','i','2','c'),
	};
	
	mKeyCbk = keyCbk;
	mBtnCbk = btnCbk;
	mPenCbk = penCbk;
	
	GPIOB->BSRR = (1 << 8) | (1 << 9);
	
	SYSCFG->EXTICR[1] = (SYSCFG->EXTICR[1] &~ 0xf0) | (9 << 4);		//EXTICR2.EXTI5 =9  (uses port J), yes index is correct!
	EXTI->FTSR |= 0x0020;	//detect faling edge on EXTI5 (aka J5)
	EXTI->IMR |= 0x0020;	//interrupt is on for EXTI5 (aka J5)
	EXTI->PR = 0x0020;		//clear current status on EXTI5 (aka J5)
	machBusyWaitDelayMsec(1);
	
	if (errNone != KALTaskCreate(&mI2cThread, &i2cThCrNfo))
		fatal("Cannot create i2c thread\n");

	if (errNone != KALTaskStart(mI2cThread, NULL))
		fatal("Cannot start i2c thread\n");
	
	logi("i2c Thread is up\n");
	
	NVIC_EnableIRQ(EXTI9_5_IRQn);

	return KERN_STATUS_OK;
}

kstatus_t drvInputPreSleep(void)
{
	return KERN_STATUS_OK;
}

kstatus_t drvInputPostWake(void)
{
	return KERN_STATUS_OK;
}
