#ifndef _EMU_JIT_INTERNAL_H_
#define _EMU_JIT_INTERNAL_H_

#include <stdbool.h>
#include <stdarg.h>
#include <stdint.h>
#include "emit.h"



/*
	* create a light "emitBuf struct" that is juts a tagged pointer storing a pointer ot one halfword
	* use callout to set pc always - much faster than loading directly
	* XXX: does c-m0 store Q flag? test!
*/

enum JitArmDpOp {		//DP ops as seen in ARM encodings
	ArmDpOpAnd,
	ArmDpOpEor,
	ArmDpOpSub,
	ArmDpOpRsb,
	ArmDpOpAdd,
	ArmDpOpAdc,
	ArmDpOpSbc,
	ArmDpOpRsc,
	ArmDpOpTst,
	ArmDpOpTeq,
	ArmDpOpCmp,
	ArmDpOpCmn,
	ArmDpOpOrr,
	ArmDpOpMov,
	ArmDpOpBic,
	ArmDpOpMvn,
};

struct JitBackendRuntimeData {
	
	#ifdef BUILD_FOR_THUMB_1
		
		struct M0backendRegState {
			
			//the order of the next 3 is required and used by prologue/epilogue!
			uint32_t regs[6];
			uint32_t r12;
			uint32_t lr;
		} regStateStorage;
		
		uint16_t prologue[14];					//must be word-aligned
		uint16_t epiloguePopPc[2];				//must be word-aligned, will pop address off the stack
		uint16_t epilogueBxLr[2];				//must be word-aligned, expects address in context->lr, MUST be right before epilogueInterworking since it falls through to it
		uint16_t epilogueInterworking[6];		//must be word-aligned, expects address in REG_NO_DST_EPILOGUE_REG
		uint16_t epilogueNoninterworking[6];	//must be word-aligned, expects address in REG_NO_DST_EPILOGUE_REG
		uint16_t epilogueThumbOnly[6];			//must be word-aligned, expects address in REG_NO_DST_EPILOGUE_REG
		
		struct M0backendRegState *regStatePtr;
		
	#else
	
		//nothing needed for v7, but cannot be empty
		
	#endif
};

struct JitSimpleSpStatus {
	uint16_t pushRegs;	//regs to pop
	int16_t srcReg		:5;		//if we need to output, this will be non-negative
	int16_t tmpReg		:5;		//for swapSp only
	int16_t nPushedRegs	:5;		//for swapSp only
};

static inline uint32_t __attribute__((pure)) jitUtilPickLowestClearBit(uint32_t val)	//return the index of the lowest clear bit in a word
{
	return __builtin_ctz(~val);
}

static inline uint32_t jitPrvPopcount16(uint32_t val)	//better than GCC's
{
	extern const uint8_t mPopcntTab[];
	
	#ifdef BUILD_FOR_THUMB_1
	
		return mPopcntTab[(uint8_t)val] + mPopcntTab[val >> 8];
	
	#else
	
		val = val - ((val >> 1) & 0x5555);
		val = (val & 0x3333) + ((val >> 2) & 0x3333);
		val = (val + (val >> 4)) & 0x0f0f;
		return (val + (val >> 8)) & 0x00ff;
		
	#endif
}

static inline uint32_t jitPrvPopcount32(uint32_t val)	//better than GCC's
{
	extern const uint8_t mPopcntTab[];
	
	#ifdef BUILD_FOR_THUMB_1
	
		return mPopcntTab[(uint8_t)val] + mPopcntTab[(uint8_t)(val >> 8)] + mPopcntTab[(uint8_t)(val >> 16)] + mPopcntTab[(uint8_t)(val >> 24)];
	
	#else
	
		val = val - ((val >> 1) & 0x55555555);
		val = (val & 0x33333333) + ((val >> 2) & 0x33333333);
		return (((val + (val >> 4)) & 0xF0F0F0F) * 0x1010101) >> 24;
		
	#endif
}

//mode 1: imm
enum EmitStatus jitEmitAluOpImm(struct EmitBuf *dest, enum EmitCc cc, enum JitArmDpOp op, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t imm, uint32_t rotBy, bool s);

//mode 1: reg shift imm
enum EmitStatus jitEmitAluOpRegShiftImm(struct EmitBuf *dest, enum EmitCc cc, enum JitArmDpOp op, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, bool s);

//mode 1: reg shift reg
enum EmitStatus jitEmitAluOpRegShiftReg(struct EmitBuf *dest, enum EmitCc cc, enum JitArmDpOp op, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t rsNo, enum EmitShiftType shiftType, bool s);

//mode 2 & 3: reg + imm
enum EmitStatus jitEmitImmMemStr(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitAddrMode adrMode, enum EmitMemOpSz size);
enum EmitStatus jitEmitImmMemLdr(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, bool sext, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitAddrMode adrMode, enum EmitMemOpSz size);

//mode 2 & 3: reg + reg shift imm
enum EmitStatus jitEmitRegRegMemStr(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rtNo, uint32_t rnNo, bool isAdd, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitAddrMode adrMode, enum EmitMemOpSz size);
enum EmitStatus jitEmitRegRegMemLdr(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, bool sext, uint32_t rtNo, uint32_t rnNo, bool isAdd, uint32_t rmNo, enum EmitShiftType shiftType, uint32_t shiftAmt, enum EmitAddrMode adrMode, enum EmitMemOpSz size);

//mode 3, two regs, reg + imm
enum EmitStatus jitEmitImmMemStrd(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitAddrMode adrMode);
enum EmitStatus jitEmitImmMemLdrd(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rtNo, uint32_t rnNo, int32_t imm, enum EmitAddrMode adrMode);

//mode 3, two regs, reg + reg
enum EmitStatus jitEmitRegRegMemStrd(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rtNo, uint32_t rnNo, bool isAdd, uint32_t rmNo, enum EmitAddrMode adrMode);
enum EmitStatus jitEmitRegRegMemLdrd(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rtNo, uint32_t rnNo, bool isAdd, uint32_t rmNo, enum EmitAddrMode adrMode);

//mode 4
enum EmitStatus jitEmitStmia(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rnNo, uint32_t regsMask, bool wbak);
enum EmitStatus jitEmitLdmia(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rnNo, uint32_t regsMask, bool wbak);
enum EmitStatus jitEmitStmib(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rnNo, uint32_t regsMask, bool wbak);
enum EmitStatus jitEmitLdmib(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rnNo, uint32_t regsMask, bool wbak);
enum EmitStatus jitEmitStmdb(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rnNo, uint32_t regsMask, bool wbak);
enum EmitStatus jitEmitLdmdb(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rnNo, uint32_t regsMask, bool wbak);
enum EmitStatus jitEmitStmda(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rnNo, uint32_t regsMask, bool wbak);
enum EmitStatus jitEmitLdmda(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rnNo, uint32_t regsMask, bool wbak);

//SWAP
enum EmitStatus jitEmitSwap(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, enum EmitMemOpSz size);

//CLZ
enum EmitStatus jitEmitClz(struct EmitBuf *dest, enum EmitCc cc, uint32_t rdNo, uint32_t rmNo);

//BX/BLX/BL_to_arm
enum EmitStatus jitEmitBxReg(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rmNo);
enum EmitStatus jitEmitBlxReg(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rmNo);
enum EmitStatus jitEmitBlToArm(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t dstAddr);

//MUL, MLA, SMULL, UMULL, SMLAL, UMLAL
enum EmitStatus jitEmitMul(struct EmitBuf *dest, enum EmitCc cc, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool s);
enum EmitStatus jitEmitMla(struct EmitBuf *dest, enum EmitCc cc, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t raNo, bool s);
enum EmitStatus jitEmitLongMul(struct EmitBuf *dest, enum EmitCc cc, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo, bool unsign, bool accum, bool s);

//MRS_reg/MSR_reg
enum EmitStatus jitEmitMrsReg(struct EmitBuf *dest, enum EmitCc cc, uint32_t rdNo);						//CPSR access only
enum EmitStatus jitEmitMsrReg(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rmNo);	//CPSR access only, top 8 bits only ever modified
enum EmitStatus jitEmitMsrImm(struct EmitBuf *dest, enum EmitCc cc, uint32_t val);						//CPSR access only, top 8 bits only ever modified

//EDSP math
enum EmitStatus jitEmitQadd(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus jitEmitQsub(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus jitEmitQdadd(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus jitEmitQdsub(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo);
enum EmitStatus jitEmitSmulxy(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool nTop, bool mTop);
enum EmitStatus jitEmitSmlaxy(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t raNo, bool nTop, bool mTop);
enum EmitStatus jitEmitSmulwy(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, bool mTop);
enum EmitStatus jitEmitSmlawy(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdNo, uint32_t rnNo, uint32_t rmNo, uint32_t raNo, bool mTop);
enum EmitStatus jitEmitSmlalxy(struct EmitBuf *dest, enum EmitCc cc, uint32_t instrAddr, uint32_t rdLoNo, uint32_t rdHiNo, uint32_t rnNo, uint32_t rmNo, bool nTop, bool mTop);

//misc math
enum EmitStatus jitEmitExtend(struct EmitBuf *dest, uint32_t rdNo, uint32_t rmNo, uint32_t rotateBy, bool byte, bool unsign);

//control flow helpers
enum EmitStatus jitEmitTuPrologue(struct EmitBuf *dest, uint32_t sourceStartAddr);	//if this backend needs a prologue for a TU, emit it here
enum EmitStatus jitEmitJumpToAnotherTu(struct EmitBuf *dest, const uint16_t *startOfCodeInOtherTu, enum EmitCc cc);
enum EmitStatus jitEmitWhereToJumpFromAnotherTu(uint16_t *startOfCodeTu, uint16_t **jumpToP);
enum EmitStatus jitEmitJumpToAbsThumbAddrNotInTu(struct EmitBuf *dest, uintptr_t to);		//eg: to a callout
enum EmitStatus jitEmitIntraTuBranch(struct EmitBuf *dest, uintptr_t to, enum EmitCc cc);		//always to thumb - likely in the same tu
enum EmitStatus jitEmitNumHalfwordsNeededForConditionalSkipover(enum EmitCc cc, uint32_t *nHalfwordsP);
enum EmitStatus jitEmitSemihostingCall(struct EmitBuf *dest, uint32_t instrAddr, enum EmitCc cc);

//useful misc
enum EmitStatus jitEmitNopFill(struct EmitBuf *dest);								//fill the buffer with NOPs till the end
enum EmitStatus jitEmitNopFillLen(struct EmitBuf *dest, uint32_t halfwords);		//fill halfwords with NOPs
enum EmitStatus jitEmitArbitraryImmAdd(struct EmitBuf *dest, uint32_t rdNo, uint32_t rnNo, uint32_t val, bool inIt);	//no flags ever set, ANY imm is possible
enum EmitStatus jitEmitLoadImmToReg(struct EmitBuf *dest, uint32_t rdNo, uint32_t val, bool canCorruptNZ, bool canCorruptC, bool isInIt); //load imm to a vreg (contrast with jitPrvEmitLoadImmToLoreg)


//backend-internal funcs

//load an imm to a real hardware loreg
enum EmitStatus jitPrvEmitLoadImmToLoreg(struct EmitBuf *dest, uint32_t rdNo, uint32_t val, bool canCorruptNZ, bool canCorruptC); //contrast with jitEmitLoadImmToReg

//literal pool load (requires one halfword for loregs, two for hiregs)
enum EmitStatus jitPrvLiteralLoad(struct EmitBuf *dest, uint32_t regNo, uint32_t val);
enum EmitStatus jitPrvLiteralLoadsFlush(struct EmitBuf *dest, uint32_t slotsNeeded);	//make sure we can emit this many loads withotu a flush


//find temp regs for use
uint32_t jitPrvFindTempRegs(uint32_t instrAddr, uint32_t unallowedRegs, uint32_t *pushRegsP, uint32_t *popRegsP, bool needPcSpacer, ...);

//swap to a new sp value if things have been pushed to stack
enum EmitStatus jitPrvEmitSwapSp(struct EmitBuf *dest, uint32_t regWithNewVal, uint32_t regTmp, uint32_t pushedRegs, uint32_t numPushedRegs);

//simple wrappers around funcs that may use SP but not PC. Not 100% efficient, but easy
enum EmitStatus jitPrvHandleSpStart(struct EmitBuf *dest, uint32_t instrAddr, struct JitSimpleSpStatus* sta, uint32_t *regNoOut1P, uint32_t *regNoOut2P, ... /* in regs pointers, NULL terminated */);
enum EmitStatus jitPrvHandleSpEnd(struct EmitBuf *dest, const struct JitSimpleSpStatus* sta);



//util funcs that the jit layer must define
uint32_t jitPrvFindClobberables(const uint32_t *curInstrP);					//based on future instrs. all regs current instr uses are NOT returned
struct JitBackendRuntimeData* jitPrvGetRuntimeDataPtr(void);
enum EmitStatus jitEmitJumpToArm(struct EmitBuf *dest, enum EmitCc cc, uint32_t dstArmAddr, bool* tuWasFoundP);


//callout funcs that the jit layer must define for v7m
void jitPrvPopPcArmOnlyCallout(void);			// only arm mode allowed
void jitPrvPopPcCallout(void);					// arm or thumb mode allowed
void jitPrvPopPcAndAdvanceSpCallout(void);		// arm or thumb mode allowed. acts as if: "add sp, #4; ldr pc, [sp], #8"
void jitPrvBxLrArmOnlyCallout(void);			// only arm mode allowed
void jitPrvBxLrCallout(void);					// arm or thumb mode allowed
void jitPrvBxR12Callout(void);					// arm or thumb mode allowed

//callout funcs that the jit layer must define for v6m
// pc is r2 because of REG_NO_DST_EPILOGUE_REG
struct M0backendRegState;
void jitPrvPopCtxAndJumpCalloutInterwork(struct M0backendRegState *ctx, uint32_t sr, uint32_t pc);
void jitPrvPopCtxAndJumpCalloutNoninterwork(struct M0backendRegState *ctx, uint32_t sr, uint32_t pc);
void jitPrvPopCtxAndJumpCalloutThumbOnly(struct M0backendRegState *ctx, uint32_t sr, uint32_t pc);


//call to allow backends to init in a new state
void jitStateBackendInit(struct JitBackendRuntimeData *rtd, bool isUiThreadPreallocatedState);

#endif

