#ifndef _KERNEL_H_
#define _KERNEL_H_

#include <stdbool.h>
#include <stdint.h>
#include "kernelCommon.h"
#include "heap.h"
#include "emu.h"

#define ZEROTH_TASK_STACK_SZ		1024

#define KERNEL_NUM_TLS_SLOTS		3

#define KERNEL_SWI_NUM				0xF0

#define KERNEL_ARM_SEMIHOSTING_SWI	0x12	//assumes ARM pc passed in r4, only r0-r3, r12, lr promised to be in actual regs, NOT CPSR or SP. only r0..r3 can be written by the handler. writes to other regs UNPREDICATBLE

struct TaskTls {
	uintptr_t slot[KERNEL_NUM_TLS_SLOTS];
};

extern struct TaskTls TLS;			//global, accessible by the task. any task's TLS is zero-inited
									//slot TLS_SLOT_JIT_STATE, TLS_SLOT_JIT_ADDR, and TLS_SLOT_JIT_JUMP_AT are used by JIT, if jit is on
#define TLS_SLOT_JIT_STATE			0
#define TLS_SLOT_JIT_ADDR			1
#define TLS_SLOT_JIT_JUMP_AT		2	

struct KernTaskCreateParams {
	uint32_t tag;
	uint32_t pc;
	void* stackMem;		//will be freed using kheapFree at thread end
	uint32_t stackSz;
	void* exinf;
	uint16_t prio;
	bool priv;		//only a priveledged task can create another priviledged task. first task created can be either
};

#define KERN_MAILBOX_STORAGE_NEEDS(depth)	(((depth) + 1) * sizeof(uint32_t))
struct KernMailboxCreateParams {
	uint32_t tag;
	uint32_t depth;
	void* storage;	//use KERN_MAILBOX_STORAGE_NEEDS	macro to get size, allocate this yourself, and pass it in
};

struct KernEvtGroupWaitParams {
	uint32_t wantedEvents;
	uint32_t *returnedEventsP;
	int32_t timeout;
	bool wantAnd;	//else we want OR
};

struct KernTimerCreateParams {
	uint32_t tag;
	KernTimerCbk cbk;
	void* cbkData;
};

//FPU flags for KTaskFpuFlags()
#define SCHED_FPU_ENABLED				1		//task has FPU access?
#define SCHED_FPU_CTX_FLAGS_TINY_STATE	2		//set if this task has opted into using tiny fpu state (nofpscr and only s0..s4 saved)


//MUTEXES ARE RECURSIVE
//	yes i know that recursive mutexes are usually a sign of bad design, sadly PalmOS needs them to be recursive
//	to support use cases that do not need recursivity, it is an option

//all syscalls are limited to 3 params and 1 return, after that, structs are used
//ABI is almost same as function call (expect r0-r3, r12 trashed)
#define SYSC_TASK_GET_TID			0x00	//	(tid_t* tidOut)	-> KERN_STATUS_*
#define SYSC_TASK_GET_INFO			0x01	//	(tid_t, KernTaskInfo*) -> KERN_STATUS_*
#define SYSC_TASK_CREATE			0x02	//	(KernTaskCreateParams*, tid_t* tidOut) -> KERN_STATUS_*
#define SYSC_TASK_DESTROY			0x03	//	(tid_t) -> KERN_STATUS_*
#define SYSC_TASK_START				0x04	//	(tid_t, void* param) -> KERN_STATUS_*
#define SYSC_TASK_SUSPEND			0x05	//	(tid_t) -> KERN_STATUS_*
#define SYSC_TASK_RESUME			0x06	//	(tid_t) -> KERN_STATUS_*
#define SYSC_TASK_WAIT				0x07	//	(i32 msec) -> KERN_STATUS_*
#define SYSC_TASK_WAIT_CLR			0x08	//	() -> KERN_STATUS_*
#define SYSC_TASK_WAKE				0x09	//	(tid_t)	-> KERN_STATUS_*
#define SYSC_TASK_DELAY				0x0A	//	(i32 msec) -> ()
#define SYSC_TASK_SWITCHING			0x0B	//	(bool on) -> ()
#define SYSC_TASK_YIELD				0x0C	//	() -> ()
#define SYSC_TASK_KILL_ME			0x0D	//	() -> ()		//actually terminates a task, SYSC_TASK_DESTROY just re-points PC to code that does this
#define SYSC_SET_TASK_FPU_FLAGS		0x0E	//	(u32 flagBitsToSet, u32 flagBitsToClear) -> u32 newFlags

#define SYSC_MUTEX_CREATE			0x10	//	(u32 tag, bool recursive, mutex_t *mutexHandleOut) -> KERN_STATUS_*
#define SYSC_MUTEX_DESTROY			0x11	//	(mutex_t) -> KERN_STATUS_*
#define SYSC_MUTEX_RESERVE			0x12	//	(mutex_t, i32 timeout) -> KERN_STATUS_*
#define SYSC_MUTEX_RELEASE			0x13	//	(mutex_t) -> KERN_STATUS_*

#define SYSC_SEMA_CREATE			0x18	//	(u32 tag, sema_t *semaHandleOut) -> KERN_STATUS_*
#define SYSC_SEMA_DESTROY			0x19	//	(sema_t) -> KERN_STATUS_*
#define SYSC_SEMA_WAIT				0x1A	//	(sema_t, i32 timeout) -> KERN_STATUS_*
#define SYSC_SEMA_POST				0x1B	//	(sema_t) -> KERN_STATUS_*

#define SYSC_MAILBOX_CREATE			0x20	//	(KernMailboxCreateParams*, mbx_t *mbxHandleOut) -> KERN_STATUS_*
#define SYSC_MAILBOX_DESTROY		0x21	//	(mbx_t) -> KERN_STATUS_*
#define SYSC_MAILBOX_SEND			0x22	//	(mbx_t, u32 msg) -> KERN_STATUS_*
#define SYSC_MAILBOX_WAIT			0x23	//	(mbx_t, u32* msgP, i32 timeout) -> KERN_STATUS_*

#define SYSC_TIMER_CREATE			0x28	//	(KernTimerCreateParams*, tmr_t *tmrHandleOutP) -> KERN_STATUS_*
#define SYSC_TIMER_DESTROY			0x29	//	(tmr_t) -> KERN_STATUS_*
#define SYSC_TIMER_SET				0x2A	//	(tmr_t, u32 msec) -> KERN_STATUS_*

#define SYSC_EVTGRP_CREATE			0x30	//	(u32 tag, u32 state, evtgrp_t *evtGrpHandleOut) -> KERN_STATUS_*
#define SYSC_EVTGRP_DESTROY			0x31	//	(evtgrp_t) -> KERN_STATUS_*
#define SYSC_EVTGRP_CLEAR			0x32	//	(evtgrp_t, u32 clearBits) -> KERN_STATUS_*
#define SYSC_EVTGRP_SIGNAL			0x33	//	(evtgrp_t, u32 setBits) -> KERN_STATUS_*
#define SYSC_EVTGRP_READ			0x34	//	(evtgrp_t, u32 *stateP) -> KERN_STATUS_*
#define SYSC_EVTGRP_WAIT			0x35	//	(evtgrp_t, KernEvtGroupWaitParams*) -> KERN_STATUS_*

#define SYSC_RTC_GET				0x38	//	(uint32_t* time) -> KERN_STATUS_*
#define SYSC_RTC_SET				0x39	//	(uint32_t time) -> KERN_STATUS_*
#define SYSC_RTC_SET_ALARM			0x3a	//	(uint32_t atTime) -> KERN_STATUS_*

#define SYSC_GET_UPTIME_MSEC		0x40	//	(uint32_t* msecs) -> KERN_STATUS_*
#define SYSC_SET_STOR_RAM_WRITABLE	0x41	//	(bool writeable) -> KERN_STATUS_*

#ifdef __cplusplus
extern "C" {
#endif

static uint32_t __attribute__((noinline, naked)) ksyscall()
{
	uint32_t dummy;
	
	asm volatile (
		"swi %1	\n\t"
		"bx lr	\n\t"
		:"=r"(dummy)
		:"I"(KERNEL_SWI_NUM)
		:"cc","memory","r0","r1","r2","r3","r12","lr"
	);
	
	return dummy;	//let's hope this doesnt confuse gcc about the return value
}

static uint32_t syscall_3p(uint32_t num, uint32_t p1, uint32_t p2, uint32_t p3) __attribute__ ((alias ("ksyscall")));
static uint32_t syscall_2p(uint32_t num, uint32_t p1, uint32_t p2) __attribute__ ((alias ("ksyscall")));
static uint32_t syscall_1p(uint32_t num, uint32_t p1) __attribute__ ((alias ("ksyscall")));
static uint32_t syscall_0p(uint32_t num) __attribute__ ((alias ("ksyscall")));

#ifdef __cplusplus
}
#endif

#define DECL_SYSCALL_0P(name, retType, num)	static inline retType name() { return (retType)syscall_0p(num); }
#define DECL_SYSCALL_1P(name, retType, param1Type, param1Name, num)	static inline retType name(param1Type param1Name) { return (retType)syscall_1p(num, (uint32_t)param1Name); }
#define DECL_SYSCALL_2P(name, retType, param1Type, param1Name, param2Type, param2Name, num)	static inline retType name(param1Type param1Name, param2Type param2Name) { return (retType)syscall_2p(num, (uint32_t)param1Name, (uint32_t)param2Name); }
#define DECL_SYSCALL_3P(name, retType, param1Type, param1Name, param2Type, param2Name, param3Type, param3Name, num)	static inline retType name(param1Type param1Name, param2Type param2Name, param3Type param3Name) { return (retType)syscall_3p(num, (uint32_t)param1Name, (uint32_t)param2Name, (uint32_t)param3Name); }

DECL_SYSCALL_1P(KTaskGetTid, kstatus_t, tid_t*, tidOutP, SYSC_TASK_GET_TID)
DECL_SYSCALL_2P(KTaskGetInfo, kstatus_t, tid_t, tid, struct KernTaskInfo*, ti, SYSC_TASK_GET_INFO)
DECL_SYSCALL_1P(KTaskDestroy, kstatus_t, tid_t, tid, SYSC_TASK_DESTROY)
DECL_SYSCALL_2P(KTaskStart, kstatus_t, tid_t, tid, void*, param, SYSC_TASK_START)
DECL_SYSCALL_1P(KTaskSuspend, kstatus_t, tid_t, tid, SYSC_TASK_SUSPEND)
DECL_SYSCALL_1P(KTaskResume, kstatus_t, tid_t, tid, SYSC_TASK_RESUME)
DECL_SYSCALL_1P(KTaskWait, kstatus_t, int32_t, msec, SYSC_TASK_WAIT)
DECL_SYSCALL_0P(KTaskWaitClr, kstatus_t, SYSC_TASK_WAIT_CLR)
DECL_SYSCALL_1P(KTaskWake, kstatus_t, tid_t, tid, SYSC_TASK_WAKE)
DECL_SYSCALL_1P(KTaskDelay, kstatus_t, int32_t, msec, SYSC_TASK_DELAY)
DECL_SYSCALL_1P(KTaskSwitching, kstatus_t, bool, on, SYSC_TASK_SWITCHING)
DECL_SYSCALL_0P(KTaskYield, void, SYSC_TASK_YIELD)
DECL_SYSCALL_2P(KTaskFpuFlags, uint32_t, uint32_t, flagBitsToSet, uint32_t, flagBitsToClear, SYSC_SET_TASK_FPU_FLAGS)

DECL_SYSCALL_3P(KMutexCreate, kstatus_t, uint32_t, tag, bool, recursive, mutex_t*, mutexHandleOut, SYSC_MUTEX_CREATE)
DECL_SYSCALL_1P(KMutexDestroy, kstatus_t, mutex_t, mut, SYSC_MUTEX_DESTROY)
DECL_SYSCALL_2P(KMutexReserve, kstatus_t, mutex_t, mut, int32_t, timeout, SYSC_MUTEX_RESERVE)
DECL_SYSCALL_1P(KMutexRelease, kstatus_t, mutex_t, mut, SYSC_MUTEX_RELEASE)

DECL_SYSCALL_3P(KSemaphoreCreate, kstatus_t, uint32_t, tag, uint32_t, initialVal, sema_t*, semaHandleOut, SYSC_SEMA_CREATE)
DECL_SYSCALL_1P(KSemaphoreDestroy, kstatus_t, sema_t, sem, SYSC_SEMA_DESTROY)
DECL_SYSCALL_2P(KSemaphoreWait, kstatus_t, sema_t, sem, int32_t, timeout, SYSC_SEMA_WAIT)
DECL_SYSCALL_1P(KSemaphorePost, kstatus_t, sema_t, sem, SYSC_SEMA_POST)

DECL_SYSCALL_2P(KMailboxDestroy, kstatus_t, mbx_t, mbx, void**, storageP, SYSC_MAILBOX_DESTROY)
DECL_SYSCALL_2P(KMailboxSend, kstatus_t, mbx_t, mbx, uint32_t, msg, SYSC_MAILBOX_SEND)
DECL_SYSCALL_3P(KMailboxWait, kstatus_t, mbx_t, mbx, uint32_t*, msgP, int32_t, timeout, SYSC_MAILBOX_WAIT)

DECL_SYSCALL_1P(KTimerDestroy, kstatus_t, tmr_t, tmr, SYSC_TIMER_DESTROY)
DECL_SYSCALL_2P(KTimerSet, kstatus_t, tmr_t, tmr, uint32_t, msec, SYSC_TIMER_SET)

DECL_SYSCALL_3P(KEventGroupCreate, kstatus_t, uint32_t, tag, uint32_t, initialState, evtgrp_t*, evtGrpHandleOut, SYSC_EVTGRP_CREATE)
DECL_SYSCALL_1P(KEventGroupDestroy, kstatus_t, evtgrp_t, evtGrp, SYSC_EVTGRP_DESTROY)
DECL_SYSCALL_2P(KEventGroupClear, kstatus_t, evtgrp_t, evtGrp, uint32_t, bitsToClear, SYSC_EVTGRP_CLEAR)
DECL_SYSCALL_2P(KEventGroupSignal, kstatus_t, evtgrp_t, evtGrp, uint32_t, bitsToSet, SYSC_EVTGRP_SIGNAL)
DECL_SYSCALL_2P(KEventGroupRead, kstatus_t, evtgrp_t, evtGrp, uint32_t*, curstateOutP, SYSC_EVTGRP_READ)

DECL_SYSCALL_1P(KGetUptimeMsec, kstatus_t, uint32_t*, secsP, SYSC_GET_UPTIME_MSEC)

DECL_SYSCALL_1P(KRtcGet, kstatus_t, uint32_t*, rtcValP, SYSC_RTC_GET)
DECL_SYSCALL_1P(KRtcSet, kstatus_t, uint32_t, rtcVal, SYSC_RTC_SET)
DECL_SYSCALL_1P(KRtcSetAlarm, kstatus_t, uint32_t, rtcVal, SYSC_RTC_SET_ALARM)

DECL_SYSCALL_1P(KSetStorageRamWriteable, kstatus_t, bool, allowStorageRamWrites, SYSC_SET_STOR_RAM_WRITABLE)

static inline kstatus_t KTaskCreate(uint32_t tag, void* pc, void* stackMem, uint32_t stackSz, void* exinf, uint16_t prio, bool priv, tid_t* tidOutP)
{
	struct KernTaskCreateParams params = {
		.tag = tag,
		.pc = (uint32_t)pc,
		.stackMem = stackMem,
		.stackSz = stackSz,
		.exinf = exinf,
		.prio = prio,
		.priv = priv,
	};
	
	return syscall_2p(SYSC_TASK_CREATE, (uint32_t)&params, (uint32_t)tidOutP);
}

static inline kstatus_t KMailboxCreate(uint32_t tag, uint32_t depth, void* storage, mbx_t* mbxHandleOutP)
{
	struct KernMailboxCreateParams params = {
		.tag = tag,
		.depth = depth,
		.storage = storage,
	};
	
	return syscall_2p(SYSC_MAILBOX_CREATE, (uint32_t)&params, (uint32_t)mbxHandleOutP);
}

static inline kstatus_t KTimerCreate(uint32_t tag, KernTimerCbk cbk, void* cbkData, tmr_t *tmrHandleOutP)
{
	struct KernTimerCreateParams params = {
		.tag = tag,
		.cbk = cbk,
		.cbkData = cbkData,
	};
	
	return syscall_2p(SYSC_TIMER_CREATE, (uint32_t)&params, (uint32_t)tmrHandleOutP);
}

static inline kstatus_t KEventGroupWait(evtgrp_t evtGrp, uint32_t wantedEvents, uint32_t *returnedEventsP, int32_t timeout, bool wantAnd)
{
	struct KernEvtGroupWaitParams params = {
		.wantedEvents = wantedEvents,
		.returnedEventsP = returnedEventsP,
		.timeout = timeout,
		.wantAnd = wantAnd,
	};
	
	return syscall_2p(SYSC_EVTGRP_WAIT, evtGrp, (uint32_t)&params);
}

#define EXC_m0_CAUSE_MEM_ACCESS_FAIL		1	//memoryaccess failed in some way (could be data misalignment or ba daddress)
#define EXC_m0_CAUSE_NMI					2
#define EXC_m0_CAUSE_UNALIGNED				3
#define EXC_m0_CAUSE_UNDEFINSTR				4	//an undefined instruction was hit
#define EXC_m0_CAUSE_BKPT_HIT				5
#define EXC_m0_CAUSE_UNCLASSIFIABLE			6	//classification failed

//fault handling code, cause is EXC_m0_CAUSE_, extraData is usually an address. both unused on C-M3
extern void faultHandlerWithExcFrame(struct CortexExcFrame *exc, uint32_t cause, uint32_t extraData);


//for internal use
void kernelLogCurTaskForExc(void);
void schedHandleUsageFaultNoCp(struct CortexExcFrame *ctx, volatile uint16_t *ufsrAddr, uint32_t ufsrVal);
bool kernelSemihostingHandle(uint32_t *r0P, uint32_t *r1P, uint32_t *r2P, uint32_t *r3P, uint32_t r12, uint32_t sp, uint32_t lr, uint32_t pc, bool fromArm);



#endif

