#ifndef _SCHED_H_
#define _SCHED_H_

#include <stdbool.h>
#include <stdint.h>
#include "timers.h"
#include "kernel.h"
#include "list.h"
#include "emu.h"


#define KERNEL_SUPPORTS_VFP


#ifndef HAVE_FPU
#undef KERNEL_SUPPORTS_VFP
#endif

#define SCHED_QUANTUM_TICKS		1048576		//let each thread do ~1M instrs per quantum or so
//#define SCHED_QUANTUM_TICKS		131072		//for emulation do less

#define SCHED_DEFAULT_PRIO		0x031e			//prio of first task - it becomes hwr event listening task so it need to be quite high
#define SCHED_MINUMUM_PRIO		0x0001			//0x0000 reserved for idle task
#define SCHED_MAXIMUM_PRIO		0xfffc			//0xffff reserved for timers task, 0xfffe to stack reaper task, 0xfffd for tasks celaning up their own shit

#define SCHED_PRIO_IDLE			0x0000
#define SCHED_PRIO_TIMERS		0xffff
#define SCHED_PRIO_STACK_REAPER	0xfffe
#define SCHED_PRIO_SELF_CLEANUP	0xfffd



//config
#define SCHED_MAX_THREADS_LOG	5				//max threads is 2 ^ this - 1
#define SCHED_MAX_MUTEXES		32
#define SCHED_MAX_SEMAPHORES	32
#define SCHED_MAX_TIMERS		64
#define SCHED_MAX_EVENT_GROUPS	32
#define SCHED_MAX_MAILBOXES		32


//XXX: TODO: comment this when we can (will break spinlock in printf and we need to fix thar
#define SCHED_TASKS_ARE_ALL_PRIVILEDGED



struct SchedFpuContext {
	uint32_t regs[32];
	uint32_t fpscr;
	uint8_t flags;
};

#ifndef BUILDING_FOR_BIG_ARM	//Cortex
	struct SchedRegs {
		uint32_t r0_r3[4];
		uint32_t r4_r11[8];
		uint32_t r12, sp, lr, pc;
		uint32_t sr;
	};
#else							//Legacy ARM
	struct SchedRegs {
		uint32_t r0_r3[4];
		uint32_t r4_r11[8];
		uint32_t r12, sr, sp, lr, pc;
	};
#endif

//higher prio values is higher priority tasks


//basic global stuff
kstatus_t schedInit(void);
kstatus_t schedStart(tid_t tid, void* param);		//create ONE task, do not "start" it, then call this with its tid, does not return in the good case

#ifdef EXPLICIT_EMU_CTX
	void schedSetCurThreadEmuCtx(struct EmuCpuState*);					//faster than syscall and equally safe
	struct EmuCpuState* schedGetCurEmuContextFromFaultContext(void);	//only safe in fault context!
#endif




//iface between kernelCommon.c and its users. do not use nuelsss your filename is "kernel*.c"
bool copyFromUser(void* dst, const void* userSrc, uint32_t len);
bool copyToUser(void* userDst, const void* src, uint32_t len);
bool readUserWord(uint32_t* dst, const uint32_t *userPtr);
bool writeUserWord(uint32_t* userDst, uint32_t src);

struct TaskContext {
	struct SchedRegs regs;
	struct TaskTls tls;
	bool priv;
};

struct Task {
	struct TaskContext ctx;
	
#ifdef KERNEL_SUPPORTS_VFP
	struct SchedFpuContext fpuCtx;
#endif

	uint32_t tag;
	tid_t tid;
	void *stackChunk;		//also start of stack memory chunk
	void *stackLimitAddr;
	void *exinf;
	ktimer_t timer;			//for various timed ops
	uint16_t runGeneration;	//for scheduling tasks of equal prio this is secondary order
	uint16_t prio;
	uint16_t blockReason;
	uint16_t wakeCount		:10;
	uint16_t unblockable	: 1;	//set when it is in its death throes

#ifdef EXPLICIT_EMU_CTX
	void* emuCtx;
#endif

	struct ListNode tasks;			//list of all tasks
	struct ListNode inBlockSource;	//when blocked, list in block source
};

void schedAssertCalledInSyscallMode(void);
bool schedAmInSyscall(void);
void schedRequestContextSwitch(void);
void __attribute__((noreturn)) schedSwitchToScheduledRegimeByCallingYield(void);

//due to this, building with LTO is much advised :)
uint8_t schedGetLockCount(void);
struct Task* schedGetNextTask(void);
struct Task* volatile* __attribute__((const)) schedGetCurTaskP(void);



#endif
