#ifndef _LIST_H_
#define _LIST_H_

#include <stddef.h>
#include <stdbool.h>

struct ListNode {
	struct ListNode *next;
	struct ListNode *prev;
};

//lists must be irq-safe, that is if we're adding or removing and an irq reads list
// in the middle anywhere, it sees a complete list. it is ok if the item being added
// isn't seen, or item being removed is, but it is not of for half a list to be
// missing or for an invalid pointer to be present

static inline void listInit(struct ListNode* n)
{
	n->next = n;
	n->prev = n;
}

static inline void listRemoveItem(struct ListNode* n)
{
	n->prev->next = n->next;
	n->next->prev = n->prev;
	asm volatile("":::"memory");	//reorder barrier for gcc
	listInit(n);	//not strictly necessary but helps avoid dangling pointers
}
static inline void listInsertAfter(struct ListNode* insertThis, struct ListNode* afterThis)
{
	insertThis->prev = afterThis;
	insertThis->next = afterThis->next;
	asm volatile("":::"memory");	//reorder barrier for gcc
	afterThis->next = insertThis;
	insertThis->next->prev = insertThis;
}

static inline void listInsertBefore(struct ListNode* insertThis, struct ListNode* beforeThis)
{
	insertThis->next = beforeThis;
	insertThis->prev = beforeThis->prev;
	asm volatile("":::"memory");	//reorder barrier for gcc
	beforeThis->prev = insertThis;
	insertThis->prev->next = insertThis;
}

static inline bool listIsEmpty(struct ListNode* n)
{
	return n->next == n;
}

#define LIST_DECLARE(_name)	struct ListNode _name = {.next = &_name, .prev = &_name, };

#define STRUCT_FROM_LIST_NODE(structType, nodeNameInStruct, node)			((structType*)(((char*)(node)) - offsetof(structType, nodeNameInStruct)))


#endif
