#include "hwTimer.h"
#include "printf.h"
#include "timers.h"
#include <stdio.h>
#include "irqs.h"
#include "cpu.h"



#define TIMERS_HEAP_DEPTH									3								//there are ((2 ^ this) - 1) timers available. we do not support over 32 anyways
#define TIMERS_MAX_AMT										((1 << TIMERS_HEAP_DEPTH) - 1)

struct TimerInfo {
	
	uint64_t expiry;
	
	TimersTimerCbk cbk;
	void* cbkData;
	
	ktimer_t handle;
};

static struct TimerInfo mTimers[TIMERS_MAX_AMT];
static uint32_t mLastTimerHandle = 0;
static uint32_t mNumTimers = 0;


void tmrShowTimers(void)
{
	uint64_t t = timerGetTime();
	uint32_t i;
	
	loge("CURRENT TIME: 0x%08x%08x\n", (uint32_t)(t >> 32), (uint32_t)t);
	for (i = 0; i < mNumTimers; i++) {
		loge("TIMER %u:\n", i);
		loge("  expires 0x%08x%08x\n", (uint32_t)(mTimers[i].expiry >> 32), (uint32_t)mTimers[i].expiry);
		loge("  cbk 0x%08x data 0x%08x\n", mTimers[i].cbk, mTimers[i].cbkData);
	}
}

static struct TimerInfo* timerFindByHandle(ktimer_t handle)	//called in non-reentrant context
{
	uint32_t i;
	
	for (i = 0; i < mNumTimers; i++) {
		if (mTimers[i].handle == handle)
			return &mTimers[i];
	}
	
	return NULL;
}

static void timerHeapDelete(uint32_t idx)
{
	mTimers[idx] = mTimers[--mNumTimers];
	
	while (idx < mNumTimers) {
	
		uint32_t childL = idx * 2 + 1;
		uint32_t childR = idx * 2 + 2;
		uint32_t smallerChildIdx;
		struct TimerInfo swp;
		
		if (childR < mNumTimers)		//have both children - repalce with smaller and retry
			smallerChildIdx = mTimers[childL].expiry < mTimers[childR].expiry ? childL : childR;
		else if (childR == mNumTimers)	//have one child - replace with it
			smallerChildIdx = childL;
		else
			break;
		
		if (mTimers[idx].expiry < mTimers[smallerChildIdx].expiry)
			break;
		
		swp = mTimers[idx];
		mTimers[idx] = mTimers[smallerChildIdx];
		mTimers[smallerChildIdx] = swp;
		idx = smallerChildIdx;
	}
}

static void timerRunAndSetNext(void)	//called in non-reentrant context
{
	uint32_t delay;
	
	while (mNumTimers && timerGetTime() >= mTimers[0].expiry) {
		
		struct TimerInfo expired = mTimers[0];
		
		timerHeapDelete(0);
		expired.cbk(expired.handle, expired.cbkData);
	}
	
	if (!mNumTimers)
		delay = 0;
	else {
		delay = (mTimers[0].expiry - timerGetTime() + SYSTICK_CLOCKS_PER_TIMER_TICK / 2) / SYSTICK_CLOCKS_PER_TIMER_TICK;
		if (!delay)				//handle unlikely underflow
			delay++;
	}
	
	hwTimerSet(delay);
}

void timerExternalIrq(void)
{
	timerRunAndSetNext();
}

ktimer_t timerCreate(uint32_t name, TimersTimerCbk cbk, void* cbkData, uint64_t ticks)
{
	uint64_t expiry = ticks + timerGetTime();
	struct TimerInfo *tmr;
	uint32_t i, irqState;
	ktimer_t ret = 0;
	
	irqState = hwTimerIntOff();		//allows re-entrancy
	
	if (mNumTimers < TIMERS_MAX_AMT) {
		
		//grab a handle
		do {			//get nonzero unused timer ID
			if (!++mLastTimerHandle)
				mLastTimerHandle++;
		} while (timerFindByHandle(mLastTimerHandle));
		ret = mLastTimerHandle;
		
		//get our slot
		tmr = &mTimers[i = mNumTimers++];
		
		//store info
		tmr->expiry = expiry;
		tmr->cbk = cbk;
		tmr->cbkData = cbkData;
		tmr->handle = ret;
		
		//re-heapify
		while (i) {
			struct TimerInfo swp;
			uint32_t parent = (i - 1) / 2;
			
			if (mTimers[parent].expiry < mTimers[i].expiry)		//cannot bubble up anymore
				break;
			
			//bubble up
			swp = mTimers[parent];
			mTimers[parent] = mTimers[i];
			mTimers[i] = swp;
			
			i = parent;
		}
		
		//re-set timer
		timerRunAndSetNext();	//only run in our interrupt context or with int off
	}
	
	hwTimerIntRestore(irqState);
	
	return ret;
}

void timerDelete(ktimer_t handle)
{
	struct TimerInfo *t;
	uint32_t irqState;
	
	irqState = hwTimerIntOff();		//allows re-entrancy
	t = timerFindByHandle(handle);
	if (t) {
		
		//delete from heap
		timerHeapDelete(t - mTimers);
		timerRunAndSetNext();	//only run in our interrupt context or with int off
	}
	
	hwTimerIntRestore(irqState);
}

void timersCommonInit(void)
{
	hwTimerInit();
	mNumTimers = 0;
	timerRunAndSetNext();
	hwTimerIntOn();
}

