#include <stdbool.h>
#include <string.h>
#include <stdint.h>
#include <stdio.h>
#include "printf.h"
#include "irqs.h"




typedef struct {
	
	void* dest;
	unsigned long maxChars;
	
	void* furtherCallback;
	void* furtherUserData;
	
} PrintData;

typedef char (*StrPrintfExCbk)(void* userData, char chr);	//return 0 to stop printing


static void prPutcharEx(char** dstP, char ch)
{
	if (*dstP)
		*(*dstP)++ = ch;
	else
		prPutchar(ch);
}

static void doubleDabble(uint8_t *dst, uint64_t val)	//dst must be 20 bytes
{
	uint_fast8_t i, j;
	
	for (i = 0; i < 20; i++)
		dst[i] = 0;
	
	for (i = 0; i < 64; i++) {
		
		uint_fast8_t shiftIn;
		
		for (j = 0; j < 20; j++) {
			
			if (dst[j] >= 0x50)
				dst[j] += 0x30;
			
			if ((dst[j] & 0x0f) >= 5)
				dst[j] += 3;
		}
		shiftIn = val >> 63;
		val <<= 1;
		
		for (j = 0; j < 20; j++) {
			
			shiftIn += 2 * dst[j];
			dst[j] = shiftIn;
			shiftIn >>= 8;
		}
	}
}

static void copyOver(uint8_t *dst, uint64_t val)	//dst must be 20 bytes
{
	uint_fast8_t i;
	
	for (i = 0; i < 20; i++, val >>= 8)
		dst[i] = val & 0xff;
}

static uint8_t getBcdDigit(const uint8_t *buf, uint_fast8_t digit)
{
	uint_fast8_t t = buf[digit / 2];
	
	if (digit & 1)
		return t >> 4;
	else
		return t & 0x0f;
}

static void StrPrvPrintfEx_number(char** dstP, uint64_t number, bool baseTen, bool zeroExtend, bool isSigned, unsigned padToLength, bool caps)
{
	uint_fast8_t numLen, printedLen, padChars, i;
	bool wasNegative = false;
	uint8_t buf[20];
	
	if (isSigned && ((int64_t)number) < 0) {
		
		wasNegative = true;
		number = -number;
	}
	
	if (baseTen)
		doubleDabble(buf, number);
	else
		copyOver(buf, number);
	
	//find the length of the number (1 minimum)
	for (numLen = sizeof(buf) * 2; numLen >= 2 && !getBcdDigit(buf, numLen - 1); numLen--);
	
	printedLen = numLen + (wasNegative ? 1 : 0);
	
	if (printedLen > padToLength)
		padChars = 0;
	else
		padChars = padToLength - printedLen;
	
	//padding works differently for zeroes and for spaced w.r.t. the minus sign
	
	if (zeroExtend && wasNegative)
		prPutcharEx(dstP, '-');
	while (padChars--)
		prPutcharEx(dstP, zeroExtend ? '0' : ' ');
	if (!zeroExtend && wasNegative)
		prPutcharEx(dstP, '-');
	for (i = 0; i < numLen; i++) {
		char ch = getBcdDigit(buf, numLen - 1 - i);
		
		ch = (ch < 10) ? (ch + '0') : (ch + (caps ? 'A' : 'a') - 10);
		prPutcharEx(dstP, ch);
	}
}

static unsigned StrVPrintf_StrLen_withMax(const char* s,unsigned max){
	
	unsigned len = 0;
	
	while((*s++) && (len < max))
		len++;
	
	return len;
}

static unsigned StrVPrintf_StrLen(const char* s){
	
	unsigned len = 0;
	
	while(*s++)
		len++;
	
	return len;
}

static uint32_t vprEx(char* dst, const char* fmtStr,va_list vl)	//length only returned when printing to a string
{
	unsigned long long val64;
	irq_state_t irqSta = 0;
	char* dstOrig = dst;
	uint32_t val32;
	char c;
	
	(void)irqSta;
	if (PRINTF_MAY_DISABLE_IRQS) {
		
		if (!dst)
			irqSta = irqsAllOff();
	}
	
	while((c = *fmtStr++) != 0){
		
		if(c == '%'){
			
			bool zeroExtend = false, useLong = false, useLongLong = false, caps = false, useHalfword = false;
			unsigned padToLength = 0, len, i;
			const char* str;
			
more_fmt:
			
			c = *fmtStr++;
			
			switch(c){
				
				case '%':
					
					prPutcharEx(&dst, c);
					break;
				
				case 'C':	//4CC
					val32 = va_arg(vl, unsigned int);
					prPutcharEx(&dst, (uint8_t)(val32 >> 24));
					prPutcharEx(&dst, (uint8_t)(val32 >> 16));
					prPutcharEx(&dst, (uint8_t)(val32 >> 8));
					prPutcharEx(&dst, (uint8_t)(val32 >> 0));
					break;
				
				case 'c':
					
					prPutcharEx(&dst, va_arg(vl, unsigned int));
					break;
				
				case 's':
					
					str = va_arg(vl, char*);
					if(!str)
						str = "(null)";
					
					if(padToLength)
						len = StrVPrintf_StrLen_withMax(str, padToLength);
					else
						padToLength = len = StrVPrintf_StrLen(str);

					if(len > padToLength)
						len = padToLength;
					else {
						
						for(i = len; i < padToLength; i++)
							prPutcharEx(&dst, L' ');
					}
					for(i = 0; i < len; i++)
						prPutcharEx(&dst, *str++);
					
					break;
				
				case '0':
				case '.':
					
					if(!zeroExtend && !padToLength) {
						
						zeroExtend = true;
						goto more_fmt;
					}
					//fallthrough
				
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
					
					padToLength = (padToLength * 10) + c - '0';
					goto more_fmt;
				
				case 'u':
					
					val64 = useLongLong ? va_arg(vl, unsigned long long) : (useLong ? va_arg(vl,unsigned long) : va_arg(vl,unsigned));
					if (useHalfword)
						val64 = (uint16_t)val64;
					StrPrvPrintfEx_number(&dst, val64, true, zeroExtend, false, padToLength, false);
					break;
					
				case 'd':
				case 'i':
					val64 = useLongLong ? va_arg(vl, signed long long) : (signed long long)(useLong ? va_arg(vl,signed long) : va_arg(vl,signed));
					if (useHalfword)
						val64 = (int64_t)(int16_t)val64;
					StrPrvPrintfEx_number(&dst, val64, true, zeroExtend, true, padToLength, false);
					break;
					
				case 'X':
					caps = true;
					//fallthrough
					
				case 'x':
					val64 = useLongLong ? va_arg(vl, unsigned long long) : (useLong ? va_arg(vl,unsigned long) : va_arg(vl,unsigned));
					if (useHalfword)
						val64 = (uint16_t)val64;
					StrPrvPrintfEx_number(&dst, val64, false, zeroExtend, false, padToLength, caps);
					break;
				
				case 'h':	//palm uses this to indicate we want a halfword used
					useHalfword = true;
					goto more_fmt;
				
				case 'L':
				case 'l':
					if (useLong)
						useLongLong = true;
					else
						useLong = true;
					goto more_fmt;
				
				default:
					
					prPutcharEx(&dst, c);
					break;
			}
		}
		else
			prPutcharEx(&dst, c);
	}

	if (dst)
		*dst = 0;	//write it but do not include in "num bytes written" return val
	else if (PRINTF_MAY_DISABLE_IRQS) {
		irqsRestoreState(irqSta);
	}
	
	return dst - dstOrig;
}

void vpr(const char* fmtStr, va_list vl)
{
	vprEx(NULL, fmtStr, vl);
}

void pr(const char* fmtStr, ...)
{
	va_list vl;
	
	va_start(vl, fmtStr);
	vprEx(NULL, fmtStr, vl);
	va_end(vl);
}

uint32_t vspr(char* dst, const char* fmtStr, va_list vl)
{
	return vprEx(dst, fmtStr, vl);
}

uint32_t spr(char* dst, const char* fmtStr, ...)
{
	uint32_t ret;
	va_list vl;
	
	va_start(vl, fmtStr);
	ret = vprEx(dst, fmtStr, vl);
	va_end(vl);
	
	return ret;
}
