#include <PalmOS.h>
#include <HostControl.h>
#include "rePalmSbarPublic.h"
#include <stdarg.h>


#define MY_CRID				'n_gs'
#define ftrNumOurButton		0
#define MY_WIDTH			40


static void log(const char* fmt, ...)		//do not emit percent signs please!
{
	va_list vl;
	
	va_start(vl, fmt);
	HostTraceOutputVT(0, fmt, vl);
	va_end(vl);
}

#define LOG(...)	log("NoGe: " __VA_ARGS__)

static UInt16 sbarLibLoad(void)
{
	UInt16 refNo;
	Err e;
	
	e = SysLibFind((const char*)RepalmSbarLibName, &refNo);
	if (e != errNone)
		e = SysLibLoad(RepalmSbarLibType, RepalmSbarLibCreator, &refNo);
	
	if (e == errNone)
		e = SbarNotifLibOpen(refNo);
	
	if (e != errNone)
		return 0xffff;
	
	return refNo;
}

static void sbarLibUnload(uint16_t refNo)
{
	uint16_t numAppsStillUsing;
	Err e;
	
	e = SbarNotifLibClose(refNo, &numAppsStillUsing);
	if (e == errNone)
		SysLibRemove(refNo);
}

static Err notifGeneralSetAlarm(void)
{
	UInt16 cardNo;
	LocalID LID;
	Err e;
	
	e = SysCurAppDatabase(&cardNo, &LID);
	if (e == errNone)
		e = AlmSetAlarm(cardNo, LID, 0, TimGetSeconds() + 30, false);
	
	return e;
}

static void notifGeneralRefresh(void)
{
	UInt16 ref = sbarLibLoad();
	void* ourButton;
	
	if (FtrGet(MY_CRID, ftrNumOurButton, (UInt32*)&ourButton) != errNone || !ourButton)
		return;
	
	SbarNotifRequestRedraw(ref, ourButton);
	sbarLibUnload(ref);
	(void)notifGeneralSetAlarm();
}

static void notifGeneralInit(void)
{
	const struct SbarNotifStateDescr dummyStates = {0, };
	UInt16 cardNo, ref = sbarLibLoad();
	UInt32 dummy;
	LocalID LID;
	Err e;
	
	e = SysCurAppDatabase(&cardNo, &LID);
	if (e != errNone)
		return;

	if (errNone != FtrGet(MY_CRID, ftrNumOurButton, &dummy))
		(void)SbarNotifCreate(ref, LID, 1, &dummyStates, REPALM_BTN_GROUP_CENTER, 0, MY_WIDTH, false, true, NULL);	//callback will set the feature

	sbarLibUnload(ref);
}

static void notifGeneralButtonCreated(void *data)
{
	struct SbarNotifCreated *bc = (struct SbarNotifCreated*)data;

	FtrSet(MY_CRID, ftrNumOurButton, (UInt32)bc->buttonHandle);
	
	(void)notifGeneralSetAlarm();
}

static void notifGeneralButtonDestroyed(void *handle)
{
	void *existingButton;
	if (FtrGet(MY_CRID, ftrNumOurButton, (UInt32*)&existingButton) != errNone || existingButton != handle)
		return;
	
	FtrUnregister(MY_CRID, ftrNumOurButton);
}

static void notifGeneralGetColor(struct RGBColorType *dst, uint16_t idxForColor, uint16_t idxForBW)
{
	uint16_t ref, idx = BmpGetBitDepth(WinGetBitmap(WinGetDisplayWindow())) >= 8 ? idxForColor : idxForBW;
	uint32_t color;
	
	ref = sbarLibLoad();
	color = SbarNotifGetColor(ref, idx);
	sbarLibUnload(ref);
	
	dst->r = color >> 16;
	dst->g = color >> 8;
	dst->b = color >> 0;
}

static void notifGeneralButtonNeedsDraw(const struct SbarNotifDrawReq *req)
{
	void *existingButton;
	
	if (FtrGet(MY_CRID, ftrNumOurButton, (UInt32*)&existingButton) == errNone && existingButton == req->buttonHandle) {
		
		TimeFormatType tf = (PrefGetPreference(prefVersion) >= preferenceDataVer2) ? PrefGetPreference(prefTimeFormat) : tfColonAMPM;
		const Int8 fontHeights[] = {7, 9, 13, 18, 28};
		struct RGBColorType color;
		UInt16 bmpW, bmpH, i;
		RectangleType rect;
		DateTimeType dt;
		MemHandle fontH;
		BitmapPtr offB;
		char time[32];
		Err err;
		
		//get time string
		TimSecondsToDateTime(TimGetSeconds(), &dt);
		TimeToAscii (dt.hour, dt.minute, tf, time);
		
		//save state
		WinPushDrawState();
		//go to native coords and get our rectangle
		WinSetCoordinateSystem(kCoordinatesNative);
		WinGetClip(&rect);
		
		//pick font
		for (i = 0; i < sizeof(fontHeights) / sizeof(*fontHeights) && fontHeights[i] < rect.extent.y; i++);
		if (i)
			i--;	//i is one more than desired font, but if our area is too small for even smallest font, use it anyways
		
		
		//install and set the desired font
		fontH = DmGet1Resource('NFNT', 1000 + i);
		err = FntDefineFont(fntAppFontCustomBase + i, MemHandleLock(fontH));
		if (err == errNone) {
			FntSetFont(fntAppFontCustomBase + i);
		
			//create an offscreen window to draw (this makes life easier due to high/low/etc resoltion screens)
			//we need to use standard coordinate system for font dimensions quesries to work
			WinSetCoordinateSystem(kCoordinatesStandard);
			bmpW = FntCharsWidth(time, StrLen(time)) - 1; // last column of last char is all blank so do not count it
			bmpH = FntCharHeight();
			
			offB = BmpCreate(bmpW, bmpH, 1, NULL, &err);
			if (err == errNone && offB) {

				BitmapTypeV3 *offBv3;
				
				offBv3 = BmpCreateBitmapV3(offB, BmpGetDensity(WinGetBitmap(WinGetDrawWindow())), BmpGetBits(offB), NULL);
				if (offBv3) {

					WinHandle offW;
					
					offW = WinCreateBitmapWindow(offB, &err);
					if (err == errNone && offW) {
					
						WinHandle drawW;
						
						drawW = WinSetDrawWindow(offW);
						
						//draw the text as black on white
						WinSetTextColor(1);
						WinSetBackColor(0);
						WinDrawChars(time, StrLen(time), 0 ,0);
						WinSetDrawWindow(drawW);
						
						//set up color(s) for real draw
						notifGeneralGetColor(&color, req->isActive ? SbarColorForeActiveColor : SbarColorForeNormalColor, req->isActive ? SbarColorForeActiveBW : SbarColorForeNormalBW);
						WinSetForeColorRGB(&color, NULL);
						color.r ^= 0xff;	//make sure it differs from fore color
						color.g ^= 0xff;
						color.b ^= 0xff;
						WinSetBackColorRGB(&color, NULL);
						WinSetDrawMode(winOverlay);
						
						//draw onscreen, centered over the button
						WinSetCoordinateSystem(kCoordinatesNative);
						WinPaintBitmap((BitmapPtr)offBv3, rect.topLeft.x + (rect.extent.x - bmpW) / 2, rect.topLeft.y + (rect.extent.y - bmpH) / 2);
					
						//cleanup
						WinDeleteWindow(offW, 0);
					}
					BmpDelete((BitmapPtr)offBv3);
				}
				BmpDelete(offB);
			}
		}
		
		MemHandleUnlock(fontH);
		DmReleaseResource(fontH);
		//undo what we did
		WinPopDrawState();
	}
}

static void notifGeneralButtonHeld(void *handle)
{
	void *existingButton;
	if (FtrGet(MY_CRID, ftrNumOurButton, (UInt32*)&existingButton) != errNone || existingButton != handle)
		return;
	
	FrmCustomAlert(InfoOKAlert, "HOLD!", NULL, NULL);
}

static void notifGeneralUiUpdateTime(FormPtr f)
{
	TimeFormatType tf = (PrefGetPreference(prefVersion) >= preferenceDataVer2) ? PrefGetPreference(prefTimeFormat) : tfColonAMPM;
	DateFormatType df = (PrefGetPreference(prefVersion) >= preferenceDataVer2) ? PrefGetPreference(prefLongDateFormat) : dfDMYLongWithComma;
	FieldPtr field = FrmGetObjectPtr(f, FrmGetObjectIndex(f, 1001));
	DateTimeType dt;
	MemHandle mh;
	char x[64];

	TimSecondsToDateTime(TimGetSeconds(), &dt);

	DateToAscii(dt.month, dt.day, dt.year, df, x);
	FrmCopyTitle(f, x);
	
	TimeToAscii (dt.hour, dt.minute, tf, x);
	mh = FldGetTextHandle(field);
	FldSetTextHandle(field, NULL);
	if (!mh)
		mh = MemHandleNew(StrLen(x) + 1);
	else
		MemHandleResize(mh, StrLen(x) + 1);
	
	StrCopy(MemHandleLock(mh), x);
	MemHandleUnlock(mh);
	FldSetTextHandle(field, mh);
	
	EvtSetNullEventTick(TimGetTicks() + SysTicksPerSecond() * 10);
}

static void notifGeneralUiDrawBmp(UInt16 id)
{
	MemHandle mh;
	
	mh = DmGetResource('Tbmp', id);
	WinDrawBitmap(MemHandleLock(mh), 93, 20);
	MemHandleUnlock(mh);
	DmReleaseResource(mh);
}

static void notifGeneralUiUpdateBattery(void)
{
	UInt16 bmpLeftId = 1001, bmpRightId = 1000;
	Boolean pluggedIn;
	RectangleType r;
	UInt8 percent;
	MemHandle mh;
	char str[16];
	
	SysBatteryInfo(false, NULL, NULL, NULL, NULL, &pluggedIn, &percent);
	if (pluggedIn)
		bmpLeftId = bmpRightId = 1002;
	
	notifGeneralUiDrawBmp(bmpLeftId);
	r.topLeft.y = 0;
	r.topLeft.x = ((percent * 31 + 50) / 100) + 94;
	r.extent.x = 1000;
	r.extent.y = 1000;
	WinSetClip(&r);
	notifGeneralUiDrawBmp(bmpRightId);
	WinResetClip();
	
	WinDrawChars(str, StrPrintF(str, "%u%%        ", percent), 130, 19);
}


static Boolean notifGeneralUiEvtHandler(EventType *e)
{
	if (e->eType == ctlRepeatEvent && e->data.ctlRepeat.controlID == 1000)
		SysLCDBrightness(true, e->data.ctlRepeat.value);
	else if (e->eType == ctlSelectEvent && e->data.ctlSelect.controlID == 1000)
		SysLCDBrightness(true, e->data.ctlSelect.value);
	else if (e->eType == nilEvent)
		notifGeneralUiUpdateTime(FrmGetActiveForm());
	else if (e->eType == firstUserEvent)
		notifGeneralUiUpdateBattery();

	return false;
}

static Err notifGeneralDockStatusChangeNotifCbk(SysNotifyParamType *notifyParamsP)
{
	UInt16 ourEvt = firstUserEvent;
	
	EvtAddUniqueEventToQueue((const EventType*)&ourEvt, 0, true);
}

static void notifGeneralShowUi(void)
{
	UInt16 cardNo;
	LocalID LID;
	FormPtr f;
	UInt32 v;
	
	if (FtrGet('n_gs', 32767, &v) == errNone && v == 1) {
		EventType e;
		
		e.eType = ctlSelectEvent;
		e.data.ctlSelect.controlID = 9000;
		e.data.ctlSelect.pControl = FrmGetObjectPtr(FrmGetActiveForm(), FrmGetObjectIndex(FrmGetActiveForm(), 9000));
		e.data.ctlSelect.on = true;
		e.data.ctlSelect.value = 0;
		EvtAddEventToQueue(&e);
	}
	else {
	
		SysCurAppDatabase(&cardNo, &LID);
		SysNotifyRegister(cardNo, LID, sysExternalConnectorAttachEvent, &notifGeneralDockStatusChangeNotifCbk, 0, NULL);
		SysNotifyRegister(cardNo, LID, sysExternalConnectorDetachEvent, &notifGeneralDockStatusChangeNotifCbk, 0, NULL);
		f = FrmInitForm(1000);
		FrmSetDIAPolicyAttr(f, frmDIAPolicyCustom);
		CtlSetValue(FrmGetObjectPtr(f, FrmGetObjectIndex(f, 1000)), SysLCDBrightness(false, 0));
		FrmSetEventHandler(f, &notifGeneralUiEvtHandler);
		notifGeneralUiUpdateTime(f);
		notifGeneralDockStatusChangeNotifCbk(NULL);	//as if connector state changes so that form can be updated
		FtrSet('n_gs', 32767, 1);
		FrmDoDialog(f);
		FtrSet('n_gs', 32767, 0);
		SysNotifyUnregister(cardNo, LID, sysExternalConnectorAttachEvent, 0);
		SysNotifyUnregister(cardNo, LID, sysExternalConnectorDetachEvent, 0);
		FrmDeleteForm(f);
	}
}


static void notifGeneralButtonTapped(void *handle)
{
	void *existingButton;
	if (FtrGet(MY_CRID, ftrNumOurButton, (UInt32*)&existingButton) != errNone || existingButton != handle)
		return;
	
	notifGeneralShowUi();
}

UInt32 PilotMain(UInt16 cmd, void* cmdPBP, UInt16 flags)
{
	switch (cmd) {
		case sysAppLaunchCmdNormalLaunch:
			notifGeneralShowUi();
			break;
	
		case sysAppLaunchCmdSystemReset:
			notifGeneralInit();
			break;
		
		case sysAppLaunchCmdAlarmTriggered:
			notifGeneralRefresh();
			((SysAlarmTriggeredParamType*)cmdPBP)->purgeAlarm = true;
			break;
		
		case REPALM_LAUNCH_CODE_BUTTON_CREATED:
			notifGeneralButtonCreated(cmdPBP);
			break;
		
		case REPALM_LAUNCH_CODE_BUTTON_DELETED:
			notifGeneralButtonDestroyed(cmdPBP);
			break;
		
		case REPALM_LAUNCH_CODE_BUTTON_TAPPED:
			notifGeneralButtonTapped(cmdPBP);
			break;
		
		case REPALM_LAUNCH_CODE_BUTTON_NEEDS_DRAW:
			notifGeneralButtonNeedsDraw((const struct SbarNotifDrawReq*)cmdPBP);
			break;
	}
	return 0;
}





