#include "palmosInternal.h"
#include "../dal/halDrawing.h"
#include "../dal/dal.h"
#include "../dal/kal.h"
#include "patching.h"
#include "private.h"
#include "printf.h"
#include "skin.h"
#include "sbar.h"

#define LONG_HOLD_TIME			2000	//msec

#define SBAR_HANDLE_MUNGE_VAL	0x825D04F5
#define SBAR_BUTTON_MAGIC		0x7AE8AA71

#define SBAR_PINS_BTN_STATE_DIS_DN	0	//down arrow disabled
#define SBAR_PINS_BTN_STATE_ENA_DN	1	//down arrow enabled
#define SBAR_PINS_BTN_STATE_DIS_UP	2	//up arrow disabled
#define SBAR_PINS_BTN_STATE_ENA_UP	3	//up arrow enabled
#define SBAR_PINS_BTN_STATE_NUM		4	//number of states

#define ANIMATE_TRANSITIONS			1	//dia only, not sbar

//XXX: handle screen lock/unlock
//XXX: grabbing field focus should pull up dia, releasing it shoudl restore it to where it was




//our structs
struct SbarNotif {
	uint32_t magic;
	struct SbarNotif *next;
	struct SbarNotif *prev;
	struct RectangleType bounds;
	int32_t order;
	LocalID handler;
	uint8_t state;
	uint8_t numStates;
	uint8_t group			: 2;
	uint8_t appDraws		: 1;
	uint8_t active			: 1;	//for drawing
	uint8_t holdSupported	: 1;
	uint32_t downAt;			//time
	struct SbarNotifStateDescr states[];
};

struct SbarNotifPenTrackingState {
	uint32_t startTime;
	struct SbarNotif *n;
	bool cancelTrigger;	//if set, we'll not trigger on release (we likely already triggered)
};


static const struct RGBColorType mColorBlack = {.r = 0, .g = 0, .b = 0}, mColorWhite = {.r = 255, .g = 255, .b = 255};

static bool sbarPrvSbarRedraw(struct SbarNotif *b);	//assumes all settings are set up. de snot flush to screen. pass NULL to redraw al lbuttons
static void sbarPrvRecalcAll(void);


static Err sbarPrvSysSetOrientation(uint16_t orientation, bool saveAsUserSelection);
static Err sbarPrvSysSetOrientationTriggerState(uint16_t state, bool saveAsUserSelection);
static Err sbarPrvShowHide(bool show, bool saveAsUserSelection);
static Err sbarPrvPINSetInputTriggerState(uint16_t state, bool saveAsUserSelection);
static Err sbarPrvPINSetInputAreaState(uint16_t state, bool saveAsUserSelection);


static WinHandle sbarPrvCreateNativeOffscreenWindowWithDepth(int16_t w, int16_t h, uint32_t depth)
{
	struct BitmapType *bmp, *dispBmp = WinGetBitmap(WinGetDisplayWindow());
	struct PalmWindow *win;
	Err err;
	
	bmp = BmpCreateVersion3(w, h, depth, NULL, BmpGetDensity(dispBmp), 0xffffffff, depth == 16 ? PALM_BMP_PIXEL_FORMAT_RGB565_LE : ((struct PalmBitmapV3*)dispBmp)->pixelFormat, &err);
	if (err != errNone)
		return NULL;
	MemPtrSetOwner(bmp, 0);	//else it will get auto-freed
	
	((struct PalmBitmap*)bmp)->noDither = 1;
	
	win = (struct PalmWindow*)WinCreateBitmapWindow(bmp, &err);
	if (err != errNone) {
		BmpDelete(bmp);
		return NULL;
	}
	MemPtrSetOwner(win, 0);	//else it will get auto-freed
	
	win->flags.format = 0;
	win->flags.freeBitmap = true;

	return (WinHandle)win;
}

static WinHandle sbarPrvCreateOurOffscreenWindow(int16_t w, int16_t h)		//dimentions are in pixels of bitmap, depth is copied form screen
{
	uint32_t depth = BmpGetBitDepth(WinGetBitmap(WinGetDisplayWindow()));
	WinHandle ret;
	
	if (depth == 8)
		depth = 16;	//our ofscreen window is never 8bpp because palettes
	
	ret = sbarPrvCreateNativeOffscreenWindowWithDepth(w, h, depth);
	if (!ret) {
		loge("Cannot create sbar offscreen window\n");
		return NULL;
	}
	
	return ret;
}

static struct RectangleType* sbarPrvGetGrafAreaRectPtr(void)
{
	struct PalmOsGobals *pg = PalmOSGetGlobalsPtr();
	struct Globals *g = globalsGet();
	
	return g->isGarnet ? &pg->evtG->v54.writingArea : &pg->evtG->v51.writingArea;
}

static bool sbarPrvUpdatePinsButtonState(void)
{
	struct Globals *g = globalsGet();
	uint32_t state;
	
	if (!g->pinsCtlButton)
		return true;
	
	if (g->diaState == SBAR_STATE_UP)
		state = g->diaTriggerEn ? SBAR_PINS_BTN_STATE_ENA_DN : SBAR_PINS_BTN_STATE_DIS_DN;
	else
		state = g->diaTriggerEn ? SBAR_PINS_BTN_STATE_ENA_UP : SBAR_PINS_BTN_STATE_DIS_UP;
	
	return sbarNotifSetState(g->pinsCtlButton, state);
}

static bool sbarPrvUpdateOsGlobalsForState(void)		//do not call this, unless you are sbarPrvSetState()
{
	struct RectangleType *writingAreaP = sbarPrvGetGrafAreaRectPtr(), screenRect, sbarAndDiaRect = {.extent = {.x = -1, .y = -1}}, notifRect = {.extent = {.x = -1, .y = -1}};
	struct PalmOsGobals *pg = PalmOSGetGlobalsPtr();
	struct Globals *g = globalsGet();
	struct PalmSilkscreenArea *area;
	uint16_t coordSys, numAreas;
	struct SbarSkinState *ss;
	uint32_t i;
	Err err;
	
	ss = &g->skin.states[g->diaState];
	pg->silkNfo = ss->aslk;
	
	coordSys = WinSetCoordinateSystem(kCoordinatesStandard);
	
	//entirely hide graf area, notif area, and sbar to start with for case when we do not have graf area
	writingAreaP->topLeft.x = 8192;
	writingAreaP->topLeft.y = 8192;
	writingAreaP->extent.x = 0;
	writingAreaP->extent.y = 0;
	
	//calc graf area rect and set screen area
	area = (struct PalmSilkscreenArea*)EvtGetSilkscreenAreaList(&numAreas);
	for (i = 0; i < numAreas; i++, area++) {
		
		switch (area->type) {
			case 'graf':
		
				//we assume the areas have the same height and that  they are next to eaxch other (though not necesarily immediately)
				// we do join them anyways so any space in between will be treated as graf area anyways
				if (area->index == 0)
					*writingAreaP = area->bounds;
				else if (area->index == 1)
					writingAreaP->extent.x = area->bounds.topLeft.x + area->bounds.extent.x - writingAreaP->topLeft.x;
				break;
			
			case 'scrn':
				screenRect = area->bounds;
				WinSetBounds(WinGetDisplayWindow(), &screenRect);
				break;
			
			case 'sbar':
				sbarAndDiaRect = area->bounds;
				break;
			
			case 'tray':
				notifRect = area->bounds;
				break;
		}
	}
	if (g->isGarnet)
		pg->evtG->v54.appAreaBottom = screenRect.topLeft.y + screenRect.extent.y;
	else
		pg->evtG->v51.appAreaBottom = screenRect.topLeft.y + screenRect.extent.y;
	
	if (sbarAndDiaRect.extent.y <= 0) {	//no dia/sbar
		
		sbarAndDiaRect.topLeft.x = 0;
		sbarAndDiaRect.topLeft.y = screenRect.topLeft.y + screenRect.extent.y;
		sbarAndDiaRect.extent.x = screenRect.extent.x + 1;
		sbarAndDiaRect.extent.y = 1;		//so we can allocate the window and such
	}
	g->sbarAndDia = sbarAndDiaRect;
	g->notifArea = notifRect;
	
	if (g->sbarWin) {
		WinDeleteWindow(g->sbarWin, false);
		g->sbarWin = NULL;
	}
	if (g->offscreenWin) {
		WinDeleteWindow(g->offscreenWin, false);
		g->offscreenWin = NULL;
	}
	
	g->sbarWin = WinCreateWindow(&sbarAndDiaRect, noFrame, false, true, &err);
	if (err != errNone || !g->sbarWin)
		return false;
	MemPtrSetOwner(g->sbarWin, 0);	//else it will get auto-freed
	
	WinSetCoordinateSystem(kCoordinatesNative);
	WinScaleRectangle(&sbarAndDiaRect);
	g->offscreenWin = sbarPrvCreateOurOffscreenWindow(sbarAndDiaRect.extent.x, sbarAndDiaRect.extent.y);
	if (!g->offscreenWin)
		return false;
	
	WinSetCoordinateSystem(coordSys);
	
	return true;
}

static void sbarPrvPrepareForDrawOnSbar(WinHandle *winBcpP, const struct RectangleType *desiredClip, bool direct)
{
	struct Globals *g = globalsGet();
	struct RectangleType nowClip;
	
	KALMutexReserve(g->mutex, -1);
	
	*winBcpP = WinSetDrawWindow(direct ? g->sbarWin : g->offscreenWin);
	WinPushDrawState();
	WinResetClip();
	WinSetCoordinateSystem(kCoordinatesStandard);
	
	if (desiredClip) {
		
		nowClip = *desiredClip;
		nowClip.topLeft.x -= g->sbarAndDia.topLeft.x;
		nowClip.topLeft.y -= g->sbarAndDia.topLeft.y;
		WinSetClip(&nowClip);
	}
	
	WinSetForeColorRGB(&mColorBlack, NULL);	//else 1bpp bitmaps do not draw right
	WinSetBackColorRGB(&mColorWhite, NULL);
}

static void sbarPrvCleanupAfterDrawOnSbar(WinHandle winBcp, const struct RectangleType *copyRect)
{
	struct Globals *g = globalsGet();
	
	WinResetClip();		//in ofscreen window
	WinSetDrawWindow(g->sbarWin);
	WinResetClip();		//in sbar window window
	
	if (copyRect) {
		WinSetCoordinateSystem(kCoordinatesStandard);
		struct RectangleType copy = *copyRect;
		copy.topLeft.x -= g->sbarAndDia.topLeft.x;
		copy.topLeft.y -= g->sbarAndDia.topLeft.y;
		WinSetClip(&copy);
	}
	else
		WinResetClip();		//in sbar window window
	WinDrawBitmap(WinGetBitmap(g->offscreenWin), 0, 0);
	WinPopDrawState();
	WinSetDrawWindow(winBcp);
	KALMutexRelease(g->mutex);
}

static bool sbarPrvRedrawEverything(void)		//global redraw - the big hammer. DO NOT CALL unless you hold the mutex
{
	struct Globals *g = globalsGet();
	WinHandle winBcp;
	
	//prepare
	sbarPrvPrepareForDrawOnSbar(&winBcp, NULL, false);
	
	//draw dia/sbar?
	if (g->skin.states[g->diaState].nor)
		WinDrawBitmap(g->skin.states[g->diaState].nor, 0, 0);

	//draw notification area
	sbarPrvSbarRedraw(NULL);
	
	//cleanup & copy to screen
	sbarPrvCleanupAfterDrawOnSbar(winBcp, NULL);
	
	return true;
}

static bool notifyAppOfScreenChange(void)
{
	struct SysNotifyDisplayResizedDetailsTag details;
	SysNotifyParamType np = {.notifyType = sysNotifyDisplayResizedEvent, .broadcaster = 'psys', .notifyDetailsP = &details};
	struct EventType evt = {.eType = winDisplayChangedEvent,} ;
	
	WinGetBounds(WinGetDisplayWindow(), &details.newBounds);
	
	//notification must be BE
	details.newBounds.topLeft.x = __builtin_bswap16(details.newBounds.topLeft.x);
	details.newBounds.topLeft.y = __builtin_bswap16(details.newBounds.topLeft.y);
	details.newBounds.extent.x = __builtin_bswap16(details.newBounds.extent.x);
	details.newBounds.extent.y = __builtin_bswap16(details.newBounds.extent.y);
	
	SysEventAddToQueue(&evt);	//technically this is for pins 1.1 only, but we send it in 1.0 since it is safe to
	return errNone == SysNotifyBroadcast(&np);
}

static bool sbarPrvSetState(uint32_t state, bool forceUpdate)
{
	bool ret = true, actualChange, redrew = false;
	struct Globals *g = globalsGet();
	uint32_t prevState, t, coordSys;
	RectangleType clip, rect;
	WinHandle wh;
	
	KALMutexReserve(g->mutex, -1);
	prevState = g->diaState;
	
	actualChange = prevState != state;
	redrew = actualChange || forceUpdate;
	
	//if dia is going up, save the area behind (only for sbar-present cases)
	if (prevState == SBAR_STATE_JUST_SBAR && state == SBAR_STATE_UP) {
		
		Err e = errNone;
		
		if (g->savedBits) {
			WinDeleteWindow(g->savedBits, false);
			g->savedBits = NULL;
		}
		
		rect = g->sbarAndDia;
		t = rect.topLeft.y + rect.extent.y;
		rect.topLeft.y = 160;
		rect.extent.y = t - rect.topLeft.y;
		
		wh = WinSetDrawWindow(WinGetDisplayWindow());
		coordSys = WinSetCoordinateSystem(kCoordinatesStandard);
		g->savedBits = WinSaveBits(&rect, &e);
		
		if (ANIMATE_TRANSITIONS) {
			
			WinGetClip(&clip);
			for (t = g->sbarAndDia.topLeft.y; t >= 160; t--) {
				
				rect.topLeft.y = t;
				rect.extent.y = g->sbarAndDia.topLeft.y - t;
				WinSetClip(&rect);
				WinDrawBitmap(g->skin.states[SBAR_STATE_UP].nor, 0, t);
				SysTaskDelay(5);
			}
			WinSetClip(&clip);
		}
		
		WinSetCoordinateSystem(coordSys);
		WinSetDrawWindow(wh);
		
		if (e != errNone)
			g->savedBits = NULL;
	}
	
	if (redrew) {
		
		g->diaState = state;
		ret = sbarPrvUpdateOsGlobalsForState();
		sbarPrvUpdatePinsButtonState();
		sbarPrvRecalcAll();
		ret = sbarPrvRedrawEverything() && ret;
		if (actualChange)
			ret = notifyAppOfScreenChange() && ret;
	}
	
	//if dia is going down, restore the area behind (only for sbar-present cases)
	if (prevState == SBAR_STATE_UP && state == SBAR_STATE_JUST_SBAR) {
		
		wh = WinSetDrawWindow(WinGetDisplayWindow());
		coordSys = WinSetCoordinateSystem(kCoordinatesStandard);
		
		if (ANIMATE_TRANSITIONS) {
			
			rect = g->sbarAndDia;
		
			WinGetClip(&clip);
			for (t = 160; t < g->sbarAndDia.topLeft.y; t++) {
				
				rect.topLeft.y = t;
				rect.extent.y = g->sbarAndDia.topLeft.y - t;
				WinSetClip(&rect);
				WinDrawBitmap(g->skin.states[SBAR_STATE_UP].nor, 0, t);
				rect.topLeft.y = 160;
				rect.extent.y = t - 160;
				WinSetClip(&rect);
				if (g->savedBits)
					WinDrawBitmap(WinGetBitmap(g->savedBits), 0, 160);
				else
					WinEraseRectangle(&rect, 0);
				SysTaskDelay(5);
			}
			WinSetClip(&clip);
		}
		
		if (g->savedBits)
			WinRestoreBits(g->savedBits, 0, 160);
		WinSetCoordinateSystem(coordSys);
		WinSetDrawWindow(wh);
		g->savedBits = NULL;
	}
	
	KALMutexRelease(g->mutex);
	
	return ret;
}

static bool sbarPrvForceRedraw(void)
{
	struct Globals *g = globalsGet();
	
	return sbarPrvSetState(g->diaState, true);
}

static Err sbarPrvScreenDepthChangedNotif(SysNotifyParamType *notif)
{
	if (!sbarPrvForceRedraw()) {
		loge("Cannot set state at depth change");
		return false;
	}
	return errNone;
}

static bool sbarPrvInstallPinsCtlButton(void)
{
	struct Globals *g = globalsGet();
	uint32_t coordSys;
	int16_t width;
	struct SbarNotifStateDescr states[] = {
		[SBAR_PINS_BTN_STATE_DIS_DN] = {
			.normalImg = g->skin.diaDn.dis,
		},
		[SBAR_PINS_BTN_STATE_ENA_DN] = {
			.normalImg = g->skin.diaDn.ena,
			.activeImg = g->skin.diaDn.act,
		},
		[SBAR_PINS_BTN_STATE_DIS_UP] = {
			.normalImg = g->skin.diaUp.dis,
		},
		[SBAR_PINS_BTN_STATE_ENA_UP] = {
			.normalImg = g->skin.diaUp.ena,
			.activeImg = g->skin.diaUp.act,
		},
	};
	
	
	BmpGetDimensions(g->skin.diaDn.dis, &width, NULL, NULL);
	coordSys = WinSetCoordinateSystem(BmpGetDensity(g->skin.diaDn.dis));
	width = WinUnscaleCoord(width, false);	//convert to standard coord sys
	WinSetCoordinateSystem(coordSys);
	
	g->pinsCtlButton = sbarNotifCreate(DmFindDatabase(MY_DB_NAME), SBAR_PINS_BTN_STATE_NUM, states, REPALM_BTN_GROUP_RIGHT, 0x7fffffff /* rightmost */, width, false, false, NULL);
	
	return !!g->pinsCtlButton;
}

bool sbarPrvInit(bool isOS50, bool isGarnet)
{
	struct PalmOsGobals *pg = PalmOSGetGlobalsPtr();
	struct Globals *g = globalsGet();
	
	//save some state
	g->isGarnet = isGarnet;
	g->isOS50 = isOS50;
	g->diaState = SBAR_STATE_UP;
	g->diaTriggerEn = false;
	g->savedBits = NULL;
	
	//user preferences
	g->userDiaState = pinInputAreaOpen;
	g->userRotState = sysOrientationPortrait;
	g->userRotTriggerState = true;
	g->userDiaTriggerState = false;
	
	//load skin
	if (!sbarPrvSkinLoad(&g->skin)) {
		loge("failed to load the skin\n");
		return false;
	}
	
	//we need a mutex
	if (errNone != KALMutexCreate(&g->mutex, 'sbar')) {
		loge("Cannot create sbar mutex\n");
		return false;
	}
	
	//free current dia info in boot globals
	if (pg->silkNfo) {
		MemPtrUnlock(pg->silkNfo);
		pg->silkNfo = NULL;
	}
	
	//record that we have no offscreen window or onscreen window
	g->sbarWin = NULL;
	g->offscreenWin = NULL;
	
	//draw
	if (!sbarPrvForceRedraw()) {
		loge("Cannot set state at init");
		return false;
	}
	
	//catch screen depth changed
	if (errNone != SysNotifyRegister((LocalID)-1, sysNotifyDisplayChangeEvent, &sbarPrvScreenDepthChangedNotif, 0, NULL)) {
		loge("Cannot register for screenchange notif");
		return false;
	}
	
	//install pins control button
	if (!sbarPrvInstallPinsCtlButton()) {
		loge("Cannot add PINS control button\n");
		return false;
	}
	
	return true;
}

static struct SbarNotif* sbarPrvCheckHandle(void* handle)
{
	struct SbarNotif *b = (struct SbarNotif*)(((uintptr_t)handle) ^ SBAR_HANDLE_MUNGE_VAL);
	
	return (b->magic == SBAR_BUTTON_MAGIC) ? b : NULL;
}

static void* sbarPrvMungeHandle(struct SbarNotif *b)
{
	return (void*)(((uintptr_t)b) ^ SBAR_HANDLE_MUNGE_VAL);
}

static bool sbarPrvButtonCallback(struct SbarNotif *b, uint16_t launchCode, void* param)
{
	uint32_t dummy;
	
	return errNone == SysAppLaunch(b->handler, 0, launchCode, param, &dummy);
}

static bool sbarPrvDrawSingleButtonInternal(struct SbarNotif *b)	//assumes everything is set up. do not call directly, unless you are sbarPrvSbarRedraw()
{
	struct Globals *g = globalsGet();
	struct RectangleType clip;
	bool ret;
	
	clip = b->bounds;
	clip.topLeft.x -= g->sbarAndDia.topLeft.x;
	clip.topLeft.y -= g->sbarAndDia.topLeft.y;
	WinSetClip(&clip);

	if (b->appDraws) {
		struct SbarNotifDrawReq req = {.buttonHandle = (void*)__builtin_bswap32((uintptr_t)sbarPrvMungeHandle(b)), .state = b->state, .isActive = b->active, };
		ret = sbarPrvButtonCallback(b, REPALM_LAUNCH_CODE_BUTTON_NEEDS_DRAW, &req);
	}
	else {
		
		struct SbarNotifStateDescr *state = &b->states[b->state];
		struct BitmapType *bmp = (b->active && state->activeImg) ? state->activeImg : state->normalImg;
		
		WinDrawBitmap(bmp, b->bounds.topLeft.x - g->sbarAndDia.topLeft.x, b->bounds.topLeft.y - g->sbarAndDia.topLeft.y);
		ret = true;
	}
	
	return ret;
}


static bool sbarPrvSbarRedraw(struct SbarNotif *b /* or NULL */)
{
	struct Globals *g = globalsGet();
	bool ret = true;
	uint32_t i;
	
	if (g->notifArea.extent.x <= 0 || g->notifArea.extent.y <= 0)
		return true;
	
	if (b)
		ret = sbarPrvDrawSingleButtonInternal(b);
	else {
		for (i = 0; i < REPALM_BTN_GROUP_NUM; i++) {
			for (b = g->notifs[i]; b; b = b->next) {
				
				ret = sbarPrvDrawSingleButtonInternal(b) && ret;
			}
		}
	}
	
	return ret;
}

static bool sbarPrvSbarRedrawSingleButton(struct SbarNotif *b)			//must draw background
{
	struct Globals *g = globalsGet();
	WinHandle winBcp;
	bool ret;
	
	sbarPrvPrepareForDrawOnSbar(&winBcp, &b->bounds, false);
	
	//single button redraw needs to draw background image
	if (g->skin.states[g->diaState].nor)
		WinDrawBitmap(g->skin.states[g->diaState].nor, 0, 0);
	
	//draw the button
	ret = sbarPrvSbarRedraw(b);
	
	//flush to screen and cleanup
	sbarPrvCleanupAfterDrawOnSbar(winBcp, &b->bounds);
	
	return ret;
}

//center group only has padding on one side (makes life easier actually
static void sbarPrvCalcCurGroupWidths(uint32_t widthsP[static REPALM_BTN_GROUP_NUM])
{
	struct Globals *g = globalsGet();
	struct SbarNotif *b;
	uint32_t i, w;
	
	for (i = 0; i < REPALM_BTN_GROUP_NUM; i++) {
		for (w = 0, b = g->notifs[i]; b; b = b->next)
			w += b->bounds.extent.x + SBAR_NOTIF_SPACING_H	;
		
		widthsP[i] = w;
	}
}

//can pass groupModified == REPALM_BTN_GROUP_NUM to recalc all
static void sbarPrvRecalcPositions(uint8_t groupModified, const uint32_t widths[static REPALM_BTN_GROUP_NUM])
{
	struct Globals *g = globalsGet();
	//recalc rectangles (depending on which group we updated, we might get away with partial recalc)
	
	struct SbarNotif *b;
	uint32_t i;
	
	//recalc rectangles (depending on which group we updated, we might get away with partial recalc)
	if (groupModified == REPALM_BTN_GROUP_LEFT || groupModified == REPALM_BTN_GROUP_NUM) {
		
		for (i = g->notifArea.topLeft.x, b = g->notifs[REPALM_BTN_GROUP_LEFT]; b; b = b->next) {
			b->bounds.topLeft.x = i;
			i += b->bounds.extent.x + SBAR_NOTIF_SPACING_H;
		}
	}
	if (groupModified == REPALM_BTN_GROUP_RIGHT || groupModified == REPALM_BTN_GROUP_NUM) {
		
		for (b = g->notifs[REPALM_BTN_GROUP_RIGHT]; b && b->next; b = b->next);	//find last right botton
		
		for (i = g->notifArea.topLeft.x + g->notifArea.extent.x; b; b = b->prev) {
			
			i -= b->bounds.extent.x;
			b->bounds.topLeft.x = i;
			i -= SBAR_NOTIF_SPACING_H;
		}
	}
	
	//we always need to recalc center group. either we updated it, or we updated another one which changed the centering of the center group
	//NOTE: all news widths are already known and are in curWidthUsed[]
	
	//see how much width we have to put around us (for centering)
	i = g->notifArea.extent.x - widths[REPALM_BTN_GROUP_LEFT] - widths[REPALM_BTN_GROUP_CENTER] - widths[REPALM_BTN_GROUP_RIGHT];
	
	//use half of it as our offset from left
	i = g->notifArea.topLeft.x + widths[REPALM_BTN_GROUP_LEFT] + i / 2;
	
	//now recalc rectangles
	for (b = g->notifs[REPALM_BTN_GROUP_CENTER]; b; b = b->next) {
		b->bounds.topLeft.x = i;
		i += b->bounds.extent.x + SBAR_NOTIF_SPACING_H;
	}
}

static void sbarPrvRecalcAll(void)
{
	uint32_t curWidthUsed[REPALM_BTN_GROUP_NUM];
	struct Globals *g = globalsGet();
	
	if (g->notifArea.extent.x > 0 && g->notifArea.extent.y > 0) {	//no point if it is hidden
	
		sbarPrvCalcCurGroupWidths(curWidthUsed);
		sbarPrvRecalcPositions(REPALM_BTN_GROUP_NUM, curWidthUsed);
	}
}

void* sbarNotifCreate(LocalID handlerApp, uint8_t numStates, const struct SbarNotifStateDescr *states, uint8_t group, int32_t order, uint16_t width, bool holdSupported, bool manualDraw, void* userData)
{
	uint32_t i, curWidthUsed[REPALM_BTN_GROUP_NUM];
	struct Globals *g = globalsGet();
	struct SbarNotifCreated nc;
	struct SbarNotif *b;
	void* ret;
	
	if (width % 2)
		width++;
	
	//sanity check
	if (!handlerApp || !width || group >= REPALM_BTN_GROUP_NUM || !numStates || width % 2)
		return NULL;
	
	sbarPrvCalcCurGroupWidths(curWidthUsed);
	
	//add this requested width to proper group
	curWidthUsed[group] += width + SBAR_NOTIF_SPACING_H;
	
	//see if we fit curently both side groups include the 2 pixel padding between buttons and from the border
	//middle group has padding on one side calculated (due to how we do this). we need to correct this. if it has any buttons,
	//it needs one extra set of padding
	if (curWidthUsed[REPALM_BTN_GROUP_CENTER])
		curWidthUsed[REPALM_BTN_GROUP_CENTER] += SBAR_NOTIF_SPACING_H;
	
	//now see if we fit
	if (curWidthUsed[REPALM_BTN_GROUP_LEFT] + curWidthUsed[REPALM_BTN_GROUP_CENTER] + curWidthUsed[REPALM_BTN_GROUP_RIGHT] > g->notifArea.extent.x)
		return NULL;
	
	//looks like we're a go for creating this button - create the struct
	b = MemChunkNew(0, sizeof(struct SbarNotif) + sizeof(struct SbarNotifStateDescr) * numStates, 0x1200);
	if (!b)
		return NULL;
	b->magic = SBAR_BUTTON_MAGIC;
	b->order = order;
	b->handler = (LocalID)handlerApp;
	b->state = 0;
	b->group = group;
	b->appDraws = manualDraw;
	b->active = false;
	b->holdSupported = holdSupported;
	b->numStates = numStates;
	b->bounds.topLeft.y = g->notifArea.topLeft.y + SBAR_NOTIF_SPACING_V;
	b->bounds.extent.x = width;
	b->bounds.extent.y = g->notifArea.extent.y;
	for (i = 0; i < numStates; i++)
		b->states[i] = states[i];
	
	ret = sbarPrvMungeHandle(b);
	
	//INSERT: if this is the only button in the group, we have a simplified flow
	if (!g->notifs[group]) {
		b->next = NULL;
		b->prev = NULL;
		g->notifs[group] = b;
	}
	else {
		struct SbarNotif *p, *t;
		
		//find insert position at the end of this, t is item before which we insert (or NULL), p is previous (or NULL)
		for (p = NULL, t = g->notifs[group]; t && t->order < order; p = t, t = t->next);
	
		b->next = t;
		b->prev = p;	
	
		if (t)
			t->prev = b;
		if (p)
			p->next = b;
		else
			g->notifs[group] = b;
	}

	nc.buttonHandle = (void*)__builtin_bswap32((uintptr_t)ret);
	nc.userData = (void*)__builtin_bswap32((uintptr_t)userData);
	(void)sbarPrvButtonCallback(b, REPALM_LAUNCH_CODE_BUTTON_CREATED, &nc);
	sbarPrvRecalcPositions(group, curWidthUsed);
	sbarPrvForceRedraw();
	
	//return handle
	return ret;
}

bool sbarNotifDestroy(void *handle)
{
	struct SbarNotif *b = sbarPrvCheckHandle(handle);
	uint32_t curWidthUsed[REPALM_BTN_GROUP_NUM];
	struct Globals *g = globalsGet();
	
	if (!b)
		return false;
	
	//unlink our button
	if (b->next)
		b->next->prev = b->prev;
	if (b->prev)
		b->prev->next = b->next;
	else
		g->notifs[b->group] = b->next;
	
	//calc resulting widths and redraw
	sbarPrvCalcCurGroupWidths(curWidthUsed);
	sbarPrvRecalcPositions(b->group, curWidthUsed);
	sbarPrvForceRedraw();
	
	//call callback
	(void)sbarPrvButtonCallback(b, REPALM_LAUNCH_CODE_BUTTON_DELETED, handle);
	
	//free state
	b->magic = 0;
	MemChunkFree(b);
	
	return true;
}

bool sbarNotifSetState(void *handle, uint8_t state)
{
	struct SbarNotif *b = sbarPrvCheckHandle(handle);
	
	if (!b || state >= b->numStates)
		return false;
	
	if (b->state == state)
		return true;
	
	b->state = state;
	return sbarPrvSbarRedrawSingleButton(b);
}

bool sbarNotifRequestRedraw(void *handle)
{
	struct SbarNotif *b = sbarPrvCheckHandle(handle);
	
	if (!b)
		return false;
	
	return sbarPrvSbarRedrawSingleButton(b);
}

uint32_t sbarNotifGetColor(uint16_t which)
{
	struct Globals *g = globalsGet();
	
	switch (which) {
		
		case SbarColorInkColor:			return g->skin.inkColor;
		case SbarColorInkBW:			return g->skin.inkBW;
		case SbarColorBackgroundColor:	return g->skin.backColor;
		case SbarColorBackgroundBW:		return g->skin.backBW;
		case SbarColorForeNormalColor:	return g->skin.foreColor;
		case SbarColorForeNormalBW:		return g->skin.foreBW;
		case SbarColorForeActiveColor:	return g->skin.foreActColor;
		case SbarColorForeActiveBW:		return g->skin.foreActBW;
		default:						return 0;
	}
}

static void sbarPrvSetForeColorFromU32(uint32_t ifColorScreen, uint32_t ifBwScreen)
{
	bool isColorScreen = BmpGetBitDepth(WinGetBitmap(WinGetDisplayWindow())) >= 8;
	struct RGBColorType rgb;
	uint32_t t;
	
	t = isColorScreen ? ifColorScreen : ifBwScreen;
	rgb.r = t >> 16;
	rgb.g = t >> 8;
	rgb.b = t >> 0;
	
	WinSetForeColorRGB(&rgb, NULL);
}

//callback can return false to finish tracking
static void sbarPrvTrackPenInArea(const struct RectangleType* area, int16_t startX, int16_t startY, bool cbkJustOnStateChange, bool (*trackCbk)(int16_t x, int16_t y, bool inside, bool penDown, void* userData), void* userData)
{
	int16_t lastX = startX, lastY = startY, x, y;
	bool active = true;	//we would not have been called if it was not active
	Boolean penDown;	//make compiler happy about types
	
	if (!trackCbk(startX, startY, true, true, userData))
		return;
	
	while (1) {
		bool nowActive;
		
		EvtGetPen (&x, &y, &penDown);
		if (penDown)
			WinWindowToDisplayPt(&x, &y);	//EvtGetPen converts to draw-window relative, we want display relative, so uncovert it
		
		if (!penDown)
			break;
		
		nowActive = RctPtInRectangle(x, y, area);
		if (nowActive != active || !cbkJustOnStateChange) {
			active = nowActive;
			if (!trackCbk(x, y, active, true, userData))
				return;
		}
		lastX = x;
		lastY = y;
	}
	
	//tell them we exited it
	(void)trackCbk(lastX, lastY, active, false, userData);
}

static bool sbarPrvSoftButtonPenTrackingCbk(int16_t x, int16_t y, bool inside, bool penDown, void* userData)
{
	const struct PalmSilkButton *btn = (const struct PalmSilkButton*)userData;
	struct Globals *g = globalsGet();
	struct BitmapType *bmp;
	WinHandle wh;

	bmp = g->skin.states[g->diaState].act;
	if (!bmp || !inside || !penDown)
		bmp = g->skin.states[g->diaState].nor;
	
	if (inside && !penDown)	//tapped
		EvtEnqueueKey(btn->asciiCode, btn->keyCode, btn->modifiers);
	
	if (!bmp)
		return true;
	
	sbarPrvPrepareForDrawOnSbar(&wh, &btn->bounds, false);
	WinDrawBitmap(bmp, 0, 0);
	sbarPrvCleanupAfterDrawOnSbar(wh, &btn->bounds);
	
	return true;
}

static bool sbarPrvNotifPenTrackingCbk(int16_t x, int16_t y, bool inside, bool penDown, void* userData)
{
	struct SbarNotifPenTrackingState *ts = (struct SbarNotifPenTrackingState*)userData;
	bool newActive = inside && penDown;
	struct SbarNotif *n = ts->n;
	
	if (!penDown)
		ts->cancelTrigger = !inside;	//no trigger if released outside
	
	if (!n->active && inside)		//just entered - record time
		ts->startTime = HALTimeGetSystemTime();
	
	if (!newActive != !n->active) {
		n->active = inside && penDown;
		sbarPrvSbarRedrawSingleButton(n);
	}
	
	if (n->active && n->holdSupported && HALTimeGetSystemTime() - ts->startTime > LONG_HOLD_TIME) {
		
		n->active = false;
		sbarPrvSbarRedrawSingleButton(n);	//redraw as inactive
		(void)sbarPrvButtonCallback(n, REPALM_LAUNCH_CODE_BUTTON_HELD, sbarPrvMungeHandle(n));
		ts->cancelTrigger = true;
		return false;
	}
	
	return true;
}

static bool sbarPrvGrafPenTrackingCbk(int16_t x, int16_t y, bool inside, bool penDown, void* userData)
{
	struct PointType *point = (struct PointType*)userData;
	struct Globals *g = globalsGet();
	
	if (penDown) {
		
		WinDrawLine(point->x - g->sbarAndDia.topLeft.x, point->y - g->sbarAndDia.topLeft.y, x - g->sbarAndDia.topLeft.x, y - g->sbarAndDia.topLeft.y);
		point->x = x;
		point->y = y;
	}
	return true;
}

static void sbarPrvHandlePenInGrafArea(const struct RectangleType* area, int16_t startX, int16_t startY)
{
	struct PointType point = {.x = startX, .y = startY};
	struct Globals *g = globalsGet();
	struct RectangleType dot;
	WinHandle wh;
	
	sbarPrvPrepareForDrawOnSbar(&wh, area, true);
	sbarPrvSetForeColorFromU32(g->skin.inkColor, g->skin.inkBW);
	
	dot.topLeft.x = startX - g->sbarAndDia.topLeft.x - 1;
	dot.topLeft.y = startY - g->sbarAndDia.topLeft.y - 1;
	dot.extent.x = 3;
	dot.extent.y = 3;
	WinDrawRectangle(&dot, 1);
	
	sbarPrvTrackPenInArea(area, startX, startY, false, &sbarPrvGrafPenTrackingCbk, &point);
	
	sbarPrvCleanupAfterDrawOnSbar(wh, area);	//this will actually erase the drawing we just did
}

void sbarPrvNotifTapped(void *notifH)
{
	struct Globals *g = globalsGet();

	if (notifH == g->pinsCtlButton) {

		if (!g->diaTriggerEn)
			return;
			
		(void)sbarPrvPINSetInputAreaState(g->diaState == SBAR_STATE_UP ? pinInputAreaClosed : pinInputAreaOpen, true);
	}
}

static void pSysEventGet(EventPtr evt, int32_t timeout)
{
	struct Globals *g = globalsGet();
	
	globalsGet()->ot_pSysEventGet(evt, timeout);
	
	if (g->fakeEnabled)
		return;
	
	if (g->eatPenEventsTillUp) {
		if (evt->eType == penMoveEvent) {
			evt->eType = nilEvent;
			return;
		}
		if (evt->eType == penUpEvent) {
			g->eatPenEventsTillUp = false;
			evt->eType = nilEvent;
			return;
		}
	}

	if (evt->eType == penDownEvent) {	//see if in our area
		
		const struct PalmSilkscreenInfo *silkNfo = PalmOSGetGlobalsPtr()->silkNfo;
		const struct PalmSilkscreenArea *areas = (const struct PalmSilkscreenArea*)(silkNfo + 1);
		const struct PalmSilkButtonList *btnNfo = (const struct PalmSilkButtonList*)(areas + silkNfo->numAreas);
		int16_t x = evt->screenX, y = evt->screenY;
		struct RectangleType grafArea;
		struct SbarNotif *b;
		uint32_t i;
		
		WinWindowToDisplayPt(&x, &y);	//event has coords as draw-window relative, we want display relative, so uncovert it
		
		//buttons take precedence over graf area
		for (i = 0; i < btnNfo->numItems; i++) {
			
			const struct PalmSilkButton *btn = &btnNfo->btns[i];
			
			if (RctPtInRectangle(x, y, &btn->bounds)) {
				
				sbarPrvTrackPenInArea(&btn->bounds, x, y, true, &sbarPrvSoftButtonPenTrackingCbk, (void*)btn);
				g->eatPenEventsTillUp = true;
				evt->eType = nilEvent;
				return;
			}
		}
		
		//then we check for status bar (only if it is visible)
		//we do not recalc notifs if there is no notif area, but they should remain inactive
		if (g->notifArea.extent.x > 0 && g->notifArea.extent.y > 0) {
			for (i = 0; i < REPALM_BTN_GROUP_NUM; i++) {
				for (b = g->notifs[i]; b; b = b->next) {
					
					if (RctPtInRectangle(x, y, &b->bounds)) {
						
						struct SbarNotifPenTrackingState ts = {.startTime = HALTimeGetSystemTime(), .n = b, .cancelTrigger = false};
						
						//we do not need constant callbacks if we're not looking for a hold
						sbarPrvTrackPenInArea(&b->bounds, x, y, !b->holdSupported, &sbarPrvNotifPenTrackingCbk, &ts);
						if (!ts.cancelTrigger)
							(void)sbarPrvButtonCallback(b, REPALM_LAUNCH_CODE_BUTTON_TAPPED, sbarPrvMungeHandle(b));
						
						g->eatPenEventsTillUp = true;
						evt->eType = nilEvent;
						return;
					}
				}
			}
		}
		
		//last we check for graf area
		grafArea = *sbarPrvGetGrafAreaRectPtr();
		if (RctPtInRectangle(x, y, &grafArea)) {
			
			sbarPrvHandlePenInGrafArea(&grafArea, x, y);
			evt->eType = nilEvent;
			return;
		}
	}
}
DEF_BOOT_PATCH_PARTIAL(pSysEventGet, 0x824);

static void sbarPrvAlertBefore(void)
{
	struct Globals *g = globalsGet();
	
	g->inAlert = true;
	g->alertPrevDiaState = g->diaState;
}

static void sbarPrvAlertAfter(void)
{
	struct Globals *g = globalsGet();
	
	g->inAlert = false;
	sbarPrvSetState(g->alertPrevDiaState, false);
}

static uint16_t pFrmCustomAlert(uint16_t resId, const char *s1, const char *s2, const char *s3)
{
	struct Globals *g = globalsGet();
	uint16_t ret;
	
	if (!g->fakeEnabled) {
		sbarPrvAlertBefore();
	}
	ret = g->ot_pFrmCustomAlert(resId, s1, s2, s3);
	if (!g->fakeEnabled) {
		sbarPrvAlertAfter();
	}
	
	return ret;
}
DEF_UI_PATCH_PARTIAL(pFrmCustomAlert, 0x204);

static uint16_t pFrmCustomResponseAlert(uint16_t resId, const char *s1, const char *s2, const char *s3, char *replyBuf, int16_t replyBufLen, FormCheckResponseFuncPtr cbk)
{
	struct Globals *g = globalsGet();
	uint16_t ret;
	
	if (!g->fakeEnabled) {
		sbarPrvAlertBefore();
		sbarPrvSetState(SBAR_STATE_UP, false);
	}
	ret = g->ot_pFrmCustomResponseAlert(resId, s1, s2, s3, replyBuf, replyBufLen, cbk);
	if (!g->fakeEnabled) {
		sbarPrvAlertAfter();
	}
	return ret;
}
DEF_UI_PATCH_PARTIAL(pFrmCustomResponseAlert, 0x208);

static void pUIReset(void)
{
	struct Globals *g = globalsGet();
	
	if (!g->fakeEnabled) {
		//reset user preferences
		g->userDiaState = pinInputAreaOpen;
		g->userRotState = sysOrientationPortrait;
		g->userRotTriggerState = true;
		g->userDiaTriggerState = false;
		
		//reset state
		g->diaTriggerEn = false;
		g->maxAppHeight = 160;
		sbarPrvSetState(SBAR_STATE_UP, true);
		
		if (g->savedBits) {
			WinDeleteWindow(g->savedBits, false);
			g->savedBits = NULL;
		}
	}
	
	g->ot_pUIReset();
}
DEF_UI_PATCH_PARTIAL(pUIReset, 0x568);

static void sbarPrvSetDiaAndSbarVisibility(uint32_t inputAreaState, uint32_t statusState)
{
	struct Globals *g = globalsGet();
	uint32_t dstState;
	
	if (inputAreaState == pinInputAreaUser)
		inputAreaState = g->userDiaState;
	
	switch (inputAreaState) {
		default:
			loge("impossible input area state requested: %d\n", inputAreaState);
			//fallthrough
		case pinInputAreaOpen:
			dstState = SBAR_STATE_UP;
			break;
		case pinInputAreaClosed:
			dstState = statusState ? SBAR_STATE_JUST_SBAR : SBAR_STATE_FULLSCREEN;
			break;
	}
	sbarPrvSetState(dstState, false);
}

static void sbarPrvEnactFormPolicy(struct PalmForm *frm)
{
	//orientation
	(void)sbarPrvSysSetOrientation(frm->diaPolicyAttr.orientation, false);
	(void)sbarPrvSysSetOrientationTriggerState(frm->diaPolicyAttr.orientationTrigger, false);
	
	//dia & sbar state
	(void)sbarPrvSetDiaAndSbarVisibility(frm->diaPolicyAttr.inputAreaState, frm->diaPolicyAttr.statusState);
	
	//dia trigger state
	(void)sbarPrvPINSetInputTriggerState(frm->diaPolicyAttr.inputTrigger, false);
}

static void pFrmDrawForm(struct PalmForm *frm)
{
	struct PalmDiaPolicyAttr *policy = &frm->diaPolicyAttr;
	bool isAlert = false, responseAlert = false;
	struct Globals *g = globalsGet();

	if (!g->fakeEnabled) {
		isAlert = g->inAlert && frm->formID == PALMOS_ALERT_FORM_ID;
		responseAlert = isAlert && frmInvalidObjectId != FrmGetObjectIndex((FormPtr)frm, PALMOS_ALERT_INPUT_FIELD_ID);
	
		if (!frm->attr.visible) {
			
			if (responseAlert) {
				
				policy->frmDIAPolicy = 1;
				policy->inputAreaState = pinInputAreaOpen;
				policy->statusState = 1;
				policy->inputTrigger = pinInputTriggerDisabled;
				policy->orientation = sysOrientationUser;
				policy->orientationTrigger = sysOrientationTriggerDisabled;
			}
			else if (isAlert) {
				
				policy->frmDIAPolicy = 1;
				policy->inputAreaState = pinInputAreaUser;
				policy->statusState = 1;
				policy->inputTrigger = pinInputTriggerDisabled;
				policy->orientation = sysOrientationUser;
				policy->orientationTrigger = sysOrientationTriggerDisabled;
			}
			else if (policy->frmDIAPolicy) {	//form has a policy
				
				policy->frmDIAPolicy = 1;
				policy->inputAreaState = pinInputAreaUser;
				policy->statusState = 1;
				policy->inputTrigger = pinInputTriggerEnabled;
				policy->orientation = sysOrientationUser;
				policy->orientationTrigger = sysOrientationTriggerEnabled;
			}
			else {								// form has no policy
				
				policy->frmDIAPolicy = 1;
				policy->inputAreaState = pinInputAreaOpen;
				policy->statusState = 1;
				policy->inputTrigger = pinInputTriggerDisabled;
				policy->orientation = sysOrientationUser;
				policy->orientationTrigger = sysOrientationTriggerEnabled;
			}
		}
		
		sbarPrvEnactFormPolicy(frm);
	}
	g->ot_pFrmDrawForm(frm);
}
DEF_UI_PATCH_PARTIAL(pFrmDrawForm, 0x218);

static void pFrmSetActiveForm(struct PalmForm *frm)
{
	struct Globals *g = globalsGet();
	
	if (!g->fakeEnabled) {
		if (frm && frm->attr.visible)
			sbarPrvEnactFormPolicy(frm);
	}
	g->ot_pFrmSetActiveForm(frm);
}
DEF_UI_PATCH_PARTIAL(pFrmSetActiveForm, 0x2B8);

void sbarFrmPrvSetActiveFormPINAttributes(uint32_t whichToAffect, bool useCustom /* use force open */)
{
	struct PalmForm *frm = (struct PalmForm*)FrmGetActiveForm();
	struct Globals *g = globalsGet();
	
	if (g->fakeEnabled)
		return;
	
	if (!frm)
		return;
	
	if (whichToAffect & 1)	//dia trigger state
		frm->diaPolicyAttr.inputTrigger = sbarPINGetInputTriggerState();
	if (whichToAffect & 2)	//dia status state
		frm->diaPolicyAttr.inputAreaState = useCustom ? pinInputAreaUser : sbarPINGetInputAreaState();
	if (whichToAffect & 4) {	//status bar state
		
		uint32_t attr;
		
		sbarStatGetAttribute(statAttrBarVisible, &attr, true);
		frm->diaPolicyAttr.statusState = attr;
	}
}

void sbarFrmPrvRedrawDisplay(const struct RectangleType *area)
{
	SysFatalAlert("FrmPrvRedrawDisplay() is not implemented!\n");
}

uint16_t sbarFrmGetDIAPolicyAttr(struct PalmForm* form)
{
	return form->diaPolicyAttr.frmDIAPolicy ? frmDIAPolicyCustom : frmDIAPolicyStayOpen;
}

Err sbarFrmSetDIAPolicyAttr(struct PalmForm* form, uint16_t policy)
{
	if (policy != frmDIAPolicyCustom && policy != frmDIAPolicyStayOpen)
		return 0x5001;
	
	form->diaPolicyAttr.frmDIAPolicy = (policy == frmDIAPolicyCustom);
	
	return errNone;
}

uint16_t sbarPINGetInputAreaState(void)
{
	struct Globals *g = globalsGet();
	
	if (g->fakeEnabled)
		return pinInputAreaNone;
	
	switch (g->diaState) {
		case SBAR_STATE_UP:
			return pinInputAreaOpen;
		case SBAR_STATE_JUST_SBAR:
		case SBAR_STATE_FULLSCREEN:
			return pinInputAreaClosed;
		default:
			SysFatalAlert("Unknown dia state\n");
			return 0;
	}	
}

uint16_t sbarPINGetInputTriggerState(void)
{
	struct Globals *g = globalsGet();
	
	if (g->fakeEnabled)
		return pinInputTriggerNone;
	
	return g->diaTriggerEn ? pinInputTriggerEnabled : pinInputTriggerDisabled;
}

static Err sbarPrvPINSetInputAreaState(uint16_t state, bool saveAsUserSelection)
{
	struct Globals *g = globalsGet();
	
	if (state != pinInputAreaOpen && state != pinInputAreaClosed && state != pinInputAreaUser)
		return 0x5001;
	
	if (g->fakeEnabled)
		return pinErrNoSoftInputArea;
	
	if (saveAsUserSelection && state != pinInputAreaUser)
		g->userDiaState = state;
	
	if (state == pinInputAreaUser)	//sort out what user last had
		state = g->userDiaState;
	
	if (state == pinInputAreaOpen) {
		if (g->diaState != SBAR_STATE_UP)
			sbarPrvSetState(SBAR_STATE_UP, false);
	}
	else {
		if (g->diaState != SBAR_STATE_JUST_SBAR && g->diaState != SBAR_STATE_FULLSCREEN)
			sbarPrvSetState(SBAR_STATE_JUST_SBAR, false);
	}
	
	return errNone;
}

Err sbarPINSetInputAreaState(uint16_t state)
{
	return sbarPrvPINSetInputAreaState(state, true);
}

static Err sbarPrvPINSetInputTriggerState(uint16_t state, bool saveAsUserSelection)
{
	struct Globals *g = globalsGet();
	
	if (state != pinInputTriggerEnabled && state != pinInputTriggerDisabled)
		return 0x5001;
	
	if (g->fakeEnabled)
		return pinErrNoSoftInputArea;
	
	if (saveAsUserSelection)
		g->userDiaTriggerState = state;
	
	if (state == pinInputTriggerEnabled) {
		if (!g->diaTriggerEn) {
			g->diaTriggerEn = true;
			sbarPrvUpdatePinsButtonState();
		}
	}
	else {
		if (g->diaTriggerEn) {
			g->diaTriggerEn = false;
			sbarPrvUpdatePinsButtonState();
		}
	}	
	
	return errNone;
}

Err sbarPINSetInputTriggerState(uint16_t state)
{
	return sbarPrvPINSetInputTriggerState(state, true);
}

Err sbarWinSetConstraintsSize(WinHandle winHandle, Coord minH, Coord prefH, Coord maxH, Coord minW, Coord prefW, Coord maxW)
{
	struct Globals *g = globalsGet();
	
	if (g->fakeEnabled)
		return pinErrNoSoftInputArea;
	
	g->maxAppHeight = maxH;
	
	return errNone;
}

Err sbarStatGetAttribute(uint16_t selector, uint32_t *dataP, bool fromArm)
{
	struct Globals *g = globalsGet();
	
	switch (selector) {
		case statAttrBarVisible:
			*dataP = (g->fakeEnabled || g->diaState != SBAR_STATE_FULLSCREEN) ? 1 : 0;
			break;
		case statAttrDimension:
			if (g->fakeEnabled) {
				
				*dataP = 0;
			}
			else if (fromArm) {
				((uint16_t*)dataP)[0] = WinScaleCoord(g->sbarAndDia.extent.x, false);
				((uint16_t*)dataP)[1] = WinScaleCoord(g->sbarAndDia.extent.y, false);
			}
			else {		//will be byteswapped but that swaps words too, so we swap them too
				((uint16_t*)dataP)[1] = WinScaleCoord(g->sbarAndDia.extent.x, false);
				((uint16_t*)dataP)[0] = WinScaleCoord(g->sbarAndDia.extent.y, false);
			}
			break;
		default:
			return sysErrParamErr;
	}
	
	return errNone;
}

static Err sbarPrvShowHide(bool show, bool saveAsUserSelection)
{
	struct Globals *g = globalsGet();
	
	if (g->fakeEnabled)
		return sysErrNotAllowed;
	
	if (show && (g->diaState == SBAR_STATE_UP || g->diaState == SBAR_STATE_JUST_SBAR))
		return errNone;
	if (!show && g->diaState == SBAR_STATE_FULLSCREEN)
		return errNone;
		
	return sbarPrvSetState(show ? SBAR_STATE_JUST_SBAR : SBAR_STATE_FULLSCREEN, false) ? errNone : 0x5001;
}

Err sbarStatShow(void)
{
	return sbarPrvShowHide(true, true);
}

Err sbarStatHide(void)
{
	return sbarPrvShowHide(false, true);
}

uint16_t sbarSysGetOrientation(void)
{
	return sysOrientationPortrait;
}

static Err sbarPrvSysSetOrientation(uint16_t orientation, bool saveAsUserSelection)
{
	struct Globals *g = globalsGet();
	
	if (orientation != sysOrientationUser && orientation != sysOrientationPortrait && orientation != sysOrientationLandscape && orientation != sysOrientationReversePortrait && orientation != sysOrientationReverseLandscape)
		return 0x5001;
	
	if (g->fakeEnabled)
		return sysErrNotAllowed;
	
	if (saveAsUserSelection && orientation != sysOrientationUser)
		g->userRotState = orientation;
	
	return sysErrNotAllowed;
}

Err sbarSysSetOrientation(uint16_t orientation)
{
	return sbarPrvSysSetOrientation(orientation, true);
}

uint16_t sbarSysGetOrientationTriggerState(void)
{
	return sysOrientationTriggerDisabled;
}

static Err sbarPrvSysSetOrientationTriggerState(uint16_t state, bool saveAsUserSelection)
{
	struct Globals *g = globalsGet();
	
	if (state != sysOrientationTriggerDisabled && state != sysOrientationTriggerEnabled)
		return 0x5001;
	
	if (g->fakeEnabled)
		return sysErrNotAllowed;
	
	if (saveAsUserSelection)
		g->userRotTriggerState = !!state;
	
	return sysErrNotAllowed;
}

Err sbarSysSetOrientationTriggerState(uint16_t state)
{
	return sbarPrvSysSetOrientationTriggerState(state, true);
}

